/* Copyright 1990 INMOS Limited */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef STD_C
#include <stdlib.h>
#include <stddef.h>
#endif
#include <toolkit.h>
#if TARG == TRANSPUTER
#include <host.h>
#endif
#include <arg.h>

#define PROG           "preocc"
#define VERSION        "4.800"
#define ENV_SEARCH_VAR "ISEARCH"
#define OUTPUT         "O"
#define DEF_VAR        "D"
#define DEBUG          "Z"
#define DEBUG_PATH     "ZI"

/*{{{  machine dependancies  */
#if     TARG == TRANSPUTER
char *switchars, *machine;
#endif

#if     TARG == PC
#define switchars      "/"
#define machine	       "PC"
#endif

#if     TARG == VAX
#define switchars      "/"
#define machine        "VAX"
#endif

#if     TARG == SUN
#define switchars      "-"
#define machine        "SUN"
#endif
/*}}}*/


#define MAXLINE        256
#define MAXDEPTH       256
#define MAX_PATH_SIZE  256
#define IOBUFSIZE      40960
#define   _ifndef      0
#define   _ifdef       1
#define   _endif       2
#define   _else        3

PRIVATE char *patterns[] = { "ifndef", "ifdef", "endif", "else", NULL };
PRIVATE char *defs[MAXDEPTH];
PRIVATE int line_number = 0;
PRIVATE int defcount = 0;
PRIVATE char *outfile = "\0";
PRIVATE char *infile = "\0";
PRIVATE char *search_path = "\0";
PRIVATE int debug;

/*{{{  arg_control fn_err (str)  */
arg_control fn_err (str)
char *str;
{
  error (ERR_SERIOUS, PROG, "","parsing command line \"%s\"", str);
  return (arg_terminate);
}
/*}}}*/
/*{{{  arg_control fn_help (str)  */
arg_control fn_help (str)
char *str;
{
  str = str;
  printf ("%s : Simple pre-processor\n", PROG);
  printf ("%s Version %s, %s\n",machine, VERSION, __DATE__);
  printf ("(c) Copyright Inmos Limited.\n\n");
  printf ("Usage: %s <filename> {%coption}\n\nOptions:\n\n", PROG, switchars[0]);
  printf ("   %s{filename}\tspecify an output file.\n", OUTPUT);
  printf ("   %s{env var}\tspecify search path.\n", DEBUG_PATH);
  printf ("   %s{define}\tspecify define variable.\n", DEF_VAR);
  printf ("   %s\t\tdebug mode.\n", DEBUG);
  exit (EXIT_SUCCESS);
  return (arg_terminate);
}
/*}}}*/
/*{{{  arg_control fn_output (str)  */
arg_control fn_output (str)
char *str;
{
  outfile = str;
  return (arg_continue);
}
/*}}}*/
/*{{{  arg_control fn_input (str)  */
arg_control fn_input (str)
char *str;
{
  if (*infile != '\0') error (ERR_SERIOUS, PROG, "",  "single input file only %s", str);
  infile = str;
  return (arg_continue);
}
/*}}}*/
/*{{{  arg_control fn_debug_path (str)  */
arg_control fn_debug_path (str)
char *str;
{
  search_path = str;
  return (arg_continue);
}
/*}}}*/
/*{{{  arg_control fn_debug (str)  */
arg_control fn_debug (str)
char *str;
{
  str = str;
  printf ("DEBUG MODE ENGAGED\n");
  debug = TRUE;
  return (arg_continue);
}
/*}}}*/
/*{{{  arg_control fn_def_var (str)  */
arg_control fn_def_var (str)
char *str;
{
  defs[defcount++] = str;
  return (arg_continue);
}
/*}}}*/
/*{{{  arg_control fn_end (str)  */
arg_control fn_end (str)
char *str;
{
  str = str;
  if (infile == "\0") error (ERR_SERIOUS, PROG, "", "must specify input file");
  defs[defcount] = NULL;
  return (arg_continue);
}
/*}}}*/

PRIVATE const arg_descriptor argd [] = 
{
  { OUTPUT,        arg_operand,  fn_output },
  { DEBUG_PATH,    arg_operand,  fn_debug_path },
  { DEBUG,         arg_single,   fn_debug },
  { DEF_VAR,       arg_operand,  fn_def_var },
  { "",            arg_token,    fn_input },
  { "",            arg_help,     fn_help },
  { "",            arg_error,    fn_err },
  { "",            arg_end,      fn_end }
};

/*{{{  PRIVATE char *findhash (line)  */
PRIVATE char *findhash (line)
char *line;
{
  while (isspace (*line)) line++;
  if (*line == '#') return (line);
  else return (NULL);
}
/*}}}*/
/*{{{  PRIVATE int search (infs, outfs, copy, arg)  */
PRIVATE int search (infs, outfs, copy, arg)
FILE *infs, *outfs;
int copy;
char *arg;
{
  char line[MAXLINE];
  char *str, *tmp, *test;
  int i, stop, res;
  /* keep copying (or not) lines until #string matches or eof */
  stop = FALSE;
  while (!stop)
  {
    /*{{{  find #  */
    test = fgets (line, MAXLINE, infs);
    line_number ++;
    if (test != NULL) str = findhash (line);
    while ((str == NULL) && (test != NULL))
    {
      if (copy) fputs (line, outfs);
      test = fgets (line, MAXLINE, infs);
      line_number ++;
      if (test != NULL) str = findhash (line);
    }
    /*}}}*/
    /*{{{  test #string  */
    if (test == NULL) /* end of file, stop */
    {
      res = EOF;
      stop = TRUE;
    }
    else /* otherwise test string */
    {
      str++;
      i = 0;
      while ((patterns[i] != NULL) && !str_semicmp (patterns[i], str)) i++;
      if (patterns[i] != NULL) /* found match, stop */
      {
        if (debug) fputs (line, stdout);
        stop = TRUE;
        res = i;
        str = &str[strlen (patterns[i])];
        while (isspace (*str)) str++;
        tmp = str;
        while (!(isspace (*str) || (*str == '\0'))) str++;
        *str = '\0';
        strcpy (arg, tmp);
      }
      else if (copy) fputs (line, outfs); /* no match, treat as ordinary line */
    }
    /*}}}*/
  }
  return (res);
}
/*}}}*/
/*{{{  PRIVATE void process (infs, outfs, defs)  */
PRIVATE void process (infs, outfs, defs)
FILE *infs, *outfs;
char *defs[];
{
  int type, level, stack[MAXDEPTH], i;
  char arg[MAXLINE];
  level = 0;
  stack[level] = TRUE;
  if (debug) printf ("level 0 copy on\n");
  type = search (infs, outfs, stack[level], arg);
  while (type != EOF)
  {
    switch (type)
    {
      /*{{{  case _ifdef:  */
      case _ifdef:
        if (++level == MAXDEPTH)
          error (ERR_FATAL, PROG, "", "conditionals exeeded maximum nested depth of %d\n", MAXDEPTH);
        i = 0;
        if (stack[level - 1])
        {
          while ((defs[i] != NULL) && (strcmp (defs[i], arg) != 0)) i++;
          if (defs[i] == NULL) stack[level] = FALSE;
          else stack[level] = TRUE;
        }
        else stack[level] = FALSE;
        break;
      /*}}}*/
      /*{{{  case _ifndef:  */
      case _ifndef:
        if (++level == MAXDEPTH)
        {
          error (ERR_FATAL, PROG, "", "conditionals exeeded maximum nested depth of %d\n", MAXDEPTH);
          exit (EXIT_FAILURE);
        }
        i = 0;
        if (stack[level - 1])
        {
          while ((defs[i] != NULL) && (strcmp (defs[i], arg) != 0)) i++;
          if (defs[i] == NULL) stack[level] = TRUE;
          else stack[level] = FALSE;
        }
        else stack[level] = FALSE;
        break;
      /*}}}*/
      /*{{{  case _endif:  */
      case _endif:
        if (--level < 0)
        {
          error (ERR_SERIOUS, PROG, "", "unexpected #endif on line %d\n", line_number);
          exit (EXIT_FAILURE);
        }
        break;
      /*}}}*/
      /*{{{  case _else:  */
      case _else:
        if (level == 0)
        {
          error (ERR_SERIOUS, PROG, "", "unexpected #else on line %d\n", line_number);
          exit (EXIT_FAILURE);
        }
        if (stack[level - 1]) stack[level] = !stack[level];
        break;
      /*}}}*/
    }
    if (debug)
    {
      if (stack[level]) printf ("level %d copy on\n", level);
      else printf ("level %d copy off\n", level);
    }
    type = search (infs, outfs, stack[level], arg);
  }
  if (level != 0)
  {
    error (ERR_SERIOUS, PROG, "", "unexpected end of file within conditional block\n");
    exit (EXIT_FAILURE);
  }
}
/*}}}*/
/*{{{  PRIVATE void determine_environment ()  (on transputer only)  */
#if TARG == TRANSPUTER
PRIVATE void determine_environment ()
{
  int host, os, board;
  host_info (&host, &os, &board);
  /* only worrying about host at the moment */
  switch (host)
  {
    case _IMS_HOST_PC:     machine = "TRANSPUTER (PC)"; break;
    case _IMS_HOST_NEC:    machine = "TRANSPUTER (NEC)"; break;
    case _IMS_HOST_VAX:    machine = "TRANSPUTER (VAX)"; break;
    case _IMS_HOST_SUN3:   machine = "TRANSPUTER (SUN3)"; break;
    case _IMS_HOST_SUN4:   machine = "TRANSPUTER (SUN4)"; break;
    default:               machine = "TRANSPUTER (unknown host)"; break;
  }
  switch (os)
  {
    case _IMS_OS_DOS:      switchars = "/"; break;
    case _IMS_OS_HELIOS:   switchars = "-"; break;
    case _IMS_OS_VMS:      switchars = "/"; break;
    case _IMS_OS_SUNOS:    switchars = "-"; break;
    default:               switchars = "-/"; break; /* better than nothing */
  }
}
#endif
/*}}}*/

/*{{{  PUBLIC int main(argc, argv)  */
PUBLIC int main(argc, argv)
int argc;
char const *argv[];
{
  FILE *infs, *outfs;
  char full_name[MAX_PATH_SIZE];
  int i;
#if TARG == TRANSPUTER
  determine_environment ();
#endif

  if (arg_parse (argc -1 , &(argv[1]), argd) != arg_parse_ok)
    error (ERR_FATAL, PROG, "", "parsing command line");
  infs = popen (infile, search_path, full_name, "r");
  if (infs == NULL) error (ERR_SERIOUS, PROG, "", "failed to find %s", infile);
  if (*search_path == '\0') search_path = ENV_SEARCH_VAR;
  if (*outfile == '\0') outfs = stdout;
  else
  {
    outfs = fopen (outfile, "w");
    if (outfs == NULL) error (ERR_SERIOUS, PROG, "", "could not open %s for output", outfile);
    i = setvbuf (outfs, NULL, _IOFBF, IOBUFSIZE);
    if (i != 0) error (ERR_FATAL, PROG, "", "Buffer broken, please report");
  }
  i = setvbuf (infs, NULL, _IOFBF, IOBUFSIZE);
  if (i != 0) error (ERR_FATAL, PROG, "", "Buffer broken, please report");
  if (debug)
  {
    printf ("defs:");
    for (i = 0; defs[i] != NULL; i++) printf (" %s", defs[i]);
    printf ("\n");
  }
  process (infs, outfs, defs);
  fclose (infs);
  if (outfs != stdout) fclose (outfs);
  return (EXIT_SUCCESS);
}
/*}}}*/
