#define DEBUG
/*****************************************************************************
 *
 *  use3 - builds up lists of free variables for procs / functions
 *         alias checks if required
 *         calls usage checker in use2 if required
 *
 *****************************************************************************/
/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "useerror.h"
# include "lexconst.h"
# include "usehdr.h"
# include "usedef.h"
# include "lex1def.h"
# include "use1def.h"
# include "use2def.h"
# include "syndef.h"
# include "chkdef.h"
/*}}}*/

/*{{{  DEBUG_MSG*/
#undef DEBUG_MSG
#ifdef DEBUG
  #define DEBUG_MSG(X) { if (debuguse) printf X; }
#else
  #define DEBUG_MSG(X)
#endif
/*}}}*/

/*{{{  PRIVATE variables*/
PRIVATE abbrevlist *inabbrevs = NULL;
PRIVATE abbrevlist *outabbrevs = NULL;
PRIVATE abbrevlist *varabbrevs = NULL;
PRIVATE abbrevlist *valabbrevs = NULL;
PRIVATE int in_abbrev = FALSE;
PRIVATE SOURCEPOSN lastloc = NOPOSN;
PRIVATE treenode *constexp_one;

typedef struct overlap_s
  {
    treenode **o_exp;
    treenode *o_check;
    struct overlap_s *o_next;
  } overlap_t;
PRIVATE overlap_t *overlapchain;
/*}}}*/

/*{{{  FORWARD*/
FORWARD PRIVATE void aliascheckexp ( treenode *n , int use_mode );
FORWARD PRIVATE void aliascheck ( treenode *n );
/*}}}*/

/*{{{  PRIVATE void insertoverlaps (void)*/
/*****************************************************************************
 *
 *  insertoverlaps inserts into the tree all the overlap checks currently
 *                 saved up.
 *
 *****************************************************************************/
PRIVATE void insertoverlaps ( void )
{
  while (overlapchain != NULL)
    {
      treenode *checkexp     = overlapchain->o_check;
      treenode **oldexp      = overlapchain->o_exp;
      overlap_t *next = overlapchain->o_next;
      *oldexp = newdopnode(S_EVAL, LocnOf(checkexp), checkexp, *oldexp, S_INT);
      memfree(overlapchain);
      overlapchain = next;
    }
}
/*}}}*/
/*{{{  PRIVATE treenode *buildsubscripts (treenode *tptr)*/
/*****************************************************************************
 *
 *  buildsubscripts generates a subscripts tree to be inserted into an
 *                  abbrevnode, from the element tptr.
 *
 *****************************************************************************/
PRIVATE treenode *buildsubscripts ( treenode *tptr )
{
  SOURCEPOSN locn = LocnOf(tptr);
  treenode *e = NULL;
  while (TRUE)
    {
      switch (TagOf(tptr))
        {
          /*{{{  S_ARRAYSUB*/
          case S_ARRAYSUB:
            e = newlistnode(S_LIST, locn, skipevals(ASIndexOf(tptr)), e);
            tptr = ASBaseOf(tptr);
            break;
          /*}}}*/
          /*{{{  S_SEGMENT*/
          case S_SEGMENT:
            if (e != NULL)
              /*{{{  add into check for current subscript*/
              {
                treenode *t;
                treenode **sptr;
              
                if (TagOf(ThisItem(e)) == S_FOR)
                  sptr = LeftOpAddr(ThisItem(e));
                else
                  sptr = ThisItemAddr(e);
              
                t = newdopnode(S_ADD, locn, *sptr, skipevals(SStartExpOf(tptr)), S_INT);
                *sptr = foldexp(t);
              }
              /*}}}*/
            else
              e = newlistnode(S_LIST, locn,
                    newdopnode(S_FOR, locn,
                      skipevals(SStartExpOf(tptr)), skipevals(SLengthExpOf(tptr)), 0),
                    NULL);
            tptr = SNameOf(tptr);
            break;
          /*}}}*/
          default:
            return e;
        }
    }
}
/*}}}*/
/*{{{  PRIVATE int subscripts_clash*/
/* This modified 1/11/90 for bug 1032.
   It used to complain that a[i][0] and a[j][0] clashed.
   Modified to only complain if ALL of the subscripts definitely clash.
*/
#define CLASH_NONE   0x0
#define CLASH_ALWAYS 0x1
#define CLASH_MAYBE  0x2
#define CLASH_NEVER  0x4
PRIVATE int subscripts_clash (treenode *lhs, treenode *rhs)
{
  /* This is used for overlap checking.
     Some 'checkable' stuff gets through the rest of the aliaschecker.
     Eg comparing a[0][i] with a[0] (or a[0][i] with a[1]).
     This takes out all such examples.
  */

  /* updated 9/10/90 by CON (bug 1010) to cope with:
     INT x IS a[1] :
     []INT y IS [a FROM 3 FOR i] :
  */
  treenode *base1, *count1, *base2, *count2;
  int test_a_always_true = FALSE, test_b_always_true = FALSE;

  /* Set up so that a[i] looks like [a FROM i FOR 1] */
  if (TagOf(lhs) == S_FOR)
    {  base1 = LeftOpOf(lhs); count1 = RightOpOf(lhs); }
  else
    {  base1 = lhs; count1 = constexp_one; }
  if (TagOf(rhs) == S_FOR)
    {  base2 = LeftOpOf(rhs); count2 = RightOpOf(rhs); }
  else
    {  base2 = rhs; count2 = constexp_one; }

  if (isconst(base1) && isconst(count1) && isconst(base2))
    {
      INT32 b1 = LoValOf(base1), c1 = LoValOf(count1);
      INT32 b2 = LoValOf(base2);
      if ((b1 + c1) <= b2)
        return CLASH_NEVER;
      test_a_always_true = TRUE;
    }
  if (isconst(base2) && isconst(count2) && isconst(base1))
    {
      INT32 b2 = LoValOf(base2), c2 = LoValOf(count2);
      INT32 b1 = LoValOf(base1);
      if ((b2 + c2) <= b1)
        return CLASH_NEVER;
      test_b_always_true = TRUE;
    }
  return (test_a_always_true && test_b_always_true) ? CLASH_ALWAYS : CLASH_MAYBE;
}
/*{{{  PRIVATE void overlaperror*/
PRIVATE void overlaperror(treenode *n, int checking_vals)
{
  usereport(checking_vals ? USE_VAR_VAL_ABBREVIATED : USE_VAR_VAR_ABBREVIATED,
            lastloc, n);
}
/*}}}*/
/*{{{  PRIVATE void overlapcheck (n, abbrevs, start, end, element)*/
/*****************************************************************************
 *
 *  Check whether accesses to variable 'n' in the range 'start' to 'end'
 *  overlap any previous accesses in the list 'abbrevs'
 *  If they do then either generate the appropriate error message if both
 *  access ranges are known or insert code into the tree to generate
 *  a runtime check.
 *
 *****************************************************************************/
PRIVATE void overlapcheck ( treenode *n , abbrevlist *abbrevs , INT32 start , INT32 end , treenode *element )
{
  int checking_vals = (abbrevs == valabbrevs);
  while (abbrevs != NULL)
    {
      if (ALNameOf(abbrevs) == n)
        {
          INT32 alfirst = ALFirstOf(abbrevs), allast = ALLastOf(abbrevs);
          int check_subs = (TagOf(element) == S_ARRAYSUB || TagOf(element) == S_SEGMENT);
          if (check_subs &&
              ((ALFirstOf(abbrevs) < 0) || (start < 0)) ) /* non constant subscripts */
            /*{{{  get into this run-time overlap check business*/
            {
              SOURCEPOSN locn = LocnOf(element);
              treenode *lhs, *rhs, *left, *right;
              int canclash = ((alfirst == 0) && (allast == MAXINDEX)) ?
                             CLASH_ALWAYS : /* CLASH_MAYBE */ CLASH_NONE; /* bug 1032 */
              DEBUG_MSG(("overlapcheck: at line %d against line %d, of %s\n",
                       FileLineOf(lastloc), FileLineOf(ALLocOf(abbrevs)), usetagstring(element)));
              lhs = buildsubscripts(element);
              rhs = ALSubscriptsOf(abbrevs);
              left = lhs; right = rhs;
              while (!EndOfList(left) && !EndOfList(right))
                /* && (canclash == CLASH_MAYBE) */ /* removed for bug 1032 */
                {
                  canclash |= subscripts_clash(ThisItem(left), ThisItem(right));
                  left  = NextItem(left);
                  right = NextItem(right);
                }
            #ifdef DEBUG
              DEBUG_MSG(("overlapcheck: canclash: "));
              if ((canclash & CLASH_NEVER ) != 0) DEBUG_MSG(("CLASH_NEVER "));
              if ((canclash & CLASH_MAYBE ) != 0) DEBUG_MSG(("CLASH_MAYBE "));
              if ((canclash & CLASH_ALWAYS) != 0) DEBUG_MSG(("CLASH_ALWAYS "));
              DEBUG_MSG(("\n"));
            #endif
              if ((canclash & CLASH_NEVER) != 0)
                ; /* skip */
              else if (((canclash & CLASH_ALWAYS) != 0) &&
                       ((canclash & CLASH_MAYBE ) == 0))
                /* We have found, eg, a clash between an abbreviation of
                   array[0], and array[0][i], which would otherwise not be found */
                {
                  overlaperror(n, checking_vals);
                  return;
                }
              else if (((canclash & CLASH_MAYBE) != 0) && RANGECHECKING
                       && !warn_on_usage_error) /* added 13/12/90 */
                /* RANGECHECKING can either be turned off, here, or it could be moved
                   into 'buildoverlapcheck' in bind1, if we wanted to keep it in the backend */
                {
                  /* element is the one we are currently working on,
                     abbrevs is the one we are overlap checking against */
                  treenode **element_base_ptr;
                  treenode *ocheck;
                  overlap_t *overlap =
                    (overlap_t *)memalloc(sizeof(overlap_t));
                  ocheck = newdopnode(S_OVERLAPCHECK, locn,
                                      copytree(lhs), copytree(rhs), S_INT);
                  element_base_ptr = (TagOf(element) == S_ARRAYSUB) ?
                                      ASIndexAddr(element) : SStartExpAddr(element);
                  overlap->o_exp   = element_base_ptr;
                  overlap->o_check = ocheck;
                  overlap->o_next  = overlapchain;
                  overlapchain     = overlap;
                }
            }
            /*}}}*/
          else if (!check_subs || ((start <= allast) && (end >= alfirst)))
            {
              overlaperror(n, checking_vals);
              return;
            }
        }
      abbrevs = ALNextOf(abbrevs);
    }
}
/*}}}*/
/*{{{  PRIVATE varlist *record_parameter_info (n, freevars)*/
/******************************************************************************
 *
 *  searches 'freevars' variable list to references to parameters of
 *  PROC / FUNCTION definition pointed to by 'n'.  References are removed
 *  from the list and linked to formal parameter name node
 *
 *****************************************************************************/
PRIVATE varlist *record_parameter_info ( treenode *n , varlist *freevars )
{
  treenode *params;       /* points to list of parameters */
  treenode *param_name;   /* points to current formal param name node */
  varlist *vlist, **prev;
  if (TagOf(n) == S_PROCDEF)
    params = NTypeOf(DNameOf(n));
  else
    params = FnParamsOf(NTypeOf(DNameOf(n)));
  while (params != NULL)
    {
      param_name = ThisItem(params);
      prev = &freevars;
      vlist = *prev;
      while ((vlist != NULL) && (VLNameOf(vlist) != param_name))
        {
          prev = &VLNextOf(vlist);
          vlist = *prev;
        }
      SetNParamUse(param_name, vlist);
#ifdef DEBUG
      if (debuguse)
        {
          int in  = paraminputon (param_name);
          int out = paramoutputon(param_name);
          if (in || out)
            printf("param %s %s input on, %s output on\n",
                    WNameOf(NNameOf(param_name)),
                    in ? "is":"is not", out ? "is":"is not");
        }
#endif
      if (vlist != NULL)
        {
          *prev = VLNextOf(vlist);
          SetVLNext(vlist, NULL);
        }
      params = NextItem(params);
    }
  return(freevars);
}
/*}}}*/
/*{{{  PRIVATE void check_channel_direction_usage (vlist, loc)*/
/*****************************************************************************
 *
 *  Check that channel parameters and free channels of a procedure are only
 *  used for either input or output.
 *
 *****************************************************************************/
PRIVATE void check_channel_direction_usage ( varlist *vlist , SOURCEPOSN loc )
{
  for (; vlist != NULL; vlist = VLNextOf(vlist))
    if ((VLInputOf(vlist) != NULL) && (VLOutputOf(vlist) != NULL))
      {
        treenode *nptr = VLNameOf(vlist);
        treenode *type = NTypeOf(nptr);
        while (TagOf(type) == S_ARRAY)
          type = ARTypeOf(type);
        if (TagOf(type) == S_PORT)
          ; /* skip */
#if 1 /* SHARED_VARS */
        /* EXPERIMENTAL way to prevent usage checking of certain variables
           CON 5/2/91 */
        else if (NVSharedOf(nptr))
          ; /* skip */
#endif
        else if (TagOf(nptr) == N_PARAM)
          usereport(USE_BAD_CHAN_PARAM, loc, nptr);
        else
          usereport(USE_BAD_FREE_CHAN, loc, nptr);
      }
}
/*}}}*/
/*{{{  PRIVATE void aliascheckuse_single*/
PRIVATE void aliascheckuse_single(abbrevlist *abbrevs, treenode *n, INT32 start, INT32 end, treenode *element, int msg)
{
  if (inabbrevlist(n, abbrevs, start, end))
    {
      if (in_abbrev)
        overlapcheck(n, abbrevs, start, end, element);
      else
        usereport(msg, lastloc, n);
    }
}
/*}}}*/
/*{{{  PRIVATE void aliascheckuse (use_mode, n, start, end, element)*/
/*****************************************************************************
 *
 *  Check that access to variable 'n' in the range 'start' to 'end' by
 *  method 'use_mode' does not break any aliasing rules.  If it does
 *  then give an error message and if it may do then insert a runtime
 *  check into the tree
 *
 *****************************************************************************/
PRIVATE void aliascheckuse ( int use_mode , treenode *n , INT32 start , INT32 end , treenode *element )
{
  switch(use_mode)
  {
    case EXP_READ:
      aliascheckuse_single(varabbrevs, n, start, end, element, USE_VAR_VAR_ABBREVIATED);
      break;
    case EXP_WRITTEN:
      aliascheckuse_single(varabbrevs, n, start, end, element, USE_VAR_VAR_ABBREVIATED);
      aliascheckuse_single(valabbrevs, n, start, end, element, USE_VAR_VAL_ABBREVIATED);
      break;
    case CHAN_INPUT:
      if (basetype(NTypeOf(n)) != S_TIMER)
        aliascheckuse_single(inabbrevs, n, start, end, element, USE_VAR_VAR_ABBREVIATED);
      break;
    case CHAN_OUTPUT:
      aliascheckuse_single(outabbrevs, n, start, end, element, USE_VAR_VAR_ABBREVIATED);
      break;
  }
}
/*}}}*/
/*{{{  PRIVATE void aliascheckinstance_single*/
PRIVATE void aliascheckinstance_single(subscrlist *subs, treenode *nptr, abbrevlist *abbrevs, int msg)
{
  while (subs != NULL)
    {
      if (inabbrevlist(nptr, abbrevs, SLFirstOf(subs), SLLastOf(subs)))
        {
          usereport(msg, lastloc, nptr);
          /*break*/ return; /* (same thing) */
        }
      subs = SLNextOf(subs);
    }
}
/*}}}*/
/*{{{  PRIVATE void aliascheckinstance (freevars)*/
/*****************************************************************************
 *
 * alias check the free variable usage in a PROC/FUNCTION instance
 *
 *****************************************************************************/
PRIVATE void aliascheckinstance ( varlist *freevars, int var_errmsg, int val_errmsg )
{
  while (freevars != NULL)
    {
      treenode *nptr = VLNameOf(freevars);
      int type = basetype(NTypeOf(nptr));
      if ((type == S_CHAN) || (type == S_PORT))
        {
          aliascheckinstance_single(VLInputOf(freevars), nptr, inabbrevs, var_errmsg);
          aliascheckinstance_single(VLOutputOf(freevars), nptr, outabbrevs, var_errmsg);
        }
      else if (type != S_TIMER)
        {
          aliascheckinstance_single(VLReadOf(freevars), nptr, varabbrevs, var_errmsg);

          aliascheckinstance_single(VLWrittenOf(freevars), nptr, varabbrevs, var_errmsg);
          aliascheckinstance_single(VLWrittenOf(freevars), nptr, valabbrevs, val_errmsg);
        }
      freevars = VLNextOf(freevars);
    }
}
/*}}}*/
/*{{{  PRIVATE abbrevlist *varsinabbrevexp(n, alist)*/
PRIVATE abbrevlist *varsin_alist;
FORWARD PRIVATE abbrevlist *varsinabbrevexp PARMS((treenode *));
/*{{{  PRIVATEPARAM abbrevlist *do_varsinabbrevexp(n)*/
/*****************************************************************************
 *
 *  Add to variable list 'varsin_alist' any variables used in the expression 'n'
 *
 *****************************************************************************/
PRIVATEPARAM int do_varsinabbrevexp ( treenode *n )
{
  DEBUG_MSG(("do_varsinabbrevexp: %s\n", usetagstring(n)));
  switch(TagOf(n))
    {
      /*{{{  subscripting / slicing*/
      case S_ARRAYSUB: case S_SEGMENT:
        {
          treenode *init_n = n;
          int all_subs_constant = TRUE;
          INT32 start, end;
          /*{{{  search subscripts*/
          while ((TagOf(n) == S_ARRAYSUB) || (TagOf(n) == S_SEGMENT))
            {
              if (TagOf(n) == S_ARRAYSUB)
                /*{{{  subscript*/
                {
                  all_subs_constant = all_subs_constant &&
                                      (TagOf(ASIndexOf(n)) == S_CONSTEXP);
                  varsin_alist = varsinabbrevexp(ASIndexOf(n));
                  n = ASBaseOf(n);
                }
                /*}}}*/
              else
                /*{{{  segment*/
                {
                  all_subs_constant = all_subs_constant &&
                                      ((TagOf(SStartExpOf(n)) == S_CONSTEXP) &&
                                      (TagOf(SLengthExpOf(n)) == S_CONSTEXP));
                  varsin_alist = varsinabbrevexp(SStartExpOf(n));
                  varsin_alist = varsinabbrevexp(SLengthExpOf(n));
                  n = SNameOf(n);
                }
                /*}}}*/
            }
          /*}}}*/
          switch (TagOf(n))
            {
              default:
                /*{{{  must be a name*/
                if (all_subs_constant && (elementsin(n) >= 0))
                  {
                    subscripts_accessed(init_n, &start, &end, FALSE);
                    if (start >= 0)
                      varsin_alist = new_abbrev_node(n, varsin_alist, start, end,
                                        lastloc, buildsubscripts(init_n));
                  }
                else
                  varsin_alist = new_abbrev_node(n, varsin_alist, -1, -1, lastloc,
                                    buildsubscripts(init_n));
                /*}}}*/
                break;
              case S_STRING: case S_CONSTCONSTRUCTOR:
                break;
              case S_CONSTRUCTOR:
              #ifdef ARRAYCONSTRUCTOR
              case S_ARRAYCONSTRUCTOR :
              #endif
                prewalktree(n, do_varsinabbrevexp);
                break;
            }
          return TRUE;
        }
      /*}}}*/
      /*{{{  name node*/
      case N_VALABBR: case N_ABBR: case N_VALRETYPE: case N_RETYPE:
      case N_DECL: case N_VALPARAM: case N_PARAM: case N_REPL:
        varsin_alist =
          new_abbrev_node(n, varsin_alist, ZERO32, MAXINDEX, lastloc, NULL);
        break;
      /*}}}*/
      /*{{{  specification / VALOF*/
      case S_VALOF: case S_ABBR: case S_RETYPE:
      case S_VALABBR: case S_VALRETYPE:
      case S_PROCDEF: case S_SFUNCDEF:
      case S_LFUNCDEF: case S_DECL: case S_TPROTDEF: case S_SPROTDEF:
        return TRUE;
      /*}}}*/
      /*{{{  SIZE*/
      case S_SIZE:
        /* Don't count the operand of a SIZE as being used */
        return TRUE;
      /*}}}*/
    }
  return FALSE;
}
/*}}}*/
/*{{{  PRIVATE abbrevlist *varsinabbrevexp(nptr)*/
/*****************************************************************************
 *
 *  Add to variable list 'alist' any variables used in the expression 'n'
 *
 *****************************************************************************/
PRIVATE abbrevlist *varsinabbrevexp ( treenode *n)
{
  abbrevlist *alist;
  abbrevlist *old_alist = varsin_alist;
  prewalktree(n, do_varsinabbrevexp);
  alist = varsin_alist;
  varsin_alist = old_alist;
  return alist;
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void val_abbreviate(n)*/
/*****************************************************************************
 *
 *  Add all variables in the expression 'n' to the val-abbreviated list
 *
 *****************************************************************************/
PRIVATE void val_abbreviate ( treenode *n )
{
  abbrevlist *vars = varsinabbrevexp(n);
  if (vars != NULL)
    {
      abbrevlist *vals = vars;
      while (ALNextOf(vals) != NULL)
        vals = ALNextOf(vals);
      SetALNext(vals, valabbrevs);
      valabbrevs = vars;
    }
}
/*}}}*/
/*{{{  PRIVATE void var_abbreviate(n)*/
/*****************************************************************************
 *
 *  Add the accessed range of the base variable in element 'n' to the
 *  var-abbreviated list and any variables in the subscripts or slice
 *  base and count expressions to the val-abbreviated list.
 *
 *****************************************************************************/
PRIVATE void var_abbreviate ( treenode *n, int use_mode )
{
  treenode *init_n = n;
  int all_subs_constant = TRUE;
  /*{{{  check subscripts*/
  while ((TagOf(n) == S_ARRAYSUB) || (TagOf(n) == S_SEGMENT))
    {
      if (TagOf(n) == S_ARRAYSUB)
        /*{{{  subscript*/
        {
          all_subs_constant = all_subs_constant &&
                              (TagOf(ASIndexOf(n)) == S_CONSTEXP);
          /* val_abbreviate(ASIndexOf(n)); */
          n = ASBaseOf(n);
        }
        /*}}}*/
      else
        /*{{{  segment*/
        {
          all_subs_constant = all_subs_constant &&
                              ((TagOf(SStartExpOf(n)) == S_CONSTEXP) &&
                              (TagOf(SLengthExpOf(n)) == S_CONSTEXP));
          /* val_abbreviate(SStartExpOf(n));
             val_abbreviate(SLengthExpOf(n)); */
          n = SNameOf(n);
        }
        /*}}}*/
    }
  /*}}}*/
  /*{{{  add to abbrev list*/
  {
    abbrevlist **abbrevs;
    switch(use_mode)
      {
        case CHAN_INPUT:  abbrevs = &inabbrevs;  break;
        case CHAN_OUTPUT: abbrevs = &outabbrevs; break;
        default:          abbrevs = &varabbrevs; break;
      }
    if (all_subs_constant && (elementsin(n) >= 0))
      {
        INT32 start, end;
        subscripts_accessed(init_n, &start, &end, FALSE);
        if (start >= 0)
          *abbrevs = new_abbrev_node(n, *abbrevs, start, end,
                                     lastloc, buildsubscripts(init_n));
      }
    else
      *abbrevs = new_abbrev_node(n, *abbrevs, -1, -1,
                                 lastloc, buildsubscripts(init_n));
  }
  /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void var_abbreviate_seg(n)*/
/*****************************************************************************
 *
 *  If 'n' is a segment, var_abbreviate it
 *
 *  We have to flag the case if so, so that we can insert an overlap check
 *  for the rhs (eg [a FROM i FOR j] := [a FROM k FOR l])
 *
 *****************************************************************************/
PRIVATE void var_abbreviate_seg ( treenode *n )
{
  if (TagOf(n) == S_SEGMENT)
    {
      var_abbreviate(n, EXP_WRITTEN);
      in_abbrev = TRUE; /* pretend we're inside an abbreviation for the rhs */
    }
}
/*}}}*/
/*{{{  PRIVATE void var_expression(n)*/
/*****************************************************************************
 *
 *  Add the accessed range of the base variable in each element of 'n' to the
 *  var-abbreviated list and any variables in the subscripts or slice
 *  base and count expressions to the val-abbreviated list.
 *  Call aliascheckexp before we add the elements to the list.
 *
 *****************************************************************************/
PRIVATE void var_expression ( treenode *n, int abbrev, int test1, int mode1, int test2, int mode2 )
{
  DEBUG_MSG(("var_expression: %s\n", usetagstring(n)));
  if (TagOf(n) == S_CONSTRUCTOR)
    {
      n = OpOf(n);
      while (!EndOfList(n))
        {
          var_expression(ThisItem(n), abbrev, test1, mode1, test2, mode2);
          n = NextItem(n);
        }
    }
  else
    {
      int saved_abbrev = in_abbrev;
      in_abbrev = abbrev;
      if (test1)
        aliascheckexp(n, mode1);
      if (test2)
        aliascheckexp(n, mode2);
      in_abbrev = saved_abbrev;
      if (test1)
        var_abbreviate(n, mode1);
      if (test2)
        var_abbreviate(n, mode2);
    }
}
/*}}}*/
/*{{{  PRIVATE varlist *freevarsinsc (n, scope, vlist)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  freevarsinsc takes a separately compiled routine body, 'n',
 *  a scope level, 'scope', and a list of
 *  free variables, 'vlist', and adds to 'vlist' any variables used in tree
 *  'n' of scope less than 'scope', which aren't already in vlist.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE varlist *freevarsinsc ( treenode *n , int scope , varlist *vlist )
{
  while (n != NULL)
    {
      DEBUG_MSG(("freevarsinsc: %s\n", usetagstring(n)));
      switch(TagOf(n))
        {
          /*{{{  SEQ PRIPAR*/
          case S_SEQ: case S_PRIPAR:
            n = CBodyOf(n);
            break;
          /*}}}*/
          /*{{{  list*/
          case S_LIST:
            vlist = freevarsinsc(ThisItem(n), scope, vlist);
            n = NextItem(n);
            break;
          /*}}}*/
          /*{{{  OUTPUT INPUT*/
          case S_OUTPUT:
            vlist = freevarsinexp(LHSOf(n), scope, vlist, CHAN_OUTPUT, FALSE);
            return(vlist);
          case S_INPUT:
            vlist = freevarsinexp(LHSOf(n), scope, vlist, CHAN_INPUT, FALSE);
            return(vlist);
          /*}}}*/
          default:
            badtag(LocnOf(n), TagOf(n), "freevarsinsc");
        }
    }
  return(vlist);
}
/*}}}*/
/*{{{  PRIVATE void aliascheckexp (n, use_mode)*/
/*****************************************************************************
 *
 *  Alias check the usage of any variables in expression tree 'n'
 *
 *****************************************************************************/
PRIVATE void aliascheckexp ( treenode *n , int use_mode )
{
  while (n != NULL)
    {
      DEBUG_MSG(("aliascheckexp: %s\n", usetagstring(n)));
      switch(TagOf(n))
        {
          /*{{{  expression*/
          /*{{{  monadics conversions*/
          case S_NEG: case S_BITNOT: case S_NOT:
          case S_UMINUS:
          case S_EXACT: case S_ROUND: case S_TRUNC:
            n = OpOf(n);
            break;
          
          case S_SIZE: case S_EVAL:
            return;
          /*}}}*/
          /*{{{  dyadics*/
          case S_AND: case S_OR:
          case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
          case S_BITAND: case S_BITOR: case S_XOR:
          case S_LSHIFT: case S_RSHIFT:
          case S_PLUS: case S_MINUS: case S_TIMES:
          case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
          case S_COLON2:
          /*case S_INTERVAL :*/
            lastloc = LocnOf(n);
            aliascheckexp(LeftOpOf(n), use_mode);
            lastloc = LocnOf(n);
            n = RightOpOf(n);
            break;
          /*}}}*/
          /*{{{  constants*/
          case S_CONSTEXP: case S_CONSTCONSTRUCTOR: case S_STRING:
            return;
          /*}}}*/
          /*{{{  FINSTANCE*/
          case S_FINSTANCE:
            {
              int save_in_abbrev;
              abbrevlist *save_val;
              lastloc = LocnOf(n);
              switch (TagOf(INameOf(n)))
                {
                  case N_SFUNCDEF: case N_LFUNCDEF: case N_INLINEFUNCDEF:
                    aliascheckinstance(NFreeVarsOf(INameOf(n)),
                      USE_VAR_FREEVAR, USE_VAL_FREEVAR);
                    break;
                  default:  /* no free variables in the others */
                    break;
                }
              /*{{{  save info*/
              save_in_abbrev = in_abbrev;
              save_val = valabbrevs;
              /*}}}*/
              /*{{{  check parameters*/
              /* N.B. all function parameters are VALs */
              {
                n = IParamListOf(n);
                in_abbrev = TRUE;
                while (n != NULL)
                  {
                    aliascheckexp(LHSOf(n), EXP_READ);
                    val_abbreviate(LHSOf(n));
                    n = RHSOf(n);
                  }
              }
              /*}}}*/
              /*{{{  restore info*/
              in_abbrev = save_in_abbrev;
              while (valabbrevs != save_val)
                valabbrevs = release_headof_abbrevlist(valabbrevs);
              /*}}}*/
              insertoverlaps();
              return;
            }
          /*}}}*/
          /*{{{  CONSTRUCTOR*/
          case S_CONSTRUCTOR:
            n = OpOf(n);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  element*/
          /*{{{  subscripting / slicing*/
          case S_ARRAYSUB: case S_SEGMENT:
            {
              treenode *init_n = n;
              int all_subs_constant = TRUE;
              INT32 start, end;
              lastloc = LocnOf(n);
              /*{{{  check subscripts*/
              {
                while ((TagOf(n) == S_ARRAYSUB) || (TagOf(n) == S_SEGMENT))
                  {
                    if (TagOf(n) == S_ARRAYSUB)
                      /*{{{  subscript*/
                      {
                        all_subs_constant = all_subs_constant &&
                                            (TagOf(ASIndexOf(n)) == S_CONSTEXP);
                        aliascheckexp(ASIndexOf(n), EXP_READ);
                        if (in_abbrev && (use_mode == EXP_WRITTEN))
                          val_abbreviate(ASIndexOf(n));
                        n = ASBaseOf(n);
                      }
                      /*}}}*/
                    else
                      /*{{{  segment*/
                      {
                        all_subs_constant = all_subs_constant &&
                                            ((TagOf(SStartExpOf(n)) == S_CONSTEXP) &&
                                            (TagOf(SLengthExpOf(n)) == S_CONSTEXP));
                        aliascheckexp(SStartExpOf(n), EXP_READ);
                        aliascheckexp(SLengthExpOf(n), EXP_READ);
                        if (in_abbrev && (use_mode == EXP_WRITTEN))
                          {
                            val_abbreviate(SStartExpOf(n));
                            val_abbreviate(SLengthExpOf(n));
                          }
                        n = SNameOf(n);
                      }
                      /*}}}*/
                  }
              }
              /*}}}*/
              switch(TagOf(n))
                {
                  default:
                     /*{{{  must be a name*/
                     if (all_subs_constant && checkalias && (elementsin(n) >= 0))
                       {
                         subscripts_accessed(init_n, &start, &end, FALSE);
                         if (start >= 0)
                           aliascheckuse(use_mode, n, start, end, init_n);
                       }
                     else
                       aliascheckuse(use_mode, n, -1, -1, init_n);
                     /*}}}*/
                     break;
                   case S_STRING: case S_CONSTCONSTRUCTOR:
                     break;
                   case S_CONSTRUCTOR:
                   /*case S_ARRAYCONSTRUCTOR :*/
                     aliascheckexp(n, use_mode);
                     break;
                 }
              return;
            }
          /*}}}*/
          /*{{{  name node*/
          case N_VALABBR: case N_ABBR:
          case N_VALRETYPE: case N_RETYPE:
          case N_DECL:
          case N_VALPARAM: case N_PARAM:
          case N_REPL:
            /* Whole variable is being used */
            aliascheckuse(use_mode, n, 0, MAXINDEX, n);
            return;
          /*}}}*/
          /*{{{  case tag def*/
          case N_TAGDEF:
            return;
          /*}}}*/
          /*}}}*/
          /*{{{  list*/
          case S_LIST:
            aliascheckexp(ThisItem(n), use_mode);
            n = NextItem(n);
            break;
          /*}}}*/
          /*{{{  else in case statement*/
          case S_ELSE:
            return;
          /*}}}*/
          /*{{{  specification / VALOF*/
          case S_ABBR: case S_RETYPE: case S_VALABBR: case S_VALRETYPE:
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_DECL: case S_TPROTDEF: case S_SPROTDEF: case S_VALOF:
            /* treat an in-line valof as a function call */
            {
              varlist *freevars;
              abbrevlist *save_valabbrevs, *save_varabbrevs;
              abbrevlist *save_inabbrevs, *save_outabbrevs;
              int save_in_abbrev;
              lastloc = LocnOf(n);
              /*{{{  save info*/
              save_in_abbrev = in_abbrev;
              save_valabbrevs = valabbrevs; save_varabbrevs = varabbrevs;
              save_inabbrevs = inabbrevs; save_outabbrevs = outabbrevs;
              valabbrevs = NULL; varabbrevs = NULL;
              inabbrevs = NULL; outabbrevs = NULL;
              in_abbrev = FALSE;
              /*}}}*/
              aliascheck(n);
              freevars = freevarsin(n, -1, NULL, FALSE);
              aliascheckinstance(freevars,
                      USE_VAR_VAR_ABBREVIATED, USE_VAR_VAL_ABBREVIATED);
              release_varlist(freevars);
              /*{{{  restore info*/
              while (valabbrevs != NULL)
                valabbrevs = release_headof_abbrevlist(valabbrevs);
              while (varabbrevs != NULL)
                varabbrevs = release_headof_abbrevlist(varabbrevs);
              while (inabbrevs != NULL)
                inabbrevs = release_headof_abbrevlist(inabbrevs);
              while (outabbrevs != NULL)
                outabbrevs = release_headof_abbrevlist(outabbrevs);
              valabbrevs = save_valabbrevs;
              varabbrevs = save_varabbrevs;
              inabbrevs = save_inabbrevs;
              outabbrevs = save_outabbrevs;
              in_abbrev = save_in_abbrev;
              /*}}}*/
              return;
            }
          /*}}}*/
          #ifdef CONFIG
          /*{{{  RECORDSUB*/
          case S_RECORDSUB:
            n = ASBaseOf(n); /* ignore the field name */
            break;
          /*}}}*/
          #endif
          #ifdef CONDEXP
          /*{{{  conditional expression*/
          case S_CONDEXP:
            {
              lastloc = LocnOf(n);
              aliascheckexp(CondExpGuardOf(n), use_mode);
              lastloc = LocnOf(n);
              aliascheckexp(CondExpTrueOf(n), use_mode);
              n = CondExpFalseOf(n);
            }
            break;
          /*}}}*/
          #endif
          #ifdef ARRAYCONSTRUCTOR
          /*{{{  arrayconstructor*/
          case S_ARRAYCONSTRUCTOR :
            {
              lastloc = LocnOf(n);
              aliascheckexp(ACStartExpOf(n), use_mode);
              lastloc = LocnOf(n);
              aliascheckexp(ACLengthExpOf(n), use_mode);
              n = ACValExpOf(n);
            }
            break;
          /*}}}*/
          #endif
          #if 0
          /*{{{  structconstructor*/
          case S_STRUCTCONSTRUCTOR :
            return;
          /*}}}*/
          #endif
          default:
            badtag(LocnOf(n), TagOf(n), "aliascheckexp");
        }
    }
  return;
}
/*}}}*/
/*{{{  PRIVATE void aliascheck (n)*/
/*****************************************************************************
 *
 * Perform alias check of variables and channels in routine 'n'
 *
 *****************************************************************************/
PRIVATE void aliascheck ( treenode *n )
{
  while (n != NULL)
    {
      DEBUG_MSG(("aliascheck: %s\n", usetagstring(n)));
      switch(TagOf(n))
        {
          /*{{{  process*/
          /*{{{  STOP SKIP END ELSE GUY ASM*/
          case S_STOP: case S_SKIP: case S_END: case S_ELSE: case S_GUY: case S_ASM:
            return;
          /*}}}*/
          /*{{{  SEQ IF PAR ALT PRIPAR PRIALT PLACEDPAR*/
          case S_SEQ: case S_IF: case S_PAR: case S_ALT:
          case S_PRIPAR: case S_PRIALT: case S_PLACEDPAR: case S_DO:
            n = CBodyOf(n);
            break;
          /*}}}*/
          /*{{{  REPLSEQ REPLIF REPLPAR REPLALT PRIREPLPAR PRIREPLALT PLACEDREPLPAR*/
          case S_REPLSEQ: case S_REPLIF: case S_REPLPAR: case S_REPLALT:
          case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
            lastloc = LocnOf(n);
            aliascheckexp(ReplCStartExpOf(n), EXP_READ);
            aliascheckexp(ReplCLengthExpOf(n), EXP_READ);
            if (parrepl(TagOf(n))) /* bug 1008 9/10/90 */
              {
                lexlevel++;
                DEBUG_MSG(("aliascheck: repl: incrementing lexlevel\n"));
                aliascheck(ReplCBodyOf(n));
                DEBUG_MSG(("aliascheck: repl: decrementing lexlevel\n"));
                lexlevel--;
                return;
              }
            n = ReplCBodyOf(n);
            break;
          /*}}}*/
          #if 0 /* temporarily disable - why is it different anyway ? */
          /*{{{  PLACEDREPLPAR*/
          case S_PLACEDREPLPAR:
          {
            INT32 base = evaluate(ReplCStartExpOf(n));
            INT32 count = evaluate(ReplCLengthExpOf(n));
            treenode *name = (treenode *) ReplCNameOf(n);
            n = ReplCBodyOf(n);
            SetNReplKnown(name, TRUE);
            while (count > 0)
              {
                SetNReplValue(name, base);
                aliascheck(n);
                base++;
                count--;
              }
            SetNReplKnown(name, FALSE);
          }
          return;
          /*}}}*/
          #endif
          /*{{{  WHILE CHOICE SELECTION*/
          case S_WHILE: case S_CHOICE:
          case S_SELECTION:
            lastloc = LocnOf(n);
            aliascheckexp(CondGuardOf(n), EXP_READ);
            n = CondBodyOf(n);
            break;
          /*}}}*/
          /*{{{  PROCESSOR*/
          case S_PROCESSOR:
            lastloc = LocnOf(n);
            aliascheckexp(ProcessorExpOf(n), EXP_READ);
            n = ProcessorBodyOf(n);
            break;
          
            return ;
          /*}}}*/
          /*{{{  PINSTANCE*/
          case S_PINSTANCE:
            {
              treenode *flist;
              treenode *initn = n;
              abbrevlist *save_val, *save_var;
              abbrevlist *save_inabbrevs, *save_outabbrevs;
              lastloc = LocnOf(n);
              /*{{{  save abbrev info*/
              save_val = valabbrevs;
              save_var = varabbrevs;
              save_inabbrevs = inabbrevs; save_outabbrevs = outabbrevs;
              /*}}}*/
              /*{{{  check parameters*/
              flist = NTypeOf(INameOf(n));
              {
                n = IParamListOf(n);
                in_abbrev = TRUE;
                while (n != NULL)
                  {
                    treenode *formal = ThisItem(flist);
                    treenode *actual = ThisItem(n);
                    int type = basetype(NTypeOf(formal));
                    if (TagOf(formal) == N_VALPARAM)
                      {
                        aliascheckexp(actual, EXP_READ);
                        val_abbreviate(actual);
                      }
                    else if (type == S_CHAN || type == S_PORT)
                      var_expression(actual, in_abbrev,
                                     ParamInputOn(formal), CHAN_INPUT,
                                     ParamOutputOn(formal), CHAN_OUTPUT);
                    else
                      var_expression(actual, in_abbrev, TRUE, EXP_WRITTEN, FALSE, 0);
                    flist = NextItem(flist);
                    n = NextItem(n);
                  }
                in_abbrev = FALSE;
              }
              /*}}}*/
              insertoverlaps ();
              switch (TagOf(INameOf(initn)))
                {
                  case N_PROCDEF: case N_INLINEPROCDEF:
                    aliascheckinstance(NFreeVarsOf(INameOf(initn)),
                      USE_VAR_FREEVAR, USE_VAL_FREEVAR);
                    break;
                  default:  /* Others won't have any free variables */
                    break;
                }
              /*{{{  restore abbrev info*/
              while (valabbrevs != save_val)
                valabbrevs = release_headof_abbrevlist(valabbrevs);
              while (varabbrevs != save_var)
                varabbrevs = release_headof_abbrevlist(varabbrevs);
              while (inabbrevs != save_inabbrevs)
                inabbrevs = release_headof_abbrevlist(inabbrevs);
              while (outabbrevs != save_outabbrevs)
                outabbrevs = release_headof_abbrevlist(outabbrevs);
              /*}}}*/
              return;
            }
          /*}}}*/
          /*{{{  ASS CASE OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT CASE_INPUT*/
          /*{{{  ass*/
          case S_ASS:
            {
              treenode *lhs = LHSOf(n), *rhs = RHSOf(n);
              /*{{{  save abbrev info*/
              abbrevlist *save_var = varabbrevs;
              abbrevlist *save_val = valabbrevs;
              /* needn't worry about channel stuff for an assignment */
              /*}}}*/
              lastloc = LocnOf(n);
          
              aliascheckexp(lhs, EXP_WRITTEN);

              in_abbrev = FALSE; /* Disable overlap checks instead of errors BUG 310 */
              if (TagOf(lhs) == S_LIST)
                walklist(var_abbreviate_seg, lhs);
              else
                var_abbreviate_seg(lhs);  /* in_abbrev is set TRUE if a segment */
          
              aliascheckexp(rhs, EXP_READ);
              in_abbrev = FALSE; /* BUG 310 ACS and CO'N */
              insertoverlaps();
          
              /*{{{  restore abbrev info*/
              while (valabbrevs != save_val)
                valabbrevs = release_headof_abbrevlist(valabbrevs);
              while (varabbrevs != save_var)
                varabbrevs = release_headof_abbrevlist(varabbrevs);
              /*}}}*/
            }
            return;
          /*}}}*/
          /*{{{  case*/
          case S_CASE:
            lastloc = LocnOf(n);
            aliascheckexp(LHSOf(n), EXP_READ);
            n = RHSOf(n);
            break;
          /*}}}*/
          /*{{{  output*/
          case S_OUTPUT:
            lastloc = LocnOf(n);
            aliascheckexp(LHSOf(n), CHAN_OUTPUT);
            aliascheckexp(RHSOf(n), EXP_READ);
            return;
          /*}}}*/
          /*{{{  input, taggedinput*/
          case S_INPUT: case S_TAGGED_INPUT:
            lastloc = LocnOf(n);
            aliascheckexp(LHSOf(n), CHAN_INPUT);
            aliascheckexp(RHSOf(n), EXP_WRITTEN);
            return;
          /*}}}*/
          /*{{{  delayedinput*/
          case S_DELAYED_INPUT:
            lastloc = LocnOf(n);
            aliascheckexp(LHSOf(n), CHAN_INPUT);
            aliascheckexp(RHSOf(n), EXP_READ);
            return;
          /*}}}*/
          /*{{{  caseinput*/
          case S_CASE_INPUT:
            lastloc = LocnOf(n);
            aliascheckexp(LHSOf(n), CHAN_INPUT);
            n = RHSOf(n);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  ALTERNATIVE*/
          case S_ALTERNATIVE:
            lastloc = LocnOf(n);
            aliascheckexp(AltGuardOf(n), EXP_READ);
            aliascheck(AltInputOf(n));
            n = AltBodyOf(n);
            break;
          /*}}}*/
          /*{{{  VARIANT*/
          case S_VARIANT:
            lastloc = LocnOf(n);
            aliascheckexp(VRTaggedListOf(n), EXP_READ);
            n = VRBodyOf(n);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  list*/
          case S_LIST:
            aliascheck(ThisItem(n));
            n = NextItem(n);
            break;
          /*}}}*/
          /*{{{  specification*/
          case S_ABBR: case S_VALABBR:
          case S_RETYPE: case S_VALRETYPE:
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_DECL:
          case S_TPROTDEF: case S_SPROTDEF:
          case S_PLACE: case S_WSPLACE: case S_VSPLACE:
          #ifdef CONFIG
          case S_PLACEON:
          CASE_CONFIG_SPEC
          #endif
            /*{{{  handle specification*/
            {
              abbrevlist *save_var = varabbrevs;
              abbrevlist *save_val = valabbrevs;
              abbrevlist *save_inabbrevs, *save_outabbrevs;
              save_inabbrevs = inabbrevs; save_outabbrevs = outabbrevs;
              while (isspecification(n))
                {
                  lastloc = LocnOf(n);
                  DEBUG_MSG(("aliascheck: (spec) %s\n", usetagstring(n)));
                  switch (TagOf(n))
                    {
                      /*{{{  abbr / retype*/
                      case S_ABBR: case S_RETYPE:
                        {
                          int type = basetype(NTypeOf(DNameOf(n)));
                          if ((type == S_CHAN) || (type == S_PORT))
                            {
                              int use = chan_use(DBodyOf(n), DNameOf(n));
                              var_expression(DValOf(n), TRUE,
                                             (use & CHAN_USE_INPUT)  != 0, CHAN_INPUT,
                                             (use & CHAN_USE_OUTPUT) != 0, CHAN_OUTPUT);
                            }
                          else
                            var_expression(DValOf(n), TRUE, TRUE, EXP_WRITTEN, FALSE, 0);
                          insertoverlaps ();
                        }
                        break;
                      /*}}}*/
                      /*{{{  valabbr / valretype*/
                      case S_VALABBR:
                      case S_VALRETYPE:
                        {
                          treenode *rhs = DValOf(n);
                          in_abbrev = TRUE;
                          aliascheckexp(rhs, EXP_READ);
                          in_abbrev = FALSE;
                          val_abbreviate(rhs);
                          insertoverlaps ();
                        }
                        break;
                      /*}}}*/
                      /*{{{  proc / func def*/
                      case S_PROCDEF:
                      case S_SFUNCDEF:
                      case S_LFUNCDEF:
                      CASE_CONFIG_SPEC
                        if (!separatelycompiled(DNameOf(n)))
                          {
                            abbrevlist *save_val = valabbrevs;
                            abbrevlist *save_var = varabbrevs;
                            abbrevlist *save_inabbrevs, *save_outabbrevs;
                            save_inabbrevs = inabbrevs; save_outabbrevs = outabbrevs;
                            valabbrevs = NULL;
                            varabbrevs = NULL;
                            inabbrevs = NULL; outabbrevs = NULL;
                            lexlevel++; /* bug 1008 9/10/90 */
                            DEBUG_MSG(("aliascheck: PROC incrementing lexlevel\n"));
                            aliascheck(DValOf(n));
                            lexlevel--;
                            DEBUG_MSG(("aliascheck: PROC decrementing lexlevel\n"));
                            valabbrevs = save_val;
                            varabbrevs = save_var;
                            inabbrevs = save_inabbrevs;
                            outabbrevs = save_outabbrevs;
                          }
                        break;
                      /*}}}*/
                      /*{{{  decl / protocol*/
                      case S_DECL:
                      case S_TPROTDEF:
                      case S_SPROTDEF:
                        break;
                      /*}}}*/
                      /*{{{  do nothings*/
                      case S_PLACE: case S_WSPLACE: case S_VSPLACE:
                      #ifdef CONFIG
                      case S_PLACEON:
                      #endif
                        break;
                      /*}}}*/
                    }
                  n = DBodyOf(n);
                }
              aliascheck(n);
              /*{{{  restore abbrev info*/
              while (valabbrevs != save_val)
                valabbrevs = release_headof_abbrevlist(valabbrevs);
              while (varabbrevs != save_var)
                varabbrevs = release_headof_abbrevlist(varabbrevs);
              while (inabbrevs != save_inabbrevs)
                inabbrevs = release_headof_abbrevlist(inabbrevs);
              while (outabbrevs != save_outabbrevs)
                outabbrevs = release_headof_abbrevlist(outabbrevs);
              /*}}}*/
            }
            /*}}}*/
            return;
          /*}}}*/
          /*{{{  VALOF - in function def*/
          case S_VALOF:
            lastloc = LocnOf(n);
            aliascheckexp(VLResultListOf(n), EXP_READ);
            n = VLBodyOf(n);
            break;
          /*}}}*/
          #ifdef CONFIG
          /*{{{  confignode*/
          case S_SET: case S_CONNECT: case S_MAP:
            lastloc = LocnOf(n);
            aliascheckexp(STDevOf(n),     EXP_READ);
            if (TagOf(n) != S_SET)
              aliascheckexp(STAttrNameOf(n), EXP_READ);
            aliascheckexp(STAttrExpOf(n), EXP_READ);
            return;
          /*}}}*/
          #endif
          default:
            badtag(LocnOf(n), TagOf(n), "aliascheck");
        }
    }
  return;
}
/*}}}*/
/*{{{  PRIVATE void genfreevars (n)*/
/*{{{  PRIVATEPARAM int do_genfreevars (n)*/
/*****************************************************************************
 *
 * Walk over tree 'n' and for each proc/function declared generate its
 * free variable list and store it in proc's name node
 *
 *****************************************************************************/
PRIVATEPARAM int do_genfreevars ( treenode *n )
{
  DEBUG_MSG(("do_genfreevars: %s\n", usetagstring(n)));
  switch(TagOf(n))
    {
      /*{{{  REPLSEQ REPLIF REPLPAR REPLALT PRIREPLPAR PRIREPLALT PLACEDREPLPAR*/
      case S_REPLSEQ: case S_REPLIF: case S_REPLPAR: case S_REPLALT:
      case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
        SetNReplKnown(ReplCNameOf(n), FALSE);
        if (parrepl(TagOf(n))) /* bug 1008 9/10/90 */
          {
            lexlevel++;
            DEBUG_MSG(("do_genfreevars: repl: incrementing lexlevel\n"));
            prewalktree(ReplCBodyOf(n), do_genfreevars);
            lexlevel--;
            DEBUG_MSG(("do_genfreevars: repl: decrementing lexlevel\n"));
            return STOP_WALK;
          }
        break; /* continue onto children */
      /*}}}*/
      /*{{{  ASS*/
      case S_ASS :
        #if 0   /* Check this later, during the usage check */
                /* to fix bug 278 in buglist (ie expand replicators first) */
        if (TagOf(LHSOf(n)) == S_LIST)
          {
            /* Check for overlap of LHS of multiple assignment */
            varlist *vars_used = NULL;
            varlist *uses_vars;
            SOURCEPOSN oldloc = lastloc;
            lastloc = LocnOf(n);
            n = LHSOf(n);
            while (n != NULL)
              {
                uses_vars = freevarsinexp(ThisItem(n), -1, NULL, EXP_WRITTEN, TRUE);
                check_no_overlaps(vars_used, uses_vars, lastloc);
                vars_used = merge(uses_vars, -1, vars_used);
                release_varlist(uses_vars);
                n = NextItem(n);
              }
            release_varlist(vars_used);
            lastloc = oldloc;
          }
        #endif /* 0 */
        break;
      /*}}}*/
      /*{{{  proc / func def; return TRUE*/
      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
        {
          treenode *routinename = DNameOf(n);
          varlist *freevars;
          lexlevel++; /* bug 1008 9/10/90 */
          DEBUG_MSG(("do_genfreevars: PROC: incrementing lexlevel\n"));
          if (separatelycompiled(routinename))
            freevars = freevarsinsc(DValOf(n), -1, NULL);
          else
            {
              prewalktree(DValOf(n), do_genfreevars);/*generate for nested routines*/
              freevars = freevarsin(DValOf(n), -1, NULL, FALSE);
            }
          DEBUG_MSG(("do_genfreevars: PROC: decrementing lexlevel\n"));
          lexlevel--;
          if (checkusage) /* condition added for bug 842 10/1/91 */
            check_channel_direction_usage(freevars, LocnOf(n));
          freevars = record_parameter_info(n, freevars);
          SetNFreeVars(routinename, freevars);
        #ifdef DEBUG
          if (debuguse)
            {
              fprintf(outfile, "free vars of %s : ", WNameOf(NNameOf(routinename)));
              printfreelist(NFreeVarsOf(routinename));
            }
        #endif
          prewalktree(DBodyOf(n), do_genfreevars);
          return STOP_WALK;
        }
      /*}}}*/
      /*{{{  abbr*/
      case S_ABBR:
        SetNParamUse(DNameOf(n), NULL);
        break;
      /*}}}*/
      /*{{{  decl*/
      case S_DECL:
        {
          treenode *list = DNameOf(n);
          if (TagOf(list) == S_LIST)
            while (list != NULL)
              {
                SetNParamUse(ThisItem(list), NULL);
                list = NextItem(list);
              }
          else
            SetNParamUse(list, NULL);
          break;
        }
      /*}}}*/
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATE void genfreevars(n)*/
/*****************************************************************************
 *
 * Walk over tree 'n' and for each proc/function declared generate its
 * free variable list and store it in proc's name node.
 * Also checks free/param channel direction usage and multiple assignment
 *
 *****************************************************************************/
PRIVATE void genfreevars ( treenode *n )
{
  prewalktree(n, do_genfreevars);
}
/*}}}*/
/*}}}*/
/*{{{  PUBLIC int paraminputon(n)*/
/*****************************************************************************
 *
 *  paraminputon returns TRUE if a process inputs on the channel parameter 'n'.
 *
 *****************************************************************************/
PUBLIC int paraminputon ( treenode *n )
{
  return ParamInputOn(n);
}
/*}}}*/
/*{{{  PUBLIC int paramoutputon(n)*/
/*****************************************************************************
 *
 *  paramoutputon returns TRUE if a process inputs on the channel parameter 'n'
 *
 *****************************************************************************/
PUBLIC int paramoutputon ( treenode *n )
{
  return ParamOutputOn(n);
}
/*}}}*/
/*{{{  PUBLIC int isafreevarof(v, n)*/
/*****************************************************************************
 *
 *  isafreevarof returns TRUE if variable 'v' is a free variable of routine
 *  'n', FALSE otherwise. This routine assumes that alias checking has
 *  already taken place.
 *
 *****************************************************************************/
PUBLIC int isafreevarof ( treenode *v , treenode *n )
{
  varlist *freevars;
  int result;
  /* freevars = freevarsin(n, -1, NULL); */
  freevars = NFreeVarsOf(n);
  result = invarlist(v, freevars, EXP_READ,     0, MAXINDEX) ||
           invarlist(v, freevars, EXP_WRITTEN,  0, MAXINDEX) ||
           invarlist(v, freevars, CHAN_INPUT,   0, MAXINDEX) ||
           invarlist(v, freevars, CHAN_OUTPUT,  0, MAXINDEX);
  /* release_varlist(freevars); */
  return(result);
}
/*}}}*/
/*{{{  PUBLIC void alias_and_usage_check(n)*/
/*****************************************************************************
 *
 *  alias_and_usage_check is an alias and usage checker
 *  which first builds up lists of free
 *  variables for routines and places them in the routines' namenodes.
 *  The routine is then alias checked and then usage checked.
 *
 *****************************************************************************/
PUBLIC void alias_and_usage_check ( treenode *n )
  {
    int error_count = 0;
    uerrors = NULL;
    genfreevars(n);  /* also checks free/param channel direction usage
                      * and multiple assignment
                      */
    if (checkalias)
      /*{{{  do the alias checking, and the usage checking, possibly*/
      {
        constexp_one = newconstexpnode(S_CONSTEXP, NOPOSN, dummyexp_p, ZERO32, 1);
        if (information)
          fprintf (outfile, "Alias%s checking\n", checkusage ? " and usage" : "");
        overlapchain = NULL;
        aliascheck(n);
        if (checkusage)
          usagecheck(n);
      }
      /*}}}*/
    /* report any errors */
    while (uerrors != NULL)
      {
        msg_out_s(warn_on_usage_error ? SEV_WARN : SEV_ERR,
                  USE, ERCodeOf(uerrors), ERLocnOf(uerrors),
                  (ERP1Of(uerrors) != NULL) ? WNameOf(NNameOf(ERP1Of(uerrors))) : NULL);
        error_count++;
        uerrors = ERNextOf(uerrors);
      }
    if (checkalias && information)
      /*{{{  give some information*/
      {
        fprintf (outfile, "Alias%s checked ", checkusage ? " and usage" : "");
        if (error_count != 0)
          {
            fprintf(outfile, "- %d %s%s\n", error_count,
                    warn_on_usage_error ? "warning" : "error",
                    error_count > 1 ? "s" : "");
          }
        else
          fputs("ok\n", outfile);
      }
      /*}}}*/
  }
/*}}}*/
