/*#define DEBUG*/
/****************************************************************************
 *
 *  path opening stuff
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  INMOS Toolsets
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/
/*{{{  include files*/
# include "imstype.h"
# include "imsmisc.h"

# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include "ochdr.h"
# include "popen.h"
/*}}}*/

/*{{{  comment about it all*/
/****************************************************************************/
/*                                                                          */
/* popen_read; A routine for opening a file in the occam Toolset, where you */
/*             search an environment variable for a list of directories     */
/*             to look in                                                   */
/* Parameters:                                                              */
/* FILE *popen_read (filename, pathname, full_name, mode)                   */
/*   char *filename, pathname, full_name;                                   */
/*   int  mode;                                                             */
/*                                                                          */
/* The filename is a string containing the basic filename.                  */
/* It first simply tries to open that filename. If it can't, but it contains*/
/* a directory part on the front, no path searching is done.                */
/* Otherwise it translates the environment variable whose name is given as  */
/* a string in pathname.  It scans through the directories on that list,    */
/* each separated by spaces or semicolons, and adds the filename onto the   */
/* end of each one in turn, trying to open it                               */
/* If it succeeds in opening the file, it returns the full name in full_name*/
/* and returns the FILE pointer.  If it can't find it then it returns null, */
/* and sets *full_name = '\0'.                                              */
/* Note that full_name must be pointing at an area big enough to hold the   */
/* longest possible filename (256 chars?)                                   */
/* The mode parameter specifies 0=text, 1=binary                            */
/*                                                                          */
/* Written by CON of INMOS, 11 August 1988                                  */
/*                                                                          */
/****************************************************************************/

/*}}}*/

/*{{{  PRIVATE FILE *try_opening*/
/* returns the file ptr if it can open filename */
PRIVATE FILE *try_opening (const char *filename, const int mode )
{
  FILE *fptr = NULL;

  DEBUG_MSG(("Trying filename \'%s\' ...\n", filename));

  /*{{{  if LLL*/
  #ifdef LLL
  {
    extern int _fmode;
    int saved_fmode = _fmode;
    if
      (mode == POPEN_MODE_TEXT)
        _fmode =  O_TEXT;
      else  /* POPEN_MODE_BINARY */
        _fmode =  O_BINARY;
    fptr = fopen (filename, "r");
    _fmode = saved_fmode;
  }
  #endif
  /*}}}*/
  /*{{{  if VAX*/
  #ifdef DEC
  if
    (mode == POPEN_MODE_TEXT)
      fptr = fopen (filename, "r");
    else  /* POPEN_MODE_BINARY */
      fptr = fopen (filename, "r", "rfm = udf");
  #endif
  /*}}}*/
  /*{{{  if SUN*/
  #if defined(SUN) || defined(GNU)
  fptr = fopen (filename, "r");  /* same for both TEXT and BINARY */
  #endif
  /*}}}*/
  /*{{{  if MSC*/
  #ifdef MSC  /*  Microsoft C - IBM or NEC */
  if
    (mode == POPEN_MODE_TEXT)
      fptr = fopen (filename, "rt");
    else  /* POPEN_MODE_BINARY */
      fptr = fopen (filename, "rb");
  #endif
  /*}}}*/
  /*{{{  if HELIOS*/
  #ifdef HELIOS
  if
    (mode == POPEN_MODE_TEXT)
      fptr = fopen (filename, "r");
    else  /* POPEN_MODE_BINARY */
      fptr = fopen (filename, "rb");
  #endif
  /*}}}*/
  /*{{{  if IMS*/
  #ifdef IMS
  if
    (mode == POPEN_MODE_TEXT)
      fptr = fopen (filename, "r");
    else  /* POPEN_MODE_BINARY */
      fptr = fopen (filename, "rb");
  #endif
  /*}}}*/

  return fptr;
}
/*}}}*/
/*{{{  PRIVATE int dirname_exists*/
/* returns true if filename includes any sort of directory spec */
PRIVATE int dirname_exists (const char *filename)
{
  DEBUG_MSG(("Looking for a directory name\n"));

  while ((*filename) != '\0')
    {
      switch (*filename)
        {
          case '[': case ']':
          case '<': case '>':
          case '/': case '\\':
          case ':':
            return (TRUE);
          default:
            break;
        }
      filename++;
    }
  return (FALSE);
}

/*}}}*/
/*{{{  PRIVATE char *get_path*/
/* returns the contents of the environment variable 'pathname' in 'path' */
PRIVATE char *get_path (const char *pathname_ptr)
{
  /*extern char *getenv(char *);*/
  char *path_string = getenv(pathname_ptr);

  DEBUG_MSG(("Looking for the path string: \'%s\'\n", pathname_ptr));

  if
    (path_string == NULL)
      {
        DEBUG_MSG(("Found NO path string\n"));
        return ("");
      }
    else
      {
        DEBUG_MSG(("Found the path string: \'%s\'\n", path_string));
        return path_string;
      }
}

/*}}}*/
/*{{{  PRIVATE int get_next_name*/
/*  gets next directory bit from the path, then */
/*  concatenates directory and filename into full_name */
/*  If there are no more dirs on the path, returns FALSE, else TRUE */
PRIVATE int get_next_name (char **path_ptr_ptr, const char *filename_ptr, char *full_name_ptr)
{
  /*{{{  test if there are any more directories on the path - return FALSE*/
  if
    ((**path_ptr_ptr) == '\0')
      {
        DEBUG_MSG(("path string is now empty\n"));
        return (FALSE);   /* no more directories on path */
      }
  /*}}}*/
  /*{{{  copy until the next space or semi-colon*/
  while (((**path_ptr_ptr) != '\0') && ((**path_ptr_ptr) != ';') &&
                                       ((**path_ptr_ptr) != ' ')   )
    {
      (*full_name_ptr++) = (**path_ptr_ptr);
      (*path_ptr_ptr)++;
    }
  /*}}}*/
  /*{{{  skip over the next spaces or semi-colons*/
  while (((**path_ptr_ptr) == ';') || ((**path_ptr_ptr) == ' '))
      (*path_ptr_ptr)++;  /* skip over the spaces or semi-colons */

  /*}}}*/
  /*{{{  now add the filename on the end*/
  /* Now simply add the filename to the end */
  while (((*full_name_ptr++) = (*filename_ptr++)) != '\0')
    ;
  /*}}}*/
  return (TRUE);
}
/*}}}*/

/*{{{  PUBLIC  FILE *popen_read*/
/* the basic filename */
/* the name of the environment variable */
/* the full name of the file which was opened */
/* 0 = TEXT, 1 = BINARY */
/* returns NULL if it couldn't open the file */
PUBLIC FILE *popen_read (const char *filename, const char *pathname,
                         char *full_name, const int mode)
{
  FILE *fptr;

  fptr = try_opening (filename, mode);  /* try opening it for reading */

  if
    /*{{{  couldn't open the file, and a pathname has been supplied*/
    ((fptr == NULL    ) &&  /* couldn't open it */
     (pathname != NULL) && ((*pathname) != '\0') )
      /* and a pathname HAS been supplied */
    /*}}}*/
      {
        if
          (!dirname_exists (filename))
            /*{{{  Search using the pathname*/
            { /* there is no directory name */
              char *path_ptr;
              int  more_on_path = TRUE;
              DEBUG_MSG(("Found no directory name, so looking down path\n"));
              path_ptr = get_path(pathname); /* point at first element of path string */
              while ((fptr == NULL) && (more_on_path))
                {
                  more_on_path = get_next_name (&path_ptr, filename, full_name);
                  if
                    (more_on_path)
                      fptr = try_opening (full_name, mode); /* try opening it */
                    else
                      {
                        DEBUG_MSG(("No more names to try\n"));
                      }
                }
            }
            /*}}}*/
          else
            /*{{{  Don't look down pathname cos there was a directory specified*/
            {
              DEBUG_MSG(("Found a directory name\n"));
              ; /* if there was a directory name, leave fptr as NULL */
            }
            /*}}}*/
      }
    else
      /*{{{  copy normal name onto full_name*/
      { 
        strcpy(full_name, filename);
      }
      /*}}}*/

  if
    (fptr == NULL)
      (*full_name) = '\0';

  return (fptr);
}
/*}}}*/
