/* Bits and pieces that are declared on some systems, but not on others */

# include <stdio.h>
# include "includes.h"

#ifdef LLL
/*{{{  PUBLIC char *strrchr (const char *s, int c)*/
PUBLIC char *strrchr ( char *s , int c )
{
  char *result = NULL;
  while (*s)
    {
      if (*s == c) result = s;
      s++;
    }
  return result;
}
/*}}}*/
#endif
/*{{{  PRIVATE void swap (a, b, size)*/
/*****************************************************************************
 *
 *  swap the two objects pointed to by 'a' and 'b': the objects are
 *       'size' bytes long
 *
 *****************************************************************************/
PRIVATE void swap ( char *a , char *b , size_t size )
{
  char temp;
  int i;
  for (i = 0; i < size; i++)
    {
      temp = *a;
      *a++ = *b;
      *b++ = temp;
    }
}
/*}}}*/
/*{{{  PUBLIC void local_qsort (base, nel, elsize, compar)*/
/* We use a local copy of qsort, because the different systems this compiler
   was compiled for had qsorts with different behaviours,
   (when the keys are equal) and we want the
   compiler to always compile the same code in the same way, no matter
   which system the compiler is running on.
*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  qsort performs a quicksort.
 *        The data to be sorted is in an array whose first element is pointed
 *        at by 'base'.
 *        'nel' is the number of elements in the array.
 *        'elsize' is the number of bytes in each element (the elements must
 *        by contiguous).
 *        'compar' is a function which takes two pointer parameters which
 *        point at two elements to be compared, and returns
 *            1 if the first element is greater than the second
 *            0 if the elements are equal
 *           -1 if the first element is less than the second.
 *
 *        This routine is compatible with the Unix System V qsort subroutine.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void local_qsort (void *b, size_t nel, size_t elsize, int (*compar)(void *, void *))
{
  char *base = (char *)b;
  char *upper = base + (nel - 1) * elsize;
  while (nel > 2)
    /*{{{  sort*/
    {
      char *i = base - elsize,
           *j = upper,
           *pivot = j;
      while (i < j)
        /*{{{  perform one iteration*/
        {
          do
            i += elsize;
          while ((*compar)(pivot, i) > 0);
        
          do
            j -= elsize;
          while ((j > i) && ((*compar)(j, pivot) > 0));
        
          if (i < j)
            swap(i, j, elsize);
        }
        /*}}}*/
    
      if (i != upper)
        swap(i, upper, elsize);
    
      /*{{{  partition; recurse on smallest end, iterate on largest end*/
      {
        unsigned int lowels = ((i - base) / elsize),
                     highels = (nel - lowels) - 1;
        if (lowels < highels)
          {
            local_qsort(base, lowels, elsize, compar);
            base = i + elsize;
            nel = highels;
          }
        else
          {
            local_qsort(i + elsize, highels, elsize, compar);
            upper = i - elsize;
            nel = lowels;
          }
      }
      /*}}}*/
    }
    /*}}}*/

  if (nel == 2)
    /*{{{  simplest case: compare the two elements*/
    if ((*compar)(base, base + elsize) > 0)
      swap(base, base + elsize, elsize);
    /*}}}*/
}
/*}}}*/

/*{{{  PUBLIC char *strlwr (char *s)*/
PUBLIC char *strlwr ( char *s )
{
  char *t;
  for (t = s; *t != '\0'; t++)
    if (isupper(*t)  /*(*t >= 'A') && (*t <= 'Z')*/ )
      *t = tolower(*t);
  return s;
}
/*}}}*/
/*{{{  PUBLIC char *strupr (char *s)*/
PUBLIC char *strupr ( char *s )
{
  char *t;
  for (t = s; *t != '\0'; t++)
    if (islower(*t) /*(*t >= 'a') && (*t <= 'z')*/ )
      *t = toupper(*t);
  return s;
}
/*}}}*/

#if defined(SUN) || defined(GNU)
/*{{{  PRIVATE char *strstr (a, b)*/
PUBLIC char *strstr ( char *a , char *b )
{
  while (TRUE)
    {
      int i;
      for (i=0;; i++)
        {
          if (b[i] == 0)
            return a;
          if (a[i] != b[i])
            break;
        }
      if (*a++ == 0)
        return NULL;
    }
}
/*}}}*/
#endif


