/*#define DEBUG*/
/*#define DEBUG_NEXTSYMB*/
/****************************************************************************
 *
 *  Occam two lexical analyser
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/
/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lexerror.h"
# include "lexconst.h"
# include "lexdef.h"
# include "lex1def.h"
# include "syn1def.h"
# include "chkdef.h"
# include "desc1def.h"
# include "genhdr.h"
# include "gen1def.h"
# include "predefhd.h"
# include "popen.h"
/*}}}*/

/*{{{  definitions*/
#define HASHSIZE 511 /* 257 */
#define TABSIZE 8

/* String termination character,
   mustn't clash with any valid occam character */
#define CLOSE_QUOTE 257

#define MINLEGALCH 32
#define MAXLEGALCH 126

/*{{{  filestruct_t*/
typedef struct
  {
    FILE *fileptr;
    char fmode;
    SOURCEPOSN ffileposn;
    int fbaseindent;
    char fname[MAX_FILENAME_LENGTH];
  } filestruct_t;
/*}}}*/
/*{{{  struct filetablestruct*/
typedef struct filetablestruct
  {
    /* I've changed this to be a linked list rather than an array */
    struct filetablestruct *ftnext;
    wordnode *ftname;
    int ftmode;
    int ftparent;
    int ftparentline;
  } filetablestruct_t;
/*}}}*/
/*}}}*/

/*{{{  global variables*/
PUBLIC int symb;
PUBLIC int symbindent, lineindent;

PUBLIC wordnode *lexword;
PUBLIC char literalv[MAXSTRING_SIZE + 1];
PUBLIC int literalp;
PUBLIC int totallines;
PUBLIC int pdnumber;  /* **SHC 5-Apr-1988 */

PUBLIC char filename[MAX_FILENAME_LENGTH];
PUBLIC int current_file;
PUBLIC SOURCEPOSN flocn;

PUBLIC int lexmode;

PUBLIC int currentfilenum;
PUBLIC int allow_asmnames;

PUBLIC int lexer_ignore_comments = FALSE; /* bug 853 18/1/91 */
/*}}}*/
/*{{{  local variables*/
PRIVATE int ch;
PRIVATE int stringcontinuation;
PRIVATE int insertlength;
PRIVATE int linep;
PRIVATE int currentindent;
PRIVATE int baseindent;

/* TRUE when inputline cannot read any more from file : there may still
   be lines in the lookaheadbuffer. */
PRIVATE int endoffile;

/* TRUE when the lexer has sent an S_END symbol: any call to nextsymb
   after this is an unexpected end of file. */
PRIVATE int sentend;

#ifndef BACKEND
/*{{{  line buffer handling*/
PRIVATE int linecount;  /* lines in this file */

PRIVATE int   linebuffer_len;        /* length of current line */
PRIVATE char *linebuffer_ptr;        /* ptr to current line */

PRIVATE int eof;                           /* True when last symb was END     */
PRIVATE int newlineflag;                   /* True when last symb was NEWLINE */
PRIVATE int desc_eof;        /* TRUE when we hit the end of a descriptor file */
PRIVATE filestruct_t filestack[FILESMAX];
PRIVATE int filestackptr = 0;
/*}}}*/
/*{{{  predefined routines defns.*/
PRIVATE struct predeflinestruct
  {
    unsigned char pnumber;
    char *pline;
  } predeflines[] =
  /* These don't HAVE to be in the correct order, but it is clearer */
  {
    /*{{{  longarithmetic*/
    {PD_LONGADD,"FI(VI,VI,VI):LONGADD"},
    {PD_LONGSUM,"FI,I(VI,VI,VI):LONGSUM"},
    {PD_LONGSUB,"FI(VI,VI,VI):LONGSUB"},
    {PD_LONGDIFF,"FI,I(VI,VI,VI):LONGDIFF"},
    {PD_LONGPROD,"FI,I(VI,VI,VI):LONGPROD"},
    {PD_LONGDIV,"FI,I(VI,VI,VI):LONGDIV"},
    /*}}}*/
    /*{{{  shift*/
    {PD_SHIFTRIGHT,"FI,I(VI,VI,VI):SHIFTRIGHT"},
    {PD_SHIFTLEFT,"FI,I(VI,VI,VI):SHIFTLEFT"},
    /*}}}*/
    /*{{{  normalise,fracmul*/
    {PD_NORMALISE,"FI,I,I(VI,VI):NORMALISE"},
    {PD_FRACMUL,"FI(VI,VI):FRACMUL"},
    /*}}}*/
    /*{{{  shift,arithmeticshift,rotate*/
    {PD_ASHIFTRIGHT,"FI(VI,VI):ASHIFTRIGHT"},
    {PD_ASHIFTLEFT,"FI(VI,VI):ASHIFTLEFT"},
    {PD_ROTATERIGHT,"FI(VI,VI):ROTATERIGHT"},
    {PD_ROTATELEFT,"FI(VI,VI):ROTATELEFT"},
    /*}}}*/
    /*{{{  causeerror*/
    {PD_CAUSEERROR,"P():CAUSEERROR"},
    /*}}}*/
    /*{{{  kernelrun,loadchannel,loadchannelvector,loadbytevector*/
    {PD_KERNELRUN,"P(V[]Y,VI,[]I,VI):KERNEL.RUN"},
    {PD_LOADINPUTCHANNEL,"P(I,C):LOAD.INPUT.CHANNEL"},
    {PD_LOADOUTPUTCHANNEL,"P(I,C):LOAD.OUTPUT.CHANNEL"},
    {PD_LOADINPUTCHANNELVECTOR,"P(I,[]C):LOAD.INPUT.CHANNEL.VECTOR"},
    {PD_LOADOUTPUTCHANNELVECTOR,"P(I,[]C):LOAD.OUTPUT.CHANNEL.VECTOR"},
    {PD_LOADBYTEVECTOR,"P(I,V[]Y):LOAD.BYTE.VECTOR"},
    /*}}}*/
    /*{{{  unpacksnroundsn*/
    {PD_UNPACKSN,"FI,I,I(VI):UNPACKSN"},
    {PD_ROUNDSN,"FI(VI,VI,VI):ROUNDSN"},
    /*}}}*/
    /*{{{  draw2d,clip2d,move2d*/
    {PD_DRAW2D,"P(V[][]Y,VI,VI,[][]Y,VI,VI,VI,VI):DRAW2D"},
    {PD_CLIP2D,"P(V[][]Y,VI,VI,[][]Y,VI,VI,VI,VI):CLIP2D"},
    {PD_MOVE2D,"P(V[][]Y,VI,VI,[][]Y,VI,VI,VI,VI):MOVE2D"},
    /*}}}*/
    /*{{{  crcbyteandword*/
    {PD_CRCWORD,"FI(VI,VI,VI):CRCWORD"},
    {PD_CRCBYTE,"FI(VI,VI,VI):CRCBYTE"},
    /*}}}*/
    /*{{{  bitops*/
    {PD_BITCOUNT,"FI(VI,VI):BITCOUNT"},
    {PD_BITREVWORD,"FI(VI):BITREVWORD"},
    {PD_BITREVNBITS,"FI(VI,VI):BITREVNBITS"},
    /*}}}*/
    /*{{{  floatingpoint*/
    {PD_ABS,"FR3(VR3):ABS"},
    {PD_ISNAN,"FB(VR3):ISNAN"},
    {PD_NOTFINITE,"FB(VR3):NOTFINITE"},
    {PD_ORDERED,"FB(VR3,VR3):ORDERED"},
    {PD_MINUSX,"FR3(VR3):MINUSX"},
    {PD_MULBY2,"FR3(VR3):MULBY2"},
    {PD_DIVBY2,"FR3(VR3):DIVBY2"},
    {PD_SQRT,"FR3(VR3):SQRT"},
    {PD_FPINT,"FR3(VR3):FPINT"},
    {PD_DABS,"FR6(VR6):DABS"},
    {PD_DISNAN,"FB(VR6):DISNAN"},
    {PD_DNOTFINITE,"FB(VR6):DNOTFINITE"},
    {PD_DORDERED,"FB(VR6,VR6):DORDERED"},
    {PD_DMINUSX,"FR6(VR6):DMINUSX"},
    {PD_DMULBY2,"FR6(VR6):DMULBY2"},
    {PD_DDIVBY2,"FR6(VR6):DDIVBY2"},
    {PD_DSQRT,"FR6(VR6):DSQRT"},
    {PD_DFPINT,"FR6(VR6):DFPINT"},
    {PD_SCALEB,"FR3(VR3,VI):SCALEB"},
    {PD_DSCALEB,"FR6(VR6,VI):DSCALEB"},
    {PD_COPYSIGN,"FR3(VR3,VR3):COPYSIGN"},
    {PD_DCOPYSIGN,"FR6(VR6,VR6):DCOPYSIGN"},
    {PD_NEXTAFTER,"FR3(VR3,VR3):NEXTAFTER"},
    {PD_DNEXTAFTER,"FR6(VR6,VR6):DNEXTAFTER"},
    {PD_LOGB,"FR3(VR3):LOGB"},
    {PD_DLOGB,"FR6(VR6):DLOGB"},
    {PD_FLOATINGUNPACK,"FI,R3(VR3):FLOATING.UNPACK"},
    {PD_DFLOATINGUNPACK,"FI,R6(VR6):DFLOATING.UNPACK"},
    {PD_ARGUMENTREDUCE,"FB,I32,R3(VR3,VR3,VR3):ARGUMENT.REDUCE"},
    {PD_DARGUMENTREDUCE,"FB,I32,R6(VR6,VR6,VR6):DARGUMENT.REDUCE"},
    /*}}}*/
    /*{{{  IEEEarithmetic*/
    {PD_REAL32OP,"FR3(VR3,VI,VR3):REAL32OP"},
    {PD_REAL64OP,"FR6(VR6,VI,VR6):REAL64OP"},
    {PD_IEEE32OP,"FB,R3(VR3,VI,VI,VR3):IEEE32OP"},
    {PD_IEEE64OP,"FB,R6(VR6,VI,VI,VR6):IEEE64OP"},
    {PD_REAL32REM,"FR3(VR3,VR3):REAL32REM"},
    {PD_REAL64REM,"FR6(VR6,VR6):REAL64REM"},
    {PD_REAL32EQ,"FB(VR3,VR3):REAL32EQ"},
    {PD_REAL64EQ,"FB(VR6,VR6):REAL64EQ"},
    {PD_REAL32GT,"FB(VR3,VR3):REAL32GT"},
    {PD_REAL64GT,"FB(VR6,VR6):REAL64GT"},
    {PD_IEEECOMPARE,"FI(VR3,VR3):IEEECOMPARE"},
    {PD_DIEEECOMPARE,"FI(VR6,VR6):DIEEECOMPARE"},
    {PD_IEEE32REM,"FB,R3(VR3,VR3):IEEE32REM"},
    {PD_IEEE64REM,"FB,R6(VR6,VR6):IEEE64REM"},
    /*}}}*/
    /*{{{  reschedule/assert*/
    {PD_RESCHEDULE,"P():RESCHEDULE"},
    {PD_ASSERT,"P(VB):ASSERT"},
    /*}}}*/
#ifdef CONFIG
    {PD_ATTR_LINK,      "X[4]E :link"},
    {PD_ATTR_TYPE,      "X[]Y :type"},
    {PD_ATTR_MEM,       "XI :memsize"},
    {PD_ATTR_ROMSIZE,   "XI :romsize"},
    {PD_ATTR_ROOT,      "XB :root"},
    {PD_ATTR_ORDERCODE, "XI :order.code"},
    {PD_ATTR_ORDERVS,   "XI :order.vs"},
    {PD_HOSTEDGE,       "E :HOST"},
#endif
    {0, ""}
};
/*}}}*/
#endif

/*}}}*/

/*{{{  name table handling*/
PRIVATE wordnode *nametable[HASHSIZE];
/*{{{  PUBLIC wordnode *lookupword(c, len)*/
/* Search for the name at *c in the symbol table. Return the node if found,
otherwise create a node and return that */
PUBLIC wordnode *lookupword ( const char *string , const int len )
{
  register wordnode *wptr;
  register unsigned int hashval;

  /*{{{  calculate hash value*/
  #if 1
  {
    /* Must be unsigned to work on a SPARC: */
    register const unsigned char *c = (const unsigned char *)string;

    if (len > 4) /* Most common case */
      /* modified 15/10/90 to look at the 'middle' char, and to include the
         length, also the third from last char - CON */
      /* Changed again to look at the first 4, the last 4, and a middle one */
      hashval = c[0] + c[1] + c[2] + c[4] + c[len >> 1] +
                c[len - 4] + c[len - 3] + c[len - 2] + c[len - 1] + len;
    else if (len == 1)
      hashval = c[0];
    else if (len == 2)
      hashval = c[0] + c[1];
    else if (len == 3)
      hashval = c[0] + c[1] + c[2];
    else
      hashval = c[0] + c[1] + c[2] + c[3];
    hashval %= HASHSIZE;
  }
  #else
  /* This is what is used in the INMOS C compiler */
  /* It looks a bit slow to me! (CON 18/10/90) */
  {
    int i;
    	char *s;
    hashval = 1;
    for (i = 0, s = string; i < len; i++, s++)
      {
        register unsigned int temp = (hashval << 7);
        hashval = ((hashval >> 25)^(temp >> 1)^(temp >> 4) ^ *s) & 0x7fffffff;
      }
    hashval %= HASHSIZE;
  }
  #endif
  /*}}}*/
  /*{{{  search for name and if found, return*/
  wptr = nametable[hashval];
  while (wptr != NULL)
    {
    #if 0
      /* This version is the old version; I don't know why it doesn't
         simply use memcmp - CON 17/10/90 */
      if (WLengthOf(wptr) == len)
        {
          char *sptr = string, *dptr = wptr->w_name;
          register int i;
          /* we can't use strcmp here, cos some of the bytes could be zero.
             We probably could (and should) use memcmp.
          */
          for (i = 0; (i < len) && (*sptr++ == *dptr++); i++)
            ;
          if (i == len)
            return(wptr);
        }
    #else
      if ((WLengthOf(wptr) == len) && (memcmp(string, WNameOf(wptr), len) == 0))
        return wptr;
    #endif
      wptr = WNextOf(wptr);
    }
  /*}}}*/
  /*{{{  create a new node and return it*/
  {
    char *nameptr = newvec(len + 1);
    /*{{{  copy string into wordnode*/
    #if 0
      /* This version doesn't use memcpy. I don't know why it doesn't. */
    {
      int i;
      char *sptr = string, *dptr = nameptr;
      for (i = 0; i < len; i++)
        *dptr++ = *sptr++;
      *dptr = '\0';
    }
    #else
      memcpy(nameptr, string, len);
      nameptr[len] = '\0';
    #endif
    /*}}}*/
    wptr = newwordnode(S_NAME, nameptr, len, nametable[hashval]);
    nametable[hashval] = wptr;
    return (wptr);
  }
  /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  error handler*/
#ifndef BACKEND
/*{{{  PUBLIC void nextline ()*/

/* Skip to start of next line */
PUBLIC void nextline ( void )
{
  while (symb != S_NEWLINE)
    nextsymb ();
  while (symb == S_NEWLINE)
    nextsymb ();
}
/*}}}*/
/*{{{  PUBLIC void skiplines (indent)*/
PUBLIC void skiplines ( int indent )
{
  /*int i = 0;*/

  if (symbindent > indent)  /* bug 1018 2/11/90 */
    nextline(); /* force at least a single new line, even if this line starts ok */

  while (lineindent > indent)
    {
      nextline ();
      /*++i;*/
    }
#if 0
  if (i != 0)
    fprintf (errfile, "Skipped %d line%s\n", i, (i=1) ? "" : "s");
#endif
  while (symb == S_COMMENT)
    nextline ();
}
/*}}}*/
#endif
/*}}}*/
/*{{{  predefine handling*/
#ifndef BACKEND
/*{{{  predecode*/
PRIVATE char *addto ( char *l , char *s )
{
  while (*s != '\0')
    *(l++) = *(s++);
  return l;
}

/* Decode the predefine string into line */
PRIVATE void predecode ( char *l , char *s )
{
  while (*s != '\0' && *s != ':')
    /*{{{  decode*/
    {
      char *expand;
      switch (*s)
        {
          case 'B' : expand = "BOOL";        break;
          case 'C' : expand = "CHAN OF ANY"; break;
          case 'F' : expand = "FUNCTION ";   break;
          case 'I' : expand = "INT";         break;
          case 'P' : expand = "PROC";        break;
          case 'R' : expand = (*++s == '3' ? "REAL32" : "REAL64"); break;
          case 'V' : expand = "VAL ";        break;
          case 'Y' : expand = "BYTE";        break;
    #ifdef CONFIG
          case 'E' : expand = "EDGE";        break;
          case 'X' : expand = "CONFIG ";     break;
    #endif
          default  : expand = NULL;          break;
        }
      if (expand != NULL)
        l = addto(l, expand);
      else
        *(l++) = *s;
      s++;
    }
    /*}}}*/
  if (*s == ':') s++;
  l = addto(l, s);
  l = addto(l, ":\n");
  *l = '\0';
}
/*}}}*/
/*{{{  PRIVATE char *readpredefline (line)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  readpredefline copies a predefined name specification into line,
 *                 and returns a pointer to the first character of line.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE char *readpredefline ( char *line )
{
  if (allowpredefs)
    while (linecount < ((sizeof(predeflines) / sizeof(struct predeflinestruct))-1))
      {
        predecode(line, predeflines[linecount].pline);
        pdnumber = predeflines[linecount].pnumber; /* **SHC 5-Apr-1988 */
        return(line);
      }
  return(NULL);
}
/*}}}*/
#endif
/*}}}*/
/*{{{  input buffer*/
PRIVATE filetablestruct_t *filenames = NULL;
PRIVATE int numfiles = 0;

/*{{{  PUBLIC int addfilename(name, mode, parent, parentline)*/
PUBLIC int addfilename ( char *name , int mode , int parent , int parentline )
{
  filetablestruct_t *file = 
    (filetablestruct_t *)newvec(sizeof(filetablestruct_t));
  if (mode == LEX_EXTERNAL)
    file->ftname = NULL;
  else
    file->ftname = lookupword(name, strlen(name));
  file->ftmode       = mode;
  file->ftparent     = parent;
  file->ftparentline = parentline;
  file->ftnext       = filenames;
  filenames = file;

  return numfiles++;
}
/*}}}*/
/*{{{  PRIVATE filetablestruct_t *lookupfilenameptr(n)*/
PRIVATE filetablestruct_t *lookupfilenameptr ( int n )
{
  filetablestruct_t *ptr = filenames;

  /* The list is in reverse order of files, so we want the n'th from last */
  n = (numfiles - n) - 1;
  while (n-- > 0)
    ptr = ptr->ftnext;
  return ptr;
}
/*}}}*/
/*{{{  PUBLIC char *lookupfilename(n)*/
PUBLIC char *lookupfilename ( int n )
{
  if ((n >= 0) && (n < numfiles))
    return(WNameOf(lookupfilenameptr(n)->ftname));
  else
    return NULL;
}
/*}}}*/
/*{{{  PUBLIC int filemodeoffile(n)*/
PUBLIC int filemodeoffile ( int n )
{
  return(lookupfilenameptr(n)->ftmode);
}
/*}}}*/
/*{{{  PUBLIC int parentoffile(n)*/
PUBLIC int parentoffile ( int n )
{
  return(lookupfilenameptr(n)->ftparent);
}
/*}}}*/
/*{{{  PUBLIC int parentposnoffile(n)*/
PUBLIC int parentposnoffile ( int n )
{
  return(lookupfilenameptr(n)->ftparentline);
}
/*}}}*/
/*{{{  PUBLIC int numberoffiles(void)*/
PUBLIC int numberoffiles(void)
{
  return(numfiles);
}
/*}}}*/

#ifndef BACKEND
#if 0 /* This version puts a hard limit on source line length */
/*{{{  PRIVATE char *readsourceline*/
PRIVATE char *readsourceline(void)
{
  static char linebuffer[LINEMAX+1]; /* buffer for current line */
  char templine[LINEMAX];
  char *ptr = fgets(templine, LINEMAX - 1, infile);
  if (ptr != NULL)
    /*{{{  copy to real line buffer, translating TABs into spaces*/
    {
      int column = 0, i = 0, nomoretabs = FALSE, eol = FALSE;
      char c;
      ptr = &(linebuffer[0]);
      while ((column < LINEMAX - 1) && !nomoretabs)
        {
          c = templine[i++];
          switch (c)
            {
              case '\0':
                nomoretabs = TRUE; eol = TRUE;
                break;
              case '\t':
                /*{{{  expand TAB into spaces*/
                {
                  int j = (TABSIZE - (column % TABSIZE));
                  while ((j > 0) && (column < LINEMAX - 1))
                    {
                      ptr[column++] = ' ';
                      j--;
                    }
                }
                /*}}}*/
                break;
              default:
                /*{{{  copy c to real line buffer*/
                {
                  ptr[column++] = c;
                  nomoretabs = (c != ' ');
                }
                /*}}}*/
                break;
            }
        }
      if (!eol)
        while ((column < LINEMAX - 1) && ((c = templine[i++]) != '\0'))
          ptr[column++] = c;

      /* test for overflow while expanding tabs, or whole line overflow */
      if (column >= (LINEMAX - 2)) /* changed bug 1019 18/10/90 */
        {
          /* insert a newline. In either case, column is nearly=LINEMAX */
          ptr[column++] = '\n';
          lexerror (LEX_LINE_TRUNCATED, flocn, LINEMAX);
        }
      else if (ptr[column-1] != '\n')
        { /* there was no newline on the end of the line */
          ptr[column++] = '\n';
        }
      ptr[column] = '\0';
    }
    /*}}}*/
  return ptr;
}
/*}}}*/
#else
/* This version allows lines of arbitrary length */
PRIVATE char *linebuffer = NULL;
PRIVATE size_t linesize  = LINEMAX;
/*{{{  PRIVATE char *readsourceline*/
PRIVATE char *readsourceline(void)
{
  int src_col = 0;
  int dest_col = 0;
  int spaces = 0;
  int expandtabs = TRUE; /* expand only until first non-whitespace character */
  int c;
  char buf[LINEMAX];
  char *ptr;

  ptr = fgets(buf, LINEMAX, infile);
  if (ptr == NULL) return NULL;

  while (TRUE)
    {
      if (spaces != 0)
        {
          c = ' '; spaces--;
        }
      else
        c = buf[src_col++];
      switch(c)
        {
          case '\n': /* hit the end of a line */
            linebuffer[dest_col  ] = '\n';
            linebuffer[dest_col+1] = '\0';
            return linebuffer;
          case '\0': /* hit the end of a buffer before end of line */
            DEBUG_MSG(("readsourceline: reading more from file\n"));
            ptr = fgets(buf, LINEMAX, infile);
            if (ptr == NULL) return NULL;
            src_col = 0;
            break;
          case '\t':
            DEBUG_MSG(("readsourceline: found a tab\n"));
            if (expandtabs)
              {
                spaces = TABSIZE - (dest_col % TABSIZE);
                break; /* break out of the whole switch */
              }
            DEBUG_MSG(("readsourceline: dropping through from tab\n"));
            /* FALL THROUGH */
          default:
            expandtabs = FALSE; /* don't expand tabs once a char has been seen */
            /* FALL THROUGH */
          case ' ':  
            linebuffer[dest_col++] = c;
            if (dest_col >= linesize)
              {
                char *new = memalloc(linesize * 2 + 2); /* 2 extra for "\n\0" */
                DEBUG_MSG(("readsourceline: extending linebuffer to %ld\n", linesize*2));
                memcpy(new, linebuffer, linesize);
                memfree(linebuffer);
                linebuffer = new;
                linesize *= 2;
              }
            break;
        }
    }
}
/*}}}*/
#endif
/*{{{  PRIVATE void inputline ()*/
/*****************************************************************************
 *
 *  inputline reads a text line from the input file into a structure element
 *            of array buf.
 *
 *****************************************************************************/
PRIVATE void inputline ( void )
{
  char *ptr;
  /*{{{  read in line*/
  switch (lexmode)
    {
      case LEX_SOURCE:
        /* ptr = fgets(linebuffer, LINEMAX - 1, infile); */
        ptr = readsourceline();
        break;
      case LEX_SC:
      case LEX_LIB:
      case LEX_STDLIB:
        ptr = readdescriptorline(linebuffer);
        break;
    #ifndef CONFIG /* no #PRAGMA EXTERNAL when configuring */
      case LEX_EXTERNAL:
        ptr = readexternalline();
        break;
    #endif
      case LEX_PREDEFS:
        ptr = readpredefline(linebuffer);
        break;
    }
  /*}}}*/
  if (ptr == NULL)
    {
    #if 0  /* This changed 23/4/90 by CO'N */
      if ((lexmode == LEX_PREDEFS) || (lexmode == LEX_EXTERNAL) || feof(infile))
        endoffile = TRUE;
      else
        lexfatal_i(LEX_READ_ERROR, flocn, ferror(infile));
    #else
      if ((lexmode == LEX_SOURCE) && (!feof(infile)))
        lexfatal_i(LEX_READ_ERROR, flocn, ferror(infile));
      else
        endoffile = TRUE;
    #endif
      linebuffer_len = 0;
    }
  else
    /*linebuffer_len = strlen(ptr);*/
    linebuffer_len = calc_len_and_hash(ptr);
  linebuffer_ptr = ptr;
}
/*}}}*/
/*{{{  PRIVATE void readline ()*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  readline updates lineptr to point to a structure containing
 *           the next line of source, reading
 *           a line from the input file if neccessary.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void readline ( void )
{
  do
    {
      inputline();
      eof |= endoffile;
      linecount++;
      /*if (lexmode == LEX_SOURCE) totallines++;*/
      totallines += (lexmode == LEX_SOURCE);
    }
  while (!eof && (linebuffer_len == 0))
    ;
  linep = 0;
}
/*}}}*/
/*{{{  PRIVATE void bufinit ()*/
PRIVATE void bufinit ( void )
{
  endoffile = FALSE;
  readline ();
}
/*}}}*/
/*{{{  PRIVATE int rch ()*/
/* Read a character from the input file */
PRIVATE int rch ( void )
{
  if (linep >= linebuffer_len)
    readline();

  currentindent = linep + baseindent;
  return (eof ? EOF : linebuffer_ptr[linep++]);
}
/*}}}*/
/*{{{  PRIVATE int rsch()*/
/* Read a character from the input file, give error message if hit EOF */
PRIVATE int rsch ( void )
{
  const int c = rch();
  if (c == EOF)
    lexfatal(LEX_EOF, flocn);
  return (c);
}
/*}}}*/

/*{{{  PRIVATE void suspend_file()*/
/*****************************************************************************
 *
 *  suspend_file saves the current state of infile in the filestack.
 *
 *****************************************************************************/
PRIVATE void suspend_file ( char *name , int mode , FILE *new_fptr )
{
  if (infile != NULL)
    {
      filestruct_t *fptr = &(filestack[filestackptr]);
      fptr->fileptr = infile;
      strcpy(fptr->fname, filename);
      fptr->fmode = lexmode;
      SetFileNum(fptr->ffileposn, currentfilenum);
      SetFileLine(fptr->ffileposn, linecount);
      fptr->fbaseindent = baseindent;
      filestackptr++;
      if (filestackptr == FILESMAX)
        lexfatal_i(LEX_TOO_MANY_FILES, flocn, FILESMAX);
    }
  if (mode != LEX_EXTERNAL)
    {
      infile = new_fptr;
      strcpy (filename, name);
    }
  currentfilenum = addfilename(name, mode, currentfilenum,
                               FileLineOf(flocn));
  SetFileLine(flocn, 0);
  SetFileNum(flocn, currentfilenum);
}
/*}}}*/
/*{{{  PRIVATE void patchup()*/
PRIVATE void patchup()
{
  if (((lexmode == LEX_SC)     || (lexmode == LEX_LIB)    ||
       (lexmode == LEX_STDLIB) || (lexmode == LEX_EXTERNAL))
      && !onlylex)
    patchdescriptors(lexmode);
}
/*}}}*/
/*{{{  PRIVATE void resume_file()*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  resume_file resumes the top entry of the file stack
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void resume_file ( void )
{
  patchup();
  {
    filestruct_t *fptr;
    filestackptr = filestackptr - 1;
    fptr = &(filestack[filestackptr]);
    if (lexmode != LEX_EXTERNAL)
      fclose(infile);

    strcpy(filename, fptr->fname);
    infile = fptr->fileptr;

    /* We resume a line after where we suspended */
    lexmode = fptr->fmode;
    sentend = FALSE;
    eof = FALSE;
    endoffile = FALSE;
    bufinit();
    flocn = fptr->ffileposn;
    linecount = FileLineOf(flocn) + 1;
    currentfilenum = FileNumOf(flocn);
    baseindent = fptr->fbaseindent;
    symbindent = 0;
    lineindent = 0;
    newlineflag = TRUE;
    ch = rch();
  }
}
/*}}}*/
/*{{{  PRIVATE void close_file()*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  close_file closes the currently open file
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void close_file ( void )
{
  patchup();
  if (lexmode != LEX_PREDEFS)
    fclose(infile);
  infile = NULL;
  sentend = FALSE;
  eof = FALSE;
  endoffile = FALSE;
}
/*}}}*/
/*{{{  PUBLIC int open_file(name, mode)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  open_file opens the file of name 'name', in mode 'mode', returns TRUE
 *            if opened ok, otherwise returns FALSE.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC int open_file ( char *name , int mode, int fileindent )
{
  FILE *fptr = NULL;
  int external_ok = FALSE;
  if ((strlen(name) >= MAX_FILENAME_LENGTH) &&
      (mode != LEX_EXTERNAL)) /* bug 331 29/8/90 */
    lexerr_i(LEX_FILE_LENGTH_ERROR, flocn, MAX_FILENAME_LENGTH);
  else
    {
      switch(mode)
        {
          case LEX_SOURCE:
            {
              char full_name[MAX_FILENAME_LENGTH];
              fptr = popen_read(name, pathname, &full_name[0], POPEN_MODE_TEXT);
            }
            break;
          case LEX_SC:
          case LEX_LIB:
          case LEX_STDLIB:   fptr = open_descfile(name, mode);
                             break;
        #ifndef CONFIG /* No #PRAGMA EXTERNAL when configuring */
          case LEX_EXTERNAL: external_ok = init_external(name);
                             break;
        #endif
        }
    }
  if ((fptr != NULL) || external_ok)
    /*{{{  set up input buffers*/
    {
      suspend_file(name, mode, fptr);
      baseindent = fileindent;
      linecount = 0;
      lexmode = mode;
      eof = FALSE;
      endoffile = FALSE;
      desc_eof = FALSE;
      symbindent = 0;
      lineindent = 0;
      newlineflag = TRUE;
      stringcontinuation = FALSE;
      sentend = FALSE;
      bufinit ();
      ch = rch();
      return(TRUE);
    }
    /*}}}*/
  else
    return(FALSE);
}
/*}}}*/

#if 0
/*{{{  PUBLIC void savefiletable(file)*/
PUBLIC void savefiletable ( FILE *file )
{
  int i;
  fwrite(&numfiles, sizeof(int), 1, file);
  /*printf("Writing %d files :\n", numfiles);*/
  for (i = 0; i < numfiles; i++)
    {
      wordnode *w = filenames[i].ftname;
      fwrite(&filenames[i], sizeof (filetablestruct_t), 1, file);
      fwrite(w, sizeof (wordnode), 1, file);
      fwrite(WNameOf(w), WLengthOf(w), 1, file);
      /*printf("ftable[%d] = \"%s\"\n", i, WNameOf(w));*/
    }
}
/*}}}*/
#endif
#endif

#if 0
/*{{{  PUBLIC void readfiletable(file)*/
PUBLIC void readfiletable ( FILE *file )
{
  int i;
  fread(&numfiles, sizeof(int), 1, file);
  for (i = 0; i < numfiles; i++)
    {
      wordnode *w = (wordnode *)newvec(sizeof(wordnode));
      fread(&filenames[i], sizeof (filetablestruct_t), 1, file);
      fread(w, sizeof (wordnode), 1, file);
      SetWName(w, (char *)newvec(WLengthOf(w)+1));
      fread(WNameOf(w), WLengthOf(w), 1, file);
      WNameOf(w)[WLengthOf(w)]='\0';
      filenames[i].ftname = w;
    }
}
/*}}}*/
#endif
/*}}}*/

/*{{{  routines*/
#ifndef BACKEND
/*{{{  PRIVATE void readtag (name)*/
PRIVATE void readtag ( char *name )
{
  int i = 0;
  while (((ch >= 'a') && (ch <= 'z')) ||  /* most common first */
         ((ch >= 'A') && (ch <= 'Z')) ||
         ((ch >= '0') && (ch <= '9')) ||
         (ch == '.')                  ||
         ((lexmode == LEX_STDLIB) &&
          ((ch == '@') || (ch == '%') || (ch == '_') ||
           (ch == '|') || (ch == '$')                  )) )
    {
      if (i < (MAXNAMELENGTH - 1)) /* added for bug 1020 18/10/90 */
        name[i++] = ch;
      ch = rch ();
    }
  name[i] = '\0';
  if (i >= (MAXNAMELENGTH - 1)) /* added for bug 1020 18/10/90 */
    synwarn_i(LEX_NAME_TRUNCATED, flocn, MAXNAMELENGTH-1);
}
/*}}}*/
#endif


#ifndef BACKEND
/*{{{  PRIVATE void readdecimal ()*/
/* Read decimal digits from the input file and place them in
literalv from position literalp onwards.
Update literalp. */
PRIVATE void readdecimal ( void )
{
  if ((ch >= '0') && (ch <= '9'))
    {
      while ((ch >= '0') && (ch <= '9'))
        {
          if (literalp < (MAXSTRING_SIZE - 5)) /* bug 1020 18/10/90 */
            literalv[literalp++] = ch;
          ch = rch ();
        }
      /* we subtract 5 here to allow for sign characters, exponents, etc */
      if (literalp >= (MAXSTRING_SIZE - 5))
        synwarn_i(LEX_NUMBER_TRUNCATED, flocn, MAXSTRING_SIZE - 5);
    }
  else
    {
      lexerr_i(LEX_NUMBER_ERROR, flocn, ch);
      literalv[literalp++] = '0';
    }
}
/*}}}*/
/*{{{  PRIVATE void rexponent ()*/
/* Read an exponent from the input file and place it in literalv
from literalp onwards */
PRIVATE void rexponent ( void )
{
  literalv[literalp++] = 'E';
  ch = rch ();
  if ((ch == '+') || (ch == '-'))
    {
      literalv[literalp++] = ch;
      ch = rch ();
    }
  else
    {
      lexerr(LEX_SIGN_ERROR, flocn);
      /*lexerror (LEX_NUMBER_ERROR, flocn, ZERO32);*/
      literalv[literalp++] = '+';
    }
  readdecimal ();
}
/*}}}*/
/*{{{  PRIVATE int hexval ()*/
/* Return the hexadecimal value of the ASCII hex digit in ch */
PRIVATE int hexval ( void )
{
  if ((ch >= '0') && (ch <= '9'))
    return (ch - '0');
  if ((ch >= 'A') && (ch <= 'F'))
    return (ch - 'A' + 10);
  lexerr_i(LEX_E_HEXDIGIT, flocn, ch);
  return (0);
}
/*}}}*/
/*{{{  PRIVATE int rescapech ()*/
PRIVATE int rescapech ( void )
{
  if (ch < MINLEGALCH || ch > MAXLEGALCH)
    /*{{{  illegal character*/
    {
      lexerr(LEX_CHAR_ERROR, flocn);
      return(ch);
    }
    /*}}}*/
  else if (ch == '*')
    /*{{{  look for special character*/
    {
      switch (ch = rsch())
        {
          /*{{{  c C*/
          case 'c': case 'C':
            return ('\r');
          /*}}}*/
          /*{{{  n N*/
          case 'n': case 'N':
            return ('\n');
          /*}}}*/
          /*{{{  t T*/
          case 't': case 'T':
            return ('\t');
          /*}}}*/
          /*{{{  s S*/
          case 's': case 'S':
            return (' ');
          /*}}}*/
          /*{{{  '*/
          case '\'':
            return ('\'');
          /*}}}*/
          /*{{{  "*/
          case '"':
            return ('"');
          /*}}}*/
          /*{{{  **/
          case '*':
            return ('*');
          /*}}}*/
          /*{{{  #*/
          case '#':
            {
              int h;
              ch = rsch();
              h = hexval();
              ch = rsch();
              return ((h << 4) | hexval());
            }
          /*}}}*/
          default:
            lexerr_i(LEX_ILLEGAL_STRING_CHAR, flocn, ch);
            return (ch);
        };
    };
    /*}}}*/
  return (ch);
}
/*}}}*/
/*{{{  PRIVATE int rsescapech ()*/
/* Read escape character present in a string */
PRIVATE int rsescapech ( void )
{
  if (ch < MINLEGALCH || ch > MAXLEGALCH || ch == '\'')
    /*{{{  illegal character in string*/
    {
      lexerr_i(LEX_STRING_ERROR, flocn, ch);
      /*return(ch);*/
      return CLOSE_QUOTE;
    }
    /*}}}*/
  else if (ch == '*')
    /*{{{  look for special character*/
    {
      switch (ch = rsch())
        {
          /*{{{  c C*/
          case 'c': case 'C':
            return ('\r');
          /*}}}*/
          /*{{{  n N*/
          case 'n': case 'N':
            return ('\n');
          /*}}}*/
          /*{{{  t T*/
          case 't': case 'T':
            return ('\t');
          /*}}}*/
          /*{{{  s S*/
          case 's': case 'S':
            return (' ');
          /*}}}*/
          /*{{{  '*/
          case '\'':
            return ('\'');
          /*}}}*/
          /*{{{  "*/
          case '"':
            return ('"');
          /*}}}*/
          /*{{{  **/
          case '*':
            return ('*');
          /*}}}*/
          /*{{{  #*/
          case '#':
            {
              int h;
              ch = rsch();
              h = hexval();
              ch = rsch();
              h = (h << 4) | hexval();
              return h;
            }
          /*}}}*/
          /*{{{  '\n'*/
          case '\n':
            {
              const int indent = lineindent;
              ch = rsch();
            #if 0
              while (ch == ' ') ch = rsch();
              SetFileLine(flocn, linecount);
            #else /* allow blank lines and comments - bug 1103 10/1/91 */
              {
                int repeating;
                do
                  {
                    repeating = FALSE;
                    while ((ch == ' ') || (ch == '\n')) ch = rsch(); /* bug 1103 10/1/91 */
                    SetFileLine(flocn, linecount);
                    if (ch == '-') /* check for comments */
                      {
                        /*{{{  check indent of comment */
                        symbindent = currentindent;
                        lineindent = symbindent;
                        if (lineindent < indent)
                          lexerr(LEX_BAD_INDENT, flocn);
                        /*}}}*/
                        ch = rsch();
                        if (ch != '-')
                          lexerr(LEX_BAD_STRING_CONTINUATION, flocn);
                        while (ch != '\n') ch = rsch();
                        repeating = TRUE;
                      }
                  }
                while (repeating);
              }
            #endif
              if (ch != '*')
                lexerr(LEX_BAD_STRING_CONTINUATION, flocn);
              /*{{{  check indent of string continuation*/
              symbindent = currentindent;
              lineindent = symbindent;
              if (lineindent < indent)
                lexerr(LEX_BAD_INDENT, flocn);
              /*}}}*/
              ch = rsch();
              return(rsescapech());
            }
          /*}}}*/
          /*{{{  l L*/
          case 'l': case 'L':
            if (!stringcontinuation && (literalp == 0))
              {
                insertlength = TRUE;
                return (' ');
              }
            else
              {
                lexerr_i(LEX_BAD_STRING_LENGTH_POSN, flocn, ch);
                return ('?');
              }
          /*}}}*/
          default:
            lexerr_i(LEX_ILLEGAL_STRING_CHAR, flocn, ch);
            return (ch);
        }
    }
    /*}}}*/
  else if (ch == '"')
    return(CLOSE_QUOTE);
  else
    return(ch);
}
/*}}}*/
/*{{{  PRIVATE int readnumber ()*/
/* Read a number from the input file.
The number may be an integer or a real.
A string representing the number is placed in literalv.
Return S_UREALLIT or S_UINTLIT */
PRIVATE int readnumber ( void )
{
  literalp = 0;
  readdecimal ();
  if (ch == '.')
    /*{{{  read rest of decimal number*/
    {
      literalv[literalp++] = '.';
      ch = rch ();
      readdecimal ();
      if (ch == 'E')
        rexponent ();
      literalv[literalp] = '\0';
      return (S_UREALLIT);
    }
    /*}}}*/
  else
    /*{{{  return integer*/
    {
      literalv[literalp] = '\0';
      return (S_UINTLIT);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE int readhex ()*/
PRIVATE int readhex ( void )
{
  while (((ch >= '0') && (ch <= '9')) ||
         ((ch >= 'A') && (ch <= 'F')))
    {
      if (literalp < (MAXSTRING_SIZE - 1)) /* bug 1020 18/10/90 */
        literalv[literalp++] = ch;
      ch = rch ();
    };
  if (literalp == 1)
    {
      lexerr_i(LEX_E_HEXDIGIT, flocn, ch);
      /*lexerror(LEX_NUMBER_ERROR, flocn, ZERO32);*/
      return (S_ILLEGALSYMB);
    }
  else if (literalp >= (MAXSTRING_SIZE - 1))
    synwarn_i(LEX_NUMBER_TRUNCATED, flocn, MAXSTRING_SIZE);
  literalv[literalp] = '\0';
  return (S_UINTLIT);
}
/*}}}*/
/*{{{  PRIVATE int rhashword (matchstring, s)*/
/* Read characters from the input stream and ensure that they match successive
characters of the string 'matchstring'.  If the whole of matchstring is
matched return 's', otherwise return S_ILLEGALSYMB. */
PRIVATE int rhashword ( char *matchstring , int s )
{
  ch = rch();
  while (*matchstring != '\0' && ch == *matchstring)
    {
      ch = rch();
      matchstring++;
    }
  if (*matchstring == '\0')
    return s;
  else
    {
      lexerr(LEX_HASHWORD_ERROR, flocn);
      return S_ILLEGALSYMB;
    }
}
/*}}}*/
/*{{{  PRIVATE int readbyteliteral ()*/
/* Read a byte literal string from the input file and place it in literalv.
Return S_UBYTELIT */
PRIVATE int readbyteliteral ( void )
{
  literalv[0] = '\'';
  literalv[1] = (ch = rescapech());
  literalv[2] = '\'';
  literalv[3] = '\0';
  literalp = 3;
  ch = rsch ();
  if (ch == '\'')
    ch = rch ();
  else
    {
      symbindent = currentindent;
      lexerr_i(LEX_E_QUOTE, flocn, ch);
      return S_ILLEGALSYMB;
    }
  return (S_UBYTELIT);
}
/*}}}*/
/*{{{  PRIVATE int readstringliteral ()*/
PRIVATE int readstringliteral ( void )
{
  insertlength = FALSE;
  literalp = 0;

  if (!stringcontinuation)  /* ch has already been escaped */
    ch = rsescapech();
  stringcontinuation = FALSE;

  while ((ch != CLOSE_QUOTE) && (literalp < MAXSTRING_SIZE))
    /*{{{  read a character into literalp*/
    {
      literalv[literalp++] = ch;
      ch = rsch();
      ch = rsescapech();
    }
    /*}}}*/
  literalv[literalp] = '\0';
  stringcontinuation = (ch != CLOSE_QUOTE);
  if (insertlength)
    /*{{{  insert the length if the string is short enough*/
    {
      if (!stringcontinuation)
        literalv[0] = literalp - 1;
      else
        {
          lexerr(LEX_BAD_STRING_LENGTH, flocn);
          literalv[0] = '?';
        }
    }
    /*}}}*/

  if (stringcontinuation)
    return (S_STRINGCONT);
  else
    /*{{{  read another character, return S_STRING*/
    {
      ch = rch ();
      return(S_STRING);
    }
    /*}}}*/
}
/*}}}*/
#endif

/*{{{  PUBLIC void lexinit ()*/
PUBLIC void lexinit ( void )
{
  int i;

  /* Initialise name table */
  for (i = 0; i < HASHSIZE; i++)
    nametable[i] = NULL;

  declkeywords ();
#ifndef BACKEND
  filestackptr = 0;
  baseindent = 0;
  numfiles = 0;
  currentfilenum = (-1);
  allow_asmnames = FALSE;

  /*{{{  set up for reading predefines*/
  lexmode = LEX_PREDEFS;
  strcpy(filename, "predefined names");
  linecount = 0;
  totallines = 0;
  eof = FALSE;
  endoffile = FALSE;
  desc_eof = FALSE;
  linebuffer = memalloc(linesize + 2); /* 2 extra for "\n\0" */
  bufinit ();
  symbindent = 0;
  lineindent = 0;
  SetFileLine(flocn, 0);
  SetFileNum(flocn, currentfilenum);
  newlineflag = TRUE;
  stringcontinuation = FALSE;
  sentend = FALSE;
  ch = rch();
  /*}}}*/
#endif
}
/*}}}*/
/*{{{  lexfinish*/
PUBLIC void lexfinish(void)
{
#ifdef DEBUG
  int i, tot = 0, m = 0, totlen = 0;
  fprintf(outfile, "Nametable hash table:\n");
  for (i = 0; i < HASHSIZE; i++)
    {
      int count = 0;
      wordnode *wptr = nametable[i];
      while (wptr != NULL)
        {
          count++;
          totlen += WLengthOf(wptr);
          wptr = WNextOf(wptr);
        }
      tot += count;
      m = max(count, m);
      fprintf(outfile, "%3d->%3d,%s", i, count, ((i % 8) == 7) ? "\n" : "");
    }
  fprintf(outfile, "Total is: %d, Average is %d, max is %d\n",
          tot, tot/HASHSIZE, m);
  fprintf(outfile, "Total length is: %d, Average is %d\n", totlen, totlen/tot);
#endif
}
/*}}}*/
/*}}}*/

#ifndef BACKEND
#ifdef DEBUG_NEXTSYMB
  #define nextsymb local_nextsymb
#endif
/*{{{  PUBLIC void nextsymb ()*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  nextsymb sets symb and symbindent to value and indentation of next symbol
 *           from source file.  If we start a new line, lineindent is set
 *           to the indentation of the first symbol on the line.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void nextsymb ( void )
{
  DEBUG_MSG(("nextsymb... "));
  while (TRUE)
    /*{{{  find the next symbol*/
    {
      if (sentend)
        lexfatal(LEX_EOF, flocn);
    
      if (newlineflag)
        /*{{{  skip leading spaces and set lineindent*/
        {
          newlineflag = FALSE;
          while (ch == ' ')  /* skip leading spaces */
            ch = rch();
          lineindent = currentindent;
        }
        /*}}}*/
    
      symbindent = currentindent;
      SetFileLine(flocn, linecount);
    
      if (stringcontinuation)
        /*{{{  read the continuing string*/
        {
          symb = readstringliteral();
          return;
        }
        /*}}}*/
      else
        /*{{{  switch on ch*/
        switch (ch)
          {
            /*{{{  newline                              return*/
            case '\n':
              ch = rch ();
              newlineflag = TRUE;
              symb = S_NEWLINE;
              return;
            /*}}}*/
            /*{{{  space                        break*/
            case ' ':
              while ((ch = rch()) == ' ')
                ;  /* SKIP */
              break;
            case '\t':
              ch = rch();
              break;
            /*}}}*/
            /*{{{  0 1 2 3 4 5 6 7 8 9                  return*/
            case '0': case '1': case '2': case '3': case '4':
            case '5': case '6': case '7': case '8': case '9':
              symb = readnumber();
              return;
            /*}}}*/
            /*{{{  a - z  A - Z                         return*/
            /*{{{  case a - z A - Z*/
            case 'a': case 'b': case 'c': case 'd': case 'e':
            case 'f': case 'g': case 'h': case 'i': case 'j':
            case 'k': case 'l': case 'm': case 'n': case 'o':
            case 'p': case 'q': case 'r': case 's': case 't':
            case 'u': case 'v': case 'w': case 'x': case 'y':
            case 'z':
            case 'A': case 'B': case 'C': case 'D': case 'E':
            case 'F': case 'G': case 'H': case 'I': case 'J':
            case 'K': case 'L': case 'M': case 'N': case 'O':
            case 'P': case 'Q': case 'R': case 'S': case 'T':
            case 'U': case 'V': case 'W': case 'X': case 'Y':
            case 'Z':
            /*}}}*/
              {
                char name [MAXNAMELENGTH];
                readtag(name);
                lexword = lookupword (name, strlen(name));
                symb = WTagOf(lexword);
                return;
              }
            /*}}}*/
            /*{{{  dot                                  return*/
            case '.':  /* All special ASM names begin with dot */
              if (allow_asmnames)
                {  /* simply read a name */
                  char name [MAXNAMELENGTH];
                  readtag(name);
                  lexword = lookupword (name, strlen(name));
                  symb = WTagOf(lexword);
                }
              else
                {
                  ch = rch ();
                  symb = S_DOT;
                }
              return;
            /*}}}*/
            /*{{{  ( ) [ ] + , = ! ? ~ & ;              return*/
            case '(': ch = rch (); symb = S_LPAREN; return;
            case ')': ch = rch (); symb = S_RPAREN; return;
            case ']': ch = rch (); symb = S_RBOX; return;
            case '+': ch = rch (); symb = S_ADD; return;
            case ',': ch = rch (); symb = S_COMMA; return;
            case '=': ch = rch (); symb = S_EQ; return;
            case '!': ch = rch (); symb = S_OUTPUT; return;
            case '?': ch = rch (); symb = S_INPUT; return;
            case '~': ch = rch (); symb = S_BITNOT; return;
            case '&': ch = rch (); symb = S_AMPERSAND; return;
            case ';': ch = rch (); symb = S_SEMICOLON; return;
            /*}}}*/
            /*{{{  *                                    return*/
            case '*':
              #if 0  /* we've checked for this outside the switch */
              if (stringcontinuation)
                {
                  ch = rsch ();
                  symb = readstringliteral();
                  return;
                }
              else
              #endif
                {
                  ch = rch ();
                  symb = S_MULT;
                  return;
                }
            /*}}}*/
            /*{{{  [                                    return*/
            case '[':
              ch = rch ();
              if (ch == ']')
                {
                  ch = rch ();
                  symb = S_BOX;
                  return;
                }
              symb = S_LBOX;
              return;
            /*}}}*/
            /*{{{  :                                    return*/
            case ':':
              ch = rch ();
              if (ch == '=')
                {
                  ch = rch ();
                  symb = S_ASS;
                  return;
                };
              if (ch == ':')
                {
                  ch = rch ();
                  symb = S_COLON2;
                  return;
                };
              symb = S_COLON;
              return;
            /*}}}*/
            /*{{{  $                                    return*/
            case '$':
              literalv[0] = '#';
              literalp = 1;
              ch = rch();
              symb = readhex();
              return;
            /*}}}*/
            /*{{{  #                                    return*/
            case '#':
              literalv[0] = '#';
              literalp = 1;
              switch(ch = rch())
                {
                  /*{{{  0-9, A, B, D, E, F must be a hex number*/
                  case '0': case '1': case '2': case '3': case '4':
                  case '5': case '6': case '7': case '8': case '9':
                  case 'A': case 'B': case 'D': case 'E': case 'F':
                    symb = readhex();
                    return;
                  /*}}}*/
                  /*{{{  C  hex number or   #COMMENT*/
                  case 'C':
                    if ((ch = rch ()) == 'O')
                      symb = rhashword("MMENT", S_HCOMMENT);
                    else
                      {
                        literalv[literalp++] = 'C';
                        symb = readhex();
                      }
                    return;
                  /*}}}*/
                  /*{{{  E  hex number or   #EXTERNAL*/
                  /* #EXTERNAL replaced by #PRAGMA
                  case 'E':
                    if ((ch = rch()) == 'X')
                      symb = rhashword("TERNAL", S_EXTERNAL);
                    else
                      {
                        literalv[literalp++] = 'E';
                        symb = readhex();
                      }
                    return; */
                  /*}}}*/
                  /*{{{  I                  #INCLUDE or #IMPORT*/
                  case 'I':
                    ch = rch();
                    if (ch == 'N')
                      symb = rhashword("CLUDE", S_INCLUDE);
                    else if (ch == 'M')
                      symb = rhashword("PORT", S_IMPORT);
                    else
                      {
                        lexerr(LEX_HASHWORD_ERROR, flocn);
                        symb = S_ILLEGALSYMB;
                      }
                    return;
                  /*}}}*/
                  /*{{{  O                  #OPTION*/
                  case 'O':
                    symb = rhashword("PTION", S_OPTION);
                    return;
                  /*}}}*/
                  /*{{{  P                  #PRAGMA*/
                  case 'P':
                    symb = rhashword("RAGMA", S_PRAGMA);
                    return;
                  /*}}}*/
                  /*{{{  S                  #SC*/
                  case 'S':
                    symb = rhashword("C", S_SC);
                    return;
                  /*}}}*/
                  /*{{{  U                  #USE*/
                  case 'U':
                    symb = rhashword("SE", S_USE);
                    return;
                  /*}}}*/
                  /*{{{  a, b, c, d, e, f   badly formed hex number*/
                  case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
                    lexerr_i(LEX_E_HEXDIGIT, flocn, ch);
                    /*lexerror(LEX_NUMBER_ERROR, flocn, ZERO32);*/
                    symb = S_ILLEGALSYMB;
                    return;
                  /*}}}*/
                  default:
                    lexerr(LEX_HASHWORD_ERROR, flocn);
                    symb = S_ILLEGALSYMB;
                    return;
                }
            /*}}}*/
            /*{{{  -                                    return/break*/
            case '-':
              ch = rsch ();
              if (ch == '-')  /* Read past comment */
                {
                  literalp = 0;
                  ch = rsch ();
                  while (ch != '\n')
                    {
                      /* don't copy into literalv, so it can't overflow:
                         bug 1020 18/10/90 */
                      /* literalv[literalp++] = ch; */
                      ch = rsch ();
                    };
                  if (!lexer_ignore_comments) /* bug 853 18/1/91 */
                    { symb = S_COMMENT; return; }
                  /* otherwise drop round loop */
                }
            #ifdef CONDEXP
              else if (ch == '>')
                { ch = rsch(); symb = S_RIGHTARROW; return; }
            #endif
              else
                { symb = S_SUBTRACT; return; }
              break;
            /*}}}*/
            /*{{{  /                                    return*/
            case '/':
              if ((ch = rch()) == '\\')
                {
                  ch = rch();
                  symb = S_BITAND;
                  return;
                };
              symb = S_DIV;
              return;
            /*}}}*/
            /*{{{  \                                    return*/
            case '\\':
              if ((ch = rch()) == '/')
                {
                  ch = rch ();
                  symb = S_BITOR;
                  return;
                };
              symb = S_REM;
              return;
            /*}}}*/
            /*{{{  <                                    return*/
            case '<':
              switch (ch = rch())
                {
                  case '<': ch = rch (); symb = S_LSHIFT; return;
                  case '=': ch = rch (); symb = S_LE; return;
                  case '>': ch = rch (); symb = S_NE; return;
                  default:  symb = S_LS; return;
                };
            /*}}}*/
            /*{{{  >                                    return*/
            case '>':
              switch (ch = rch())
                {
                  case '<': ch = rch (); symb = S_XOR; return;
                  case '=': ch = rch (); symb = S_GE; return;
                  case '>': ch = rch (); symb = S_RSHIFT; return;
                  default:  symb = S_GR; return;
                };
            /*}}}*/
            /*{{{  '                                    return*/
            case '\'':
              ch = rsch ();
              symb = readbyteliteral();
              return;
            /*}}}*/
            /*{{{  "                                    return*/
            case '"':
              ch = rsch ();
              symb = readstringliteral();
              return;
            /*}}}*/
            /*{{{  EOF                                  return*/
            case EOF:
              if (filestackptr > 0)
                {
                  resume_file();
                  break;
                }
              else
                {
                  close_file();
                  symb = S_END;
                  sentend = TRUE;
                  return;
                }
            /*}}}*/
            /*{{{  default                      break*/
            default:
              lexerr_i(LEX_ILLEGAL_CHAR, flocn, ch);
              symb = S_ILLEGALSYMB;
              ch = rch ();
              DEBUG_MSG(("nextsymb: looping for next char... "));
              break;
            /*}}}*/
          }
        /*}}}*/
    }
    /*}}}*/
}
/*}}}*/
/*{{{  lextest routines*/
/*{{{  PUBLIC void printlex ()*/
PUBLIC void printlex ( void )
{
#ifdef DEBUG_NEXTSYMB
  fprintf(outfile, "%d:", symbindent);
#endif
  switch(symb)
    {
      default:
        fprintf(outfile, "%s ", tagstring(symb));
        break;
      /*{{{  S_NEWLINE;*/
      case S_NEWLINE:
        fprintf(outfile, "NEWLINE\n");
        break;
      /*}}}*/
      /*{{{  S_END*/
      case S_END:
        fprintf(outfile, "END\n");
        break;
      /*}}}*/
      /*{{{  S_UBYTELIT*/
      case S_UBYTELIT:
        fprintf(outfile,"BYTELIT:");
        printbyte (literalv[1]);
        break;
      /*}}}*/
      /*{{{  S_INTLIT*/
      case S_INTLIT:
        fprintf (outfile, "INTLIT:");
        fprintf (outfile,"%.*s ", literalp, literalv);
        break;
      /*}}}*/
      /*{{{  S_UINTLIT*/
      case S_UINTLIT:
        fprintf (outfile, "UINTLIT:");
        fprintf (outfile,"%.*s ", literalp, literalv);
        break;
      /*}}}*/
      /*{{{  S_NAME*/
      case S_NAME:
        fprintf (outfile, "NAME:");
        fprintf (outfile,"%s ", WNameOf(lexword));
        break;
      /*}}}*/
      /*{{{  S_STRING*/
      case S_STRING:
        fprintf (outfile, "STRING:");
        printstring (literalv, literalp);
        break;
      /*}}}*/
      /*{{{  S_STRINGCONT*/
      case S_STRINGCONT:
        fprintf(outfile, "STRINGCONT:");
        printstring (literalv, literalp);
        break;
      /*}}}*/
      /*{{{  S_UREALLIT*/
      case S_UREALLIT:
        fprintf (outfile, "UREALLIT:");
        fprintf (outfile,"%.*s ", literalp, literalv);
        break;
      /*}}}*/
      /*{{{  S_COMMENT*/
      case S_COMMENT:
        fprintf(outfile, "-- %.*s", literalp, literalv);
        break;
      /*}}}*/
    }
}
/*}}}*/
#ifdef DEBUG_NEXTSYMB
#undef nextsymb
PUBLIC void nextsymb ( void )
{
  local_nextsymb();
  printlex();
}
#endif
/*}}}*/
#endif

