/*****************************************************************************
 *
 *  Occam 2 compiler  inst.c Instruction handling for GUY code inserts
 *
 *****************************************************************************/

/*{{{  copyright*/
/*****************************************************************************
 *
 *  occam 2 compiler
 *
 *  copyright Inmos Limited 1987
 *
 *****************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "instdef.h"
# include "instruct.h"
/*}}}*/

/*{{{  definitions*/
#define N_PRIMARIES 16
/*}}}*/
/*{{{  private variables*/
PRIVATE int trueval = TRUE;
PRIVATE char instname[20];
/*}}}*/
/*{{{  instruction table*/
#define GUYS_ATTR (0x1000) /* Set if only allowed in FULL GUYs */
#define PTRS_ATTR (0x0100) /* Set if this allows ADDRESSOF in ASM */
#define ELEM_ATTR (0x0010) /* Set if this is a store */
#define OPRD_ATTR (0x000F) /* Count of expected number of operands */
/* (when = OPRD_ATTR, it means 'any number' */
PRIVATE struct instruction_struct
  {
    INT32 ivalue;
    char *iname;
    int  *ivalid;
    int   iattr; /* 0xGABC, where A=ptrs_allowed,
                                  B=element_required, C=operands,
                                  G=Full guys/asm only */
  } instructions[] =
  {
    /*{{{  primaries*/
    { I_LDL     | I_PRIMARY, "LDL",  &trueval, 1 },
    { I_STL     | I_PRIMARY, "STL",  &trueval, 1 },
    { I_LDLP    | I_PRIMARY, "LDLP", &trueval, 1 },
    { I_LDNL    | I_PRIMARY, "LDNL", &trueval, 1 },
    { I_STNL    | I_PRIMARY, "STNL", &trueval, 1 },
    { I_LDNLP   | I_PRIMARY, "LDNLP",&trueval, 1 },
    { I_EQC     | I_PRIMARY, "EQC",  &trueval, 1 },
    { I_LDC     | I_PRIMARY, "LDC",  &trueval, 1 },
    { I_ADC     | I_PRIMARY, "ADC",  &trueval, 1 },
    { I_J       | I_PRIMARY, "J",    &trueval, 1 },
    { I_CJ      | I_PRIMARY, "CJ",   &trueval, 1 },
    { I_CALL    | I_PRIMARY, "CALL", &trueval, GUYS_ATTR | 1 },
    { I_AJW     | I_PRIMARY, "AJW",  &trueval, GUYS_ATTR | 1 },
    { I_PFIX    | I_PRIMARY, "PFIX", &trueval, GUYS_ATTR | 1 },
    { I_NFIX    | I_PRIMARY, "NFIX", &trueval, GUYS_ATTR | 1 },
    { I_OPR     | I_PRIMARY, "OPR",  &trueval, GUYS_ATTR | 1 },
    /*}}}*/
    /*{{{  secondaries*/
    /*{{{  rev, ret, ldpi, gajw, gcall, mint, lend*/
    { I_REV,   "REV",   &trueval, 0 },
    { I_RET  , "RET",   &trueval, GUYS_ATTR },
    { I_LDPI,  "LDPI",  &trueval, 0 },
    { I_GAJW , "GAJW",  &trueval, GUYS_ATTR },
    { I_GCALL, "GCALL", &trueval, GUYS_ATTR },
    { I_MINT,  "MINT",  &trueval, 0 },
    { I_LEND , "LEND",  &trueval, GUYS_ATTR },
    /*}}}*/
    /*{{{  csub0, ccnt1*/
    { I_CSUB0,   "CSUB0",   &trueval, 0 },
    { I_CCNT1,   "CCNT1",   &trueval, 0 },
    /*}}}*/
    /*{{{  testerr, seterr, stoperr, clrhalterr, sethalterr, testhalterr*/
    { I_TESTERR,    "TESTERR",     &trueval, 0 },
    { I_SETERR,     "SETERR",      &trueval, 0 },
    { I_STOPERR   , "STOPERR",     &trueval, GUYS_ATTR },
    { I_CLRHALTERR, "CLRHALTERR",  &trueval, GUYS_ATTR },
    { I_SETHALTERR, "SETHALTERR",  &trueval, GUYS_ATTR },
    { I_TESTHALTERR,"TESTHALTERR", &trueval, 0 },
    /*}}}*/
    /*{{{  bsub, wsub, bcnt, wcnt*/
    { I_BSUB,    "BSUB",    &trueval, 0 },
    { I_WSUB,    "WSUB",    &trueval, 0 },
    { I_BCNT,    "BCNT",    &trueval, 0 },
    { I_WCNT,    "WCNT",    &trueval, 0 },
    /*}}}*/
    /*{{{  lb, sb, move*/
    { I_LB,      "LB",      &trueval, 0 },
    { I_SB,      "SB",      &trueval, 0 },
    { I_MOVE,    "MOVE",    &trueval, 0 },
    /*}}}*/
    /*{{{  and, or, xor, not, shl, shr, add, sub, mul, div, rem, gt, diff, sum, prod*/
    { I_AND,     "AND",     &trueval, 0 },
    { I_OR,      "OR",      &trueval, 0 },
    { I_XOR,     "XOR",     &trueval, 0 },
    { I_NOT,     "NOT",     &trueval, 0 },
    { I_SHL,     "SHL",     &trueval, 0 },
    { I_SHR,     "SHR",     &trueval, 0 },
    { I_ADD,     "ADD",     &trueval, 0 },
    { I_SUB,     "SUB",     &trueval, 0 },
    { I_MUL,     "MUL",     &trueval, 0 },
    { I_DIV,     "DIV",     &trueval, 0 },
    { I_REM,     "REM",     &trueval, 0 },
    { I_GT,      "GT",      &trueval, 0 },
    { I_DIFF,    "DIFF",    &trueval, 0 },
    { I_SUM,     "SUM",     &trueval, 0 },
    { I_PROD,    "PROD",    &trueval, 0 },
    /*}}}*/
    /*{{{  startp, endp, runp, stopp, ldpri*/
    { I_STARTP,   "STARTP",  &trueval, GUYS_ATTR },
    { I_ENDP  ,   "ENDP",    &trueval, GUYS_ATTR },
    { I_RUNP  ,   "RUNP",    &trueval, GUYS_ATTR },
    { I_STOPP ,   "STOPP",   &trueval, GUYS_ATTR },
    { I_LDPRI,    "LDPRI",   &trueval, 0 },
    /*}}}*/
    /*{{{  in, out, outword, outbyte, resetch*/
    { I_IN     , "IN",      &trueval, GUYS_ATTR },
    { I_OUT    , "OUT",     &trueval, GUYS_ATTR },
    { I_OUTWORD, "OUTWORD", &trueval, GUYS_ATTR },
    { I_OUTBYTE, "OUTBYTE", &trueval, GUYS_ATTR },
    { I_RESETCH, "RESETCH", &trueval, GUYS_ATTR },
    /*}}}*/
    /*{{{  alt, altwt, altend, enbs, diss, enbc, ldtimer, tin, talt, taltwt,enbt,dist*/
    { I_ALT    , "ALT",     &trueval, GUYS_ATTR },
    { I_ALTWT  , "ALTWT",   &trueval, GUYS_ATTR },
    { I_ALTEND , "ALTEND",  &trueval, GUYS_ATTR },
    { I_ENBS   , "ENBS",    &trueval, GUYS_ATTR },
    { I_DISS   , "DISS",    &trueval, GUYS_ATTR },
    { I_ENBC   , "ENBC",    &trueval, GUYS_ATTR },
    { I_DISC   , "DISC",    &trueval, GUYS_ATTR },
    { I_LDTIMER, "LDTIMER", &trueval, 0 },
    { I_TIN    , "TIN",     &trueval, GUYS_ATTR },
    { I_TALT   , "TALT",    &trueval, GUYS_ATTR },
    { I_TALTWT , "TALTWT",  &trueval, GUYS_ATTR },
    { I_ENBT   , "ENBT",    &trueval, GUYS_ATTR },
    { I_DIST   , "DIST",    &trueval, GUYS_ATTR },
    /*}}}*/
    /*{{{  xword, cword, xdble, csngl, norm*/
    { I_XWORD,   "XWORD",   &trueval, 0 },
    { I_CWORD,   "CWORD",   &trueval, 0 },
    { I_XDBLE,   "XDBLE",   &trueval, 0 },
    { I_CSNGL,   "CSNGL",   &trueval, 0 },
    { I_NORM,    "NORM",    &trueval, 0 },
    /*}}}*/
    /*{{{  ladd, lsub, lsum, ldiff, lmul, ldiv, lshl,lshr*/
    { I_LADD,    "LADD",    &trueval, 0 },
    { I_LSUB,    "LSUB",    &trueval, 0 },
    { I_LSUM,    "LSUM",    &trueval, 0 },
    { I_LDIFF,   "LDIFF",   &trueval, 0 },
    { I_LMUL,    "LMUL",    &trueval, 0 },
    { I_LDIV,    "LDIV",    &trueval, 0 },
    { I_LSHL,    "LSHL",    &trueval, 0 },
    { I_LSHR,    "LSHR",    &trueval, 0 },
    /*}}}*/
    /*{{{  testpranal, saveh, savel, sthf, sthb, stlf, stlb, sttimer*/
    { I_TESTPRANAL, "TESTPRANAL", &trueval, 0 },
    { I_SAVEH,      "SAVEH",      &trueval, GUYS_ATTR },
    { I_SAVEL,      "SAVEL",      &trueval, GUYS_ATTR },
    { I_STHF ,      "STHF",       &trueval, GUYS_ATTR },
    { I_STHB ,      "STHB",       &trueval, GUYS_ATTR },
    { I_STLF ,      "STLF",       &trueval, GUYS_ATTR },
    { I_STLB ,      "STLB",       &trueval, GUYS_ATTR },
    { I_STTIMER,    "STTIMER",    &trueval, 0 },
    /*}}}*/
    /*{{{  unpacksn, roundsn, postnormsn, ldinf, cflerr*/
    { I_UNPACKSN,   "UNPACKSN",   &fpsupport, 0 },
    { I_ROUNDSN,    "ROUNDSN",    &fpsupport, 0 },
    { I_POSTNORMSN, "POSTNORMSN", &fpsupport, 0 },
    { I_LDINF,      "LDINF",      &fpsupport, 0 },
    { I_CFLERR,     "CFLERR",     &fpsupport, 0 },
    /*}}}*/
    /*{{{  fmul*/
    { I_FMUL,    "FMUL",    &fracmul, 0 },
    /*}}}*/
    /*{{{  start, testhardchan, testldd, testlde, testlds, teststd, testste, teststs*/
    { I_START       , "START",        &trueval, GUYS_ATTR },
    { I_TESTHARDCHAN, "TESTHARDCHAN", &trueval, GUYS_ATTR },
    { I_TESTLDD     , "TESTLDD",      &trueval, GUYS_ATTR },
    { I_TESTLDE     , "TESTLDE",      &trueval, GUYS_ATTR },
    { I_TESTLDS     , "TESTLDS",      &trueval, GUYS_ATTR },
    { I_TESTSTD     , "TESTSTD",      &trueval, GUYS_ATTR },
    { I_TESTSTE     , "TESTSTE",      &trueval, GUYS_ATTR },
    { I_TESTSTS     , "TESTSTS",      &trueval, GUYS_ATTR },
    /*}}}*/
    /*{{{  move2dinit, move2dall, move2dnonzero, move2dzero*/
    { I_MOVE2DINIT,    "MOVE2DINIT",    &graphicsmove, 0 },
    { I_MOVE2DALL,     "MOVE2DALL",     &graphicsmove, 0 },
    { I_MOVE2DNONZERO, "MOVE2DNONZERO", &graphicsmove, 0 },
    { I_MOVE2DZERO,    "MOVE2DZERO",    &graphicsmove, 0 },
    /*}}}*/
    /*{{{  crcword, crcbyte, bitcnt, bitrevword, bitrevnbits*/
    { I_CRCWORD,     "CRCWORD",     &hascrc, 0 },
    { I_CRCBYTE,     "CRCBYTE",     &hascrc, 0 },
    { I_BITCNT,      "BITCNT",      &hasbitops, 0 },
    { I_BITREVWORD,  "BITREVWORD",  &hasbitops, 0 },
    { I_BITREVNBITS, "BITREVNBITS", &hasbitops, 0 },
    /*}}}*/
    /*{{{  dup, wsubdb*/
    { I_DUP,     "DUP",      &hasdup, 0 },
    { I_WSUBDB,  "WSUBDB",   &haswsubdb, 0 },
    /*}}}*/
    /*{{{  inline fp codes*/
    { I_FPLDNLDBI,  "FPLDNLDBI",  &fpinline, 0 },
    { I_FPCHKERR,   "FPCHKERR",   &fpinline, 0 },
    { I_FPSTNLDB,   "FPSTNLDB",   &fpinline, 0 },
    { I_FPLDNLSNI,  "FPLDNLSNI",  &fpinline, 0 },
    { I_FPADD,      "FPADD",      &fpinline, 0 },
    { I_FPSTNLSN,   "FPSTNLSN",   &fpinline, 0 },
    { I_FPSUB,      "FPSUB",      &fpinline, 0 },
    { I_FPLDNLDB,   "FPLDNLDB",   &fpinline, 0 },
    { I_FPMUL,      "FPMUL",      &fpinline, 0 },
    { I_FPDIV,      "FPDIV",      &fpinline, 0 },
    { I_FPLDNLSN,   "FPLDNLSN",   &fpinline, 0 },
    { I_FPREMFIRST, "FPREMFIRST", &fpinline, 0 },
    { I_FPREMSTEP,  "FPREMSTEP",  &fpinline, 0 },
    { I_FPNAN,      "FPNAN",      &fpinline, 0 },
    { I_FPORDERED,  "FPORDERED",  &fpinline, 0 },
    { I_FPNOTFINITE,"FPNOTFINITE",&fpinline, 0 },
    { I_FPGT,       "FPGT",       &fpinline, 0 },
    { I_FPEQ,       "FPEQ",       &fpinline, 0 },
    { I_FPI32TOR32, "FPI32TOR32", &fpinline, 0 },
    { I_FPI32TOR64, "FPI32TOR64", &fpinline, 0 },
    { I_FPB32TOR64, "FPB32TOR64", &fpinline, 0 },
    { I_FPRTOI32,   "FPRTOI32",   &fpinline, 0 },
    { I_FPSTNLI32,  "FPSTNLI32",  &fpinline, 0 },
    { I_FPLDZEROSN, "FPLDZEROSN", &fpinline, 0 },
    { I_FPLDZERODB, "FPLDZERODB", &fpinline, 0 },
    { I_FPINT,      "FPINT",      &fpinline, 0 },
    { I_FPDUP,      "FPDUP",      &fpinline, 0 },
    { I_FPREV,      "FPREV",      &fpinline, 0 },
    { I_FPLDNLADDDB,"FPLDNLADDDB",&fpinline, 0 },
    { I_FPLDNLMULDB,"FPLDNLMULDB",&fpinline, 0 },
    { I_FPLDNLADDSN,"FPLDNLADDSN",&fpinline, 0 },
    { I_FPLDNLMULSN,"FPLDNLMULSN",&fpinline, 0 },
    /*}}}*/
    /*{{{  fptesterr*/
    { I_FPTESTERR,  "FPTESTERR",  &fptesterr, 0 },
    /*}}}*/
    /*{{{  fpu entry codes*/
    { I_FPUSQRTFIRST,"FPUSQRTFIRST",&fpinline, 0 },
    { I_FPUSQRTSTEP, "FPUSQRTSTEP", &fpinline, 0 },
    { I_FPUSQRTLAST, "FPUSQRTLAST", &fpinline, 0 },
    { I_FPURP,       "FPURP",       &fpinline, 0 },
    { I_FPURM,       "FPURM",       &fpinline, 0 },
    { I_FPURZ,       "FPURZ",       &fpinline, 0 },
    { I_FPUR32TOR64, "FPUR32TOR64", &fpinline, 0 },
    { I_FPUR64TOR32, "FPUR64TOR32", &fpinline, 0 },
    { I_FPUEXPDEC32, "FPUEXPDEC32", &fpinline, 0 },
    { I_FPUEXPINC32, "FPUEXPINC32", &fpinline, 0 },
    { I_FPUABS,      "FPUABS",      &fpinline, 0 },
    { I_FPUNOROUND,  "FPUNOROUND",  &fpinline, 0 },
    { I_FPUCHKI32,   "FPUCHKI32",   &fpinline, 0 },
    { I_FPUCHKI64,   "FPUCHKI64",   &fpinline, 0 },
    { I_FPUDIVBY2,   "FPUDIVBY2",   &fpinline, 0 },
    { I_FPUMULBY2,   "FPUMULBY2",   &fpinline, 0 },
    { I_FPURN,       "FPURN",       &fpinline, 0 },
    { I_FPUSETERR,   "FPUSETERR",   &fpinline, 0 },
    { I_FPUCLRERR,   "FPUCLRERR",   &fpinline, 0 },
    /*}}}*/
    /*{{{  fp codes, full guy only*/
    { I_FPSTTEST, "FPSTTEST", &fpinline, GUYS_ATTR },
    { I_FPLDTEST, "FPLDTEST", &fpinline, GUYS_ATTR },
    { I_FPENTRY3, "FPENTRY3", &fpinline, GUYS_ATTR },
    { I_FPENTRY2, "FPENTRY2", &fpinline, GUYS_ATTR },
    { I_FPENTRY , "FPENTRY",  &fpinline, GUYS_ATTR },
    /*}}}*/
    /*{{{  debugger support*/
    { I_BREAK        , "BREAK",         &debugsupport, GUYS_ATTR },
    { I_CLRJ0BREAK   , "CLRJ0BREAK",    &debugsupport, GUYS_ATTR },
    { I_SETJ0BREAK   , "SETJ0BREAK",    &debugsupport, GUYS_ATTR },
    { I_TESTJ0BREAK  , "TESTJ0BREAK",   &debugsupport, GUYS_ATTR },
    { I_TIMERDISABLEH, "TIMERDISABLEH", &hastimerdis, GUYS_ATTR },
    { I_TIMERDISABLEL, "TIMERDISABLEL", &hastimerdis, GUYS_ATTR },
    { I_TIMERENABLEH , "TIMERENABLEH",  &hastimerdis, GUYS_ATTR },
    { I_TIMERENABLEL , "TIMERENABLEL",  &hastimerdis, GUYS_ATTR },
    { I_LDMEMSTARTVAL, "LDMEMSTARTVAL", &hasldmemstart, 0 },
    /*}}}*/
    /*{{{  pop*/
    { I_POP,     "POP",           &haspop, 0 },
    /*}}}*/
    /*{{{  loader support*/
    { I_LDDEVID, "LDDEVID",       &haslddevid, 0 },
    /*}}}*/
    /*{{{  Recoded FPU instructions*/
    { I_FPSETERR    , "FPSETERR",     &H1_instr, 0 },
    { I_FPRN        , "FPRN",         &H1_instr, 0 },
    { I_FPDIVBY2    , "FPDIVBY2",     &H1_instr, 0 },
    { I_FPMULBY2    , "FPMULBY2",     &H1_instr, 0 },
    { I_FPSQRT      , "FPSQRT",       &H1_instr, 0 },
    { I_FPRP        , "FPRP",         &H1_instr, 0 },
    { I_FPRM        , "FPRM",         &H1_instr, 0 },
    { I_FPRZ        , "FPRZ",         &H1_instr, 0 },
    { I_FPR32TOR64  , "FPR32TOR64",   &H1_instr, 0 },
    { I_FPR64TOR32  , "FPR64TOR32",   &H1_instr, 0 },
    { I_FPEXPDEC32  , "FPEXPDEC32",   &H1_instr, 0 },
    { I_FPEXPINC32  , "FPEXPINC32",   &H1_instr, 0 },
    { I_FPABS       , "FPABS",        &H1_instr, 0 },
    { I_FPCLRERR    , "FPCLRERR",     &H1_instr, 0 },
    { I_FPNOROUND   , "FPNOROUND",    &H1_instr, 0 },
    { I_FPCHKI32    , "FPCHKI32",     &H1_instr, 0 },
    { I_FPCHKI64    , "FPCHKI64",     &H1_instr, 0 },
    /*}}}*/
    /*{{{  New sequential code*/
    { I_GTU         , "GTU",          &H1_instr, 0 },
    { I_FPRANGE     , "FPRANGE",      &H1_instr, 0 },
    { I_XBWORD      , "XBWORD",       &H1_instr, 0 },
    { I_LBX         , "LBX",          &H1_instr, 0 },
    { I_CB          , "CB",           &H1_instr, 0 },
    { I_CBU         , "CBU",          &H1_instr, 0 },
    { I_SSUB        , "SSUB",         &H1_instr, 0 },
    { I_CIR         , "CIR",          &H1_instr, 0 },
    { I_SS          , "SS",           &H1_instr, 0 },
    { I_LS          , "LS",           &H1_instr, 0 },
    { I_CIRU        , "CIRU",         &H1_instr, 0 },
    { I_FPREM       , "FPREM",        &H1_instr, 0 },
    { I_FPGE        , "FPGE",         &H1_instr, 0 },
    { I_FPLG        , "FPLG",         &H1_instr, 0 },
    { I_XSWORD      , "XSWORD",       &H1_instr, 0 },
    { I_LSX         , "LSX",          &H1_instr, 0 },
    { I_CS          , "CS",           &H1_instr, 0 },
    { I_CSU         , "CSU",          &H1_instr, 0 },
    /*}}}*/
    /*{{{  Error handling*/
    { I_LDFLAGS     , "LDFLAGS",      &H1_instr, 0 },
    { I_STFLAGS     , "STFLAGS",      &H1_instr, 0 },
    { I_LDTH        , "LDTH",         &H1_instr, 0 },
    { I_SYSCALL     , "SYSCALL",      &H1_instr, 0 },
    { I_SELTH       , "SELTH",        &H1_instr, 0 },
    { I_GOPROT      , "GOPROT",       &H1_instr, 0 },
    { I_TRET        , "TRET",         &H1_instr, 0 },
    { I_LDSHADOW    , "LDSHADOW",     &H1_instr, 0 },
    { I_STSHADOW    , "STSHADOW",     &H1_instr, 0 },
    /*}}}*/
    /*{{{  Process model*/
    { I_SETTIMESLICE, "SETTIMESLICE", &H1_instr, 0 },
    { I_LDPROC      , "LDPROC",       &H1_instr, 0 },
    { I_SWAPQUEUE   , "SWAPQUEUE",    &H1_instr, 0 },
    { I_SWAPTIMER   , "SWAPTIMER",    &H1_instr, 0 },
    { I_INSERTQUEUE , "INSERTQUEUE",  &H1_instr, 0 },
    { I_TIMESLICE   , "TIMESLICE",    &H1_instr, 0 },
    /*}}}*/
    /*{{{  Communication*/
    { I_INSPHDR     , "INSPHDR",      &H1_instr, 0 },
    { I_READBFR     , "READBFR",      &H1_instr, 0 },
    { I_LDCNT       , "LDCNT",        &H1_instr, 0 },
    { I_LDCHSTATUS  , "LDCHSTATUS",   &H1_instr, 0 },
    { I_READHDR     , "READHDR",      &H1_instr, 0 },
    { I_WRITEHDR    , "WRITEHDR",     &H1_instr, 0 },
    { I_INITVLCB    , "INITVLCB",     &H1_instr, 0 },
    { I_SETCHMODE   , "SETCHMODE",    &H1_instr, 0 },
    { I_SETHDR      , "SETHDR",       &H1_instr, 0 },
    { I_SWAPBFR     , "SWAPBFR",      &H1_instr, 0 },
    { I_LDVLCB      , "LDVLCB",       &H1_instr, 0 },
    { I_STVLCB      , "STVLCB",       &H1_instr, 0 },
    { I_VIN         , "VIN",          &H1_instr, 0 },
    { I_VOUT        , "VOUT",         &H1_instr, 0 },
    { I_STOPCH      , "STOPCH",       &H1_instr, 0 },
    /*}}}*/
    /*{{{  Semaphores and resources*/
    { I_SIGNAL      , "SIGNAL",       &H1_instr, 0 },
    { I_WAIT        , "WAIT",         &H1_instr, 0 },
    { I_GRANT       , "GRANT",        &H1_instr, 0 },
    { I_ENBG        , "ENBG",         &H1_instr, 0 },
    { I_DISG        , "DISG",         &H1_instr, 0 },
    /*}}}*/
    /*{{{  Cache and configuration*/
    { I_LDCONF      , "LDCONF",       &H1_instr, 0 },
    { I_STCONF      , "STCONF",       &H1_instr, 0 },
    { I_FDCA        , "FDCA",         &H1_instr, 0 },
    { I_ICA         , "ICA",          &H1_instr, 0 },
    /*}}}*/
    /*{{{  Added 13/12/90 */
    { I_INTDIS      , "INTDIS",       &H1_instr, 0 },
    { I_INTENB      , "INTENB",       &H1_instr, 0 },
    { I_CHANTYPE    , "CHANTYPE",     &H1_instr, 0 },
    { I_FPLDALL     , "FPLDALL",      &H1_instr, 0 },
    { I_FPSTALL     , "FPSTALL",      &H1_instr, 0 },
    { I_FDCL        , "FDCL",         &H1_instr, 0 },
    { I_ICL         , "ICL",          &H1_instr, 0 },
    { I_LDRESPTR    , "LDRESPTR",     &H1_instr, 0 },
    { I_STRESPTR    , "STRESPTR",     &H1_instr, 0 },
    { I_ERDSQ       , "ERDSQ",        &H1_instr, 0 },
    { I_IRDSQ       , "IRDSQ",        &H1_instr, 0 },
    { I_MKRC        , "MKRC",         &H1_instr, 0 },
    { I_UNMKRC      , "UNMKRC",       &H1_instr, 0 },
    { I_STMOVE2DINIT, "STMOVE2DINIT", &H1_instr, 0 },
    /*}}}*/
    /*{{{  Added 15/2/91 */
    { I_SWAPGSTATUS , "SWAPGSTATUS",  &H1_instr, 0 },
    /*}}}*/
    /*{{{  pseudo ops*/
    { I_AJWRET | I_PSEUDO_OP, "AJWRET",  &trueval, GUYS_ATTR },
    { I_LDRETP | I_PSEUDO_OP, "LDRETP",  &trueval, 0 },
    { I_LD     | I_PSEUDO_OP, "LD",      &trueval, PTRS_ATTR | 1 },
    { I_LDAB   | I_PSEUDO_OP, "LDAB",    &trueval, PTRS_ATTR | 2 },
    { I_LDABC  | I_PSEUDO_OP, "LDABC",   &trueval, PTRS_ATTR | 3 },
    { I_ST     | I_PSEUDO_OP, "ST",      &trueval, ELEM_ATTR | 1 },
    { I_STAB   | I_PSEUDO_OP, "STAB",    &trueval, ELEM_ATTR | 2 },
    { I_STABC  | I_PSEUDO_OP, "STABC",   &trueval, ELEM_ATTR | 3 },
    { I_BYTE   | I_PSEUDO_OP, "BYTE",    &trueval, OPRD_ATTR }, /* unlimited operands */
    { I_WORD   | I_PSEUDO_OP, "WORD",    &trueval, OPRD_ATTR }, /* ditto */
    { I_ALIGN  | I_PSEUDO_OP, "ALIGN",   &trueval, 0 },
    { I_LDLABELDIFF | I_PSEUDO_OP, "LDLABELDIFF", &trueval, 2 }
    /*}}}*/
    /*}}}*/
  };
/*}}}*/

/*{{{  PUBLIC INT32 lookupinstruction (char *inst)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  lookupinstruction looks up the instruction name 'inst' in the instruction
 *                    table and returns the instruction value if found,
 *                    otherwise an error value :
 *                      INSTRUCTION_NOT_VALID means the instruction exists
 *                        on some processors, but not on the current target.
 *                      INSTRUCTION_NOT_ENABLED means the instruction exists
 *                        but cannot be used in the current guyinserts mode.
 *                      INSTRUCTION_NOT_DECLARED means that the instruction
 *                        does not exist for any target processor.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC INT32 lookupinstruction (char *inst, int guy_not_asm,
				int *err, int *operands, int *ptrs_allowed,
                                int *element_required)
{
  int i;
  *err              = 0;
  *operands         = 0;
  *ptrs_allowed     = FALSE;
  *element_required = FALSE;
  for (i = 0; i < (sizeof(instructions) / sizeof(struct instruction_struct));
       i++)
    if (strcmp(instructions[i].iname, inst) == 0)
      {
	/* if the instruction is defined for this target, and legal
	   in our guyinserts mode ... */
	if (*(instructions[i].ivalid))
	  {
            INT32 instval = instructions[i].ivalue;
            int   attr    = instructions[i].iattr;
	    if ((guyinserts == SEQUENTIALGUYS) && (attr & GUYS_ATTR))
	      *err = INSTRUCTION_NOT_ENABLED;
	    if (guy_not_asm && instval & I_PSEUDO_OP)
              *err = INSTRUCTION_NOT_VALID;

            *ptrs_allowed     = (attr & PTRS_ATTR) != 0;
            *element_required = (attr & ELEM_ATTR) != 0;
            *operands         = (attr & OPRD_ATTR);
            if (*operands == OPRD_ATTR) *operands = (-1);

	    return (instval);
	  }
	else
	  {
	    *err = INSTRUCTION_NOT_VALID;
	    return (ZERO32);
	  }
      }
  *err = INSTRUCTION_NOT_DECLARED;
  return (ZERO32);
}
/*}}}*/
/*{{{  PUBLIC char *primaryname(int inst)*/
/*****************************************************************************
 *
 *  primaryname takes a primary instruction number 'inst' and returns
 *              the instruction name.
 *
 *****************************************************************************/
PUBLIC char *primaryname ( int inst )
{
  int i;
  for (i = 0; i < N_PRIMARIES; i++)
    if ((instructions[i].ivalue & ~I_PRIMARY) == inst)
      {
	strcpy(instname, instructions[i].iname);
	strlwr(instname);
	return instname;
      }
  sprintf(instname, "???%d", inst);
  return instname;
}
/*}}}*/
/*{{{  PUBLIC char *secondaryname (int inst)*/
/*****************************************************************************
 *
 *  secondaryname takes a secondary instruction number 'inst' and
 *                returns the instruction name.
 *
 *****************************************************************************/
PUBLIC char *secondaryname ( int inst )
{
  int i;
  for (i = N_PRIMARIES;
       i < (sizeof(instructions) / sizeof(struct instruction_struct));
       i++)
    if (instructions[i].ivalue == inst)
      {
	strcpy(instname, instructions[i].iname);
	strlwr(instname);
	return instname;
      }
  sprintf(instname, "opr %d", inst);
  return instname;
}
/*}}}*/
