/*#define DEBUG*/
/******************************************************************************
*
*  configurer interface to backend
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1990
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <stdlib.h>
# include "includes.h"
# include "tcoff.h"

# include "usehdr.h"
# include "confhdr.h"
# include "confdef.h"
# include "conf1def.h"
# include "conf2def.h"
# include "lexconst.h"
# include "lex1def.h"
# include "deschdr.h"
# include "desc1def.h"
# include "chkdef.h"
# include "code1def.h"
# include "syn3def.h"

#if 0
  #include "../adk/tagdefs.h"
  #include "../adk/protos.h"
  #include "../adk/typedef.h"
#else
  #include "tagdefs.h"
  #include "protos.h"
  #include "typedef.h"
#endif
/*}}}*/

/*{{{  PUBLIC data*/
PUBLIC int config_target_type;

PUBLIC int config_backend      = TRUE;
PUBLIC int config_backend_dump = FALSE;
PUBLIC int config_postmortem   = FALSE; /* Postmortem debugging only */
PUBLIC int config_overlay      = TRUE;
PUBLIC int config_rom          = FALSE;
PUBLIC int config_rom_in_rom;
PUBLIC int config_repeat       = FALSE;
PUBLIC int config_profile      = FALSE;
/*}}}*/

/*{{{  PUBLIC data for the configurer backend */

/* These are only ever defined in this file.
   Where the "same" flag is used or setup elsewhere, I use a separate flag,
   (eg config_repeat), and copy that into the backend's flag.

   This means that this file is the ONLY file which needs to include
   the backend's include files.

   CON 11/10/90
*/

PUBLIC BOOL WarningFlag     = FALSE;
PUBLIC BOOL PedanticFlag    = FALSE;
PUBLIC BOOL InformationFlag = FALSE;
PUBLIC BOOL RomRunRamFlag   = FALSE;
PUBLIC BOOL RomRunRomFlag   = FALSE;
PUBLIC BOOL DebugModeFlag   = FALSE; /* this just disables 'ordering' */
PUBLIC INT32 RomMemorySize  = 0;
PUBLIC CHAR *OutputFileName   = NULL;
PUBLIC FILE *OutputFileStream = NULL;

PUBLIC BOOL SimpleProcessFlag  = FALSE;
PUBLIC BOOL OverlayProcessFlag = FALSE;
PUBLIC BOOL ProfileModeFlag    = FALSE;

PUBLIC BOOL ExecuteManyFlag = FALSE;

PUBLIC CHAR *SearchPathName = pathname;

PUBLIC BOOL VirtualModeFlag   = FALSE;
PUBLIC BOOL VirtualEnableFlag = FALSE;
PUBLIC INT VInputBlockSize   = 0;
PUBLIC INT VOutputBlockSize  = 0;
PUBLIC INT VInputBufferSize  = 0;
PUBLIC INT VOutputBufferSize = 0;
PUBLIC CHAR *VInputChannelName  = "";
PUBLIC CHAR *VOutputChannelName = "";
/*}}}*/

/*{{{  PRIVATE data*/
#if 0
PRIVATE Identifier *processor_id;      /* value of the "processor" identifier */
PRIVATE Identifier *process_id;        /* value of the "process"   identifier */
#else
/* we cache a copy of the token too */
PRIVATE Token *processor_id;           /* value of the "processor" identifier */
PRIVATE Token *process_id;             /* value of the "process"   identifier */
#endif
PRIVATE Constant   *filename_constant; /* Constant representing objfilename */
PRIVATE Token      *filename_token;    /* Token    representing objfilename */
PRIVATE List **next_statement;  /* pointer to where to attach the next statement */

typedef struct procid_chain_s
  {
    struct procid_chain_s *i_next; /* ptr to next on the list */
  /*Identifier            *i_id;*/ /* Identifier for following type and mem */
    Token                 *i_id;   /* Token for following type and mem */
    const wordnode        *i_str;  /* wordnode describing identifier */
    const wordnode        *i_type;
    INT32                  i_mem;
  } procid_chain_t;

PRIVATE procid_chain_t *procid_chain = NULL; /* list of created Identifiers */
PRIVATE INT32 hw_proc_count = 0;
PRIVATE INT32 sw_proc_count = 0;
/*}}}*/

/*{{{  configurer backend attribute strings*/
/* Keywords which must be recognised by the backend */
#define HIDE_BACKEND_NAMES TRUE

#if HIDE_BACKEND_NAMES
  /* we hide these so that we needn't bother about user names clashing
     with them
  */
  #define NODE_KEYWORD   "%node%"
  #define INPUT_KEYWORD  "%input%"
  #define OUTPUT_KEYWORD "%output%"

  #define PROCESS_TYPE_WORD   "%process%"
  #define PROCESSOR_TYPE_WORD "%processor%"
#else
  #define NODE_KEYWORD   "node"
  #define INPUT_KEYWORD  "input"
  #define OUTPUT_KEYWORD "output"

  #define PROCESS_TYPE_WORD   "process"
  #define PROCESSOR_TYPE_WORD "processor"

  #define HOSTNAME "host"
#endif

/* These attributes are all recognised by context: */
#define ELEMENT_ATTR_WORD   "element"
#define LINK_ATTR_WORD      "link"
#define TYPE_ATTR_WORD      "type"
#define MEMORY_ATTR_WORD    "memory"
#define ORDERCODE_ATTR_WORD "code"
#define ORDERVS_ATTR_WORD   "vector"
#define INTERFACE_ATTR_WORD "interface"

/* These are used to cache Tokens for each of the above words */
PRIVATE Token *element_token;
PRIVATE Token *type_token;
PRIVATE Token *memory_token;
PRIVATE Token *ordercode_token;
PRIVATE Token *ordervs_token;
PRIVATE Token *interface_token;
PRIVATE Expression *link_exp; /* we always create an Expression for the link token */

/* These are specific strings for the 'element' attribute */
#define PROCESS_ATTR_WORD   "process"
#define PROCESSOR_ATTR_WORD "processor"

/* cache copies of all constant expressions less than this value */
#define MAX_CACHE_CONSTANTS (20)
/*}}}*/

/*{{{  PRIVATE List *create_list*/
PRIVATE List *create_list(void *const item, List *const list)
{
  List *l = CreateList();
  l->Data = item;
  l->Link = list;
  return l;
}
/*}}}*/
/*{{{  PRIVATE void add_statement*/
PRIVATE void add_statement(Statement *const s)
{
  *next_statement = create_list(s, NULL);
  next_statement = &((*next_statement)->Link);
}
/*}}}*/
/*{{{  PRIVATE Constant *create_constant_str*/
PRIVATE Constant *create_constant_str(const char *const s, const int len)
/* This creates a Constant node, given a string */
{
  Constant *c = CreateConstant(T_STRING);
  c->Data.SVal.Size = len;
/*c->Data.SVal.Data = (BYTE *)s;*/
/*c->Data.SVal.Data = (BYTE *)memalloc(len + 1);*/
  c->Data.SVal.Data = (BYTE *)MallocChk(len + 1); /* use the backend's malloc */
  memcpy(c->Data.SVal.Data, s, len+1);
  return c;
}
/*}}}*/
/*{{{  PRIVATE Token *create_token_const*/
PRIVATE Token *create_token_const(Constant *const c)
/* This creates a Token node, given a Constant node */
{
  Token      *t = CreateToken(L_CONSTANT);
  t->Data.Value = c;
  return t;
}
/*}}}*/
/*{{{  PRIVATE Expression *create_exp_token*/
PRIVATE Expression *create_exp_token(Token *const t)
/* This creates an Expression node, given a Token node */
{
  Expression *e = CreateExpression((t->Type == L_CONSTANT) ? E_CONSTANT : E_IDENTIFIER);
  e->Data.Operand = t;
  return e;
}
/*}}}*/
/*{{{  PRIVATE Expression *create_exp_constant*/
PRIVATE Expression *create_exp_constant(Constant *const c)
/* This creates an Expression node, given a Constant node */
{
  return create_exp_token(create_token_const(c));
}
/*}}}*/
/*{{{  PRIVATE Expression *create_exp_INT32*/
PRIVATE Expression *create_exp_INT32(const INT32 value)
/* This creates an Expression node, given an INT32 */
{
  static Expression *cached_values[MAX_CACHE_CONSTANTS]; /* initialised to NULLs */
  const int cacheable = (value >= 0) && (value < MAX_CACHE_CONSTANTS);
  if (cacheable && (cached_values[value] != NULL))
    return cached_values[value];
  else
    {
      Expression *e;
      Constant *c = CreateConstant(T_INT32);
      c->Data.IVal = value;
      e = create_exp_constant(c);
      if (cacheable)
        cached_values[value] = e;
      return e;
    }
}
/*}}}*/
/*{{{  PRIVATE Expression *create_exp_str*/
PRIVATE Expression *create_exp_str(const char *const str, const int len)
/* This creates an Expression node, given a string */
{
  return create_exp_constant(create_constant_str(str, len));
}
/*}}}*/
/*{{{  PRIVATE Expression *create_binary_exp*/
PRIVATE Expression *create_binary_exp(const int tag, Expression *const e1,
                                                     Expression *const e2)
/* This creates a binary Expression node, given two Expressions */
{
  Expression *e = CreateExpression(tag);
  e->Data.Binary.Expr1 = e1;
  e->Data.Binary.Expr2 = e2;
  return e;
}
/*}}}*/
/*{{{  PRIVATE Attribute *create_attr_exp*/
PRIVATE Attribute *create_attr_exp(Token *const t, Expression *const e)
{
  Attribute *a = CreateAttribute(A_EXPRESSION);
  a->Name = t;
  a->Data.Value = e;
  return a;
}
/*}}}*/
/*{{{  PRIVATE Token *create_token_id*/
PRIVATE Token *create_token_id(Identifier *const i)
/* This creates a Token, given an Identifier */
{
  Token *t = CreateToken(L_IDENTIFIER);
  t->Data.Name = i;
  return t;  
}
/*}}}*/
/*{{{  PRIVATE Token *create_token*/
PRIVATE Token *create_token(const char *const s)
/* This creates a Token, given a string (which is inserted as an Identifier) */
{
  return create_token_id(InsertIdentifier((char *)s, L_IDENTIFIER));
}
/*}}}*/
/*{{{  PRIVATE Parameter *create_param*/
PRIVATE Parameter *create_param(Expression *const name, Expression *const value)
{
  Parameter *p = CreateParameter();
  p->Name = name;
  p->Value = value;
  return p;
}
/*}}}*/
/*{{{  PRIVATE Expression *name_to_exp*/
PRIVATE Expression *name_to_exp(const namedesc_t *const name)
/*****************************************************************************
*
* name_to_exp returns an Expression representing (subscripted) name
*
*****************************************************************************/
{
  Expression *e = CreateExpression(E_IDENTIFIER);

  treenode *t = NTypeOf(name->n_name);
  INT32 i     = name->n_index;

  e->Data.Operand = create_token(WNameOf(NNameOf(name->n_name)));

  while (TagOf(t) == S_ARRAY)
    {
      INT32 size;
      t = ARTypeOf(t);
      size = elementsin(t);
      e = create_binary_exp(E_SUBSCRIPT, e, create_exp_INT32(i/size));

      i = i % size;
    }
  return e;
}
/*}}}*/
/*{{{  PRIVATE Expression *write_name_and_dims*/
PRIVATE Expression *write_name_and_dims(treenode *const nptr)
{
  Expression *e = CreateExpression(E_IDENTIFIER);
  treenode *type = NTypeOf(nptr);
  e->Data.Operand = create_token(WNameOf(NNameOf(nptr)));
  if (config_srcout) fputs(WNameOf(NNameOf(nptr)), outfile);

  while (TagOf(type) == S_ARRAY)
    {
      e = create_binary_exp(E_SUBSCRIPT, e, create_exp_INT32(ARDimOf(type))); /* index expression */
      if (config_srcout) fprintf(outfile, "[%ld]", ARDimOf(type));
      type = ARTypeOf(type);
    }
  return e;
}
/*}}}*/
/*{{{  PRIVATE INT *occamsymbol_dims*/
PRIVATE INT *occamsymbol_dims(treenode *const nptr)
{
  treenode *type;
  int dims = 0;
  INT *ptr;
  for (type = NTypeOf(nptr); TagOf(type) == S_ARRAY; type = ARTypeOf(type))
    dims++;
  if (dims != 0)
    {
    /*ptr = (INT *)memalloc((dims + 1) * (sizeof(INT)));*/
      ptr = (INT *)MallocChk((dims + 1) * (sizeof(INT))); /* use backend's malloc */
      dims = 0;
      for (type = NTypeOf(nptr); TagOf(type) == S_ARRAY; type = ARTypeOf(type))
        ptr[dims++] = (INT)ARDimOf(type);
      ptr[dims] = (-1);
    }
  else
    ptr = NULL;
  return ptr;
}
/*}}}*/
/*{{{  PRIVATE List *write_decl*/
PRIVATE List *write_decl(const namedesc_t *const name, const char *const declstr)
{
  List *l;
  if (config_srcout) fprintf(outfile, "%s ", declstr);
  l = create_list (write_name_and_dims(name->n_name), NULL);
  if (config_srcout) fputs(";\n", outfile);
  return l;
}
/*}}}*/
/*{{{  PRIVATE Identifier/Token *declare_node_subtype*/
/*PRIVATE Identifier *declare_node_subtype(char *const name,
                          const wordnode *const typestr, const INT32 mem)*/
PRIVATE Token *declare_node_subtype(char *const name,
                          const wordnode *const typestr, const INT32 mem)
{
/* This does the equivalent of:
  define processor(type = "typestr", memory = mem) name;
*/

/*Identifier *i  = InsertIdentifier(name, L_IDENTIFIER);*/
  Token      *i  = create_token(name);
  Statement  *s  = CreateStatement(D_NODEDEF);
  Attribute  *ta = create_attr_exp(/*create_token(TYPE_ATTR_WORD)*/ type_token,
                                   create_exp_str(WNameOf(typestr), WLengthOf(typestr)));
  Attribute  *ma = create_attr_exp(/*create_token(MEMORY_ATTR_WORD)*/ memory_token,
                                   create_exp_INT32(mem));

  s->Data.NodeDef.Type       = create_token(PROCESSOR_TYPE_WORD);
  s->Data.NodeDef.Attributes = create_list(ta, create_list(ma, NULL));
/*s->Data.NodeDef.Name       = create_token_id(i);*/
  s->Data.NodeDef.Name       = i;
  add_statement(s);

  if (config_srcout)
    fprintf(outfile, "define %s (%s = \"%s\", %s = %ld) %s;\n",
            PROCESSOR_TYPE_WORD, TYPE_ATTR_WORD, WNameOf(typestr),
                                 MEMORY_ATTR_WORD, mem, name);

  {
    procid_chain_t *new = newvec(sizeof(procid_chain_t));
    new->i_next = procid_chain;
    new->i_type = typestr;
    new->i_mem  = mem;
    new->i_id   = i;
    new->i_str  = lookupword(name, strlen(name));
    procid_chain = new;
  }
  return i;
}
/*}}}*/
/*{{{  PRIVATEPARAM void write_proc_hw(namedesc_t *name)*/
PRIVATEPARAM void write_proc_hw(const namedesc_t *const name)
{
  prochwnode_t *procnode = prochwnodeof(name);

  if (name->n_index == 0)
    {
      char processortype_str[MAXSTRING_SIZE];
      char       *processortype    = PROCESSOR_TYPE_WORD; /* default - no attrs set */
    /*Identifier *processortype_id = processor_id;*/      /* ditto */
      Token *processortype_id = processor_id;             /* ditto */
      int allempty = empty_proc(procnode);
      const INT32 size = elementsin(NTypeOf(name->n_name));

      {
        /* see if we can `abbreviate' the processors because they are
           all of the same type and memory size
        */
        const INT32 mem  = procnode->p_mem;
        wordnode *type   = procnode->p_type;
        int allsame = TRUE; /* the first is the same as itself! */
        int i;
        for (i = 1; i < size; i++)
          {
            allsame  = allsame  && (procnode[i].p_mem  == mem)
                                && (procnode[i].p_type == type);
            allempty = allempty && empty_proc(&procnode[i]); /* bug 847 15/1/91 */
          }

        for (i = 0; i < size; i++)
          procnode[i].p_connected = allsame; /* mark each processor */

        if (allsame && !allempty) /* declare a `node type' with processor type and memsize */
          {
            procid_chain_t *chain;

            processortype_id = NULL;
            for (chain = procid_chain; chain != NULL; chain = chain->i_next)
              if (chain->i_type == type && chain->i_mem == mem)
                {
                  processortype_id =         chain->i_id;
                  processortype    = WNameOf(chain->i_str);
                }

            if (processortype_id == NULL) /* not already created */
              {
                static int decl_count = 0;
                sprintf(processortype_str, "%s%s%d", processortype,
                                           PARSEABLE_SEPARATOR, decl_count);
                decl_count++; /* ensure a unique name */
                processortype_id = declare_node_subtype(processortype_str, type, mem);
                processortype    = processortype_str;
              }
          }
      }
      if (!allempty) /* bug 847 15/1/91 */
        {
          List      *l = write_decl(name, processortype);
          Statement *s = CreateStatement(D_NODE);
        /*s->Data.Node.Type       = create_token_id(processortype_id);*/
          s->Data.Node.Type       = processortype_id;
          s->Data.Node.Attributes = NULL;
          s->Data.Node.Names      = l;
          add_statement(s);
          hw_proc_count += size;
        }
    }

  /* The `p_connected' field is TRUE if the type and memsize has been declared
     as part of the processor definition.
  */
  if (!empty_proc(procnode) && !procnode->p_connected)
    {
      Statement *s    = CreateStatement(D_NODEATTR);
      Attribute *type = create_attr_exp(/*create_token(TYPE_ATTR_WORD)*/ type_token,
                                        create_exp_str(WNameOf(procnode->p_type),
                                                       WLengthOf(procnode->p_type)));
      Attribute *mem  = create_attr_exp(/*create_token(MEMORY_ATTR_WORD)*/ memory_token,
                                        create_exp_INT32(procnode->p_mem));

      procnode->p_nameexp   = name_to_exp(name); /* save a copy for later */
      s->Data.NodeAttr.Name = procnode->p_nameexp;
      s->Data.NodeAttr.Attributes = create_list(type, create_list(mem, NULL));
      add_statement(s);

      if (config_srcout)
        {
          print_name_to_str(name);
          fprintf(outfile, "(type = \"%s\", memory = %ld);\n",
                  WNameOf(procnode->p_type), procnode->p_mem);
        }
    }
}
/*}}}*/
#if 0 /* No need to describe arcs to the backend */
/*{{{  PRIVATEPARAM void write_arc_hw(namedesc_t *name)*/
PRIVATEPARAM void write_arc_hw(const namedesc_t *const name)
{
  if (name->n_index == 0)
    {
      List *l = write_decl(name, "connection");
      Statement *s = CreateStatement(D_CONNECTION);
      s->Data.Names = l;
      add_statement(s);
    }
}
/*}}}*/
#endif
/*{{{  PRIVATEPARAM void write_edge_hw(namedesc_t *name)*/
PRIVATEPARAM void write_edge_hw(const namedesc_t *const name)
{
  if (name->n_index == 0)
    {
      List *l = write_decl(name, "edge");
      Statement *s = CreateStatement(D_EDGE);
      s->Data.Names = l;
      add_statement(s);
    }
}
/*}}}*/
/*{{{  PRIVATE Expression *write_edgedesc*/
PRIVATE Expression *write_edgedesc(const edgedesc_t *const edge)
{
  Expression *e;
  namedesc_t *name;

  if (edge->e_link == EDGENODE_NO_LINK)
    {
      name = (namedesc_t *)&(edge->e_un.e_edge);
      e = name_to_exp(name);
    }
  else
    {
      prochwnode_t *procnode = edge->e_un.e_proc;
      name = &(procnode->p_name);
      e = procnode->p_nameexp; /* use the cached version */
      if (e == NULL)
        {
          e = name_to_exp(name);
          procnode->p_nameexp = e;
        }
    }
  if (config_srcout)
    print_name_to_str(name);

  if (edge->e_link != EDGENODE_NO_LINK)
    {
      e = create_binary_exp(E_SUBFIELD, e,
                            /*create_exp_token(create_token(LINK_ATTR_WORD))*/ link_exp);
      e = create_binary_exp(E_SUBSCRIPT, e, create_exp_INT32(edge->e_link));
      if (config_srcout)
        fprintf(outfile, ".link[%d]", edge->e_link);
    }

  return e;
}
/*}}}*/
/*{{{  PRIVATE void write_connection*/
PRIVATE void write_connection(const connection_t *const conn)
{
  Statement *s = CreateStatement(D_CONNECT);

  if (config_srcout) fputs("connect ", outfile);
  s->Data.Connect.Channel1 = write_edgedesc(&(conn->c_edge[0]));
  if (config_srcout) fputs(", ", outfile);
  s->Data.Connect.Channel2 = write_edgedesc(&(conn->c_edge[1]));

#if 0 /* No need to name the arcs to the backend */
  if (conn->c_arc != NULL)
    {
      s->Data.Connect.Connector = name_to_exp(&(conn->c_arc->a_name));
      if (config_srcout)
        {
          fputs(" by ", outfile);
          print_name_to_str(&(conn->c_arc->a_name));
        }
    }
#endif

  add_statement(s);

  if (config_srcout) fputs(";\n", outfile);
}
/*}}}*/
/*{{{  PRIVATE Identifier/Token *declare_node*/
/*PRIVATE Identifier *declare_node(char *const name, const char *const string)*/
PRIVATE Token *declare_node(char *const name, const char *const string)
{
/* This does the equivalent of:
  define node(element = string) name;
*/

  /*Identifier *i = InsertIdentifier(name, L_IDENTIFIER);*/
  Token      *i = create_token(name);
  Statement  *s = CreateStatement(D_NODEDEF);
  Attribute  *a = create_attr_exp(/*create_token(ELEMENT_ATTR_WORD)*/ element_token,
                                  create_exp_str(string, strlen(string)));

  s->Data.NodeDef.Type       = create_token(NODE_KEYWORD);
  s->Data.NodeDef.Attributes = create_list(a, NULL);
/*s->Data.NodeDef.Name       = create_token_id(i);*/
  s->Data.NodeDef.Name       = i;
  add_statement(s);

  if (config_srcout)
    fprintf(outfile, "define %s (%s = \"%s\") %s;\n",
            NODE_KEYWORD, ELEMENT_ATTR_WORD, string, name);

  return i;
}
/*}}}*/
/*{{{  PRIVATE void write_config_hw*/
PRIVATE void write_config_hw(void)
{
  connection_t *conn = connectionchain;

  /* Cache copies of these tokens, cos they are used a lot */
  element_token = create_token(ELEMENT_ATTR_WORD);
  memory_token  = create_token(MEMORY_ATTR_WORD);
  type_token    = create_token(TYPE_ATTR_WORD);
  link_exp      = create_exp_token(create_token(LINK_ATTR_WORD));
  
  processor_id = declare_node(PROCESSOR_TYPE_WORD, PROCESSOR_ATTR_WORD);

  walk_config_list(proclist, write_proc_hw);
  walk_config_list(edgelist, write_edge_hw);

#if 0 /* no need to name all the arcs for the backend */
  walk_config_list(arclist,  write_arc_hw);
#endif

  while (conn != NULL)
    {
      write_connection(conn);
      conn = conn->c_next;
    }
}
/*}}}*/
/*{{{  PRIVATE INT32 create_interface*/
PRIVATE INT32 create_interface(const procswnode_t *const procnode,
                               OccamSymbol **interface)
{
  const prochwnode_t *const hwnode = prochwnodeof(&(procnode->p_name));

  Statement *s = CreateStatement(D_NODE);

  Attribute *al = CreateAttribute(A_PARAMETER);
  List **l = &(al->Data.Attributes);

  treenode *t = procnode->p_fparams;
  INT32 no_of_params = 0;

  if (config_srcout) fprintf(outfile, "%s (", PROCESS_TYPE_WORD);

  while (!EndOfList(t))
    {
      treenode *param = ThisItem(t);
      if ((TagOf(param) == N_PARAM) || (TagOf(param) == N_VALPARAM))
        {
          Attribute   *a;
          Expression  *name_exp;
          const char *direction = (NVDirectionOf(param) == CHAN_USE_OUTPUT) ?
                                   OUTPUT_KEYWORD : INPUT_KEYWORD;

          if (config_srcout)
            fprintf(outfile,"%s%s ", no_of_params == 0 ? "interface (" : ", ", direction);

          a = CreateAttribute(A_DECLARATION);
          a->Name            = create_token(direction);
          name_exp           = write_name_and_dims(param);
          a->Data.Parameters = create_list(create_param(name_exp, NULL), NULL);

          /* add to _end_ of param list */
          *l = create_list(a, NULL);
          l = &((*l)->Link);

          /* add to interface list */
          {
            OccamSymbol *this = CreateOccamSymbol();
            
            /* Find the 'basename' of the parameter name expression */
            while (name_exp->Type == E_SUBSCRIPT)
              name_exp = name_exp->Data.Binary.Expr1;

            this->Type = (NVDirectionOf(param) == CHAN_USE_OUTPUT) ? T_OUTPUT : T_INPUT;
          #if 0 /* now we just use the same character string as the Identifier */
          /*this->Name = memalloc(WLengthOf(NNameOf(param))+1);*/
            this->Name = MallocChk(WLengthOf(NNameOf(param))+1); /* use backend's malloc */
            memcpy(this->Name, WNameOf(NNameOf(param)), WLengthOf(NNameOf(param))+1);
          #else
            /* Expression -> Token -> Identifier -> char* */
            this->Name = name_exp->Data.Operand->Data.Name->Name;
          #endif
            this->ValueType  = FALSE;
            this->Dimensions = occamsymbol_dims(param);
            this->Link       = NULL;

            *interface = this;
            interface  = &(this->Link);
          }

          no_of_params ++; /* No unsized parameters */
        }
      t = NextItem(t);
    }
  if (no_of_params != 0)
    {
      al->Name = /*create_token(INTERFACE_ATTR_WORD)*/ interface_token;
      s->Data.Node.Attributes = create_list(al, NULL);
      if (config_srcout) fputc(')', outfile);
    }
  /* else leave s->Data.Node.Attributes as NULL */

  if (config_reorderable && (hwnode->p_ordercode != 0))
    {
      Attribute *a = create_attr_exp(/*create_token(ORDERCODE_ATTR_WORD)*/ ordercode_token,
                                     create_exp_INT32(hwnode->p_ordercode));
      s->Data.Node.Attributes = create_list(a, s->Data.Node.Attributes);
      if (config_srcout) fprintf(outfile, ", code = %ld", hwnode->p_ordercode);
    }
  if (config_reorderable && (hwnode->p_ordervs != 0))
    {
      Attribute *a = create_attr_exp(/*create_token(ORDERVS_ATTR_WORD)*/ ordervs_token,
                                     create_exp_INT32(hwnode->p_ordervs));
      s->Data.Node.Attributes = create_list(a, s->Data.Node.Attributes);
      if (config_srcout) fprintf(outfile, ", vector = %ld", hwnode->p_ordervs);
    }

  if (config_srcout) fprintf(outfile, ") %s;\n", WNameOf(procnode->p_procname));

  /*s->Data.Node.Type  = create_token_id(process_id);*/
  s->Data.Node.Type  = process_id;
  /*s->Data.Node.Names = create_list(create_exp_token(create_token(WNameOf(procnode->p_procname))),
                                   NULL);*/
  s->Data.Node.Names = create_list(procnode->p_procnameexp, NULL);

  add_statement(s);

  return no_of_params;
}
/*}}}*/
/*{{{  PRIVATE void create_place_statement*/
PRIVATE void create_place_statement(Expression *const e1, Expression *const e2)
{
  Statement *s = CreateStatement(D_PLACE);
  s->Data.Place.Element1 = e1;
  s->Data.Place.Element2 = e2;
  add_statement(s);
}
/*}}}*/
/*{{{  PRIVATE void place_sw_on_hw*/
PRIVATE void place_sw_on_hw(const procswnode_t *const procnode, const namedesc_t *const name)
{
#if 0 /* This doesn't use the 'cached' version of the processor's name Expression */
  create_place_statement(create_exp_token(create_token(WNameOf(procnode->p_procname))),
                         name_to_exp(name));
#else
  create_place_statement(procnode->p_procnameexp, name_to_exp(name));
#endif

  if (config_srcout)
    {
      fprintf(outfile, "place %s on ", WNameOf(procnode->p_procname));
      print_name_to_str(name);
      fputs(";\n", outfile);
    }
}
/*}}}*/
/*{{{  PRIVATE BIT32 origin_num_from_origin_str*/
PRIVATE BIT32 origin_num_from_origin_str(const wordnode *const string)
{
  /* origin string looks like:  "filename:hexnum" */

  char *s, *ss;
  if (string == NULL) return 0;

  s = WNameOf(string);

  /* search backwards for colon */
  ss = strrchr(s, ':');
  if (ss == NULL)
    ss = s;
  else
    ss++;  /* skip past the colon */

  return strtol(ss, (char **)NULL, 16);
}
/*}}}*/
/*{{{  PRIVATE int errormodeof*/
PRIVATE int errormodeof(const INT32 attr)
{
  DEBUG_MSG(("errormodeof: attr is %lX\n", attr));
  switch(attr & ATTRIB_ERROR_MASK)
    {
      case ATTRIB_UNIVERSAL: return M_UNIVERSAL;
      case ATTRIB_HALT:      return M_HALT;
      case ATTRIB_STOP:      return M_STOP;
      default:               return M_NONE;
    }
}
/*}}}*/
/*{{{  PRIVATE int targettypeof*/
PRIVATE int targettypeof(const INT32 instr, const INT32 attr)
{
  setprocessor(processorstring(instr, attr));
  setprocessorattr();
  return config_target_type;
}
/*}}}*/
/*{{{  PRIVATEPARAM void write_proc_sw*/
PRIVATEPARAM void write_proc_sw(const namedesc_t *const name)
{
  Statement *s;
  OccamSymbol *interface = NULL;
  INT32 no_of_params;
  procswnode_t *procnode = procswnodeof(name);
  treenode *list;
  OccamObject *occam_object;
  Object      *object_file; /* an 'object file' */


  if (procnode->p_process == NULL)
    return;  /* nothing on that processor */

  /* we cache a copy of the Expression node which holds this processor's name */
  procnode->p_procnameexp = create_exp_token(create_token(WNameOf(procnode->p_procname)));

  no_of_params = create_interface(procnode, &interface);
  place_sw_on_hw(procnode, name);
  sw_proc_count++;

  occam_object = CreateOccamObject();

  if (config_srcout)
    fprintf(outfile, "use \"%s\" for %s; /*(WS:%ld, VS:%ld, sz:%ld, off:%ld, sk:%ld)*/\n",
            objfilename, WNameOf(procnode->p_procname),
            procnode->p_ws, procnode->p_vs, procnode->p_size, procnode->p_offset, procnode->p_seek);

  setproctypeto(name); /* sets processor attr, etc */

  /*{{{  `master' or cluedo file*/
  object_file = CreateObject();
  object_file->OccamProcess    = TRUE;
  object_file->HighPriority    = FALSE;
  object_file->Interface       = interface;
  /*object_file->Id;*/
  object_file->Index           = procnode->p_seek;
  object_file->Origin          = get_sourcehash();
  object_file->TargetType      = targettypeof(processortype, processorattr);
  object_file->ErrorMode       = errormodeof(processorattr);
  object_file->CodeSize        = procnode->p_size;
  object_file->CodeEntryOffset = procnode->p_offset;
  object_file->StackSize       = procnode->p_ws * bytesperword;
  object_file->StackParamSize  = (no_of_params + 2) * bytesperword;
  object_file->VectorSize      = procnode->p_vs * bytesperword;
  object_file->StaticSize      = 0;
  /*}}}*/

  occam_object->FileDatas = create_list(object_file, NULL);
  /* Apparently we are not really supposed to use the same one of these
     for each occam_object, but since the backend never reclaims memory,
     we can.
  */
  occam_object->FileNames = create_list(filename_constant, NULL);
  occam_object->CodeSize  = procnode->p_size;

  for (list = procnode->p_modules; !EndOfList(list); list = NextItem(list))
    {
      module_t *module = (module_t *)ThisItem(list);

      /* Added for bug 1089 and 1068 4/1/91: */
      if (module->m_config == NULL) /* this module has not yet been processed */
      /*{{{  data for a called module*/
      {
        object_file = CreateObject();
        object_file->OccamProcess    = TRUE;
        object_file->HighPriority    = FALSE;
        object_file->Interface       = NULL;
        /*object_file->Id;*/ /* do not set this field */
        object_file->Index           = module->m_seek_ptr;
        object_file->Origin          = origin_num_from_origin_str(module->m_name);
        object_file->TargetType      = targettypeof(module->m_instr, module->m_attr);
        object_file->ErrorMode       = errormodeof(module->m_attr);
        object_file->CodeSize        = module->m_size;
        object_file->CodeEntryOffset = (-1); /* means NOT the entry point */
        object_file->StackSize       = 0;
        object_file->StackParamSize  = 0;
        object_file->VectorSize      = 0;
        object_file->StaticSize      = 0;
        {
          const char *const thisfile = lookupfilename(module->m_filenum);
          /* see comment about filename_constant above */
          module->m_filestr = create_constant_str(thisfile, strlen(thisfile));
        }
        module->m_config  = object_file;
      }
      /*}}}*/
      else
        object_file = module->m_config; /* Added for bug 1089 and 1068 4/1/91 */

      occam_object->FileDatas = create_list(object_file, occam_object->FileDatas);
      occam_object->FileNames = create_list(module->m_filestr,
                                            occam_object->FileNames);
      occam_object->CodeSize += module->m_size;

      if (config_srcout)
        fprintf(outfile, "use \"%s\" for %s; /*org:\"%s\", sk:%ld, sz:%ld*/\n",
                lookupfilename(module->m_filenum),
                WNameOf(procnode->p_procname),
                WNameOf(module->m_name), module->m_seek_ptr, module->m_size);
    }

  s = CreateStatement(D_USE);
  s->Data.Use.Name      = filename_token; /* see comment about filename_constant above */
  /*s->Data.Use.Element = create_exp_token(create_token(WNameOf(procnode->p_procname)));*/
  s->Data.Use.Element   = procnode->p_procnameexp;
  s->Data.Use.OccamData = occam_object;

  add_statement(s);
}
/*}}}*/
/*{{{  PRIVATE namedesc_t *find_arc_edge*/
PRIVATE namedesc_t *find_arc_edge(const connection_t *const conn)
/* returns the namedesc of the connecting edge */
{
  namedesc_t *edgename = NULL;
  int i;
  for (i = 0; i < 2; i++)
    if (conn->c_edge[i].e_link == EDGENODE_NO_LINK)
      edgename = (namedesc_t *)&conn->c_edge[i].e_un.e_edge;
  return edgename;
}
/*}}}*/
/*{{{  PRIVATE void create_and_place_soft_edge*/
PRIVATE void create_and_place_soft_edge(const namedesc_t *const edgename, const int input)
{
  Statement *s;
  char edge_chan[MAXSTRING_SIZE];

  parseable_str(edge_chan, edgename);
  strcat(edge_chan, input ? INPUT_SUFFIX : OUTPUT_SUFFIX);

  s = CreateStatement(input ? D_INPUT : D_OUTPUT);
  s->Data.Names = create_list(create_exp_token(create_token(edge_chan)), NULL);
  add_statement(s);

  create_place_statement(create_exp_token(create_token(edge_chan)),
                         name_to_exp(edgename));

  if (config_srcout)
    {
      fprintf(outfile, "%s %s;\n", input ? INPUT_KEYWORD : OUTPUT_KEYWORD, edge_chan);
      fprintf(outfile, "place %s on ", edge_chan);
      print_name_to_str(edgename);
      fputs(";\n", outfile);
    }
}
/*}}}*/
/*{{{  PRIVATE Expression *write_channame*/
PRIVATE Expression *write_channame(const chan_dir_t *const chandir,
                                   const namedesc_t *const edgename,
                                   const char *const suffix)
{
  if (chandir->c_proc == NULL)
    {
      char str[MAXSTRING_SIZE];
      parseable_str(str, edgename);
      strcat(str, suffix);
      if (config_srcout) fputs(str, outfile);
      return create_exp_token(create_token(str));
    }
  else
    {
      Expression *e = name_to_exp(&chandir->c_id);
      Expression **base = &e;

      while ((*base)->Type == E_SUBSCRIPT)
        base = &((*base)->Data.Binary.Expr1);

      *base = create_binary_exp(E_SUBFIELD,
                /*create_exp_token(create_token(WNameOf(chandir->c_proc->p_procname))),*/
                chandir->c_proc->p_procnameexp,
                *base);

      if (config_srcout)
        {
          fputs(WNameOf(chandir->c_proc->p_procname), outfile);
          fputc('.', outfile);
          print_name_to_str(&chandir->c_id);
        }
      return e;
    }
}
/*}}}*/
/*{{{  PRIVATE void place_arc*/
PRIVATE void place_arc(const chanswnode_t *const channode, const int dirn_index)
{
  if (channode->c_dirn[dirn_index].c_proc != NULL)
    {
      /* We have to be careful about a channel being placed onto a link
         which goes from a processor and straight back to it.
         So we use the arc usage info which was created earlier.
      */
      Expression *channel, *link;
      archwnode_t *arc = channode->c_arc;
      edgedesc_t *thisedge = NULL;
      int i;

      for (i = 0; i < 2; i++)
        if (arc->a_usage[i][dirn_index] == channode)
          thisedge = &arc->a_conn->c_edge[i];
      /* Now thisedge will point to an edge of this processor */

      if (config_srcout) fputs("place ", outfile);
      channel = write_channame(&(channode->c_dirn[dirn_index]), NULL, NULL);
      if (config_srcout) fputs(" on ", outfile);
      link    = write_edgedesc(thisedge);
      if (config_srcout) fputs(";\n", outfile);
      create_place_statement(channel, link);
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM void write_chan_sw*/
PRIVATEPARAM void write_chan_sw(const namedesc_t *const name)
{
  const chanswnode_t *const channode = channodeof(name);
  namedesc_t *edgename = NULL;

  /* set edgename if this channel joins a processor to an edge */
  if ((channode->c_arc != NULL) &&
      ((channode->c_dirn[DIRN_IN ].c_proc != NULL) ||
       (channode->c_dirn[DIRN_OUT].c_proc != NULL)))
    {
      /* If the channel is used, and it is placed onto an arc, we have to
         make sure that the placement occurs.
         If connected to an edge, declare a 'soft channel' mapped to that edge.
         Otherwise explicitly place the two ends of the channel onto
         the required links.
      */
      edgename = find_arc_edge(channode->c_arc->a_conn);
      if (edgename != NULL)
        create_and_place_soft_edge(edgename, channode->c_dirn[DIRN_IN].c_proc != NULL);
      else
        {
          place_arc(channode, DIRN_IN);
          place_arc(channode, DIRN_OUT);
        }
    }

  /* we generate a connect statement if:
       a) the channel is placed on an arc to an edge, or
       b) the channel is used for input and for output, and the channel has not
          been optimised to a local.
     it has been optimised to a local if:
       a) it is a scalar, and b) it connects same processor to same, and
       c) it has not been placed onto an arc.
  */
  if ((edgename != NULL) ||
      (!empty_name(&(channode->c_dirn[DIRN_IN ].c_id)) &&
       !empty_name(&(channode->c_dirn[DIRN_OUT].c_id)) &&
       !((TagOf(NTypeOf(name->n_name)) == S_CHAN)    &&
         (channode->c_dirn[DIRN_IN].c_proc == channode->c_dirn[DIRN_OUT].c_proc) &&
         (channode->c_arc == NULL))                     ) )
    {
      Statement *s = CreateStatement(D_CONNECT);

      if (config_srcout) fputs("connect ", outfile);
      s->Data.Connect.Channel1 = write_channame(&(channode->c_dirn[DIRN_IN]),
                                 edgename, OUTPUT_SUFFIX);
      if (config_srcout) fputs(", ", outfile);
      s->Data.Connect.Channel2 = write_channame(&(channode->c_dirn[DIRN_OUT]),
                                 edgename, INPUT_SUFFIX);
      if (config_srcout) fputs(";\n", outfile);

      add_statement(s);
    }
}
/*}}}*/
/*{{{  PRIVATE void write_config_sw*/
PRIVATE void write_config_sw(void)
{
  ordercode_token = create_token(ORDERCODE_ATTR_WORD);
  ordervs_token   = create_token(ORDERVS_ATTR_WORD);
  interface_token = create_token(INTERFACE_ATTR_WORD);

  process_id        = declare_node(PROCESS_TYPE_WORD, PROCESS_ATTR_WORD);
  filename_constant = create_constant_str(objfilename, strlen(objfilename));
  filename_token    = create_token_const(filename_constant);

  walk_config_list(proclist, write_proc_sw);
  walk_config_list(chanlist, write_chan_sw);
}
/*}}}*/
/*{{{  PRIVATE Statement *create_compound_statement*/
PRIVATE Statement *create_compound_statement(void)
{
  Statement *s;
  FastMalloc = TRUE; /* This turns on the backend's memory manager */
  s = CreateStatement(D_COMPOUND);
  next_statement = &(s->Data.Statements);
  return s;
}
/*}}}*/
/*{{{  PRIVATE void end_compound_statement*/
#if 1
PRIVATE void end_compound_statement(void)
{
  /* nothing to do */
  FastMalloc = FALSE; /* This turns off the backend's memory manager */
}
#else
#define end_compound_statement() /* empty */
#endif
/*}}}*/

/*{{{  PRIVATE void call_backend*/
PRIVATE void call_backend(Statement *const statement)
{
  HardChannel *Channel;

  if (information || config_info)
    {
      print_memstats(); /* supplies some diagnostics */
      ToolName = ""; /* set backend's string */
      fputs("Processing final configuration with backend", outfile);
      /*fputs("Backend version: ", outfile);*/
      OutputVersion(outfile);
      ToolName = compilername; /* set backend's string to same as the frontend's */
    }

  if (config_info)
    fputs("Calling EvalStatement\n", outfile);
  EvalStatement(statement);

  if (config_info)
    fputs("Calling CheckNetwork\n", outfile);
  if ((Channel = CheckNetwork()) != NULL)
    {
      if (config_info)
        fputs("Calling SetNetworkMap\n", outfile);
      SetNetworkMap(Channel);
    }

  if (ErrorCount == 0)
    {
      if (information || config_info)
        fprintf(outfile, "Writing file \"%s\"\n", cfbfilename);

      /* open the output file */
      OutputFileName = cfbfilename;
      OutputFileStream = open_object_file(OutputFileName);

      if (config_info)
        fputs("Calling WriteBinaryFile\n", outfile);
      WriteBinaryFile(OutputFileStream, Channel);

      /* close binary file */
      close_object_file(OutputFileStream, OutputFileName);
    }
  if (ErrorCount != 0)
    {
      remove(objfilename);
      remove(OutputFileName);
      end_compiler(EXIT_FAILURE);
    }
}
/*}}}*/

/*{{{  PUBLIC void write_config_desc*/
PUBLIC void write_config_desc(void)
{
  Statement *backend_statement;

  if (information || config_info)
    {
      print_memstats(); /* supplies some diagnostics */
      fputs("Creating final configuration\n", outfile);
    }

  if (config_backend)
    {
      char *root_proc_name = NULL;
      if (config_rom)
        {
          root_proc_name = WNameOf(NNameOf(root_processor->p_name.n_name));
          RomRunRomFlag  =  config_rom_in_rom;
          RomRunRamFlag  = !config_rom_in_rom;
          RomMemorySize  = romsize;
        }

      ProfileModeFlag    = config_profile; /* this is almost totally unsupported */
      SimpleProcessFlag  = config_postmortem && !ProfileModeFlag;
      OverlayProcessFlag = config_overlay    && !ProfileModeFlag;
      ExecuteManyFlag    = config_repeat;
      ToolName = compilername; /* set backend's string to same as the frontend's */
      SetHostDefaults();   /* sets stderr etc */
    #if HIDE_BACKEND_NAMES
      SetOccamIdentifierTable(WNameOf(NNameOf(hostedgeptr)), root_proc_name);
    #else
      /* modify HOST's name to be compatible to with the backend's name */
      SetNName(hostedgeptr, lookupword(HOSTNAME, strlen(HOSTNAME)));
      SetIdentifierTable(root_proc_name);
    #endif
      SetSymbolTable();
    }

  backend_statement = create_compound_statement();

  if (config_srcout)
    fprintf(outfile, "/* %s */\n/* Configuration source generated from \"%s\" */\n",
            C_VERSION, sourcefilename);

  if (config_srcout) fputs("\n/* Hardware description: */\n", outfile);
  write_config_hw();

  if (config_srcout) fputs("\n/* Software description: */\n", outfile);
  write_config_sw();

  end_compound_statement();

  if (information || config_info)
    {
      fprintf(outfile, "Created configuration for %ld processors", hw_proc_count);
      if (config_info)
        fprintf(outfile, " and %ld processes", sw_proc_count);
      fputc('\n', outfile);
    }

  if (config_reclaim_mem)
    freeup_all_workspace(); /* clobbers ALL `newvec' memory */

  if (config_backend_dump)
    {
      DiagnosticStream = outfile;
      DumpStatement(backend_statement);
    }

  if (config_backend)
    call_backend(backend_statement);
}
/*}}}*/
