/*#define DEBUG*/
/******************************************************************************
*
*  configurer main body, and software parts
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1990
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lexconst.h"
# include "lexdef.h"
# include "lex1def.h"
# include "chkdef.h"
# include "syndef.h"
# include "syn1def.h"
# include "chkerror.h"
# include "usehdr.h"
# include "usedef.h"
# include "use1def.h"
# include "confhdr.h"
# include "confdef.h"
# include "conf1def.h"
# include "conf2def.h"
# include "gen1def.h"
# include "deschdr.h"
# include "desc1def.h"
# include "code1def.h"
# include "debugdef.h"
# include "bind3def.h"
/*}}}*/

/*{{{  PUBLIC variables*/
PUBLIC int config_info           = FALSE;
PUBLIC int config_code           = TRUE;
PUBLIC int config_srcout         = FALSE;
PUBLIC int config_reclaim_mem    = TRUE;
PUBLIC int config_readable_names = FALSE; /* formal param names */

PUBLIC char cfbfilename[MAX_FILENAME_LENGTH];
/*}}}*/

/*{{{  PRIVATE variables*/
/* this is used to remember the function which was passed to
   expandwalkproctree
*/
PRIVATE int (*expandwalkproctree_f)(treenode *);
PRIVATE treenode *expand_speclist;
PRIVATE treenode **expand_lastspec;
PUBLIC  treenode *expand_repllist;

PRIVATE treenode *speclist;
PRIVATE treenode *netlist;
PRIVATE treenode *configptr = NULL;
PRIVATE treenode *mapptr    = NULL;
PRIVATE treenode **lastspec;
PRIVATE treenode **lastnet;

#define EXTERNAL_CHAN_LEXLEVEL 1  /* inside a CONFIG construct */

PRIVATE int modify_chan_mode;
PRIVATE int any_unknown_dirn = FALSE;

 /* This is used to generate unique names for formal parameters */
PRIVATE int formalparam_count;

/* This is TRUE when collecting and immediately compiling processor bodies
   for those processors which are only `used' once.
*/
PRIVATE int collect_singles;

/* This holds the node of the proc being compiled */
PRIVATE procswnode_t *current_procswnode;
/* This holds that proc's name */
/*PRIVATE const namedesc_t *current_procname;*/

/* This flags whether to turn accesses to single elements of global channel
   arrays into scalar parameters rather than [1]CHAN parameters.
*/
/* Temporarily set to FALSE while investigating bug 835 */
#define OPTIMISE_SINGLES TRUE
/*}}}*/

/*{{{  forward declarations*/
PRIVATEPARAM int do_expandwalkproctree(treenode *tptr);
/*}}}*/

/*{{{  PRIVATE int expandwalkproctree_if(treenode *tptr)*/
PRIVATE int expandwalkproctree_if(treenode *tptr)
/*
  Interprets an IF construct.
  Returns TRUE if a branch was chosen
*/

{
  if (TagOf(tptr) != S_IF)
    badtag(LocnOf(tptr), TagOf(tptr), "expandwalkproctree_if");
  tptr = CBodyOf(tptr);
  while (!EndOfList(tptr))
    {
      treenode *t = ThisItem(tptr);
      while (isspecification(t))
        {
          if ((*expandwalkproctree_f)(t) != CONTINUE_WALK)
            {
              tptr = NextItem(tptr);  /* point at next guard */
              if (EndOfList(tptr))
                return FALSE;
              t = ThisItem(tptr);
            }
          else
            t = DBodyOf(t);
        }
      if (TagOf(t) == S_CHOICE)
        {
          if (!is_evaluable(CondGuardOf(t)))
            chkreport(CONF_CANNOT_EVAL_EXP, LocnOf(CondGuardOf(t)));
          if (evaluate(CondGuardOf(t)) != 0)
            {
              prewalkproctree(CondBodyOf(t), do_expandwalkproctree);
              return TRUE;
            }
        }
      else  /* S_IF or S_REPLIF */
        {
          if (expandwalkproctree_if(t))
            return TRUE;
        }
      tptr = NextItem(tptr);  /* point at next guard */
    }
  return FALSE;
}
/*}}}*/
/*{{{  PRIVATE int spec_needed_for_compile*/
PRIVATE int spec_needed_for_compile(treenode *tptr)
{
  switch(TagOf(tptr))
    {
      case S_PROCDEF: case S_LFUNCDEF: case S_SFUNCDEF:
        return TRUE;
      case S_VALABBR: case S_VALRETYPE:
        return (TagOf(NTypeOf(DNameOf(tptr))) == S_ARRAY);
      default:
        return FALSE;
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM int do_expandwalkproctree*/
PRIVATEPARAM int do_expandwalkproctree(treenode *tptr)
{
  /*DEBUG_MSG(("do_expandwalkproctree: %s\n", itagstring(TagOf(tptr))));*/
  if ((*expandwalkproctree_f)(tptr) != CONTINUE_WALK)
    return STOP_WALK;
  switch(nodetypeoftag(TagOf(tptr)))
    {
      default:
        break;
      /*{{{  abbreviations etc*/
      case DECLNODE:
        if (spec_needed_for_compile(tptr)) /* save this spec onto expand_speclist */
          {
            treenode **saved_lastspec = expand_lastspec;
            treenode *body = DBodyOf(tptr);
            *expand_lastspec = tptr;
            SetDBody(tptr, NULL); /* terminate the list */
            expand_lastspec = DBodyAddr(tptr); /* point at next hole */
      
            prewalkproctree(body, do_expandwalkproctree);
      
            SetDBody(tptr, body);
            expand_lastspec = saved_lastspec; /* Take tptr off the list */
            *expand_lastspec = NULL; /* terminate the one above it in the list */
      
            return STOP_WALK;
          }
        break;
      /*}}}*/
      /*{{{  replicators*/
      case REPLCNODE:
        {
          treenode *name = ReplCNameOf(tptr);
          assert(TagOf(tptr) != S_REPLIF);
          if (is_evaluable(ReplCStartExpOf(tptr)) &&
              is_evaluable(ReplCLengthExpOf(tptr)))
            {
              INT32 base  = evaluate(ReplCStartExpOf(tptr));
              INT32 count = evaluate(ReplCLengthExpOf(tptr));
              DEBUG_MSG(("expanding repl: base: %ld, count: %ld\n", base, count));
              expand_repllist = newlistnode(S_LIST, NOPOSN, name, expand_repllist);
              tptr = ReplCBodyOf(tptr);
              SetNReplKnown(name, TRUE);
              for (; count > 0; count--)
                {
                  if (config_info)
                    fprintf(outfile, "Setting replicator %s to %ld\n",
                            WNameOf(NNameOf(name)), base);
                  SetNReplValue(name, base);
                  prewalkproctree(tptr, do_expandwalkproctree);
                  base++;
                }
              if (config_info)
                fprintf(outfile, "Unsetting replicator %s\n", WNameOf(NNameOf(name)));
              SetNReplKnown(name, FALSE);
              {
                treenode *old_repllist = expand_repllist;
                expand_repllist = NextItem(expand_repllist);
                if (config_reclaim_mem)
                  freenode(&old_repllist);
              }
              return STOP_WALK;
            }
          else
            {
              if (config_info)
                fprintf(outfile, "Setting replicator %s to unknown\n",
                        WNameOf(NNameOf(name)));
              SetNReplKnown(name, FALSE);
            }
        }
        break;
      /*}}}*/
      /*{{{  CNODE*/
      case CNODE:
        if (TagOf(tptr) == S_IF)
          {
            if (!expandwalkproctree_if(tptr))
              chkreport(CONF_EVAL_STOP, LocnOf(tptr));
            return STOP_WALK;
          }
        break;
      /*}}}*/
      /*{{{  leaf*/
      case LEAFNODE:
        if (TagOf(tptr) == S_STOP)
          chkreport(CONF_EVAL_STOP, LocnOf(tptr));
        break;
      /*}}}*/
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PUBLIC void expandwalkproctree*/
PUBLIC void expandwalkproctree(treenode *tptr, int (*f1)(treenode *))
/*****************************************************************************
*
* expandwalkproctree
*  This is designed to be called on a tree whose replicators are all
*  constant, and it will go through IF constructs, just pulling
*  out the correct branches.
*
*****************************************************************************/
{
  int (*saved_f)(treenode *) = expandwalkproctree_f;
  expandwalkproctree_f = f1;
  expand_repllist = NULL;
  expand_speclist = NULL;
  expand_lastspec = &expand_speclist;
  prewalkproctree(tptr, do_expandwalkproctree);
  expandwalkproctree_f = saved_f;
}
/*}}}*/


/*{{{  PUBLIC void map_to_basename*/
PUBLIC void map_to_basename(treenode *tptr, namedesc_t *name)
/*****************************************************************************
*
* map_to_basename takes a (treenode *) element and returns the name descriptor
*                 of the first item used.
*
*****************************************************************************/
{
  treenode *const basename = nameof(tptr);
  INT32 index;
  if (TagOf(NTypeOf(basename)) == S_ARRAY)
    {
      INT32 last;
      subscripts_accessed(tptr, &index, &last, TRUE); /* report errors */
      /*DEBUG_MSG(("map_to_basename: %s subscripts accessed are: %ld to %ld\n",
                 WNameOf(NNameOf(basename)), index, last));
      */
    }
  else
    index = 0;
  name->n_name  = basename;
  name->n_index = index;
}
/*}}}*/
/*{{{  PUBLIC void name_to_str*/
PRIVATE void name_to_str_either(char *string, const namedesc_t *const name,
                                const int parseable)
{
  if (empty_name(name))
    strcpy(string, "NULL"); /* safety check */
  else
    {
      char *s;
      treenode *t = NTypeOf(name->n_name);
      INT32 i     = name->n_index;

      strcpy(string, WNameOf(NNameOf(name->n_name)));
      s = (&string[0]) + WLengthOf(NNameOf(name->n_name));

      if (parseable)
        {
          if (i != 0)
            {
              sprintf(s, "%s%ld", PARSEABLE_SEPARATOR, i);  /* parseable for C */
              s += strlen(s);
            }
        }
      else
        while (TagOf(t) == S_ARRAY)
          {
            INT32 size;
            t = ARTypeOf(t);
            size = elementsin(t);
            sprintf(s, "[%ld]", i / size);
            s += strlen(s);
            i = i % size;
          }
    }
}
PUBLIC void name_to_str(char *string, const namedesc_t *const name)
/*****************************************************************************
*
* name_to_str returns a printable representation of a name descriptor.
*
*****************************************************************************/
{
  name_to_str_either(string, name, FALSE);
}
PUBLIC void parseable_str(char *string, const namedesc_t *const name)
/*****************************************************************************
*
* parseable_str returns an occam syntax name for a name descriptor
*
*****************************************************************************/
{
  name_to_str_either(string, name, TRUE);
}
PUBLIC void print_name_to_str(const namedesc_t *const name)
{
  char str[MAXSTRING_SIZE];
  name_to_str_either(str, name, FALSE);
  fputs(str, outfile);
}
/*}}}*/

/*{{{  PRIVATE int protocol_has_int()*/
PRIVATE int protocol_has_int(const treenode *const p)
/*****************************************************************************
*
* protocol_has_int returns TRUE if the protocol has any INT inside it
*
*****************************************************************************/
{
  if (p == NULL)  /* end of a list */
    return FALSE;
  switch (TagOf(p))
    {
      case S_INT:
        return TRUE;
      case S_COLON2:
        return protocol_has_int(LeftOpOf(p)) || protocol_has_int(RightOpOf(p));
      case N_SPROTDEF: case N_TPROTDEF:
        return protocol_has_int(NTypeOf(p));
      case S_LIST: /* part of a sequential or tagged protocol */
        return protocol_has_int(ThisItem(p)) || protocol_has_int(NextItem(p));
      case N_TAGDEF:
        return protocol_has_int(NTypeOf(p)); /* The seqential bit following the tag */
      default:
        return FALSE;
    }
}
/*}}}*/
/*{{{  PRIVATE int dirn_from_mode*/
PRIVATE int dirn_from_mode(const int mode)
{
  return (mode == CHAN_USE_INPUT) ? DIRN_IN : DIRN_OUT;
}
/*}}}*/
/*{{{  PRIVATE int mode_from_dirn*/
PRIVATE int mode_from_dirn(const int dirn)
{
  return (dirn == DIRN_IN) ? CHAN_USE_INPUT : CHAN_USE_OUTPUT;
}
/*}}}*/
/*{{{  PRIVATE int determine_unknown_direction*/
PRIVATE int determine_unknown_direction(const chanswnode_t *const channode,
                                        const procswnode_t *const procnode)
{
  int i;
  int dirn = (-1);
  for (i = DIRN_START; i < DIRN_END; i++)
    if (channode->c_dirn[i].c_proc == procnode)
      dirn = i;
  return dirn;
}
/*}}}*/
/*{{{  PRIVATE void mark_chan_direction*/
PRIVATE void mark_chan_direction(const SOURCEPOSN locn, const char *const procstr,
  procswnode_t *procnode, const namedesc_t *const channame, int mode)
/*****************************************************************************
*
* mark_chan_direction marks a channel as being used by that processor
*
*          if this completes a connection, we also check the properties
*          of that connection
*
*****************************************************************************/
{
  chanswnode_t *const channode = channodeof(channame);
  char chanstr[MAXSTRING_SIZE];
  name_to_str(chanstr, channame);

  if (mode == 0) /* unknown direction, so 'guess' */
    {
      int i;
      if (determine_unknown_direction(channode, procnode) != (-1))
        return; /* already mapped to this processor; leave it at that */
        
      mode = mode_from_dirn(DIRN_IN); /* if none empty, force an error a little later */
      for (i = DIRN_START; i < DIRN_END; i++)
        if (channode->c_dirn[i].c_proc == NULL)
          mode = mode_from_dirn(i);
    }

  {
    procswnode_t **connection;
    char *dirstring = (mode == CHAN_USE_INPUT) ? "input" : "output";

    if (config_info)
      fprintf(outfile, "Channel %s used for %s by processor %s\n", chanstr,
              dirstring, procstr);

    connection = &(channode->c_dirn[dirn_from_mode(mode)].c_proc);
    if (*connection != NULL)
      chkerr_ss(CONF_CHAN_MULTIPLY_USED, locn, chanstr, dirstring);

    (*connection) = procnode;
  }

  check_arc_status(procnode, locn, channode, dirn_from_mode(mode));

  if ((channode->c_dirn[DIRN_IN].c_proc != NULL) && (channode->c_dirn[DIRN_OUT].c_proc != NULL) &&
      (channode->c_dirn[DIRN_IN].c_proc != channode->c_dirn[DIRN_OUT].c_proc))
    {
      int status;
      int bpw[2];
      int i;
      for (i = DIRN_START; i < DIRN_END; i++)
        bpw[i] = proc_wordlength(&(channode->c_dirn[i].c_proc->p_name));

      /* check that a protocol containing INT is not used between two different
         wordlengths */
      if (bpw[DIRN_IN] != bpw[DIRN_OUT])
        {
          const treenode *const protocol = ProtocolOf(basetype_tree(NTypeOf(channame->n_name)));
          if (protocol_has_int(protocol))
            chkerr_s(CONF_INVALID_PROTOCOL, LocnOf(channame->n_name),
                    chanstr);
        }

      /* check that a hardware connection is ok */
      status = connection_status(&(channode->c_dirn[DIRN_OUT].c_proc->p_name),
                                 &(channode->c_dirn[DIRN_IN ].c_proc->p_name));
      if (status != CHK_OK)
        chkerr_ss(status, locn, procstr, chanstr);
    }
}
/*}}}*/
/*{{{  PRIVATE void mark_chan_directions*/
PRIVATE void mark_chan_directions(const SOURCEPOSN locn, const char *const procstr,
           procswnode_t *procnode,
           treenode *const name, const subscrlist *subs, const int mode)
/*****************************************************************************
*
* mark_chan_directions marks a list of channels as being used by that processor
*
*****************************************************************************/
{
  namedesc_t channame;

  channame.n_name  = name;
  channame.n_index = 0;

  if (TagOf(NTypeOf(name)) == S_ARRAY)
    {
      const INT32 len = elementsin(NTypeOf(name));
      while (subs != NULL)
        {
          INT32 first = SLFirstOf(subs), last = SLLastOf(subs);
          if ((first == 0) && (last == MAXINDEX))
            last = len - 1;
          for (; first <= last; first++)
            {
              channame.n_index = first;
              /* range check added for bug 846 15/1/91 */
              if (first >= len)
                {
                  char str[MAXSTRING_SIZE];
                  name_to_str(str, &channame);
                  chkerr_s(CONF_SUBSCRIPT_OUT_OF_RANGE, locn, str);
                  return; /* don't do any more for this channel */
                }
              mark_chan_direction(locn, procstr, procnode, &channame, mode);
            }
          subs = SLNextOf(subs);
        }
    }
  else
    mark_chan_direction(locn, procstr, procnode, &channame, mode);
}
/*}}}*/
/*{{{  PUBLIC void declare_process*/
PUBLIC void declare_process(treenode *const tptr)
/*****************************************************************************
*
* declare_process collects data about free channels used by this
*                   processor.
*
*****************************************************************************/
{
  namedesc_t procname;
  procswnode_t *procnode;
  varlist *free;
  char procstr[MAXSTRING_SIZE];

  map_to_basename(ProcessorExpOf(tptr), &procname);
  if (procname.n_index < 0) return;
  name_to_str(procstr, &procname);

  free = freevarsin(ProcessorBodyOf(tptr), -1, NULL, FALSE);
  if (config_info)
    {
      fprintf(outfile, "PROCESSOR %s, freelist: ", procstr);
      printfreelist(free);
    }

  if (!valid_procswnode(&procname))
    {
      chkerr_s(CONF_LOGICAL_NOT_MAPPED, LocnOf(tptr), procstr);
      return;
    }

  procnode = procswnodeof(&procname);
  check_proc_hw(&procnode->p_name); /* check that the processor is ready to accept software */

  procnode->p_varlist = merge(free, -1, procnode->p_varlist);
  procnode->p_count++;

  for (; free != NULL; free = VLNextOf(free))
    {
      treenode *const name = VLNameOf(free);
      if (basetype(NTypeOf(name)) == S_CHAN)
        {
          if (VLReadOf(free) != NULL)
            {
              msg_out_ss(SEV_WARN, CHK, CONF_CHAN_NO_DIRECTION, LocnOf(tptr),
                         WNameOf(NNameOf(name)), procstr);
              any_unknown_dirn = TRUE;
            }
          if (VLInputOf(free) != NULL)
            mark_chan_directions(LocnOf(tptr), procstr, procnode,
                                 name, VLInputOf(free), CHAN_USE_INPUT);
          if (VLOutputOf(free) != NULL)
            mark_chan_directions(LocnOf(tptr), procstr, procnode,
                                 name, VLOutputOf(free), CHAN_USE_OUTPUT);
        }
    }
  /*if (testflag)*/
  freeup_temp_workspace();
}
/*}}}*/
/*{{{  PRIVATEPARAM void do_process_unknown_directions*/
PRIVATEPARAM void do_process_unknown_directions(const namedesc_t *const procname)
{
  procswnode_t *const procnode = procswnodeof(procname);
  const varlist *free = procnode->p_varlist;
  if (free != NULL)
    {
      char procstr[MAXSTRING_SIZE];
      name_to_str(procstr, procname);
      for (; free != NULL; free = VLNextOf(free))
        {
          treenode *name = VLNameOf(free);
          if ((basetype(NTypeOf(name)) == S_CHAN) && (VLReadOf(free) != NULL))
            {
            /*  mark_chan_directions(LocnOf(name), procstr, procnode,
                                   name, VLInputOf(free), 0); */
              mark_chan_directions(LocnOf(name), procstr, procnode,
                                   name, VLReadOf(free), 0); /* bug 1084 2/1/91 */
            }
        }
      /*if (testflag)*/
      freeup_temp_workspace();
    }
}
/*}}}*/
/*{{{  PRIVATE void process_unknown_directions*/
PRIVATE void process_unknown_directions(void)
/*****************************************************************************
*
* process_unknown_directions decides which directions to use for any channels
*                            which aren't explicitly used.
*
*****************************************************************************/
{
  if (any_unknown_dirn)
    {
      if (config_info)
        fputs("Processing unknown directions\n", outfile);
      walk_config_list(proclist, do_process_unknown_directions);
    }
}
/*}}}*/

/*{{{  PRIVATE void set_name_translations*/
PRIVATE void set_name_translations(const namedesc_t *const thisname, const int mode,
                                   treenode *const new, const int i)
{
  chanswnode_t *const thisnode = channodeof(thisname);
  int dirn;
  for (dirn = DIRN_START; dirn < DIRN_END; dirn++)
    if ((mode & mode_from_dirn(dirn)) != 0)
      {
        thisnode->c_dirn[dirn].c_id.n_name  = new;
        thisnode->c_dirn[dirn].c_id.n_index = i;
      }
}
/*}}}*/
/*{{{  PRIVATE void create_formal*/
PRIVATE void create_formal(procswnode_t *const procnode,
                           const namedesc_t *const channame, const int newname,
                           treenode *const type, int mode)
/*****************************************************************************
*
* create_formal creates a new formal parameter, and adds
*               its name into the formal parameter list
*
*****************************************************************************/
{
  treenode *new;
  wordnode *name = NNameOf(channame->n_name);

  if (mode == 0) /* pick the direction from the known details of the channel */
    mode = mode_from_dirn(determine_unknown_direction(channodeof(channame), procnode));

  /* if newname is TRUE, we MUST modify the name so that we can distinguish
     input and output names */
  /* OR: a simple channel name might clash with a reserved attribute if it
     doesn't already have a dot in it! */
  if (!config_readable_names)
    {
      /* for large networks, this is more memory efficient for the backend */
      char str[MAXSTRING_SIZE];
      sprintf(str, "p%s%d", PARSEABLE_SEPARATOR, formalparam_count);
      formalparam_count++;
      name = lookupword(str, strlen(str));
    }
  else if (newname || (strchr(WNameOf(name), '.') == NULL))
    {
      char str[MAXSTRING_SIZE];
      parseable_str(str, channame);
      strcat(str, (mode == CHAN_USE_INPUT) ? INPUT_SUFFIX : OUTPUT_SUFFIX);
      name = lookupword(str, strlen(str));
    }

  lexlevel++;  /* to make these look like formal parameters */
  new = declname(N_PARAM, NOPOSN, name, copytree(type), NULL);
  SetNVDirectionOf(new, mode);
  lexlevel--;

  {
    int i;
    namedesc_t thisname = *channame;
    const INT32 length = elementsin(type);
    for (i = 0; i < length; i++)
      {
        set_name_translations(&thisname, mode, new, i);
        thisname.n_index++;
      }
  }

  procnode->p_fparams = addtofront(new, procnode->p_fparams);
}
/*}}}*/
/*{{{  PRIVATE void create_decl*/
PRIVATE void create_decl(procswnode_t *const procnode, const namedesc_t *const channame,
                         wordnode *const newname, treenode *const type)
/*****************************************************************************
*
* create_decl creates a new local declaration, and adds
*               its name into the locals list
*
*****************************************************************************/
{
  treenode *dptr;
  lexlevel++;  /* to make these look like local vars */
  dptr = declare(S_DECL, NOPOSN, copytree(type), newname, NULL);
  lexlevel--;

  set_name_translations(channame, CHAN_USE_BIDIRECTIONAL, DNameOf(dptr), 0);

  SetDBody(dptr, procnode->p_locals);
  procnode->p_locals = dptr;
}
/*}}}*/
/*{{{  PRIVATE void createchanarray*/
/*****************************************************************************
 *
 *  createchanarray: this converts a name which is used as an array of channels
 *                   into a formal parameter
 *
 *****************************************************************************/
PRIVATE void createchanarray ( procswnode_t *const procnode, 
        const namedesc_t *const channame, treenode *type, const int mode)
{
  const INT32 length = elementsin(type);

#if OPTIMISE_SINGLES /* while investigating bug 835 */
  if (length == 1)  /* optimisation to turn into a single element */
    {
    #ifdef DEBUG
      if (config_info)
        fprintf(outfile, "createchanarray: %s, converted to single element\n",
                         WNameOf(NNameOf(channame->n_name)));
    #endif
      while (TagOf(type) == S_ARRAY)
        type = ARTypeOf(type);
    }
#endif

#ifdef DEBUG
  if (config_info)
    {
      fprintf(outfile, "createchanarray: %s, length: %ld",
                       WNameOf(NNameOf(channame->n_name)), length);
      if (prtree)
        printtree(0, type);
      fputc('\n', outfile);
    }
#endif

  create_formal(procnode, channame, TRUE, type, mode);
}
/*}}}*/
/*{{{  PRIVATE void factor()*/
/*****************************************************************************
 * factor :
 *   Given a dimension tree and a segment of a 1D array
 *   produce a set of new arrays which factorise the
 *   segment using the dimension tree.
 *****************************************************************************/
PRIVATE void factor (procswnode_t *const procnode, namedesc_t *const name,
                     const INT32 start, const INT32 length, treenode *const dims, const int mode)
{
  assert(start >= 0 && length >= 0);
#ifdef DEBUG
  if (config_info)
    {
      fprintf(outfile, "factoring %s; start: %ld, length: %ld",
              WNameOf(NNameOf(name->n_name)), start, length);
      if (prtree)
        printtree(0, dims);
      fputc('\n', outfile);
    }
#endif
  if (length > 0)
    {
      const INT32 size = elementsin(ARTypeOf(dims));
      if (TagOf(ARTypeOf(dims)) != S_ARRAY)
        /*{{{  declare a 1d array for remaining elements*/
        {
          treenode *newtype =
                    newarraynode(S_ARRAY, NOPOSN, newconstant(length), ARTypeOf(dims));
          SetARDim(newtype, length);
          /* fprintf(outfile, "(%d)[%d]\n", start, length);*/
          name->n_index = start;
          createchanarray(procnode, name, newtype, mode);
        }
        /*}}}*/
      else if (length < size)
        factor(procnode, name, start, length, ARTypeOf(dims), mode);
      else
        /*{{{  factorise the array*/
        {
          const INT32 end        = start+length;
          const INT32 startalign = (size - (start % size)) % size;
          const INT32 endalign   = end % size;
          const INT32 arraysize  = (length-(startalign+endalign))/size;
          factor(procnode, name, start, startalign, ARTypeOf(dims), mode);
          if (arraysize > 0)
            /*{{{  declare array*/
            {
              treenode *newtype =
                 newarraynode(S_ARRAY, NOPOSN, newconstant(arraysize), ARTypeOf(dims));
              SetARDim(newtype, arraysize);
              name->n_index = start+startalign;
              createchanarray(procnode, name, newtype, mode);
            }
            /*}}}*/
          factor(procnode, name,
                 end - endalign, endalign, ARTypeOf(dims), mode);
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PRIVATE void process_free_chan_array()*/
PRIVATE void process_free_chan_array(procswnode_t *const procnode,
        treenode *const nptr, const subscrlist *subscr, const int mode)
/*****************************************************************************
*
* process_free_chan_array takes an array which is used free, and converts it
*                         into multiple formal parameters
*
*****************************************************************************/
{
  namedesc_t channame;
  channame.n_name  = nptr;

  if ((SLFirstOf(subscr) == 0) && (SLLastOf(subscr) == MAXINDEX))
    /* the whole array is used */
    {
      channame.n_index = 0;
      createchanarray(procnode, &channame, NTypeOf(nptr), mode);
    }
  else
    {
      while (subscr != NULL)
        {
          factor(procnode, &channame,
                 SLFirstOf(subscr), SLLastOf(subscr)-SLFirstOf(subscr)+1,
                 NTypeOf(nptr), mode);
          subscr = SLNextOf(subscr);
        }
    }
}
/*}}}*/
/*{{{  PRIVATE int subscripts_are_included (sptr, start, end)*/
/*****************************************************************************
 *
 *  Test if the range 'start' to 'end' is completely included inside
 *  any subscript accesses in the list 'sptr'
 *  (c.f. function subscr_overlaps)
 *
 *****************************************************************************/
PRIVATE int subscripts_are_included (const subscrlist *sptr, const INT32 start, const INT32 end )
{
  for (;sptr != NULL; sptr = SLNextOf(sptr))
    {
      if ((start >= SLFirstOf(sptr)) && (end <= SLLastOf(sptr)))
        return(TRUE);
    }
  return(FALSE);
}
/*}}}*/
/*{{{  PRIVATE void process_unknown_chan_array*/
PRIVATE void process_unknown_chan_array(procswnode_t *const procnode,
                                        treenode *const nptr, varlist *free)
/* This goes through the list of `unknown' direction channels.
   For any which will not be covered by a real direction, it adds them to the list.
   NOTE - it relies on the fact that the unknown stuff will never be actually
   de-referenced, if the two segments are `overlapping'.
*/
{
  subscrlist *read;
  for (read = VLReadOf(free); read != NULL; read = SLNextOf(read))
    {
      const INT32 start = SLFirstOf(read);
      const INT32 end   = SLLastOf(read);
      if (!subscripts_are_included(VLInputOf(free),  start, end) &&
          !subscripts_are_included(VLOutputOf(free), start, end))
        {
          subscrlist temp;
          SetSLFirst(&temp, start);
          SetSLLast(&temp,  end);
          SetSLNext(&temp,  NULL);
          process_free_chan_array(procnode, nptr, &temp, 0);
        }
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM void create_translations*/
PRIVATEPARAM void do_create_translations(const namedesc_t *const name)
/*****************************************************************************
*
* do_create_translations adds translations for a single processor
*
*****************************************************************************/
{
  procswnode_t *procnode = procswnodeof(name);
  varlist *free = procnode->p_varlist;

  if (procnode->p_count == 0)
    return;

  if (config_info)
    {
      char procstr[MAXSTRING_SIZE];
      name_to_str(procstr, name);
      fprintf(outfile, "Creating translations for processor %s (current mem %ld)\n", procstr, tablesize());
    }

  formalparam_count = 0;

  while (free != NULL)
    {
      treenode *nptr = VLNameOf(free);
      if (basetype(NTypeOf(nptr)) == S_CHAN)
        {
          if (TagOf(NTypeOf(nptr)) == S_ARRAY)
            {
              if (config_info)
                fprintf(outfile, "Creating formal parameters for channel array %s\n",
                        WNameOf(NNameOf(nptr)));
              /*if (VLReadOf(free) != NULL)  -- removed for bug 1084 2/1/91
                process_free_chan_array(procnode, nptr, VLReadOf(free), 0);*/

              /* Note that we do the known ones first, so that if an unknown
                 range and a known range overlap, the known range ends up
                 with preference! (bug 1084 2/1/91)
                 Note too that bug 1072 is the same problem as bug 1084.
              */
              if (VLInputOf(free) != NULL)
                process_free_chan_array(procnode, nptr, VLInputOf(free), CHAN_USE_INPUT);
              if (VLOutputOf(free) != NULL)
                process_free_chan_array(procnode, nptr, VLOutputOf(free), CHAN_USE_OUTPUT);
              /* added for bug 1084 - 2/1/91 */
              process_unknown_chan_array(procnode, nptr, free);
            }
          else  /* single channels */
            {
              namedesc_t channame;
              int mode = 0;
              if (VLInputOf(free)  != NULL) mode |= CHAN_USE_INPUT;
              if (VLOutputOf(free) != NULL) mode |= CHAN_USE_OUTPUT;
              channame.n_name  = nptr;
              channame.n_index = 0;

              if (mode != CHAN_USE_BIDIRECTIONAL)
                {
                  if (config_info)
                    fprintf(outfile, "Creating formal parameter for channel %s\n",
                            WNameOf(NNameOf(nptr)));
                  create_formal(procnode, &channame, FALSE, NTypeOf(nptr), mode);
                }
              else if (channodeof(&channame)->c_arc != NULL)
                {
                  if (config_info)
                    fprintf(outfile, "Creating two formal parameters for channel %s\n",
                            WNameOf(NNameOf(nptr)));
                  create_formal(procnode, &channame, TRUE,
                                NTypeOf(nptr), CHAN_USE_INPUT);
                  create_formal(procnode, &channame, TRUE,
                                NTypeOf(nptr), CHAN_USE_OUTPUT);
                }
              else
                {
                  if (config_info)
                    fprintf(outfile, "Creating local declaration for channel %s\n",
                            WNameOf(NNameOf(nptr)));
                    create_decl(procnode, &channame, NNameOf(nptr), NTypeOf(nptr));
                }
            }
        }
      free = VLNextOf(free);
    }

#if 0 /* now we only do this as we're building the process bodies */
  /* create a copy of the outer level speclist */
  procnode->p_specs = copytree(speclist);  /* works ok on NULL trees */
#endif

  if (procnode->p_procname == NULL)
    {
      char s[MAXSTRING_SIZE];
      parseable_str(s, name);
      strcpy(&s[strlen(s)], PROCNAME_SUFFIX);
      procnode->p_procname = lookupword(s, strlen(s));
    }
  /*if (testflag)*/
    freeup_temp_workspace();
}
/*}}}*/
/*{{{  PRIVATE void create_translations*/
PRIVATE void create_translations(void)
/*****************************************************************************
*
* create_translations sets up the data for translating channels to parameters
*
*****************************************************************************/
{
  walk_config_list(proclist, do_create_translations);
}
/*}}}*/
/*{{{  PRIVATEPARAM int check_lib_instances*/
PRIVATEPARAM int check_lib_instances(treenode *tptr)
{
  switch (TagOf(tptr))
    {
      case S_PINSTANCE: case S_FINSTANCE:
        {
          treenode *name = INameOf(tptr);
          if (separatelycompiled(name))
            checklibproctype(name, LocnOf(tptr));
        }
        break;
      case S_RECORDSUB:
        chkerr_s(CHK_ILLEGAL_CONSTRUCT, LocnOf(tptr), tagstring(S_PROCESSOR));
        break;
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATEPARAM int check_nested_pri_par*/
PRIVATEPARAM int check_nested_pri_par(treenode *const tptr)
{
  switch (TagOf(tptr))
    {
      case S_PINSTANCE: case S_FINSTANCE:
        {
          treenode *const name = INameOf(tptr);
          if (separatelycompiled(name) && (NNestedPriParOf(name)))
            chkerr(CONF_NESTED_PRI_PAR, LocnOf(tptr));
        }
        break;
      case S_PRIPAR: case S_PRIREPLPAR:
        chkerr(CONF_NESTED_PRI_PAR, LocnOf(tptr));
        break;
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATE void merge_process*/
PRIVATE void merge_process(treenode **root, treenode *const new, const INT32 pri)
{
  if (pri == PRIORITY_HI)
    {
      if (*root == NULL)
        *root = newleafnode(S_SKIP, NOPOSN); /* create low priority branch */
      if (TagOf(*root) != S_PRIPAR) /* create a PRI PAR */
        *root = newcnode(S_PRIPAR, NOPOSN, addtofront(NULL, addtofront(*root, NULL)));
      merge_process(ThisItemAddr(CBodyOf(*root)), new, PRIORITY_LO);
    }
  else if (*root == NULL || (TagOf(*root) == S_SKIP)) /* Nothing there yet */
    *root = new;
  else if (TagOf(*root) == S_PRIPAR) /* add to low priority half */
    merge_process(ThisItemAddr(NextItem(CBodyOf(*root))), new, PRIORITY_LO);
  else if (TagOf(*root) == S_PAR)    /* Already more than one */
    SetCBody(*root, addtofront(new, CBodyOf(*root)));
  else /* create a PAR */
    *root = newcnode(S_PAR, NOPOSN, addtofront(new, addtofront(*root, NULL)));
}
/*}}}*/
/*{{{  PRIVATE void add_speclist_translations*/
PRIVATE void add_speclist_translations(treenode *old, treenode *new)
{
  /* run through speclist, setting up translations */
  while (old != NULL)
    {
      addnametrans(DNameOf(old), DNameOf(new));
      old = DBodyOf(old); new = DBodyOf(new);
    }
}
/*}}}*/
/*{{{  PRIVATE treenode *add_specs_to_front*/
PRIVATE treenode *add_specs_to_front(treenode *tptr, treenode *const specs)
{
  if (specs != NULL)
    {
      treenode *s = specs;
      while (DBodyOf(s) != NULL)
        s = DBodyOf(s);
      SetDBody(s, tptr);
      tptr = specs;
    }
  return tptr;
}
/*}}}*/
/*{{{  PRIVATE void merge_processor_body*/
PRIVATE void merge_processor_body(treenode *tptr, const namedesc_t *const procname,
                                  procswnode_t *const procnode)
{
  treenode *newtree, *newspecs;
  if ((information && (procnode->p_count > 1)) || config_info)
    {
      fputs("Merging body of PROCESSOR ", outfile);
      print_name_to_str(procname);
      if (config_info)
        fprintf(outfile, " (current mem %ld)", tablesize());
      fputc('\n', outfile);
    }

  setproctypeto(&procnode->p_name);

  /* create a copy of the outer level speclist */
  if (procnode->p_specs == NULL)
    procnode->p_specs = copytree(speclist);  /* works ok on NULL trees */

  lexlevel++;

  /* decls inside CONFIG but outside PROCESSOR */
  newspecs = copytree(expand_speclist);

  marknametrans();
  add_speclist_translations(expand_speclist, newspecs);
  add_speclist_translations(speclist, procnode->p_specs);
  newtree = transcopytree(ProcessorBodyOf(tptr), NOPOSN); /* keeps old Locns */
  freenametrans();

  newtree = add_specs_to_front(newtree, newspecs);

  lexlevel--;

  foldtree(newtree);  /* fold any references to replicator variables */
  prewalktree(newtree, check_lib_instances);

  {
    const INT32 pri = procswnode_priority(procname);
    if (pri == PRIORITY_HI)
      prewalktree(newtree, check_nested_pri_par);

    merge_process(&procnode->p_process, newtree, pri);
  }

  if (prtree)
    { printtree(0, procnode->p_process); fputc('\n', outfile); }

  /*if (testflag)*/
    freeup_temp_workspace();
}
/*}}}*/

/*{{{  PRIVATE int dimensiondepth(tptr)*/
/* return the dimensionality of the tree tptr */
PRIVATE int dimensiondepth ( treenode *tptr )
{
  int depth = 0;
  while (tptr != NULL && (TagOf(tptr) == S_ARRAY || TagOf(tptr) == S_ARRAYSUB))
    {
      depth++;
      tptr = (TagOf(tptr) == S_ARRAY) ? ARTypeOf(tptr) : ASBaseOf(tptr);
    }
  assert(tptr != NULL);
  return depth;
}
/*}}}*/
/*{{{  PRIVATE int dirn_from_modify_mode*/
PRIVATE int dirn_from_modify_mode(const chanswnode_t *const channode)
{
  return (modify_chan_mode == 0) ?
         determine_unknown_direction(channode, current_procswnode) :
         dirn_from_mode(modify_chan_mode);
}
/*}}}*/
/*{{{  PRIVATE void modify_name*/
PRIVATE void modify_name(treenode **tptr)
{
  namedesc_t name;
  chanswnode_t *channode;
  namedesc_t *newname;
  name.n_name  = *tptr;
  name.n_index = 0;
  channode = channodeof(&name);
  newname  = &(channode->c_dirn[dirn_from_modify_mode(channode)].c_id);
  if (config_info)
    fprintf(outfile, "modifying name: %s to %s\n", WNameOf(NNameOf(name.n_name)),
            WNameOf(NNameOf(newname->n_name)));

  if (config_reclaim_mem)
    freetree(tptr);

  *tptr = newname->n_name;
#if OPTIMISE_SINGLES
  /* This added 20/12/90 for bug 835 */
  if ((TagOf(NTypeOf(name.n_name    )) == S_ARRAY) && /* original was an array */
      (TagOf(NTypeOf(newname->n_name)) != S_ARRAY))   /* but new one isn't */
    {
      /* create a constructor with the right number of dimensions */
      treenode *type;
      for (type = NTypeOf(name.n_name); TagOf(type) == S_ARRAY; type = ARTypeOf(type))
        *tptr = newmopnode(S_CONSTRUCTOR, LocnOf(*tptr), addtofront(*tptr, NULL), 0);
    }
#endif
}
/*}}}*/
/*{{{  PRIVATE void modify_base*/
PRIVATE void modify_base(treenode **tptr, treenode *newname)
{
  /* This routine changes the name of the array at the base of
     a subscript or segment expression */
  /* for bug 1143 - 13/2/91 - Eg when the array is accessed by a variable
     or a replicator */
  DEBUG_MSG(("modify_base\n"));
  while (TRUE)
    switch(nodetypeoftag(TagOf(*tptr)))
      {
        default: /* we've arrived at the name of the expression */
          *tptr = newname;
          return;
        case ARRAYSUBNODE: tptr = ASBaseAddr(*tptr);  break;
        case SEGMENTNODE:  tptr = SNameAddr(*tptr);   break;
      }  
}
/*{{{  PRIVATE void modify_arraysub*/
PRIVATE void modify_arraysub(treenode **tptr, treenode *const channame)
{
  treenode *t = *tptr;
  namedesc_t name;
  chanswnode_t *channode;
  namedesc_t *newname;
  INT32 first, last;
  int whole_array_accessed = FALSE;
  subscripts_accessed(t, &first, &last, FALSE);
  if (first < 0)
    {
      first = 0;
      last  = elementsin(NTypeOf(channame));
      whole_array_accessed = TRUE; /* bug 1143 13/2/91 */
    }

  if (config_info)
    {
      fprintf(outfile, "modifying arraysub: %s[%ld-%ld]",
        WNameOf(NNameOf(channame)), first, last);
      if (whole_array_accessed) fputs(" (whole)", outfile);
      if (prtree)
        printtree(0, t);
      fputc('\n', outfile);
    }

  name.n_name  = channame;
  name.n_index = first;
  channode = channodeof(&name);
  newname  = &(channode->c_dirn[dirn_from_modify_mode(channode)].c_id);
  {
    treenode *oldtype = NTypeOf(channame);
    treenode *newtype = NTypeOf(newname->n_name);
    const int depthold       = dimensiondepth(oldtype);
    const int depthnew       = dimensiondepth(newtype);
    const int depthindex     = dimensiondepth(t);
    const INT32 newoffset    = newname->n_index;
    const int finaldimension = depthold - depthindex;

    assert((depthold >= depthnew) && (depthold >= depthindex));

    DEBUG_MSG(("modifying arraysub: depthold:%d, depthnew:%d, depthindex:%d, newoffset:%d, finaldimension:%d\n",
               depthold, depthnew, depthindex, newoffset, finaldimension));
  #if OPTIMISE_SINGLES
    if ((elementsin(newtype) == 1) /* optimisation to turn into single element */
        || (depthnew <= finaldimension))
      {
        if (config_reclaim_mem)
          freetree(tptr);

        *tptr = newname->n_name;
      }
    else
  #endif
    if (whole_array_accessed) /* bug 1143 13/2/91 */
      modify_base(tptr, newname->n_name);
    else
      {  /* newname must be indexed */
        const INT32 newindex = newoffset / elementsin(ARTypeOf(newtype));
        int c;
        for (c = depthnew - 1; c > finaldimension; c--)
          t = ASBaseOf(t);

        if (config_reclaim_mem)
          { freetree(ASIndexAddr(t)); freetree(ASBaseAddr(t)); }

        SetASIndex(t, newconstant(newindex));
        SetASBase(t, newname->n_name);
      }
  }

  if (config_info && prtree)
    {
      fprintf(outfile, "arraysub is now:");
      printtree(0, *tptr);
      fputc('\n', outfile);
    }
}
/*}}}*/
/*{{{  PRIVATE void modify_segment*/
PRIVATE void modify_segment(treenode **tptr, treenode *const channame)
{
  treenode *t = *tptr;
  INT32 first, last;
  namedesc_t name;
  chanswnode_t *channode;
  namedesc_t *newname;
  int whole_array_accessed = FALSE;

#if 1
  subscripts_accessed(t, &first, &last, FALSE);
  if (first < 0)
    {
      first = 0;
      whole_array_accessed = TRUE; /* bug 1143 13/2/91 */
    }
#else
  if (TagOf(SStartExpOf(t)) != S_CONSTEXP)
    badtag(LocnOf(t), TagOf(SStartExpOf(t)), "modify_segment");
  if (TagOf(SLengthExpOf(t)) != S_CONSTEXP)
    badtag(LocnOf(t), TagOf(SLengthExpOf(t)), "modify_segment");
  first = LoValOf(SStartExpOf(t));
#endif

  name.n_name  = channame;
  name.n_index = first;
  channode = channodeof(&name);
  newname  = &(channode->c_dirn[dirn_from_modify_mode(channode)].c_id);

  if (config_info)
    {
      fprintf(outfile, "modifying segment: %s[%ld...]",
        WNameOf(NNameOf(channame)), first);
      if (whole_array_accessed) fputs(" (whole)", outfile);
      if (prtree)
        printtree(0, t);
      fputc('\n', outfile);
    }
#if OPTIMISE_SINGLES
  if (elementsin(NTypeOf(newname->n_name)) == 1)
    {
      /* create a constructor with the right number of dimensions */
      treenode *oldtype;
      if (config_reclaim_mem)
        freetree(tptr);
      *tptr = newname->n_name;
      for (oldtype = NTypeOf(name.n_name); TagOf(oldtype) == S_ARRAY; oldtype = ARTypeOf(oldtype))
        *tptr = newmopnode(S_CONSTRUCTOR, LocnOf(*tptr), addtofront(*tptr, NULL), 0);
    }
  else
#endif
  if (whole_array_accessed) /* bug 1143 13/2/91 */
    modify_base(tptr, newname->n_name);
  else
    {
      INT32 newoffset = newname->n_index;
      INT32 newindex  = newoffset/elementsin(ARTypeOf(NTypeOf(newname->n_name)));
      if (config_reclaim_mem)
        { freetree(SNameAddr(t)); freetree(SStartExpAddr(t)); }
      SetSName(t, newname->n_name);
      SetSStartExp(t, newconstant(newindex));
    }

  if (config_info && prtree)
    {
      fprintf(outfile, "segment is now:");
      printtree(0, t);
      fputc('\n', outfile);
    }
}
/*}}}*/
/*{{{  PRIVATE int isonlist*/
PRIVATE int isonlist(treenode *const tptr, treenode *list)
{
  while (!EndOfList(list) && (tptr != ThisItem(list)))
    list = NextItem(list);
  return !EndOfList(list);
}
/*}}}*/
/*{{{  PRIVATE int name_must_be_changed*/
PRIVATE int name_must_be_changed(treenode *const nptr)
{
#if 0
  return ((NLexLevelOf(nptr) == EXTERNAL_CHAN_LEXLEVEL) &&
          (basetype(NTypeOf(nptr)) == S_CHAN) &&
          !isformalparam(nptr));
#endif
  return (basetype(NTypeOf(nptr)) == S_CHAN) &&
         (isonlist(nptr, chanlist) || isonlist(nptr, chanabbrlist));
}
/*}}}*/
/*{{{  PRIVATEPARAM int modify_chan_subscripts*/
PRIVATEPARAM int modify_chan_subscripts(treenode **tptr)
{
  treenode *t = *tptr;
  switch(nodetypeoftag(TagOf(t)))
    {
      default:
        return CONTINUE_WALK;
      case NAMENODE:
        if (name_must_be_changed(t))
          modify_name(tptr);
        return STOP_WALK;
      case ARRAYSUBNODE:
        {
          treenode *const channame = nameof(t);
          if (name_must_be_changed(channame))
            modify_arraysub(tptr, channame);
        }
        return STOP_WALK;
      case SEGMENTNODE:
        {
          treenode *const channame = SNameOf(t);
          if ((TagOf(channame) != S_SEGMENT) && (TagOf(channame) != S_ARRAYSUB) &&
              name_must_be_changed(channame))
            modify_segment(tptr, channame);
        }
        return STOP_WALK;
      case ACTIONNODE:
        {
          const int saved_mode = modify_chan_mode;
          modify_chan_mode = (TagOf(t) == S_OUTPUT) ? CHAN_USE_OUTPUT : CHAN_USE_INPUT;
          modprewalktree(LHSAddr(t), modify_chan_subscripts);
          modprewalktree(RHSAddr(t), modify_chan_subscripts);
          modify_chan_mode = saved_mode;
        }
        return STOP_WALK;
      case DECLNODE:
        if (((TagOf(t) == S_ABBR) || (TagOf(t) == S_RETYPE)) &&
            basetype(NTypeOf(DNameOf(t))) == S_CHAN )
          {
            const int saved_mode = modify_chan_mode;
            modify_chan_mode = chan_use(DBodyOf(t), DNameOf(t));

            modprewalktree(DValAddr(t), modify_chan_subscripts);
            modify_chan_mode = saved_mode;
            modprewalktree(DBodyAddr(t), modify_chan_subscripts);
            return STOP_WALK;
          }
        else
          return CONTINUE_WALK;
      case INSTANCENODE:
        if (TagOf(t) == S_PINSTANCE) /* functions can't have channel params */
          {
            treenode *fparams = NTypeOf(INameOf(t));
            treenode *aparams = IParamListOf(t);

            /* skip leading hidden params */
            while (!EndOfList(fparams) && 
                   (TagOf(ThisItem(fparams)) != N_PARAM) &&
                   (TagOf(ThisItem(fparams)) != N_VALPARAM))
              fparams = NextItem(fparams);

            while (!EndOfList(fparams))
              {
                treenode *thisparam = ThisItem(fparams);
                if (basetype(NTypeOf(thisparam)) == S_CHAN)
                  {
                    const int saved_mode = modify_chan_mode;
                    if (paraminputon(thisparam))
                      modify_chan_mode = CHAN_USE_INPUT;
                    else if (paramoutputon(thisparam))
                      modify_chan_mode = CHAN_USE_OUTPUT;
                    else
                      modify_chan_mode = 0; /* unknown direction */
                    modprewalktree(ThisItemAddr(aparams), modify_chan_subscripts);
                    modify_chan_mode = saved_mode;
                  }
                fparams = NextItem(fparams); aparams = NextItem(aparams);
              }
          }
        /* we don't need to bother to go inside function param lists */
        return STOP_WALK;
    }
}
/*}}}*/
/*{{{  PRIVATE void compile_processor*/
PRIVATE void compile_processor(const namedesc_t *const procname,
                               procswnode_t *const procnode)
{
  treenode *tptr = procnode->p_process;
  treenode *proc_nptr;

  if (information || config_info)
    {
      /*fprintf(outfile, "Compiling PROCESSOR %s", procstr);*/
      fputs("Compiling NODE ", outfile);
      print_name_to_str(procname);
      if (config_info)
        fprintf(outfile, " (current mem %ld)", tablesize());
      fputc('\n', outfile);
    }
  current_procswnode = procnode;
  /*current_procname   = procname;*/

  setproctypeto(&(procnode->p_name));

  modify_chan_mode = 0; /* set to 'unknown direction' */
  modprewalktree(&tptr, modify_chan_subscripts);

  lexmode = LEX_SOURCE; /* This probably isn't necessary */
  tptr = add_specs_to_front(tptr, procnode->p_locals);
  tptr = declare(S_PROCDEF, NOPOSN, procnode->p_fparams, procnode->p_procname, tptr);
  proc_nptr = DNameOf(tptr);

  tptr = add_specs_to_front(tptr, procnode->p_specs);

  if (prtree)
    { printtree(0, tptr); fputc('\n', outfile); }

  bereinit(lexlevel); /*descreinit(); - don't need to, and it breaks sourcehash*/
  allocateworkspace(tptr);
  codegenerate(FALSE, tptr);
  procnode->p_seek = saved_seek_position();

  if (config_reclaim_mem)
    {
      if (config_info)
        fprintf(outfile, "Freeing up processor's memory\n");
      /*if (prtree)
        { printtree(0, tptr); fputc('\n', outfile); }*/

      /* ensure that we don't freeup the formal parameter lists, because
         we use them to generate the backend's information */
      SetNType(proc_nptr, NULL);
      
      freetree(&tptr);
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM void do_collect_process_bodies*/
PRIVATEPARAM int do_collect_process_bodies(treenode *tptr)
/*****************************************************************************
*
* do_collect_process_bodies; if collect_singles is TRUE, it collects the
*                            bodies of processors which are only used once,
*                            and immediately compiles them.
*               otherwise    it collects the bodies of processors which are
*                            used more than once, for later compilation.
*
*****************************************************************************/
{
  namedesc_t procname;
  procswnode_t *procnode;

  if (errors != 0) /* in the second pass, a recent compilation may have failed */
    return STOP_WALK;

  if (TagOf(tptr) != S_PROCESSOR)
    return CONTINUE_WALK;

  map_to_basename(ProcessorExpOf(tptr), &procname);
  if (procname.n_index < 0) /* subscript overflow etc */
    return STOP_WALK;

  procnode = procswnodeof(&procname);

  if (collect_singles)
    {
      if (procnode->p_count == 1)
        {
          merge_processor_body(tptr, &procname, procnode);
          if (errors == 0)
            compile_processor(&procname, procnode);
        }
    }
  else /* just merge the `multiply used' processors */
    if (procnode->p_count > 1)
      merge_processor_body(tptr, &procname, procnode);
  return STOP_WALK;
}
/*}}}*/
/*{{{  PRIVATEPARAM void do_compile_processor*/
PRIVATEPARAM void do_compile_processor(const namedesc_t *const procname)
/*****************************************************************************
*
* do_compile_processor compiles the main body for the named processor,
*                      ONLY if this processor is accessed more than once.
*
*****************************************************************************/
{
  procswnode_t *const procnode = procswnodeof(procname);
  if (procnode->p_process != NULL /* empty if this hardware node is never used */
      && procnode->p_count > 1)
    compile_processor(procname, procnode);
}
/*}}}*/
/*{{{  PRIVATE void collect_process_bodies*/
PRIVATE void collect_process_bodies(treenode *config_tptr)
/*****************************************************************************
*
* collect_process_bodies walks the software description, combining the
*                        process bodies for each processor
*
*****************************************************************************/
{
  collect_singles = FALSE;
  expandwalkproctree(config_tptr, do_collect_process_bodies);
}
/*}}}*/
/*{{{  PRIVATE void compile_processor_bodies*/
PRIVATE void compile_processor_bodies(treenode *config_tptr)
/*****************************************************************************
*
* compile_processor_bodies compiles each processor's main body code
*
*****************************************************************************/
{
  walk_config_list(proclist, do_compile_processor);
  collect_singles = TRUE;
  expandwalkproctree(config_tptr, do_collect_process_bodies);
  freecode();   /* free up the code buffer */
  debugfree();  /* free up the debug info buffers */
  freevarmap(); /* free up the workspace allocator buffers */
}
/*}}}*/

/*{{{  PUBLIC void pseudo_link()*/
PUBLIC void pseudo_link(treenode *libcalls)
{
  while (libcalls != NULL)
    {
      const INT32 target_address =
            add_to_module_list(&current_procswnode->p_modules, libcalls);
      patch_jump(NLEntryOffsetOf(libcalls), target_address);
      libcalls = NLEntryNextOf(libcalls);
    }
}
/*}}}*/
/*{{{  PUBLIC void save_entrydata*/
PUBLIC void save_entrydata(const treenode *const nptr,
                           const INT32 offset, const INT32 code_size)
{
  if (/*information ||*/ config_info)
    {
      /*char procstr[MAXSTRING_SIZE];
      name_to_str(procstr, current_procname);*/
      fprintf(outfile,
       "NODE requires: workspace %ld, vectorspace %ld words; %ld byte code stub\n",
       /*procstr,*/ NPDatasizeOf(nptr), NPVSUsageOf(nptr), code_size);
    }
  current_procswnode->p_ws     = NPDatasizeOf(nptr);
  current_procswnode->p_vs     = NPVSUsageOf(nptr);
  current_procswnode->p_offset = offset;
  current_procswnode->p_size   = code_size;
}
/*}}}*/
/*{{{  PUBLIC void config_link_map*/
PUBLIC void config_link_map(void)
/* This is called just before finishing the object file, so that we can
   insert a link map
*/
{
  write_linker_map(objfilename, current_procswnode->p_size, saved_seek_position(),
                   current_procswnode->p_modules);
}
/*}}}*/

/*{{{  PRIVATE treenode *splitup_specs*/
PRIVATE treenode *splitup_specs(treenode *t)
{
  while (isspecification(t))
    {
      if (spec_needed_for_compile(t))
        { *lastspec = t; lastspec = DBodyAddr(t); }
      else
        switch (TagOf(t))
          {
             case S_DECL: /* can only be hardware nodes, and channels */
             case S_ABBR: /* ditto */
             case S_NETWORK:
             case S_PLACEON:
               { *lastnet = t; lastnet = DBodyAddr(t); break; }
               break;
             case S_CONFIG:
               { configptr = DValOf(t); break; }
             case S_MAPPING:
               { mapptr    = DValOf(t); break; }
             default:
               break;
        }
      t = DBodyOf(t);
    }
  return t;
}
/*}}}*/
/*{{{  PUBLIC void config*/
PUBLIC void config (treenode *const tptr)
{
  /*DEBUG_MSG(("config\n"));*/
  /*{{{  attach all outer level declarations to the network description*/
  lastspec = &speclist;
  lastnet  = &netlist;
  
  (void) splitup_specs(tptr);
  
  /* Now we move all leading specifications off the CONFIG sub-tree */
  /* This is optional, but makes for more efficient memory usage later */
  configptr = splitup_specs(configptr);
  
  *lastspec = NULL; /* terminate the 'tree' of leading specifications */
  *lastnet  = NULL; /* terminate the 'tree' of hardware specifications */
  /*}}}*/

  if (assembly_output && prtree && comp_error == 0)
    {
      fputs("speclist is ",    outfile); printtree(0, speclist);
      fputs("\nnetlist is ",   outfile); printtree(0, netlist);
      fputs("\nconfigptr is ", outfile); printtree(0, configptr);
      fputs("\nmapptr is ",    outfile); printtree(0, mapptr);
      fputc('\n', outfile);
    }

  if (information || config_info)
    fputs("Processing hardware description\n", outfile);
  process_config(netlist, TRUE); /* first pass sets up hardware descriptions */

  if ((errors == 0) && (mapptr != NULL))
    {
      if (information || config_info)
        fputs("Processing mapping\n", outfile);
      process_mapping(mapptr);
    }
  if (errors == 0)
    {
      if (information || config_info)
        fputs("Checking hardware consistency\n", outfile);
      check_hw_consistency();
      /*check_sw_consistency();*/
    }
  if (errors == 0)
    { /* second pass sets up CHANs and PROCESSORs */
      if (information || config_info)
        fputs("Processing software description\n", outfile);
      process_config(configptr, FALSE);
      process_unknown_directions(); 
      check_host_connection();
    }
  if (errors == 0)
    {
      if (information || config_info)
        fputs("Creating translations\n", outfile);
      create_translations();
    }
  if (errors == 0)
    {
      if (information || config_info)
        fputs("Collecting process bodies\n", outfile);
      collect_process_bodies(configptr);
    }
  if ((errors == 0) && config_code)
    {
      if (information || config_info)
        fprintf(outfile, "Compiling processor bodies to file \"%s\"\n", objfilename);
      if (!assembly_output && !disassemble)
        objfile = open_object_file(objfilename);
      compile_processor_bodies(configptr);
      if (!assembly_output && !disassemble)
        close_object_file(objfile, objfilename);
    }

  dump_config();  /* debugging info */

  if ((errors == 0) && (config_srcout || config_backend || config_backend_dump))
    write_config_desc();
}
/*}}}*/
