/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two checker    chk4 - driver for scoping and semantic analysis
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lexconst.h"
# include "lexdef.h"
# include "lex1def.h"
# include "chkerror.h"
# include "syn1def.h"
# include "chk1def.h"
# include "chk2def.h"
# include "chk4def.h"
# include "chkdef.h"
# include "desc1def.h"
# include "genhdr.h"
/*}}}*/

/*{{{  definitions*/
/*#define NAME_STACK_SIZE 2000*/
/*#define UNDEC_NAME_STACK_SIZE 50*/

/*{{{  struct namestackentry*/
typedef struct namestackentry
  {
    struct namestackentry *n_next;
    treenode *namedesc;
    wordnode *wordptr;
  } namestackentry_t;

typedef int namestack_t;
/*}}}*/
/*}}}*/

/*{{{  PRIVATE variables*/
PRIVATE BIT16 namep;

PRIVATE namestackentry_t *namestack;
PRIVATE namestackentry_t *namestack_base;
PRIVATE namestackentry_t *undeclarednamestack;
PRIVATE namestackentry_t *undeclarednamestack_base;
PRIVATE namestackentry_t *freenamestack = NULL;

PRIVATE int insidepripar, nestedpripar;
/*PRIVATE int insideinline;*/ /* no longer necessary to distinguish */
PRIVATE int insideseparatelycompiledsc = FALSE;

#ifdef CONFIG
PRIVATE int configcheckstate;
PRIVATE int configcheckscope;
PRIVATE int configcount;
PRIVATE int networkcount;
PRIVATE int mappingcount;

PRIVATE namestackentry_t *attributenamestack;
PRIVATE namestackentry_t *attributenamestack_base;
#endif
/*}}}*/

/*{{{  forward declarations*/
PRIVATE void scopeandcheck(treenode **tptr);
#ifdef CONFIG
PRIVATEPARAM int configcheck(treenode *tptr);
#endif
/*}}}*/

/*{{{  routines*/
/*{{{  name stack handling*/
/*{{{  PRIVATE namestackentry_t *newnamestackentry()*/
PRIVATE namestackentry_t *newnamestackentry(namestackentry_t *old,
        treenode *defn, wordnode *name)
{
  namestackentry_t *ptr;
  if (freenamestack == NULL)
    ptr = (namestackentry_t *)newvec(sizeof(namestackentry_t));
  else
    {
      ptr = freenamestack;
      freenamestack = ptr->n_next;
    }
  ptr->wordptr  = name;
  ptr->namedesc = defn;
  ptr->n_next   = old;
  DEBUG_MSG(("newnamestackentry: namep is %d, returning %x\n", namep, ptr));
  return ptr;
}
/*}}}*/
/*{{{  PRIVATE treenode *searchnamestack (n)*/
PRIVATE treenode *searchnamestack (namestackentry_t *stack,
                                   wordnode *n )
{
  /* we MUST ensure that it's on the stack somewhere */

  while (stack->wordptr != n)
    stack = stack->n_next;
  return stack->namedesc;
}
/*}}}*/
/*{{{  PUBLIC treenode *findname (n)*/
PUBLIC treenode *findname ( wordnode *n )
{
  /* Look up the name pointed to by n on the name stack.
     Return NULL if not found */

  namestack_base->wordptr = n;  /* its namedesc is already NULL */
  return searchnamestack(namestack, n);
}
/*}}}*/
/*{{{  PUBLIC void addname (nptr)*/
PUBLIC void addname ( treenode *nptr )
{
  if ((warning_flags & (WARNING_DESCOPED_N | WARNING_DESCOPED_P)) &&
      (findname(NNameOf(nptr)) != NULL) && 
      /* Don't warn of 'descoped' parameter names inside a library formal param list */
      !insideseparatelycompiledsc)
    {
      int param = ((TagOf(nptr) == N_PARAM) || (TagOf(nptr) == N_VALPARAM));
      if (( param && (warning_flags & WARNING_DESCOPED_P) ) ||
          (!param && (warning_flags & WARNING_DESCOPED_N) ) )
            msg_out_s(SEV_WARN, CHK, CHK_NAME_IS_DESCOPED, LocnOf(nptr), WNameOf(NNameOf(nptr)));
    }
  SetNScope(nptr, namep);
  namep++;

  namestack = newnamestackentry(namestack, nptr, NNameOf(nptr));

  DEBUG_MSG(("addname: %s, ptr is %x, namep is %d\n",
             WNameOf(namestack->wordptr), namestack, namep));
}
/*}}}*/
/*{{{  PRIVATE treenode *addundeclaredname (name)*/
/* Pointer to node containing text of name */
PRIVATE treenode *addundeclaredname ( wordnode *name )
{
  treenode *nptr = declname(N_DECL, chklocn, name, undeclaredp, NULL);
  SetNScope(nptr, namep);

  undeclarednamestack = newnamestackentry(undeclarednamestack, nptr, NNameOf(nptr));

  return (nptr);
}
/*}}}*/
/*{{{  PRIVATE treenode *findundeclaredname (n)*/
PRIVATE treenode *findundeclaredname ( wordnode *n )
{
  /* Look up the name pointed to by n on the name stack.
     Return NULL if not found */
  undeclarednamestack_base->wordptr = n;  /* its namedesc is already NULL */
  return searchnamestack(undeclarednamestack, n);
}
/*}}}*/
/*{{{  PRIVATE treenode *lookupname (n)*/
PRIVATE treenode *lookupname ( wordnode *n, treenode *tptr )
{
  /* Look up the name pointed to by n on the name stack.
     If found, return a pointer to the name record,
     otherwise make new entry. */
  /* tptr is the namenode returned from findname previously */
  if (tptr == NULL)
    /*{{{  check to see if this undeclared variable has already been seen*/
    {
      SOURCEPOSN locn = sourcefileof(chklocn);
      tptr = findundeclaredname (n);
      if (tptr == NULL)
        /*{{{  Never been seen before, create an undeclared name node*/
        {
          tptr = addundeclaredname(n);
          chkerr_s(CHK_NAME_NOT_DECLARED, locn, WNameOf(n));
        }
        /*}}}*/
      {
        treenode *old = NDeclOf(tptr);
        /* This test added 16/10/90 so we don't repeat previous lines */
        if ((old == NULL) || (LocnOf(old) != locn))
          { /* add to list of undeclared instances */
            treenode *dptr = newlistnode(S_LIST, locn, NULL, old);
            SetNDecl(tptr, dptr);
          }
      }
    }
    /*}}}*/
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE void printlinenums (file, listptr)*/
PRIVATE void printlinenums ( FILE *file , treenode *listptr )
{
  int current_file = (-1);
  while (listptr != NULL)
    {
      if (current_file != (-1))
        fputc(',', file);
      if (FileNumOf(LocnOf(listptr)) != current_file)
        {
          current_file = FileNumOf(LocnOf(listptr));
          fprintf(file, " \"%s\" line%s ", lookupfilename(current_file),
          ((NextItem(listptr) != NULL) &&
           (FileNumOf(LocnOf(NextItem(listptr))) == current_file)) ? "s" : "");
        }
      fprintf(file, "%d", FileLineOf(LocnOf(listptr)));
      listptr = NextItem(listptr);
    }
}
/*}}}*/
/*{{{  PUBLIC void printundeclarednames (file)*/
PUBLIC void printundeclarednames ( FILE *file )
{
  namestackentry_t *nmptr = undeclarednamestack;
  while (nmptr != undeclarednamestack_base)
    {
      treenode *list = reverselist(NDeclOf(nmptr->namedesc));
      fprintf (file, "%s undeclared on", WNameOf(nmptr->wordptr));
      printlinenums (file, list);
      fputc('\n', file);
      nmptr = nmptr->n_next;
    }
}
/*}}}*/
/*{{{  PUBLIC int inscope (nptr)*/
/* Return TRUE if the symbol nptr is in scope, FALSE otherwise */
PUBLIC int inscope ( treenode *nptr )
{
  return (nptr == findname(NNameOf(nptr)));
}
/*}}}*/
/*}}}*/

#ifdef CONFIG
/*{{{  PRIVATE treenode *find_attribute (n)*/
PRIVATE treenode *find_attribute ( wordnode *n )
{
  /* Look up the name pointed to by n on the name stack.
     Return NULL if not found */

  attributenamestack_base->wordptr = n;  /* its namedesc is already NULL */
  return searchnamestack(attributenamestack, n);
}
/*}}}*/
/*{{{  PUBLIC void add_attribute*/
PUBLIC void add_attribute(treenode *nptr)
{
  attributenamestack = newnamestackentry(attributenamestack, nptr, NNameOf(nptr));
}
/*}}}*/
/*{{{  PRIVATE init_attributes*/
PRIVATE void init_attributes(void)
{
  /* prepare the attributes for S_NODE */
  attributenamestack_base = newnamestackentry(NULL, NULL, NULL);
  attributenamestack      = attributenamestack_base;
}
/*}}}*/
#endif

/*{{{  PRIVATE int current_scope()*/
#if 0 /* make it a macro instead */
PRIVATE BIT16 current_scope(void)
{
  return namep;
}
#else
#define current_scope() namep
#endif
/*}}}*/
/*{{{  PRIVATE namestack_t markscopenames()*/
#if 0 /* make it a macro instead */
PRIVATE namestack_t markscopenames(void)
{
  DEBUG_MSG(("markscopenames: namestack is %x (%s), namep is %d\n", namestack, WNameOf(namestack->wordptr), namep));
  return namep;
}
#else
#define markscopenames() namep
#endif
/*}}}*/
/*{{{  PRIVATE void descopenames (namestack_t n)*/
PRIVATE void descopenames ( namestack_t n )
{
  DEBUG_MSG(("descopenames: marker is %d\n", n));
  assert(namep >= n);
  while (namep > n)
    {
      treenode *nptr = namestack->namedesc;
      DEBUG_MSG(("descopenames: removing %x (%s) (%d)\n", namestack, WNameOf(namestack->wordptr), namep));
      if ((!NUsedOf(nptr)) &&
          (NTypeOf(nptr) == NULL || TagOf(NTypeOf(nptr)) != S_UNDECLARED))
        {
          char *namestring = WNameOf(NNameOf(nptr));
          SOURCEPOSN locn = LocnOf(nptr);
          switch (TagOf(nptr))
            /*{{{  cases*/
            {
              default:          break;
              case N_DECL:
              case N_ABBR:
              case N_RETYPE:    if (warning_flags & WARNING_UNUSED_V)
                                  msg_out_s(SEV_WARN, CHK, CHK_NAME_NOT_USED, locn, namestring);
                                break;
              case N_PARAM:
              case N_VALPARAM:  if (warning_flags & WARNING_UNUSED_P)
                                  msg_out_s(SEV_WARN, CHK, CHK_PARAM_NOT_USED, locn, namestring);
                                break;
              case N_VALABBR:
              case N_VALRETYPE: if ((warning_flags & WARNING_UNUSED_V) && !isconst(nptr))
                                  msg_out_s(SEV_WARN, CHK, CHK_NAME_NOT_USED, locn, namestring);
                                break;
              case N_PROCDEF:
              case N_SFUNCDEF:
              case N_LFUNCDEF:  if ((NLexLevelOf(nptr) != 0) && (warning_flags & WARNING_UNUSED_R))
                                  msg_out_s(SEV_WARN, CHK, CHK_ROUTINE_NOT_USED, locn,namestring);
                                break;
            }
            /*}}}*/
        }
      --namep;
      {  /* put that namestack entry onto the freelist */
        namestackentry_t *temp = namestack;
        namestack = namestack->n_next;
        temp->n_next = freenamestack;
        freenamestack = temp;
      }
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM int decllab(treenode *tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  decllab is the routine called by the tree walker on each node
 *          when we are declaring GUY code labels.
 *          Don't go into routine bodies, but do go into expressions.
 *
 *****************************************************************************/
/*}}}*/
PRIVATEPARAM int decllab ( treenode *tptr )
{
  /* We do successive specifications in a loop to avoid excessive recursion */
  if (isspecification(tptr))
    {
      /* S_LABELDEF will only ever appear as a specification on its own */
      if (TagOf(tptr) == S_LABELDEF)
        {
          addname(DNameOf(tptr));
          return STOP_WALK; /* body of S_LABELDEF is always NULL */
        }
      tptr = skipspecifications(tptr); /* skip nested PROCs etc */
      prewalktree(tptr, decllab);
      return STOP_WALK;
    }
  else
    return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATE void decllabels(treenode *tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  decllabels takes a parse tree and brings into scope all the labels in
 *             that tree. It does not look inside nested routines.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void decllabels ( treenode *tptr )
{
  prewalktree(tptr, decllab);
}
/*}}}*/
/*{{{  PRIVATE void scopeandchecksc (tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  scopeandchecksc applies scoping and type checking to separately
 *                  compiled routine body tptr.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void scopeandchecksc ( treenode **tptr )
{
  while (*tptr != NULL)
    {
      treenode *t = *tptr;
      if (TagOf(t) != S_NAME) chklocn = LocnOf(t);
      switch(TagOf(t))
        {
          default:
            return;
          /*{{{  SEQ PRIPAR          break*/
          case S_SEQ:
            tptr = CBodyAddr(t);
            break;
          case S_PRIPAR:
            nestedpripar = TRUE;
            tptr = CBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  OUTPUT, INPUT       break*/
          case S_OUTPUT:
          case S_INPUT:
            tptr = LHSAddr(t);
            break;
          /*}}}*/
          /*{{{  NAME                return*/
          case S_NAME :
            {
              wordnode *n = (wordnode *)t;
              *tptr = lookupname(n, findname(n));
            }
            return;
          /*}}}*/
          /*{{{  LIST                break*/
          case S_LIST :
            scopeandchecksc(ThisItemAddr(t));
            tptr = NextItemAddr(t);
            break;
          /*}}}*/
        }
    }
}
/*}}}*/

#ifdef CONFIG
/*{{{  PRIVATE treenode *cset(tptr)*/
/*****************************************************************************
 *
 *  cset performs semantic checking on a set node
 *
 *****************************************************************************/
PRIVATE treenode *cset ( treenode *tptr )
{
  DEBUG_MSG(("cset\n"));
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *t, *lhs, *rhs;
          int varno = 1;
          chklocn = LocnOf(tptr);
      
          t = typecheck (STDevOf(tptr), S_NODE);
          if (!sametype(TagOf(t), S_NODE))
            chkreport_s(CHK_INVTYPE_DEVICE, chklocn, tagstring(S_NODE));

          /* we're basically doing a checkasslist here, but we can't call
             typecheck_main on lhs cos fields on their own are disallowed */
          /*checkasslist(lhs, rhs);*/
          lhs = STAttrNameOf(tptr);
          rhs = STAttrExpOf(tptr);
          while (!EndOfList(lhs) && !EndOfList(rhs))
            {
              treenode *type, *attr;
              int base;
              char buf[50];
              wordnode *n = (wordnode *)ThisItem(lhs);
              NewItem(lookupname(n, find_attribute(n)), lhs);
              attr = ThisItem(lhs);
              if (TagOf(attr) != N_FIELD)
                chkreport_s(CHK_INVTYPE_ATTR, chklocn, WNameOf(NNameOf(attr)));
              type = NTypeOf(attr);
              base = basetype(type);
              sprintf(buf, "%d", varno);
              if (!typesequivalent(type, typecheck(ThisItem(rhs), base), FALSE))
                chkreport_s(CHK_INVTYPE_ASS, chklocn, buf);
              lhs = NextItem(lhs);
              rhs = NextItem(rhs);
              varno++;
            }
          if (!EndOfList(lhs))
            chkreport (CHK_TOO_MANY_VARS, chklocn);
          else if (!EndOfList(rhs))
            chkreport (CHK_TOO_FEW_VARS, chklocn);

          SetSTDev(tptr, foldexp(STDevOf(tptr)));
          if (STAttrExpOf(tptr) != NULL)
            SetSTAttrExp(tptr, foldexplist(STAttrExpOf(tptr)));
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
        DEBUG_MSG(("cset finished\n"));
      }
      /*}}}*/
    }
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE treenode *cconnect(tptr)*/
/*****************************************************************************
 *
 *  cconnect performs semantic checking on a connect node
 *
 *****************************************************************************/
PRIVATE treenode *cconnect ( treenode *tptr )
{
  DEBUG_MSG(("cconnect\n"));
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *t;
          chklocn = LocnOf(tptr);
      
          t = typecheck (ConnectFromEdgeOf(tptr), S_EDGE);
          if (!sametype(TagOf(t), S_EDGE))
            chkreport_s(CHK_INVTYPE_EDGE, chklocn, tagstring(S_EDGE));
          SetConnectFromEdge(tptr, foldexp(ConnectFromEdgeOf(tptr)));
      
          t = typecheck (ConnectToEdgeOf(tptr), S_EDGE);
          if (!sametype(TagOf(t), S_EDGE))
            chkreport_s(CHK_INVTYPE_EDGE, chklocn, tagstring(S_EDGE));
          SetConnectToEdge(tptr,   foldexp(ConnectToEdgeOf(tptr)));
      
          if (ConnectArcOf(tptr) != NULL)
            {
              t = typecheck (ConnectArcOf(tptr), S_ARC);
              if (!sametype(TagOf(t), S_ARC))
                chkreport_s(CHK_INVTYPE_ARC, chklocn, tagstring(S_ARC));
              SetConnectArc(tptr,      foldexp(ConnectArcOf(tptr)));
            }
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
        DEBUG_MSG(("cconnect finished\n"));
      }
      /*}}}*/
    }
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE treenode *cmap(tptr)*/
/*****************************************************************************
 *
 *  cmap performs semantic checking on a map node
 *
 *****************************************************************************/
PRIVATE treenode *cmap ( treenode *tptr )
{
  DEBUG_MSG(("cmap\n"));
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *t, *maplist;
          int sourcetype;
          chklocn = LocnOf(tptr);
      
          t = typecheck(MapDestOf(tptr), S_NODE);
          if (sametype(TagOf(t), S_NODE))
            sourcetype = S_NODE;
          else if (sametype(TagOf(t), S_ARC))
            sourcetype = S_CHAN;
          else
            chkreport(CHK_INV_MAPPING_RHS, chklocn);
          SetMapDest(tptr, foldexp(MapDestOf(tptr)));
      
          maplist = MapSourceOf(tptr);
          while (!EndOfList(maplist))
            {
              t = typecheck(ThisItem(maplist), sourcetype);
              if (!sametype(TagOf(t), sourcetype))
                chkreport_s(CHK_INV_MAPPING_LHS, chklocn, tagstring(sourcetype));
              NewItem(foldexp(ThisItem(maplist)), maplist);
              maplist = NextItem(maplist);
            }

          if (MapPriOf(tptr) != NULL)
            {
              if (sourcetype != S_NODE)
                chkreport(CHK_INV_MAPPING_NOPRI, chklocn);
              t = typecheck(MapPriOf(tptr), S_INT);
              if (!sametype(TagOf(t), S_INT))
                chkreport(CHK_INV_MAPPING_PRI, chklocn);
              SetMapPri(tptr, foldexp(MapPriOf(tptr)));
            }
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
        DEBUG_MSG(("cmap finished\n"));
      }
      /*}}}*/
    }
  return (tptr);
}
/*}}}*/
#endif

/*{{{  PRIVATE treenode *csubscript(tptr)*/
/*****************************************************************************
 *
 *  csubscript turns array subscripting into record subscripting if necessary
 *
 *****************************************************************************/
PRIVATE treenode *csubscript ( treenode *tptr )
{
#ifdef CONFIG
  int type = TagOf(typecheck(ASBaseOf(tptr), S_UNKNOWN));
  if (type == S_NODE /* || type == S_RECORD */ )
    {
      if (TagOf(ASIndexOf(tptr)) == S_NAME)
        {
          wordnode *n = (wordnode *)ASIndexOf(tptr);
          SetASIndex(tptr, lookupname(n, find_attribute(n)));
        }
      else
        scopeandcheck(ASIndexAddr(tptr));

      if (TagOf(ASIndexOf(tptr)) != N_FIELD)
        chkerr(CHK_RSUB_NOT_FIELD, LocnOf(tptr));
      SetTag(tptr, S_RECORDSUB);
    }
  else
#endif
    scopeandcheck(ASIndexAddr(tptr));
  return (tptr);
}
/*}}}*/

/*{{{  PRIVATE void scopeprocorfunc (tptr)*/
PRIVATE void scopeprocorfunc ( treenode *tptr )
{
  treenode *name = DNameOf(tptr);
  treenode *params = NParamListOf(name);
  namestack_t namestackmarker = markscopenames();
  int oldinsidepripar = insidepripar;
  int oldnestedpripar = nestedpripar;
  /*int oldinsideinline = insideinline;*/
  insidepripar = FALSE;
  nestedpripar = FALSE;
  /*insideinline = inline(name);*/
  insideseparatelycompiledsc = separatelycompiled(name);
  if (params != NULL)
    {
      /*{{{  scope parameters*/
      {
        treenode *p = params;
        while (!EndOfList(p))
          {
            scopeandcheck(NTypeAddr(ThisItem(p)));
            p = NextItem(p);
          }
      }
      /*}}}*/
      params = cparmlist(params, TagOf(tptr), LocnOf(tptr));
      if (params != NULL)
        walklist(addname, params);                /* Add parameters  */
      SetNParamList(name, params);
    }
  if (guyinserts) decllabels(DValOf(tptr));   /* Bring guy labels into scope */
  if (separatelycompiled(name))
    /*{{{  mark parameters as used, scopeandcheck the body*/
    {
      for (params = NParamListOf(name); params != NULL; params = NextItem(params))
        SetNUsed(ThisItem(params), TRUE);
      scopeandchecksc(DValAddr(tptr));
    }
    /*}}}*/
  else
    scopeandcheck(DValAddr(tptr));               /* Check procedure/function */
  cdeclaration(tptr);
  if (nestedpripar) SetNNestedPriPar(name, TRUE);
  insidepripar = oldinsidepripar;
  nestedpripar = oldnestedpripar;
  /*insideinline = oldinsideinline;*/
  insideseparatelycompiledsc = FALSE;
  descopenames(namestackmarker);                                 /* Descope names   */
}
/*}}}*/
/*{{{  PRIVATE void scopeandcheck (tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  scopeandcheck applies scoping and type checking to parse tree
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void scopeandcheck ( treenode **tptr )
{
  static int guy_not_asm = FALSE;  /* YES - static but local to this function! */
  if (tptr != NULL)
    while (*tptr != NULL)
      {
        treenode *t = *tptr;
        int nodetype = nodetypeoftag(TagOf(t));
        if (nodetype != WORDNODE) chklocn = LocnOf(t);
        DEBUG_MSG(("scopeandcheck: %s\n", itagstring(TagOf(t))));
        switch (nodetype)
          {
            default:
              return;
            /*{{{  specification*/
            case DECLNODE:
              {
                namestack_t namestackmarker = markscopenames();
                treenode **root = tptr;
                while ((*tptr != NULL) && isspecification(*tptr))
                  {
                    t = *tptr;
                    chklocn = LocnOf(t);
                    switch (TagOf(t))
                    {
                      /*{{{  ABBR, VALABBR, RETYPE, VALRETYPE, TPROTDEF, SPROTDEF*/
                      case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
                      case S_SPROTDEF: CASE_CONFIG_SPEC
                        {
                          treenode *name = DNameOf(t);
                          scopeandcheck(NTypeAddr(name));
                          scopeandcheck(DValAddr(t));
                          if (cdeclaration(t) != NULL)
                            addname(name);
                          break;
                        }
                      case S_TPROTDEF:
                        {
                          treenode *name = DNameOf(t);
                          treenode *n;
                          for (n = NTypeOf(name); !EndOfList(n); n = NextItem(n))
                            {
                              addname(ThisItem(n));
                              scopeandcheck(NTypeAddr(ThisItem(n)));
                            }
                          if (cdeclaration(t) != NULL)      /* Scope the protocol name */
                            addname(name);
                          break;
                        }
                      /*}}}*/
                      /*{{{  PROCDEF FUNCDEF*/
                      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
                        {
                          int savednestedpripar = nestedpripar;
                          nestedpripar = FALSE;
                          scopeprocorfunc(t);
                          nestedpripar = savednestedpripar;
                          addname(DNameOf(t));
                        }
                        break;
                      /*}}}*/
                      /*{{{  DECL*/
                      case S_DECL:
                        {
                          treenode *name = DNameOf(t);
                          #ifdef DECL_EQ
                          if (DValOf(t) == NULL)
                            synerror(TRUE, CHK_MISSING_INIT, LocnOf(t), ZERO32);
                          else
                          #endif
                          if (TagOf(name) == S_LIST)
                            /* They all share the same type tree so do this once */
                            scopeandcheck(NTypeAddr(ThisItem(name)));
                          else
                            scopeandcheck(NTypeAddr(name));
                      
                          if (cdeclaration(t) != NULL) /* Check type and name */
                            {
                              name = DNameOf(t);
                              if (TagOf(name) == S_LIST)
                                walklist(addname, name);
                              else
                                addname(name);
                            }
                        }
                        break;
                      /*}}}*/
                      /*{{{  PLACE WSPLACE VSPLACE*/
                      case S_PLACE: case S_WSPLACE: case S_VSPLACE:
                      #ifdef CONFIG
                      case S_PLACEON:
                      #endif
                        scopeandcheck(DNameAddr(t));
                        scopeandcheck(PlaceExpAddr(t));
                        *tptr = callocation(t);
                        break;
                      /*}}}*/
                      /*{{{  COMMENT RECURSIVE*/
                      /**********************  Start comment out ****************************
                      |*{{{  RECURSIVE*|
                      case S_RECURSIVE :
                        {
                          treenode *item = DValOf(t);
                          while (!EndOfList(item))
                            {
                              treenode *name = DNameOf(ThisItem(item));
                              treenode *rest = NextItem(item);
                              wordnode *id = NNameOf(name);
                              while (!EndOfList(rest))
                              {
                                if (id == NNameOf(DNameOf(ThisItem(rest))))
                                  synerror (SYN_MULTIPLE_RECURSIVE, flocn, (BIT32)WNameOf(id));
                                rest = NextItem(rest);
                              }
                              addname(name);
                              item = NextItem(item);
                            }
                          item = DValOf(t);
                          while (!EndOfList(item))
                            {
                              scopeprocorfunc(ThisItem(item));
                              item = NextItem(item);
                            }
                        }
                        break;
                      |*}}}*|
                       **********************   End comment out  ****************************/
                      /*}}}*/
                      /*{{{  labeldef*/
                      case S_LABELDEF:
                        /*if (insideinline)
                          chkrecoverreport(CHK_LABEL_INSIDE_INLINE, chklocn, ZERO32);*/
                        break;
                      /*}}}*/
                    }
                    tptr = DBodyAddr(t);
                  }
                if ((*tptr != NULL) && (TagOf(*tptr) == S_VALOF))
                  /*{{{  do checking*/
                  {
                    /* Scope and check body and result list here so we can include the
                     * preceeding specs in check of valof
                     */
                    namestack_t valofnamestackmarker = markscopenames();
                    t = *tptr;
                    scopeandcheck(VLBodyAddr(t));
                    descopenames(valofnamestackmarker);            /* Descope names within Valof */
                    scopeandcheck(VLResultListAddr(t));
                    descopenames(namestackmarker);                 /* Descope names before Valof */
                    *root = cvalof(*root, current_scope());
                  }
                  /*}}}*/
                else
                  {
                    scopeandcheck(tptr);
                    descopenames(namestackmarker);
                  }
                return;
              }
            /*}}}*/
            /*{{{  process*/
            /*{{{  Constructor*/
            case CNODE:
              if (TagOf(t) == S_PRIPAR)
              {
                int savedinsidepripar = insidepripar;
                if (insidepripar)
                  chkerr(CHK_NESTED_PRI_PAR, chklocn);
                insidepripar = TRUE;
                nestedpripar = TRUE;
                scopeandcheck(CBodyAddr(t));
                if (listitems(CBodyOf(t)) != 2)
                  chkerr(CHK_INV_PRI_PAR, chklocn);
                insidepripar = savedinsidepripar;
                return;
              }
              guy_not_asm = (TagOf(t) == S_GUY);
              tptr = CBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  REPLSEQ REPLPAR REPLIF REPLALT PRIREPLALT   return*/
            case REPLCNODE:
              {
                /*int savedinsidepripar = insidepripar;*/
                namestack_t namestackmarker = markscopenames();
                scopeandcheck(ReplCStartExpAddr(t));
                scopeandcheck(ReplCLengthExpAddr(t));
                if (crepl(t) != NULL)
                  {
                    /*{{{  Check repl is constant*/
                    if (parrepl(TagOf(t)) && !isconst(ReplCLengthExpOf(t)))
                      chkerr(CHK_REPL_NOT_CONST, chklocn);
                    /*}}}*/
                    addname(ReplCNameOf(t));
                    if (TagOf(t) == S_PRIREPLPAR)
                      {
                        chkerr(CHK_NO_PRI_REPL_PAR, chklocn);
                        #if 0  /* no point in leaving this here! */
                        if (insidepripar)
                          chkerr(CHK_NESTED_PRI_PAR, chklocn);
                        insidepripar = TRUE;
                        nestedpripar = TRUE;
                        #endif
                      }
                    scopeandcheck(ReplCBodyAddr(t));
                    #if 0
                    if (TagOf(t) == S_PRIREPLPAR)
                      insidepripar = savedinsidepripar;
                    #endif
                  }
                descopenames(namestackmarker);
              }
              return;
            /*}}}*/
            /*{{{  WHILE CHOICE                                return*/
            case CONDNODE:
              scopeandcheck(CondGuardAddr(t));
              scopeandcheck(CondBodyAddr(t));
              if (TagOf(t) != S_SELECTION)
                *tptr = ccond(t);
              return;
            /*}}}*/
            /*{{{  ALTERNATIVE                                 return*/
            case ALTNODE:
              scopeandcheck(AltGuardAddr(t));
              scopeandcheck(AltInputAddr(t));
              scopeandcheck(AltBodyAddr(t));
              *tptr = calt(t);
              return;
            /*}}}*/
            /*{{{  PINSTANCE FINSTANCE                         return*/
            case INSTANCENODE:
              {
                treenode **params = IParamListAddr(t);
                treenode *name;
                scopeandcheck(INameAddr(t));
                if (*params != NULL)
                  scopeandcheck(params);
            
                name = INameOf(t);
                if (insidepripar && TagOf(name) != N_DECL && NNestedPriParOf(name))
                  /* N.B. tag is N_DECL if the lookupname failed */
                  chkerr_s(CHK_NESTED_PRI_PROC, chklocn, WNameOf(NNameOf(name)));
            
              #ifndef CONFIG
                if (separatelycompiled(name))
                  checklibproctype(name, LocnOf(t));
              #endif
                if (TagOf(t) == S_PINSTANCE)
                  *tptr = cinstance(t);
                return ;
              }
            /*}}}*/
            /*{{{  CASE_INPUT, DELAYED_INPUT, OUTPUT, INPUT, TAGGED_INPUT, ASS, CASE  return*/
            case ACTIONNODE:
              scopeandcheck(LHSAddr(t));
              scopeandcheck(RHSAddr(t));
              *tptr = (TagOf(t) == S_CASE) ? ccase(t) : caction(t);
              return;
            /*}}}*/
            /*{{{  VARIANT                                     break*/
            case VARIANTNODE:
              scopeandcheck (VRTaggedListAddr(t));
              tptr = VRBodyAddr(t);
              break;
            /*}}}*/
            /*}}}*/
            /*{{{  expression*/
            /*{{{  monadics            return*/
            case MOPNODE:
              tptr = OpAddr(t);
              break;
            /*}}}*/
            /*{{{  dyadics             break*/
            case DOPNODE:
              if ((TagOf(t) == S_GUYCODE) || (TagOf(t) == S_GUYSTEP))
                {
                  scopeandcheck(RightOpAddr(t));
                  /* guy_not_asm was set up by enclosing S_GUY or S_ASM */
                  *tptr = cguy_or_asm(t, guy_not_asm);
                  return;
                }
              scopeandcheck(LeftOpAddr(t));
              tptr = RightOpAddr(t);
              break;
            /*}}}*/
            /*{{{  VALOF                       return*/
            case VALOFNODE:
              {
                namestack_t namestackmarker = markscopenames();
                scopeandcheck(VLBodyAddr(t));
                descopenames(namestackmarker);      /* Descope names within Valof */
                scopeandcheck(VLResultListAddr(t));
                *tptr = cvalof(t, current_scope());
              }
              return;
            /*}}}*/
            #ifdef ARRAYCONSTRUCTOR
            /*{{{  arrayconstructor*/
            case S_ARRAYCONSTRUCTOR :  unknown
              {
                namestack_t namestackmarker = markscopenames();
                scopeandcheck(ACStartExpAddr(t));
                scopeandcheck(ACLengthExpAddr(t));
                addname(ACNameOf(t));
                scopeandcheck(ACValExpAddr(t));
                descopenames(namestackmarker);
              }
              return;
            /*}}}*/
            #endif
            #ifdef CONDEXP
            /*{{{  conditional expression*/
            case CONDEXPNODE:
              {
                scopeandcheck(CondExpGuardAddr(t));
                scopeandcheck(CondExpTrueAddr(t));
                tptr = CondExpFalseAddr(t);
              }
              break;
            /*}}}*/
            #endif
            /*}}}*/
            /*{{{  element*/
            case ARRAYSUBNODE:
              scopeandcheck(ASBaseAddr(t));
              *tptr = csubscript(t);  /* convert to S_RECORDSUB if necessary */
              return;
            case SEGMENTNODE:
              scopeandcheck(SNameAddr(t));
              scopeandcheck(SStartExpAddr(t));
              scopeandcheck(SLengthExpAddr(t));
              tptr = SSubscriptExpAddr(t);
              break;
            /*}}}*/
            /*{{{  type*/
            /*{{{  arraynode*/
            case ARRAYNODE :
              scopeandcheck(ARDimLengthAddr(t));
              tptr = ARTypeAddr(t);
              break;
            /*}}}*/
            /*{{{  channel/port*/
            case CHANNODE:
              tptr = ProtocolAddr(t);
              break;
            /*}}}*/
            /*}}}*/
            /*{{{  configuration*/
            /*{{{  PROCESSOR*/
            case PROCESSORNODE:
              scopeandcheck(ProcessorExpAddr(t));
              scopeandcheck(ProcessorBodyAddr(t));
              SetProcessorScope(t, current_scope());
              *tptr = cprocessor(t);
              return;
            /*}}}*/
            /*}}}*/
            /*{{{  NAME*/
            case WORDNODE :
              if (TagOf(t) != S_ASMNAME)
                {
                  wordnode *n = (wordnode *)t;
                  *tptr = lookupname(n, findname(n));
                  SetNUsed(*tptr, TRUE);
                }
              return;
            /*}}}*/
            /*{{{  List*/
            case LISTNODE :
              while (!EndOfList(t))
                {
                  scopeandcheck(ThisItemAddr(t));
                  t = NextItem(t);
                }
              return;
            /*}}}*/
            #ifdef CONFIG
            #if 0
            /*{{{  structconstructor*/
            case S_STRUCTCONSTRUCTOR : unknown
              {
                if (MOpTypeOf(t) == S_NODE)
                  {
                    t = OpOf(t);              /* This will be a list */
                    if (!EndOfList(t))        /* Skip device type    */
                      t = NextItem(t);
                    while (!EndOfList(t))
                      {
                        scopeandcheck(ThisItemAddr(t));
                        t = NextItem(t);
                      }
                    *tptr = cnetnode(*tptr);
                  }
              }
              return;
            /*}}}*/
            #endif
            /*{{{  config*/
            case CONFIGNODE:
              scopeandcheck(STDevAddr(t));
              if (TagOf(t) != S_SET)
                scopeandcheck(STAttrNameAddr(t));
              scopeandcheck(STAttrExpAddr(t));
              switch(TagOf(t))
                {
                  case S_CONNECT: *tptr = cconnect(t); break;
                  case S_MAP:     *tptr = cmap(t);     break;
                  case S_SET:     *tptr = cset(t);     break;
                }
              return;
            /*}}}*/
            #endif
          }
      }
}
/*}}}*/
#ifdef CONFIG

/* Checking states:
 S_END  outermost VALs
 S_NETWORK   inside a network description
 S_MAPPING   inside a MAPPING construct
 S_CONFIG    inside a CONFIG construct
 S_PROCESSOR inside a PROCESSOR construct
*/
/*{{{  PRIVATE void invalidconfigconstruct()*/
PRIVATE void invalidconfigconstruct(treenode *tptr)
{
  if (configcheckstate == S_END)
    chkerr(CHK_ILLEGAL_CONFIG_CONSTRUCT, LocnOf(tptr));
  else
    chkerr_s(CHK_ILLEGAL_CONSTRUCT, LocnOf(tptr), tagstring(configcheckstate));
}
/*}}}*/
/*{{{  PRIVATE void changestate()*/
PRIVATE void changestate(treenode *tptr, treenode *new_tptr, int req_state, int new_state)
{
  if (configcheckstate == req_state)
    {
      DEBUG_MSG(("changestate: from %s to %s\n", tagstring(req_state), tagstring(new_state)));
      configcheckstate = new_state;
      prewalkproctree(new_tptr, configcheck);
      configcheckstate = req_state;
      DEBUG_MSG(("changestate: back %s to %s\n", tagstring(new_state), tagstring(req_state)));
    }
  else
    invalidconfigconstruct(tptr);
}
/*}}}*/
/*{{{  PRIVATEPARAM int configcheck()*/
PRIVATEPARAM int configcheck(treenode *tptr)
{
  /* rscunit has ensured that if we're genuinely at the top level, all
     we will see are S_VALABBR, S_VALRETYPE, S_TPROTDEF, S_SPROTDEF,
     S_PROCDEF, S_LFUNCDEF, S_SFUNCDEF, S_NETWORK, S_CONFIG, S_MAPPING
  */
  DEBUG_MSG(("configcheck: %s\n", itagstring(TagOf(tptr))));
  switch (nodetypeoftag(TagOf(tptr)))
    {
      case CONSTEXPNODE: case CONSTTABLENODE: case LITNODE:
      case MOPNODE: case DOPNODE:
      case SEGMENTNODE:
      case NAMENODE:
        return CONTINUE_WALK;
      default:
        break;
    }
  switch (TagOf(tptr))
    {
      default:
        if (configcheckstate != S_PROCESSOR)
          invalidconfigconstruct(tptr);
        break;
      /*{{{  S_CONFIG*/
      case S_CONFIG:
        configcount++;
        if (configcount > 1)
          chkerr_s(CHK_DUPLICATE_CONSTRUCT, LocnOf(tptr), tagstring(S_CONFIG));
        changestate(tptr, DValOf(tptr), S_END, S_CONFIG);
        break; /* continue after the CONFIG */
      /*}}}*/
      /*{{{  S_NETWORK*/
      case S_NETWORK:
        networkcount++;
        if (networkcount > 1)
          chkerr_s(CHK_DUPLICATE_CONSTRUCT, LocnOf(tptr), tagstring(S_NETWORK));
        changestate(tptr, DValOf(tptr), S_END, S_NETWORK);
        break; /* continue after the NETWORK */
      /*}}}*/
      /*{{{  S_MAPPING*/
      case S_MAPPING:
        mappingcount++;
        if (mappingcount > 1)
          chkerr_s(CHK_DUPLICATE_CONSTRUCT, LocnOf(tptr), tagstring(S_NETWORK));
        changestate(tptr, DValOf(tptr), S_END, S_MAPPING);
        break; /* continue after the MAPPING */
      /*}}}*/
      /*{{{  S_PROCESSOR*/
      case S_PROCESSOR:
        configcheckscope = ProcessorScopeOf(tptr);
        changestate(tptr, ProcessorBodyOf(tptr), S_CONFIG, S_PROCESSOR);
        return STOP_WALK;
      /*}}}*/
      /*{{{  NETWORK or MAPPING only*/
      case S_DO: case S_REPLDO: case S_SET:
        if (configcheckstate != S_NETWORK && configcheckstate != S_MAPPING)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  CONFIG or PROCESSOR only*/
      case S_PAR: case S_REPLPAR:
        if (configcheckstate != S_CONFIG && configcheckstate != S_PROCESSOR)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  toplevel or CONFIG only*/
      case S_PLACEON:
        if (configcheckstate != S_CONFIG && configcheckstate != S_END)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  nowhere except config*/
      case S_PLACEDPAR: case S_PLACEDREPLPAR:
        if (configcheckstate != S_CONFIG)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  nowhere except network*/
      case S_CONNECT: case S_RECORDSUB:
        if (configcheckstate != S_NETWORK)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  nowhere except mapping*/
      case S_MAP:
        if (configcheckstate != S_MAPPING)
          invalidconfigconstruct(tptr);
        break;
      /*}}}*/
      /*{{{  PROC/FUNCTION*/
      case S_PROCDEF: case S_LFUNCDEF: case S_SFUNCDEF:
        DEBUG_MSG(("configcheck: PROC/FUNCTION %s (%s)\n",
                   WNameOf(NNameOf(DNameOf(tptr))), itagstring(TagOf(DNameOf(tptr)))));
        if (separatelycompiled(DNameOf(tptr)))
          {
            if (configcheckstate == S_NETWORK || configcheckstate == S_MAPPING)
              invalidconfigconstruct(tptr);
          }
        else /* locally defined */
          {
            if (configcheckstate != S_PROCESSOR)
              invalidconfigconstruct(tptr);
            prewalkproctree(DValOf(tptr), configcheck);
          }
        break;
      /*}}}*/
      /*{{{  declarations*/
      case S_DECL: case S_ABBR: /* case S_RETYPE: */
        {
          treenode *name = DNameOf(tptr);
          int type;
          if (TagOf(name) == S_LIST) name = ThisItem(name);
          type = basetype(NTypeOf(name));
          if (((configcheckstate == S_END || configcheckstate == S_NETWORK) &&
               network_datatype(type)                                     ) ||
              ((configcheckstate == S_CONFIG || configcheckstate == S_END ) &&
               (type == S_CHAN)                                           ) ||
              ((configcheckstate == S_PROCESSOR) && !network_datatype(type))    )
            ;
          else
            invalidconfigconstruct(tptr);
        }
        break;
      /*}}}*/
      /*{{{  PLACEMENTs*/
      case S_PLACE: case S_WSPLACE: case S_VSPLACE:
        if (configcheckstate != S_PROCESSOR)
          invalidconfigconstruct(tptr);
        else if (NScopeOf(DNameOf(tptr)) < configcheckscope)
          chkerr_s(CHK_ILLEGAL_PLACE, LocnOf(tptr), WNameOf(NNameOf(DNameOf(tptr))));
        break;
      /*}}}*/
      /*{{{  Always valid*/
      case S_VALABBR: case S_VALRETYPE:
      case S_TPROTDEF: case S_SPROTDEF:
      case S_IF: /*case S_REPLIF:*/ case S_CHOICE:
      case S_SKIP: case S_STOP: case S_LIST: case S_END:
      case S_ARRAYSUB:
        break;
      /*}}}*/
    }
  return CONTINUE_WALK;
}
/*{{{*/
#endif
/*{{{  PUBLIC void scopeandcheck_main()*/
PUBLIC void scopeandcheck_main(treenode **tptr)
{
  scopeandcheck(tptr);
#ifdef CONFIG
  if ((errors == 0) && !nochecking && (lexmode != LEX_STDLIB))
    {
      configcheckstate = S_END; /*(compilemode == COMP_PROGRAM ? S_END : S_PROCESSOR);*/
      configcount = 0; networkcount = 0; mappingcount = 0;
      prewalkproctree(*tptr, configcheck);
      if (configcount == 0)
        chkerr_s(CHK_ZERO_CONSTRUCT, ZERO32, tagstring(S_CONFIG));
      if (networkcount == 0)
        chkerr_s(CHK_ZERO_CONSTRUCT, ZERO32, tagstring(S_NETWORK));
    }
#endif
}
/*}}}*/

/*{{{  PUBLIC void scopeinit ()*/
PUBLIC void scopeinit ( void )
{
  /*{{{  initialise real namestack*/
  namep = 0;
  namestack_base = newnamestackentry(NULL, NULL, NULL);
  namestack = namestack_base;
  /*}}}*/
  /*{{{  initialise undeclared namestack*/
  /*undecnamep = 0;*/
  undeclarednamestack_base = newnamestackentry(NULL, NULL, NULL);
  undeclarednamestack = undeclarednamestack_base;
  /*}}}*/
  insidepripar = FALSE;
  /*insideinline = FALSE;*/
#ifdef CONFIG
  init_attributes();
#endif
}
/*}}}*/
/*}}}*/
