/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two checker Constant expression evaluation
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>

# include "includes.h"
# include "extlib.h"

# include "chkerror.h"
# include "lexdef.h"
# include "lex1def.h"
# include "chk1def.h"
# include "chk3def.h"
# include "chkdef.h"
# include "predefhd.h"
# include "genhdr.h"
# include "gen1def.h"
/*}}}*/

/*{{{  local variables*/
typedef struct
  {
    tagtype_s tag;  /* same tagtype as treenodes */
    unsigned char *val;
  } bytenode_t;
PRIVATE bytenode_t bytenode;

PRIVATE int overlaps_are_const = FALSE;
/*}}}*/
/*{{{  PUBLIC int checkbounds (sourcetype, desttype, shi, slo)*/

/* Check that the byte or integer value of type sourcetype held in shi:slo
can be represented as a byte or integer value of type desttype */
PUBLIC int checkbounds ( int sourcetype , int desttype , BIT32 shi , BIT32 slo )
{
  BIT32 maxhi, maxlo, minhi, minlo;
  int toobig, toosmall;

  if (sourcetype != S_INT64)
    /*{{{  sign extend slo into shi for 64-bit comparison*/
    I32ToI64 (&shi, &slo, slo);
    /*}}}*/
  /*{{{  set up maxhi, maxlo, minhi, minlo*/
  switch (desttype)
    {
      /*{{{  S_BOOL*/
      case S_BOOL:
        maxhi = 0; maxlo = 1;
        minhi = 0; minlo = 0;
        break;
      /*}}}*/
      /*{{{  S_BYTE*/
      case S_BYTE:
        maxhi = 0; maxlo = 255;
        minhi = 0; minlo = 0;
        break;
      /*}}}*/
      /*{{{  S_INT16*/
      case S_INT16:
        maxhi = 0; maxlo = MOSTPOS_INT16;
        minhi = 0xffffffffl; minlo = MOSTNEG_INT16;
        break;
      /*}}}*/
      /*{{{  S_INT32*/
      case S_INT32:
        maxhi = 0; maxlo = MOSTPOS_INT32;
        minhi = 0xffffffffl; minlo = MOSTNEG_INT32;
        break;
      /*}}}*/
      /*{{{  S_INT64*/
      case S_INT64:
        maxhi = MOSTPOS_INT32; maxlo = 0xffffffffl;
        minhi = MOSTNEG_INT32; minlo = 0;
        break;
      /*}}}*/
      /* Should never get here */
      default:
        badtag(chklocn, (BIT32)desttype, "checkbounds");
    }
  /*}}}*/
  Int64Gt (&toobig, shi, slo, maxhi, maxlo);
  Int64Gt (&toosmall, minhi, minlo, shi, slo);
  if (toobig)
    return (1);
  else if (toosmall)
    return (-1);
  else
    return (0);
}
/*}}}*/
/*{{{  PRIVATE void IStrToInt (inttype, error, hi, lo, string)*/
PRIVATE void IStrToInt ( int inttype , int *error , BIT32 *hi , BIT32 *lo , char *string )
{
  *hi = 0; /* clean up messy constants - 23/11/90 */
  if (*string == '#')
    /*{{{  read a hex string*/
    {
      string++;
      switch (inttype)
        {
          case S_BYTELIT:  StrToH8 (error, lo, string); break;
          case S_INT16LIT: StrToH16 (error, lo, string); break;
          case S_INT32LIT: StrToHex (error, lo, string); break;
          case S_INT64LIT: StrToH64 (error, hi, lo, string); break;
        }
    }
    /*}}}*/
  else if (*string == '\'')
    /*{{{  read a converted byte literal*/
    {
    /* *hi = 0; */ /* this has been moved up a bit */
      *lo = (BIT32)(string[1] & 0xff);
      *error = FALSE;
    }
    /*}}}*/
  else
    /*{{{  read a decimal string*/
    switch (inttype)
      {
        case S_BYTELIT:  StrToI8 (error, lo, string); break;
        case S_INT16LIT: StrToI16 (error, lo, string); break;
        case S_INT32LIT: StrToInt (error, lo, string); break;
        case S_INT64LIT: StrToI64 (error, hi, lo, string); break;
      }
    /*}}}*/
}
/*}}}*/
/*{{{  constant folding*/
/*{{{  PRIVATE void exactconversion (sourcetype, desttype, dhi, dlo, shi, slo)*/
PRIVATE void exactconversion ( int sourcetype , int desttype , BIT32 *dhi , BIT32 *dlo , BIT32 shi , BIT32 slo )
{
   if (isreal (desttype))
    /*{{{  real to real conversion*/
    {
      if ((desttype == S_REAL64) && (sourcetype == S_REAL32))
      /* The only legal conversion which has any effect */
        {
          int error;
          R32ToR64 (&error, dhi, dlo, slo);
          if (error)
            chkreport (CHK_CFOLD_OFLOW, chklocn);
        }
    }
    /*}}}*/
  else
    /*{{{  bool byte or int conversion*/
    {
      if (sourcetype == S_INT)
        sourcetype = targetintsize;
      if (desttype == S_INT)
        desttype = targetintsize;
    
      if ((desttype == S_INT64) && (sourcetype != S_INT64))
        I32ToI64(&shi, &slo, slo);
      if (checkbounds (sourcetype, desttype, shi, slo) != 0)
        chkreport (CHK_CFOLD_OFLOW, chklocn);
      else
        {
          *dlo = slo;
          *dhi = shi;
        }
    }
    /*}}}*/
}
/*}}}*/
/*{{{  pointing into contructors and constant tables*/
/*{{{  PUBLIC treenode *point_at_list_item(tptr, n)*/
/*****************************************************************************
 *
 *  point_at_list_item takes a pointer to a list, 'tptr', and moves
 *                     n items along the list, and returns a pointer to the
 *                     corresponding list item.
 *
 *****************************************************************************/
PUBLIC treenode *point_at_list_item ( treenode *tptr , BIT32 n )
{
  int i;
  for (i = 0; i < n; i++)
    {
      if (EndOfList(tptr))    /* We have fallen off the end of the list */
        chkreport_i(CHK_SUBSCRIPT_RANGE, chklocn, n);
      tptr = NextItem(tptr);
    }
  return (ThisItem(tptr));
}
/*}}}*/
/*{{{  PUBLIC treenode *point_at_subscript(sptr, type, subscript)*/
/*****************************************************************************
 *
 * point_at_subscript takes a pointer to a construct 'sptr' of type 'type'
 *                    and a subscript value and returns a pointer to that
 *                    element of the construct.
 *
 *****************************************************************************/
PUBLIC treenode *point_at_subscript ( treenode *sptr , treenode *type , BIT32 subscript )
{
  switch (TagOf(sptr))
    {
      /*{{{  CONSTRUCTOR*/
      case S_CONSTRUCTOR:
        sptr = OpOf(sptr);
        return (point_at_list_item (sptr, subscript));
      /*}}}*/
      /*{{{  CONSTCONSTRUCTOR STRING*/
      case S_CONSTCONSTRUCTOR:
      case S_STRING:
        {
          INT32 element_size = bytesin(type);
          bytenode.tag = S_CONSTPTR;
          bytenode.val = (unsigned char *)WNameOf(CTValOf(sptr)) +
                                             (subscript * element_size);
          return((treenode *)(&bytenode));
        }
      /*}}}*/
      /*{{{  CONSTPTR*/
      case S_CONSTPTR:
        {
          INT32 element_size = bytesin(type);
          bytenode.val += subscript * element_size;
          return((treenode *)(&bytenode));
        }
      /*}}}*/
      /*{{{  LIST*/
      case S_LIST:
        return (point_at_list_item(sptr, subscript));
      /*}}}*/
      #ifdef ARRAYCONSTRUCTOR
      /*{{{  ARRAYCONSTRUCTOR*/
      case S_ARRAYCONSTRUCTOR :
        {
          treenode *newtree, *index;
          index = newconstexpnode (S_CONSTEXP, LocnOf(sptr),
                                   dummyexp_p, 0, subscript);
          index = newdopnode (S_ADD, LocnOf(sptr),
                              copytree(ACStartExpOf(sptr)), index, S_INT);
          marknametrans();
          addnametrans(ACNameOf(sptr), index);
          newtree = transcopytree(ACValExpOf(sptr), 0);
          freenametrans();
          return newtree;
        }
      /*}}}*/
      #endif
      default:
        chkreport_i(CHK_INV_CONSTRUCT, chklocn, TagOf(sptr));
    }
  return ((treenode *)NULL); /* Not reached */
}
/*}}}*/
/*{{{  PUBLIC treenode *point_at_construct (tptr, checked)*/
/*****************************************************************************
 *
 * point_at_construct descends the element tree tptr, to
 *                    reach the value of the element.
 *                    If checked is TRUE, it will check that subscripts
 *                    and segment ranges are legal.
 *
 *****************************************************************************/
PUBLIC treenode *point_at_construct ( treenode *tptr , int checked )
{
  switch (TagOf(tptr))
    {
      /*{{{  case S_ARRAYSUB*/
      case S_ARRAYSUB:
        {
          treenode *sptr = point_at_construct(ASBaseOf(tptr), checked);
          /* Type of the thing we are subscripting */
          treenode *type = gettype(ASBaseOf(tptr));
          BIT32 shi, slo;
      
          foldconstexp (ASIndexOf(tptr), &shi, &slo, CHK_EXP_NOT_CONST);
          if (!checked)
            /*{{{  check subcript range*/
            {
              if (slo >= ARDimOf(type))
                chkreport_i(CHK_SUBSCRIPT_RANGE, chklocn, slo);
            }
            /*}}}*/
          return(point_at_subscript(sptr, ARTypeOf(type), slo));
        }
      /*}}}*/
      /*{{{  case S_SEGMENT*/
      case S_SEGMENT:
        {
          treenode *sptr = point_at_construct (SNameOf(tptr), checked);
          /* Type of the thing we are subscripting */
          treenode *type = gettype(ThisItem(tptr));
          BIT32 shi, slo, lhi, llo;
      
          /*{{{  fold start and length*/
          foldconstexp (SStartExpOf(tptr), &shi, &slo, CHK_EXP_NOT_CONST);
          foldconstexp (SLengthExpOf(tptr), &lhi, &llo, CHK_EXP_NOT_CONST);
          /*}}}*/
          if (!checked)
            /*{{{  check start and length are legal*/
            {
              INT32 i = (INT32)slo, j = (INT32)llo;
              INT32 d = ARDimOf(type);
              if (d == (-1)) /* dimension is unknown */
                d = MOSTPOS_INT32;
              if (i < 0 || i > d)
                chkreport_i(CHK_SEG_START_RANGE, chklocn, i);
              if (j < 0)
                chkreport_i(CHK_SEG_LENGTH_RANGE, chklocn, j);
              if (j > (d - i))
                chkreport_i(CHK_SEG_RANGE, chklocn, i + j - 1);
            }
            /*}}}*/
      
          return(point_at_subscript(sptr, ARTypeOf(type), slo));
        }
      /*}}}*/
      /*{{{  case N_VALABBR case N_VALRETYPE*/
      case N_VALABBR:
      case N_VALRETYPE:
        return(point_at_construct(DValOf(NDeclOf(tptr)), TRUE));
      /*}}}*/
      /*{{{  COMMENT case S_CONSTRUCTOR*/
      /**********************  Start comment out ****************************
      |*{{{  case S_CONSTRUCTOR*|
      case S_CONSTRUCTOR:
        |* fold it and  return pointer to the folded version *|
        return(point_at_construct(foldexp(tptr)));
      |*}}}*|
       **********************   End comment out  ****************************/
      /*}}}*/
      default:
        return (tptr);
    }
}
/*}}}*/
/*}}}*/
/*{{{  folding constant tables                         ***/
/*{{{  PRIVATE void foldscalar_into_array (BIT32 hi, lo, BYTE *cptr, int len)*/
/*****************************************************************************
 *
 *  foldscalar_into_array takes a scalar value in (hi, lo) and puts it
 *                        in the array cptr, of length len.
 *
 *****************************************************************************/
PRIVATE void foldscalar_into_array ( BIT32 hi , BIT32 lo , BYTE *cptr , int len )
{
  int i;
  BIT32 v = lo;
#if TARGET_LITTLEENDIAN
  for (i = 0; i < len; i++)
    {
      cptr[i] = (BYTE)(v & 0xffl);
      v >>= 8;
      if (i == 3) v = hi;
    }
#else
  for (i = len - 1; i > 0; i--)
    {
      cptr[i] = (BYTE)(v & 0xffl);
      v >>= 8;
      if (i == 4) v = hi;
    }
#endif
}
/*}}}*/
/*{{{  PRIVATE void foldarray_into_scalar (BYTE *cptr, int len, BIT32 *hi, *lo)*/
/*****************************************************************************
 *
 *  foldarray_into_scalar takes an array cptr, of length len, and puts it
 *                        in the scalar (*hi, *lo).
 *
 *****************************************************************************/
PRIVATE void foldarray_into_scalar ( BYTE *cptr , int len , BIT32 *hi , BIT32 *lo )
{
  BYTE ar[8];
  BYTE extendbyte;
  int i;

  for (i = 0; i < len; i++)
    ar[i] = *cptr++;

  extendbyte = (ar[len - 1] & 0x80) ? 0xff : 0;
  for (/* i = len */; i < 8; i++)
    ar[i] = extendbyte;

#if TARGET_LITTLEENDIAN
  *lo = (((((ar[3] << 8) | ar[2]) << 8) | ar[1]) << 8) | ar[0];
  *hi = (((((ar[7] << 8) | ar[6]) << 8) | ar[5]) << 8) | ar[4];
#else
  *lo = (((((ar[4] << 8) | ar[5]) << 8) | ar[6]) << 8) | ar[7];
  *hi = (((((ar[0] << 8) | ar[1]) << 8) | ar[2]) << 8) | ar[3];
#endif
}
/*}}}*/
/*{{{  forward declaration of foldconstructor*/
PRIVATE BYTE *foldconstructor PARMS((treenode *tptr, BYTE *cptr, int e));
/*}}}*/
/*{{{  PRIVATE BYTE *foldconstructorelement(tptr, cptr, e)         ***/
/*****************************************************************************
 *
 * foldconstructorelement takes an element of a constructor in tptr.
 *                        If the element is an expression it is constant
 *                        folded, and stored in e bytes beginning at cptr.
 *                        If the element is a constructor, the routine
 *                        foldconstructor is called to fold each nested
 *                        element.
 *                        The updated cptr is returned.
 *
 *****************************************************************************/
PRIVATE BYTE *foldconstructorelement ( treenode *tptr , BYTE *cptr , int e )
{
  /*{{{  comment*/
  /* The element could be
       exp                    foldconstexp
       CONSTRUCTOR            foldconstructor
       STRING                 do it explicitly
       SEGMENT
       VAL abbrev or retype   walk through to original declaration
                              - this means we can now have CONSTCONSTRUCTORs
         hmm... point_at_construct will point to the first element
                then foldconstructorelement for length times.
                See constraint on what is segmented, below.
    */
  /*}}}*/
  switch(TagOf(tptr))
    /*{{{  cases*/
    {
      default:
        /*{{{  constant fold and store*/
        {
          BIT32 rhi, rlo;
          foldconstexp(tptr, &rhi, &rlo, CHK_EXP_NOT_CONST);
          foldscalar_into_array(rhi, rlo, cptr, e);
          return cptr + e;
        }
        /*}}}*/
      /*{{{  S_CONSTRUCTOR*/
      case S_CONSTRUCTOR:
      #ifdef ARRAYCONSTRUCTOR
      case S_ARRAYCONSTRUCTOR :
      #endif
        return(foldconstructor(tptr, cptr, e));
      /*}}}*/
      /*{{{  S_STRING S_CONSTCONSTRUCTOR*/
      case S_STRING : case S_CONSTCONSTRUCTOR:
        {
          BYTE *sptr = (BYTE *)WNameOf(CTValOf(tptr));
          int len = WLengthOf(CTValOf(tptr));
          memcpy(cptr, sptr, len);
          cptr += len;
          return cptr;
        }
      /*}}}*/
      /*{{{  S_SEGMENT                           ***/
      case S_SEGMENT:
        /* Here I assume you can only segment a string or constant
           constructor, ie.
             i.  You can't expect a constant segment of a non-constant
                 constructor to be constant
             ii. You can't segment (or subscript) a constructor before it
                 has been folded. The language definition covers this at
                 the moment.
                 **  The new Prentice-Hall version of the language allows
                     a constructor to be subscripted. */
        {
          bytenode_t *sptr = (bytenode_t *)point_at_construct(tptr, FALSE);
          BIT32 lhi, llo, i;
          if (TagOf(sptr) != S_CONSTPTR)
            badtag(chklocn, (BIT32)TagOf(sptr), "foldconstructorelement");
          foldconstexp (SLengthExpOf(tptr), &lhi, &llo, CHK_EXP_NOT_CONST);
          for (i = 0; i < llo; i++)
            {
              cptr = foldconstructorelement((treenode *)sptr, cptr, e);
              sptr->val += e;
            }
          return(cptr);
        }
      /*}}}*/
      /*{{{  S_CONSTPTR*/
      case S_CONSTPTR:
        {
          BYTE *sptr = ((bytenode_t *)tptr)->val;
          *cptr++ = *sptr;
          return(cptr);
        }
      /*}}}*/
      /*{{{  case N_VALABBR case N_VALRETYPE*/
      case N_VALABBR: case N_VALRETYPE:
        return (foldconstructorelement(DValOf(NDeclOf(tptr)), cptr, e));
      /*}}}*/
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE BYTE *foldconstructor(tptr, cptr, e)*/
/*****************************************************************************
 *
 * foldconstructor constant folds a constructor into a BYTE array.
 *                 tptr points to the constant constructor tree to be folded,
 *                 cptr points to the next free byte of the array,
 *                 e is the number of bytes to be occupied by each base
 *                 element of the constructor.
 *                 The updated value of cptr is returned.
 *
 *****************************************************************************/
PRIVATE BYTE *foldconstructor ( treenode *tptr , BYTE *cptr , int e )
{
#ifdef ARRAYCONSTRUCTOR
  if (TagOf(tptr) == S_ARRAYCONSTRUCTOR)
    {
      INT32 i;
      treenode *index  = ACNameOf(tptr);
      INT32 start      = evaluate(ACStartExpOf(tptr));
      INT32 end        = start + evaluate(ACLengthExpOf(tptr)) ;
      SetNReplKnown(index, TRUE);
      for (i = start;  i < end; i ++)
        {
          SetNReplValue(index, i);
          cptr = foldconstructorelement(ACValExpOf(tptr), cptr, e);
        }
      SetNReplKnown(index, FALSE);
    }
  else
#endif
    {
      treenode *t = OpOf(tptr);
      while (!EndOfList(t))
        {
          cptr = foldconstructorelement(ThisItem(t), cptr, e);
          t = NextItem(t);
        }
    }
  return(cptr);
}
/*}}}*/
/*}}}*/
/*{{{  folding expression operators*/
/*{{{  PRIVATE void foldmonadic (tptr, reshi, reslo, e)*/
PRIVATE void foldmonadic ( treenode *tptr , BIT32 *reshi , BIT32 *reslo , int e)
{
  int type = MOpTypeOf(tptr), error = FALSE;
  BIT32 ophi, oplo;

  foldconstexp (OpOf(tptr), &ophi, &oplo, e);
  if (type == S_INT)
    type = targetintsize;
  switch (TagOf(tptr))
    {
      /*{{{  S_NEG*/
      case S_NEG:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Sub (&error, reslo, ZERO32, oplo);
              break;
            case S_INT32:
              Int32Sub (&error, reslo, ZERO32, oplo);
              break;
            case S_INT64:
              Int64Sub (&error, reshi, reslo, ZERO32, ZERO32, ophi, oplo);
              break;
            case S_REAL32:
              Real32Op (&error, reslo, ZERO32, Op_Sub + (RN << 2), oplo);
              break;
            case S_REAL64:
              Real64Op (&error, reshi, reslo, ZERO32, ZERO32,
                        Op_Sub + (RN << 2), ophi, oplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_UMINUS*/
      case S_UMINUS:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16Minus (reslo, ZERO32, oplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32Minus (reslo, ZERO32, oplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64Minus (reshi, reslo, ZERO32, ZERO32, ophi, oplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_BITNOT*/
      case S_BITNOT:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16BitNot (reslo, oplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32BitNot (reslo, oplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64BitNot (reshi, reslo, ophi, oplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_NOT*/
      case S_NOT:
        if (oplo == TRUE)
          *reslo = FALSE;
        else
          *reslo = TRUE;
        break;
      /*}}}*/
    }
}
/*}}}*/
/*{{{  PRIVATE void folddyadic (tptr, reshi, reslo, e)*/
PRIVATE void folddyadic ( treenode *tptr , BIT32 *reshi , BIT32 *reslo , int e)
{
  int type = DOpTypeOf(tptr), error = FALSE;
  BIT32 leftophi, leftoplo, rightophi, rightoplo;

  foldconstexp (LeftOpOf(tptr), &leftophi, &leftoplo, e);
  foldconstexp (RightOpOf(tptr), &rightophi, &rightoplo, e);

  if (type == S_INT)
    type = targetintsize;
  switch (TagOf(tptr))
    {
      /*{{{  operator*/
      /*{{{  S_AND*/
      case S_AND:
        if ((leftoplo == TRUE) && (rightoplo == TRUE))
          *reslo = TRUE;
        else
          *reslo = FALSE;
        break;
      /*}}}*/
      /*{{{  S_OR*/
      case S_OR:
        if ((leftoplo == TRUE) || (rightoplo == TRUE))
          *reslo = TRUE;
        else
          *reslo = FALSE;
        break;
      /*}}}*/
      /*{{{  S_ADD*/
      case S_ADD:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Add (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT32:
              Int32Add (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT64:
              Int64Add (&error, reshi, reslo, leftophi, leftoplo,
                        rightophi, rightoplo);
              break;
            case S_REAL32:
              Real32Op (&error, reslo, leftoplo, Op_Add + (RN << 2), rightoplo);
              break;
            case S_REAL64:
              Real64Op (&error, reshi, reslo, leftophi, leftoplo,
                        Op_Add + (RN << 2), rightophi, rightoplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_CSUB0*/
      /* This can only occur when the backend is trying to fold something */
      case S_CSUB0:
        {
          int toobig, toosmall, equal;
      
          if (type != S_INT64)
            {
            /*I32ToI64 (&leftophi, &leftoplo, leftoplo);
              I32ToI64 (&rightophi, &rightoplo, rightoplo);*/
              leftophi = 0;  /* do an UNSIGNED conversion to 64 bits */
              rightophi = 0; /* ditto */
            }
          Int64Gt (&toobig, leftophi, leftoplo, rightophi, rightoplo);
          Int64Eq (&equal,  leftophi, leftoplo, rightophi, rightoplo);
          Int64Gt (&toosmall, ZERO32, ZERO32, leftophi, leftoplo);
          if (toobig || equal || toosmall)
            chkreport(CHK_SUBSCRIPT_OUT_OF_RANGE, LocnOf(tptr));
          else
            {
              *reslo = leftoplo;
              *reshi = leftophi;
            }
        }
        break;
      /*}}}*/
      /*{{{  S_SUBTRACT*/
      case S_SUBTRACT:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Sub (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT32:
              Int32Sub (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT64:
              Int64Sub (&error, reshi, reslo, leftophi, leftoplo,
                        rightophi, rightoplo);
              break;
            case S_REAL32:
              Real32Op (&error, reslo, leftoplo, Op_Sub + (RN << 2), rightoplo);
              break;
            case S_REAL64:
              Real64Op (&error, reshi, reslo, leftophi, leftoplo,
                        Op_Sub + (RN << 2), rightophi, rightoplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_MULT*/
      case S_MULT:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Mul (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT32:
              Int32Mul (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT64:
              Int64Mul (&error, reshi, reslo, leftophi, leftoplo,
                        rightophi, rightoplo);
              break;
            case S_REAL32:
              Real32Op (&error, reslo, leftoplo, Op_Mul + (RN << 2), rightoplo);
              break;
            case S_REAL64:
              Real64Op (&error, reshi, reslo, leftophi, leftoplo,
                        Op_Mul + (RN << 2), rightophi, rightoplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_DIV*/
      case S_DIV:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Div (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT32:
              Int32Div (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT64:
              Int64Div (&error, reshi, reslo, leftophi, leftoplo,
                        rightophi, rightoplo);
              break;
            case S_REAL32:
              Real32Op (&error, reslo, leftoplo, Op_Div + (RN << 2), rightoplo);
              break;
            case S_REAL64:
              Real64Op (&error, reshi, reslo, leftophi, leftoplo,
                        Op_Div + (RN << 2), rightophi, rightoplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_REM*/
      case S_REM:
        /*{{{  perform operation depending on type*/
        switch (type)
          {
            case S_INT16:
              Int16Rem (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT32:
              Int32Rem (&error, reslo, leftoplo, rightoplo);
              break;
            case S_INT64:
              Int64Rem (&error, reshi, reslo, leftophi, leftoplo,
                        rightophi, rightoplo);
              break;
            case S_REAL32:
              Real32Rem (&error, reslo, leftoplo, rightoplo);
              break;
            case S_REAL64:
              Real64Rem (&error, reshi, reslo, leftophi, leftoplo,
                         rightophi, rightoplo);
              break;
          }
        /*}}}*/
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  S_BITAND*/
      case S_BITAND:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16BitAnd (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32BitAnd (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64BitAnd (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_BITOR*/
      case S_BITOR:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16BitOr (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32BitOr (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64BitOr (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_XOR*/
      case S_XOR:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16Xor (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32Xor (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64Xor (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_LSHIFT*/
      case S_LSHIFT:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16LShift (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32LShift (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64LShift (reshi, reslo, leftophi, leftoplo, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_RSHIFT*/
      case S_RSHIFT:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16RShift (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32RShift (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64RShift (reshi, reslo, leftophi, leftoplo, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_PLUS*/
      case S_PLUS:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16Plus (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32Plus (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64Plus (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_MINUS*/
      case S_MINUS:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16Minus (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32Minus (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64Minus (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_TIMES*/
      case S_TIMES:
        switch (type)
          {
            /*{{{  S_INT16*/
            case S_INT16:
              Int16Times (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT32*/
            case S_INT32:
              Int32Times (reslo, leftoplo, rightoplo);
              break;
            /*}}}*/
            /*{{{  S_INT64*/
            case S_INT64:
              Int64Times (reshi, reslo, leftophi, leftoplo, rightophi, rightoplo);
              break;
            /*}}}*/
          }
        break;
      /*}}}*/
      /*{{{  S_EQ S_NE*/
      case S_EQ:
      case S_NE:
        {
          int res;
          switch (type)
            {
              /*{{{  S_BOOL*/
              case S_BOOL:
                res = (leftoplo == rightoplo) ? TRUE : FALSE;
                break;
              /*}}}*/
              /*{{{  S_BYTE*/
              case S_BYTE:
                Int8Eq (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT16*/
              case S_INT16:
                Int16Eq (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT32*/
              case S_INT32:
                Int32Eq (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT64*/
              case S_INT64:
                Int64Eq (&res, leftophi, leftoplo, rightophi, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_REAL32*/
              case S_REAL32:
                Real32Eq (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_REAL64*/
              case S_REAL64:
                Real64Eq (&res, leftophi, leftoplo, rightophi, rightoplo);
                break;
              /*}}}*/
            }
            if (TagOf(tptr) == S_NE)
              /*{{{  invert result*/
              res = !res;
              /*}}}*/
            *reslo = res;
        }
        break;
      /*}}}*/
      /*{{{  S_GR S_LE S_GE S_LS*/
      case S_GR:
      case S_LE:
      case S_GE:
      case S_LS:
        {
          int res;
          if ((TagOf(tptr) == S_GE) || (TagOf(tptr) == S_LS))
            /*{{{  swap the operands*/
            {
              BIT32 temp;
              temp = leftoplo;
              leftoplo = rightoplo;
              rightoplo = temp;
              temp = leftophi;
              leftophi = rightophi;
              rightophi = temp;
            }
            /*}}}*/
          switch (type)
            /*{{{  types*/
            {
              /*{{{  COMMENT S_BOOL*/
              /**********************  Start comment out ****************************
              |*{{{  S_BOOL*|
              |*
              case S_BOOL:
                *reslo = ((leftoplo == TRUE) && (rightoplo == FALSE)) ? TRUE : FALSE;
                break;
              *|
              |*}}}*|
               **********************   End comment out  ****************************/
              /*}}}*/
              /*{{{  S_BYTE*/
              case S_BYTE:
                Int8Gt (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT16*/
              case S_INT16:
                Int16Gt (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT32*/
              case S_INT32:
                Int32Gt (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_INT64*/
              case S_INT64:
                Int64Gt (&res, leftophi, leftoplo, rightophi, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_REAL32*/
              case S_REAL32:
                Real32Gt (&res, leftoplo, rightoplo);
                break;
              /*}}}*/
              /*{{{  S_REAL64*/
              case S_REAL64:
                Real64Gt (&res, leftophi, leftoplo, rightophi, rightoplo);
                break;
              /*}}}*/
            }
            /*}}}*/
          if ((TagOf(tptr) == S_LE) || (TagOf(tptr) == S_GE))
            /*{{{  invert result*/
            res = !res;
            /*}}}*/
          *reslo = res;
        }
        break;
      /*}}}*/
      /*{{{  S_AFTER*/
      case S_AFTER:
        {
          int res;
          switch (type)
            {
              /*{{{  S_INT16*/
              case S_INT16:
                {
                  BIT32 r;
                  Int16Minus (&r, leftoplo, rightoplo);
                  Int16Gt (&res, r, ZERO32);
                }
                break;
              /*}}}*/
              /*{{{  S_INT32*/
              case S_INT32:
                {
                  BIT32 r;
                  Int32Minus (&r, leftoplo, rightoplo);
                  Int32Gt (&res, r, ZERO32);
                }
                break;
              /*}}}*/
              /*{{{  S_INT64*/
              case S_INT64:
                {
                  BIT32 rhi, rlo;
                  Int64Minus (&rhi, &rlo, leftophi, leftoplo, rightophi, rightoplo);
                  Int64Gt (&res, rhi, rlo, ZERO32, ZERO32);
                }
                break;
              /*}}}*/
            }
          *reslo = res;
        }
        break;
      /*}}}*/
      /*}}}*/
    }
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE treenode *foldconstarrayexp*/
/*****************************************************************************
 *
 *  foldconstarrayexp folds expression 'tptr' into a constant array node
 *
 *****************************************************************************/
PRIVATE wordnode *foldconstarrayexp ( treenode *tptr)
{
  switch (TagOf(tptr))
    {
      /*{{{  N_VALABBR N_VALRETYPE*/
      case N_VALABBR: case N_VALRETYPE:
        {
          treenode *v = DValOf(NDeclOf(tptr));
          assert(TagOf(v) == S_CONSTCONSTRUCTOR || TagOf(v) == S_STRING);
          return CTValOf(v);
        }
      /*}}}*/
      /*{{{  S_ARRAYSUB S_SEGMENT*/
      case S_ARRAYSUB: case S_SEGMENT:
        {
          int arraysub = (TagOf(tptr) == S_ARRAYSUB);
          treenode *base = arraysub ? ASBaseOf(tptr) : SNameOf(tptr);
          treenode *basetype = gettype(base);
          BIT32 bytesinbasetype = bytesin(ARTypeOf(basetype));
          BYTE *baseimage = (BYTE *)WNameOf(foldconstarrayexp(base));
          treenode *startexp = arraysub ? ASIndexOf(tptr) : SStartExpOf(tptr);
          BIT32 start, count, temp;
          foldconstexp(startexp, &temp, &start, CHK_EXP_NOT_CONST);
          if (arraysub)
            count = 1;
          else
            {
              foldconstexp(SLengthExpOf(tptr), &temp, &count, CHK_EXP_NOT_CONST);
              /*{{{  check start and length are legal*/
              {
                INT32 i = start,
                      j = count,
                      d = ARDimOf(basetype);
                if (d == (-1)) /* if dimension is unknown */
                  d = MOSTPOS_INT32;
                if (i < 0 || i > d)
                  chkreport_i(CHK_SEG_START_RANGE, chklocn, i);
                if (j < 0)
                  chkreport_i(CHK_SEG_LENGTH_RANGE, chklocn, j);
                if (j > (d - i))
                  chkreport_i(CHK_SEG_RANGE, chklocn, i + j - 1);
              }
              /*}}}*/
            }
          start *= bytesinbasetype;
          count *= bytesinbasetype;
        #if 0
          {
            BYTE *cptr = (BYTE *)memalloc((size_t)(count + 1)); /* Don't use newvec as it may not give enough space */
            memcpy(cptr, baseimage + start, (size_t)count);
            return newwordnode(S_NAME, (char *)cptr, (int)count, NULL);
          }
        #else
          return lookupword((char *)baseimage + start, count); /* 8/1/91 reclaim mem */
        #endif
        }
      /*}}}*/
      /*{{{  S_CONSTRUCTOR*/
      case S_CONSTRUCTOR:
      #ifdef ARRAYCONSTRUCTOR
      case S_ARRAYCONSTRUCTOR :
      #endif
        {
          treenode *type = gettype(tptr);
          const INT32 len = bytesin(type);
          const int e = bytesinscalar(basetype(type));
          BYTE *cptr = (BYTE *)memalloc((size_t)(len + 1));
          (void) foldconstructor(tptr, cptr, e);
        #if 0
          return newwordnode(S_NAME, (char *)cptr, (int) len, NULL);
        #else
          {
            wordnode *result = lookupword((char *)cptr, len);
            memfree(cptr); /* added 8/1/91 for regaining memory */
            return result;
          }
        #endif
        }
      /*}}}*/
      /*{{{  S_CONSTCONSTRUCTOR S_STRING*/
      case S_CONSTCONSTRUCTOR: case S_STRING:
        return CTValOf(tptr);
      /*}}}*/
      default:
        chkreport(CHK_EXP_NOT_CONST, chklocn);
    }
  return (NULL); /* Not reached */
}
/*}}}*/
/*{{{  PUBLIC void foldconstexp (tptr, reshi, reslo, e)*/
PUBLIC void foldconstexp ( treenode *tptr , BIT32 *reshi , BIT32 *reslo , int e)
{
  int error = FALSE;
  switch (TagOf(tptr))
    {
      /*{{{  tags*/
      /*{{{  dyadic operator*/
      case S_AND: case S_OR:
      case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
      case S_BITAND: case S_BITOR: case S_XOR:
      case S_LSHIFT: case S_RSHIFT:
      case S_PLUS: case S_MINUS: case S_TIMES:
      case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
      case S_CSUB0:
        folddyadic (tptr, reshi, reslo, e);
        break;
      /*}}}*/
      /*{{{  monadic operator*/
      case S_NEG:
      case S_UMINUS:
      case S_BITNOT:
      case S_NOT:
        foldmonadic (tptr, reshi, reslo, e);
        break;
      /*}}}*/
      /*{{{  case S_SIZE S_ELSIZE*/
      case S_SIZE:
      case S_ELSIZE:
        {
          treenode *t = gettype (OpOf(tptr));
          if (TagOf(t) == S_ARRAY)
            {
              *reslo = ARDimOf(t);
              if (*reslo == (-1))
                /* Array size is unknown, so not constant */
                chkreport (e, chklocn);
            }
          else if (TagOf(t) == S_UNDECLARED)
            chkreport (e, chklocn);
          else
            /* Should NEVER be here because it is already type-checked */
            badtag(chklocn, (BIT32)TagOf(t), "fold-SIZE");
          break;
        }
      /*}}}*/
      /*{{{  case S_TRUE*/
      case S_TRUE:
        *reslo = 1;
        break;
      /*}}}*/
      /*{{{  case S_FALSE*/
      case S_FALSE:
        *reslo = 0;
        break;
      /*}}}*/
      /*{{{  case S_INTLIT*/
      case S_INTLIT:
        {
          int littype;
          /*{{{  littype = target int literal size*/
          switch (targetintsize)
            {
              case S_INT16: littype = S_INT16LIT; break;
              case S_INT32: littype = S_INT32LIT; break;
              /*case S_INT64: littype = S_INT64LIT; break;*/ /* not supported */
            }
          /*}}}*/
          IStrToInt (littype, &error, reshi, reslo, WNameOf(StringPtrOf(tptr)));
          if (error)
            chkreport (CHK_CFOLD_OFLOW, chklocn);
          break;
        }
      /*}}}*/
      /*{{{  case S_BYTELIT case S_INT16LIT case S_INT32LIT case S_INT64LIT*/
      case S_BYTELIT:
      case S_INT16LIT:
      case S_INT32LIT:
      case S_INT64LIT:
        IStrToInt (TagOf(tptr), &error, reshi, reslo, WNameOf(StringPtrOf(tptr)));
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  case S_REAL32LIT*/
      case S_REAL32LIT:
        StrToR32 (&error, reslo, WNameOf(StringPtrOf(tptr)));
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  case S_REAL64LIT*/
      case S_REAL64LIT:
        StrToR64 (&error, reshi, reslo, WNameOf(StringPtrOf(tptr)));
        if (error)
          chkreport (CHK_CFOLD_OFLOW, chklocn);
        break;
      /*}}}*/
      /*{{{  ASMNAME*/
      case S_ASMNAME:
        {
          int i = which_asmname(tptr);
          switch (asmvalids[i])
            {
              case ASMNAME_VALID: /* Now value is known */
                *reslo = asmvalues[i];
                break;
              default:
                chkreport (e, chklocn);
            }
        }
        break;
      /*}}}*/
      /*{{{  case N_VALABBR N_VALRETYPE*/
      case N_VALABBR:
      case N_VALRETYPE:
        foldconstexp (DValOf(NDeclOf(tptr)), reshi, reslo, e);
        break;
      /*}}}*/
      /*{{{  case N_TAGDEF*/
      case N_TAGDEF:
        *reshi = ZERO32;
        *reslo = NTValueOf(tptr);
        break;
      /*}}}*/
      /*{{{  case S_MOSTPOS*/
      case S_MOSTPOS:
        {
          int tsize = MOpTypeOf(tptr);
          if (tsize == S_INT)
            tsize = targetintsize;
          switch (tsize)
            {
              /*{{{  short*/
              case S_INT16:
                *reslo = MOSTPOS_INT16;
                break;
              case S_INT32:
                *reslo = MOSTPOS_INT32;
                break;
              /*}}}*/
              /*{{{  long*/
              case S_INT64:
                *reshi = MOSTPOS_INT32;
                *reslo = 0xffffffffl;
                break;
              /*}}}*/
              /* Type checked, so we cannot fall off, theoretically */
              default:
                badtag(chklocn, (BIT32)tsize, "fold-mostpos");
            }
          break;
        }
      /*}}}*/
      /*{{{  case S_MOSTNEG*/
      case S_MOSTNEG:
        {
          int tsize = MOpTypeOf(tptr);
          if (tsize == S_INT)
            tsize = targetintsize;
          switch (tsize)
            {
              /*{{{  short*/
              case S_INT16:
                *reslo = MOSTNEG_INT16;
                break;
              case S_INT32:
                *reslo = MOSTNEG_INT32;
                break;
              /*}}}*/
              /*{{{  long*/
              case S_INT64:
                *reshi = MOSTNEG_INT32;
                *reslo = 0;
                break;
              /*}}}*/
              /* Type checked, so we cannot fall off, theoretically */
              default:
                badtag(chklocn, (BIT32)tsize, "fold-mostneg");
            }
          break;
        }
      /*}}}*/
      /*{{{  conversions*/
      /*{{{  case S_EXACT*/
      case S_EXACT:
        {
          int sourcetype = typeof(OpOf(tptr)),
              desttype = MOpTypeOf(tptr);
          BIT32 rhi, rlo;
          foldconstexp (OpOf(tptr), &rhi, &rlo, e);
          exactconversion (sourcetype, desttype, reshi, reslo, rhi, rlo);
        }
        break;
      /*}}}*/
      /*{{{  case S_ROUND S_TRUNC*/
      case S_ROUND:
      case S_TRUNC:
        {
          int sourcetype = typeof(OpOf(tptr)),
              desttype = MOpTypeOf(tptr);
          BIT32 rhi, rlo;
          int error = FALSE;
          int roundmode = (TagOf(tptr) == S_ROUND) ? Nearest : Truncate;
          foldconstexp (OpOf(tptr), &rhi, &rlo, e);
      
          /*{{{  convert from sourcetype to desttype*/
          if (sourcetype == S_INT)
            sourcetype = targetintsize;
          if (desttype == S_INT)
            desttype = targetintsize;
          switch (desttype)
            {
              /*{{{  case S_INT16*/
              case S_INT16:
                if (sourcetype == S_REAL32)
                  {
                    BIT32 n;
                    R32ToI32 (&error, &n, roundmode, rlo);
                    if (error)
                      chkreport (CHK_CFOLD_OFLOW, chklocn);
                    exactconversion (S_INT32, S_INT16, reshi, reslo, 0, n);
                  }
                else if (sourcetype == S_REAL64)
                  {
                    BIT32 n;
                    R64ToI32 (&error, &n, roundmode, rhi, rlo);
                    if (error)
                      chkreport (CHK_CFOLD_OFLOW, chklocn);
                    exactconversion (S_INT32, S_INT16, reshi, reslo, 0, n);
                  }
                else
                  badtag(chklocn, (BIT32)sourcetype, "fold-convert");
                break;
              /*}}}*/
              /*{{{  case S_INT32*/
              case S_INT32:
                if (sourcetype == S_REAL32)
                  R32ToI32 (&error, reslo, roundmode, rlo);
                else if (sourcetype == S_REAL64)
                  R64ToI32 (&error, reslo, roundmode, rhi, rlo);
                else
                  badtag(chklocn, (BIT32)sourcetype, "fold-convert");
                break;
              /*}}}*/
              /*{{{  case S_INT64*/
              case S_INT64:
                if (sourcetype == S_REAL32)
                  R32ToI64 (&error, reshi, reslo, roundmode, rlo);
                else if (sourcetype == S_REAL64)
                  R64ToI64 (&error, reshi, reslo, roundmode, rhi, rlo);
                else
                  badtag(chklocn, (BIT32)sourcetype, "fold-convert");
                break;
              /*}}}*/
              /*{{{  case S_REAL32*/
              case S_REAL32:
                switch (sourcetype)
                  {
                    case S_INT16:
                    case S_INT32:  I32ToR32 (reslo, roundmode, rlo);
                                   break;
                    case S_INT64:  I64ToR32 (reslo, roundmode, rhi, rlo);
                                   break;
                    case S_REAL32: *reslo = rlo;
                                   break;
                    case S_REAL64: R64ToR32(&error, reslo, roundmode, rhi, rlo);
                                   break;
                    default:
                      badtag(chklocn, (BIT32)sourcetype, "fold-convert");
                  }
                break;
              /*}}}*/
              /*{{{  case S_REAL64*/
              case S_REAL64:
                switch (sourcetype)
                  {
                    case S_INT16:
                    case S_INT32:  I32ToR64 (reshi, reslo, rlo);
                                   break;
                    case S_INT64:  I64ToR64 (reshi, reslo, roundmode, rhi, rlo);
                                   break;
                    case S_REAL32: R32ToR64(&error, reshi, reslo, rlo);
                                   break;
                    case S_REAL64: *reshi = rhi; *reslo = rlo;
                                   break;
                    default:
                      badtag(chklocn, (BIT32)sourcetype, "fold-convert");
                  }
                break;
              /*}}}*/
              default:
                  badtag(chklocn, (BIT32)desttype, "fold-convert");
            }
          /*}}}*/
      
          if (error)
            chkreport (CHK_CFOLD_OFLOW, chklocn);
          break;
        }
      /*}}}*/
      /*}}}*/
      /*{{{  case S_ARRAYSUB*/
      case S_ARRAYSUB :
        {
          treenode *cptr = point_at_construct(tptr, FALSE);
          if (TagOf(cptr) != S_CONSTPTR)
            foldconstexp (cptr, reshi, reslo, e);
          else
            /*{{{  pick value out of BYTE array*/
            {
              treenode *type = nametype(tptr);
              BYTE *p = ((bytenode_t *)cptr)->val;
              int b = bytesinscalar(basetype(type));
              int e = (int)min(4, b) * 8;
              int shift;
              *reslo = ZERO32;
              *reshi = ZERO32;
              for (shift = 0; shift < e ; shift += 8)
                {
                  *reslo |= ((BIT32)(*p)) << shift;
                  p++;
                }
              if (b > 4)
                {
                  e = (b - 4) * 8;
                  for (shift = 0; shift < e; shift += 8)
                    {
                      *reshi |= (BIT32)(*p) << shift;
                      p++;
                    }
                }
              else if (b == 2) /* must be either INT on T2, or INT16 */
                { /* sign extend reslo just in case */ /* bug 1011 24/10/90 */
                  if (*reslo & 0x8000)
                    *reslo |= 0xFFFF0000;
                }
            }
            /*}}}*/
          break;
        }
      /*}}}*/
      /*{{{  case S_CONSTEXP  - it is already folded*/
      case S_CONSTEXP:
        *reshi = HiValOf(tptr);
        *reslo = LoValOf(tptr);
        break;
      /*}}}*/
      /*{{{  case N_DECL - if it is an undeclared name*/
      case N_DECL:
        if (TagOf(NTypeOf(tptr)) == S_UNDECLARED)
          {
            *reshi = ZERO32; *reslo = ZERO32;
          }
        else
          chkreport(e, chklocn);
      break;
      /*}}}*/
      /*{{{  case N_REPL - used by usage checker*/
      case N_REPL:
        DEBUG_MSG(("foldconstexp: N_REPL: known? %d", NReplKnownOf(tptr)));
        if (NReplKnownOf(tptr))
          *reslo = NReplValueOf(tptr);
        else
          chkreport(e, chklocn);
        break;
      /*}}}*/
      /*{{{  case S_SEGSTART*/
      case S_SEGSTART:
        foldconstexp(SStartExpOf(OpOf(tptr)), reshi, reslo, e);
        break;
      /*}}}*/
      #ifdef CONDEXP
      /*{{{  case S_CONDEXP*/
      case S_CONDEXP :
        {
          BIT32 ophi, oplo;
          foldconstexp (CondExpGuardOf(tptr), &ophi, &oplo, e);
          if (oplo == TRUE)
            foldconstexp (CondExpTrueOf(tptr), reshi, reslo, e);
          else
            foldconstexp (CondExpFalseOf(tptr), reshi, reslo, e);
        }
        break;
      /*}}}*/
      #endif
      /*{{{  case S_EVAL*/
      case S_EVAL:
        /* Won't be constant, if we used 'isconst', but will be if we used
           'is_evaluable' in the useage checker. */
        foldconstexp(RightOpOf(tptr), reshi, reslo, e);
        break;
      /*}}}*/
      /*}}}*/
      default:
        chkreport (e, chklocn);
    }
  ;
}
/*}}}*/
/*{{{  PUBLIC treenode *newconstexp (tptr)*/
PUBLIC treenode *newconstexp ( treenode *tptr )
{
  treenode *r = tptr;
  if (typeof(tptr) == S_ARRAY)
    {
      if (TagOf(tptr) != S_STRING && TagOf(tptr) != S_CONSTCONSTRUCTOR)
        r = newconsttablenode(S_CONSTCONSTRUCTOR, LocnOf(tptr),
                              foldconstarrayexp(tptr),
                              tptr);
    }
  else
    {
      BIT32 rhi, rlo;
      if (TagOf(tptr) != S_CONSTEXP)
        {
          foldconstexp(tptr, &rhi, &rlo, CHK_EXP_NOT_CONST);
          r = newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo);
        }
    }
  return r;
}
/*}}}*/
/*{{{  PUBLIC int wouldbeconst (tptr)  without folding*/
PUBLIC int wouldbeconst ( treenode *tptr )
/* This returns TRUE if the tree will be constant once special names
   such as .WSSIZE have been translated */
{
  int spare_valids[ASMNAMES_COUNT];
  int constant;
  int i;
    
  /* To allow things like .WSSIZE, we mark all the special ASM names
     as valid, so that isconst can return TRUE */
  memcpy(spare_valids, asmvalids, sizeof(int)*ASMNAMES_COUNT);

  for (i = 0; i < ASMNAMES_COUNT; i++)
    asmvalids[i] = ASMNAME_VALID;

  constant = isconst(tptr);
  memcpy(asmvalids, spare_valids, sizeof(int)*ASMNAMES_COUNT);

  return constant;
}
/*}}}*/
/*{{{  PUBLIC int isconst (tptr)  without folding*/
/* Test whether the expression tree tptr is constant, return TRUE if it is,
FALSE otherwise.
*/
PUBLIC int isconst ( treenode *tptr )
{
  if (tptr != NULL) /* added for bug 1120 25/1/91 */
  switch (TagOf(tptr))
    {
      /*{{{  MOSTPOS MOSTNEG literal constant expression or constructor*/
      case S_MOSTPOS: case S_MOSTNEG: case S_INTLIT:
      case S_INT16LIT: case S_INT32LIT: case S_INT64LIT:
      case S_REAL32LIT: case S_REAL64LIT:
      case S_BYTELIT: case S_TRUE: case S_FALSE:
      case S_STRING: case S_CONSTEXP: case S_CONSTCONSTRUCTOR: case S_DUMMYEXP:
        return TRUE;
      /*}}}*/
      /*{{{  ASMNAME*/
      case S_ASMNAME:
        return (asmvalids[which_asmname(tptr)] == ASMNAME_VALID);
      /*}}}*/
      /*{{{  monadic operator*/
      case S_NEG: case S_UMINUS: case S_BITNOT: case S_NOT:
      case S_EXACT: case S_ROUND: case S_TRUNC:
        return isconst(OpOf(tptr));
      case S_SIZE: case S_ELSIZE: /* introduced by the backend */
        {
          treenode *t = gettype (OpOf(tptr));
          return ((TagOf(t) != S_UNDECLARED) && (ARDimOf(t) != (-1)));
        }
      case S_SEGSTART: /* introduced by the backend */
        return isconst(SStartExpOf(OpOf(tptr)));
      case S_ADDRESSOF:
        return FALSE;
      /*}}}*/
      /*{{{  dyadic operator*/
      case S_AND: case S_OR:
      case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
      case S_BITAND: case S_BITOR: case S_XOR:
      case S_LSHIFT: case S_RSHIFT:
      case S_PLUS: case S_MINUS: case S_TIMES:
      case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE:
      case S_AFTER:
      case S_CSUB0: case S_CCNT1:  /* introduced by the back end */
      case S_EVAL:
        return (isconst(LeftOpOf(tptr)) && isconst(RightOpOf(tptr)));
      /*}}}*/
      #ifdef CONDEXP
      /*{{{  conditional expression*/
      case S_CONDEXP:
        return (isconst(CondExpGuardOf(tptr)) && isconst(CondExpTrueOf(tptr)) &&
                isconst(CondExpFalseOf(tptr)));
      /*}}}*/
      #endif
      /*{{{  constructor*/
      case S_CONSTRUCTOR:
        tptr = OpOf(tptr);
        while (!EndOfList(tptr))
          {
            if (!isconst(ThisItem(tptr)))
              return FALSE;
            tptr = NextItem(tptr);
          }
        return TRUE; /* fallen off the end, so it must be constant */
      #if 0      
      case S_STRUCTCONSTRUCTOR:
        return FALSE;
      #endif
      /*}}}*/
      #ifdef ARRAYCONSTRUCTOR
      /*{{{  array constructor*/
      case S_ARRAYCONSTRUCTOR :
        if (isconst(ACStartExpOf(tptr)) && isconst(ACLengthExpOf(tptr)))
          {
          #if 0
            /* Nasty little hack to get isconst to return true for
             * replicator in the value expression
             */
            int result;
            SetTag(ACNameOf(tptr), S_CONSTEXP);
            result = isconst(ACValExpOf(tptr));
            SetTag(ACNameOf(tptr), N_REPL);
            return result;
          #else
            int result;
            SetNReplKnownOf(ACNameOf(tptr), TRUE);
            result = isconst(ACValExpOf(tptr));
            SetNReplKnownOf(ACNameOf(tptr), FALSE);
            return result;
          #endif
          }
        else
          return FALSE;
      /*}}}*/
      #endif
      /*{{{  array subscript*/
      case S_ARRAYSUB:
      case S_ARRAYITEM:
      case S_RECORDSUB:
        return (isconst(ASIndexOf(tptr)) && isconst(ASBaseOf(tptr)));
      /*}}}*/
      /*{{{  array dimension*/
      case S_ARRAY:
        return (ARDimOf(tptr) != (-1));
      /*}}}*/
      /*{{{  segment*/
      case S_SEGMENT:
        return (isconst(SNameOf(tptr)) && isconst(SStartExpOf(tptr)) &&
                                         isconst(SLengthExpOf(tptr)));
      case S_SEGMENTITEM:
        return FALSE;
      /*}}}*/
      /*{{{  val abbrev  val retype*/
      case N_VALABBR: case N_VALRETYPE:
        return isconst(DValOf(NDeclOf(tptr)));
      /*}}}*/
      /*{{{  decl abbrev retype param repl specification valof function ...*/
      case N_ABBR: case N_RETYPE: case N_DECL: case N_VALPARAM:
      case N_PARAM:
      case S_DECL: case S_VALABBR: case S_ABBR: case S_RETYPE: case S_VALRETYPE:
      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
      case S_TPROTDEF: case S_SPROTDEF: case S_COLON2:
      case S_VALOF: case S_FINSTANCE: case S_FNFORMALRESULT:
      case T_TEMP: case T_PREEVALTEMP:
        return FALSE;
      case N_TAGDEF: case N_FIELD:
        return TRUE;
      case N_REPL:
        DEBUG_MSG(("isconst: N_REPL: known? %d", NReplKnownOf(tptr)));
        return NReplKnownOf(tptr);
      /*}}}*/
      /*{{{  overlapcheck*/
      case S_OVERLAPCHECK:
        return overlaps_are_const;
      /*}}}*/
      default:
        badtag(chklocn, (BIT32)TagOf(tptr), "isconst");
    }
  return (FALSE);
}
/*}}}*/
/*{{{  PUBLIC treenode *foldexp (tptr)*/
/*
Fold constant expressions, return a pointer to the folded tree.
Does not refold previously folded trees.
*/
PUBLIC treenode *foldexp ( treenode *tptr )
{
  if (tptr == NULL)
    return NULL;   /* bug 838 20/12/90 */

  switch (TagOf(tptr))
    {
      /*{{{  cases*/
      /*{{{  val names*/
      case N_VALABBR: case N_VALRETYPE:
        if (DValOf(NDeclOf(tptr)) == NULL)
          break;
        /* The original declaration will already be folded. */
        /* If the name is not scalar then we do not insert another copy in the tree
           here */
        /* This is changed so that it DOES fold arrays: 28/2/90 by CO'N */
        /* Arghh! This breaks the way debug info is generated, and causes */
        /* Duplicate copies of the arrays to exist. Take it out again */
        /* if (isconst(tptr)) */
        if (isconst(tptr) && TagOf(NTypeOf(tptr)) != S_ARRAY)
          tptr = newconstexp(tptr);
        break;
      /*}}}*/
      /*{{{  literals MOSTPOS MOSTNEG*/
      case S_INTLIT: case S_INT16LIT: case S_INT32LIT: case S_INT64LIT:
      case S_REAL32LIT: case S_REAL64LIT: case S_BYTELIT: case S_TRUE: case S_FALSE:
      case S_MOSTPOS: case S_MOSTNEG:
        tptr = newconstexp(tptr);
        break;
      /*}}}*/
      /*{{{  ASMNAME*/
      case S_ASMNAME:
        if (isconst(tptr)) /* true if has been set to 'valid' */
          tptr = newconstexp(tptr);
        break;   /* Don't yet convert it! */
      /*}}}*/
      /*{{{  monadic operator*/
      case S_ADDRESSOF:
        SetOp(tptr, foldexp(OpOf(tptr)));
        break;
      case S_EXACT: case S_ROUND: case S_TRUNC:
      case S_NEG: case S_UMINUS: case S_BITNOT: case S_NOT:
        if (isconst(OpOf(tptr)))
          tptr = newconstexp(tptr);
        else
          SetOp(tptr, foldexp(OpOf(tptr)));
        break;
      case S_SIZE:
        {
          treenode *t = gettype(OpOf(tptr));
          if (TagOf(t) != S_UNDECLARED)
            {
              if (ARDimOf(t) != (-1))
                tptr = newconstexpnode(S_CONSTEXP, chklocn, tptr, ZERO32, ARDimOf(t));
              else
                SetOp(tptr, foldexp(OpOf(tptr)));
            }
        }
        break;
      /*}}}*/
      /*{{{  dyadic operator*/
      /*{{{  S_AND S_OR*/
      case S_AND: case S_OR:
        if (isconst(LeftOpOf(tptr)) && isconst(RightOpOf(tptr)))
          tptr = newconstexp(tptr);
        else if (isconst(LeftOpOf(tptr)))
          /*{{{  fold left, fold out   ( FALSE | TRUE )  ( AND | OR )*/
          {
            BIT32 rhi, rlo;
            foldconstexp(LeftOpOf(tptr), &rhi, &rlo, CHK_EXP_NOT_CONST);
            if (TagOf(tptr) == S_AND)
              {
                if (rlo == ZERO32)     /* FALSE AND ... */
                  return(newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo));
                else                   /* TRUE  AND ... */
                  return foldexp(RightOpOf(tptr));
              }
            else /* TagOf(tptr) == S_OR */
              {
                if (rlo == ONE32)      /* TRUE OR ... */
                  return(newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo));
                else                   /* FALSE OR ... */
                  return foldexp(RightOpOf(tptr));
              }
            /*{{{  COMMENT */
            /**********************  Start comment out ****************************
            @*{{{  *@
            if (((TagOf(tptr) == S_AND) && (rlo == ZERO32)) ||
                ((TagOf(tptr) == S_OR) && (rlo == ONE32)))
              return(newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo));
            else
              {
                SetLeftOp(tptr,
                         newconstexpnode(S_CONSTEXP, chklocn, LeftOpOf(tptr), rhi, rlo));
                SetRightOp(tptr, foldexp(RightOpOf(tptr)));
                return(tptr);
              }
            @*}}}*@
             **********************   End comment out  ****************************/
            /*}}}*/
          }
          /*}}}*/
        else if (isconst(RightOpOf(tptr)))
          /*{{{  fold right, fold out '... AND FALSE' and '... OR TRUE'*/
          {
            BIT32 rhi, rlo;
            foldconstexp(RightOpOf(tptr), &rhi, &rlo, CHK_EXP_NOT_CONST);
            if (TagOf(tptr) == S_AND)
              {
                if (rlo == ZERO32)  /* ... AND FALSE */
                  return newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo);
                else                /* ... AND TRUE  */
                  return foldexp(LeftOpOf(tptr));
              }
            else /* TagOf(tptr) == S_OR */
              {
                if (rlo == ONE32)  /* ... OR TRUE */
                  return(newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo));
                else               /* ... OR FALSE */
                  return foldexp(LeftOpOf(tptr));
              }
            /*{{{  COMMENT */
            /**********************  Start comment out ****************************
            @*{{{  *@
            if (((TagOf(tptr) == S_AND) && (rlo == ZERO32)) ||
                ((TagOf(tptr) == S_OR) && (rlo == ONE32)))
              return(newconstexpnode (S_CONSTEXP, chklocn, tptr, rhi, rlo));
            else
              {
                SetRightOp(tptr,
                       newconstexpnode (S_CONSTEXP, chklocn, RightOpOf(tptr), rhi, rlo));
                SetLeftOp(tptr, foldexp(LeftOpOf(tptr)));
                return(tptr);
              }
            @*}}}*@
             **********************   End comment out  ****************************/
            /*}}}*/
          }
          /*}}}*/
        else
          {
            SetLeftOp(tptr, foldexp(LeftOpOf(tptr)));
            SetRightOp(tptr, foldexp(RightOpOf(tptr)));
          }
        break;
      /*}}}*/
      case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
      case S_BITAND: case S_BITOR: case S_XOR:
      case S_LSHIFT: case S_RSHIFT:
      case S_PLUS: case S_MINUS: case S_TIMES:
      case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE:
      case S_AFTER:
      case S_CSUB0:
      case S_EVAL:
        {
          int rightisconst = isconst(RightOpOf(tptr));
          if (isconst(LeftOpOf(tptr)) && rightisconst)
            tptr = newconstexp(tptr);
          else
            {
              SetLeftOp(tptr, foldexp(LeftOpOf(tptr)));
              SetRightOp(tptr, foldexp(RightOpOf(tptr)));
              if (rightisconst &&
                  ((TagOf(tptr) == S_LSHIFT) || (TagOf(tptr) == S_RSHIFT)) &&
                  /* This is an unsigned test, so it catches negative values */
                  /* We know that (HiValOf(RightOpOf(tptr)) == ZERO32) cos type is INT */
                  (LoValOf(RightOpOf(tptr)) > (8 * bytesinscalar(DOpTypeOf(tptr))) ))
                    chkreport (CHK_INV_SHIFT, chklocn);
            }
        }
        break;
      /*}}}*/
      #ifdef CONDEXP
      /*{{{  conditional expression*/
      case S_CONDEXP:
        if (isconst(CondExpGuardOf(tptr)) && isconst(CondExpTrueOf(tptr)) &&
            isconst(CondExpFalseOf(tptr)))
          tptr = newconstexp(tptr);
        else
          {
            SetCondExpGuard(tptr, foldexp(CondExpGuardOf(tptr)));
            SetCondExpTrue(tptr, foldexp(CondExpTrueOf(tptr)));
            SetCondExpFalse(tptr, foldexp(CondExpFalseOf(tptr)));
          }
        break;
      /*}}}*/
      #endif
      /*{{{  constructor*/
      case S_CONSTRUCTOR:
        if (isconst(tptr))
          tptr = newconstexp(tptr);
        else
          (void) foldexplist(OpOf(tptr));
        break;
      /*}}}*/
      #if 0
      /*{{{  structconstructor*/
      case S_STRUCTCONSTRUCTOR:
        {
          break;
          /*
          treenode *t = OpOf(tptr);
          while (!EndOfList(t))
            {
              NewItem(foldexp(ThisItem(t)), t);
              t = NextItem(t);
            }
          */
        }
      /*}}}*/
      #endif
      #ifdef ARRAYCONSTRUCTOR
      /*{{{  arrayconstructor*/
      case S_ARRAYCONSTRUCTOR :
        SetACStartExp(tptr, foldexp(ACStartExpOf(tptr)));
        SetACLengthExp(tptr, foldexp(ACLengthExpOf(tptr)));
        if (TagOf(ACStartExpOf(tptr)) == S_CONSTEXP &&
            TagOf(ACLengthExpOf(tptr)) == S_CONSTEXP)
          {
            INT32 i;
            treenode *list = NULL;
            treenode *index  = ACNameOf(tptr);
            INT32 start      = evaluate(ACStartExpOf(tptr));
            INT32 end        = start + evaluate(ACLengthExpOf(tptr)) ;
            treenode *name = newnamenode(TagOf(index),
                                       LocnOf(index),
                                       NNameOf(index),
                                       NTypeOf(index),
                                       NDeclOf(index),
                                       NLexLevelOf(index),
                                       NScopeOf(index),
                                       NModeOf(index));
            SetTag(index, S_CONSTEXP);
            SetCExp(index, name);
            for (i = start;  i < end; i ++)
              {
                SetLoVal(index, i);
                list = appendnode(foldexp(copytree(ACValExpOf(tptr))), list);
              }
            SetTag(tptr, S_CONSTRUCTOR);
            SetOp(tptr, list);
          }
        break;
      /*}}}*/
      #endif
      /*{{{  function instance*/
      case S_FINSTANCE:
        {
          treenode *fname = INameOf(tptr);
          foldexplist(IParamListOf(tptr));

          if (TagOf(fname) == N_PREDEFFUNCTION)
            {
              switch (NModeOf(fname))
                {
                  case PD_ASHIFTLEFT: case PD_ASHIFTRIGHT:
                  case PD_ROTATELEFT: case PD_ROTATERIGHT:
                  case PD_BITREVNBITS:
                    {  /* check the second parameter - places to shift */
                      treenode *param = ThisItem(NextItem(IParamListOf(tptr)));
                      if (isconst(param) &&
                          (LoValOf(param) > (8 * bytesperword)))
                         chkreport(CHK_INV_SHIFT, chklocn);
                    }
                    break;
                  default: break;
                }
            }
        }
        break;
      /*}}}*/
      /*{{{  array subscript*/
      case S_ARRAYSUB:
        {
          int rightconst = isconst(ASIndexOf(tptr));
      
          /* We can fold at this level if both right-hand and left-hand
             sides are constant */
          if (rightconst && isconst(ASBaseOf(tptr)))
            tptr = newconstexp(tptr);
          else
            {
              if (!issimple(ASBaseOf(tptr)))
                SetASBase(tptr, foldexp(ASBaseOf(tptr)));
              SetASIndex(tptr, foldexp(ASIndexOf(tptr)));
              if (rightconst)
                /*{{{  check subcript range*/
                {
                  treenode *t = gettype(ASBaseOf(tptr));
                  if (LoValOf(ASIndexOf(tptr)) >= ARDimOf(t))
                    chkreport_i(CHK_SUBSCRIPT_RANGE, chklocn, LoValOf(ASIndexOf(tptr)));
                }
                /*}}}*/
            }
          break;
        }
      case S_RECORDSUB:
        if (isconst(ASBaseOf(tptr)))
          tptr = newconstexp(tptr);
        break;
      /*}}}*/
      /*{{{  colon2*/
      case S_COLON2:
        SetLeftOp(tptr, foldexp(LeftOpOf(tptr)));
        SetRightOp(tptr, foldexp(RightOpOf(tptr)));
        break;
      /*}}}*/
      /*{{{  segment*/
      case S_SEGMENT:
        if (isconst(tptr))
          tptr = newconstexp(tptr);
        else
          {
            if (!issimple(SNameOf(tptr)))
              SetSName(tptr, foldexp(SNameOf(tptr)));
            SetSStartExp(tptr, foldexp(SStartExpOf(tptr)));
            SetSLengthExp(tptr, foldexp(SLengthExpOf(tptr)));
            {
              int startconst = isconst(SStartExpOf(tptr));
              int lengthconst = isconst(SLengthExpOf(tptr));
              /*{{{  if start or length are constant, check they are in range*/
              if (startconst || lengthconst)
                {
                  treenode *t = gettype(SNameOf(tptr));
                  /*{{{  check start and length are legal*/
                  {
                    INT32 i = startconst ? (INT32)LoValOf(SStartExpOf(tptr)) : ZERO32,
                          j = lengthconst ? (INT32)LoValOf(SLengthExpOf(tptr)) : ZERO32,
                          d = ARDimOf(t);
                    if (d == (-1)) /* if dimension is unknown */
                      d = MOSTPOS_INT32;
                    if (i < 0 || i > d)
                      chkreport_i(CHK_SEG_START_RANGE, chklocn, i);
                    if (j < 0)
                      chkreport_i(CHK_SEG_LENGTH_RANGE, chklocn, j);
                    if (j > (d - i))
                      chkreport_i(CHK_SEG_RANGE, chklocn, i + j - 1);
                  }
                  /*}}}*/
                }
              /*}}}*/
            }
          }
        break;
      /*}}}*/
      /*{{{  var name, string, already folded trees*/
      case N_DECL: case N_ABBR: case N_RETYPE: case N_FIELD:
      case N_VALPARAM: case N_PARAM:
      case N_LABELDEF:
      case S_CONSTEXP: case S_CONSTCONSTRUCTOR: case S_STRING:
      case T_TEMP: case T_PREEVALTEMP:
      case S_OVERLAPCHECK :
        break;
      case N_TAGDEF:
        tptr = newconstexp(tptr);
        break;
      case N_REPL:
        DEBUG_MSG(("foldexp: N_REPL: known? %d", NReplKnownOf(tptr)));
        if (NReplKnownOf(tptr))
          tptr = newconstexp(tptr);
        break;
      /*}}}*/
      /*{{{  specification*/
      case S_VALABBR: case S_ABBR:
      case S_PROCDEF:
      case S_TPROTDEF: case S_SPROTDEF:
      case S_DECL:
      case S_SFUNCDEF: case S_LFUNCDEF:
      case S_RETYPE: case S_VALRETYPE:
        {
          treenode *t = tptr, *spec;
          while (isspecification(t))
            {
              spec = t;
              t = DBodyOf(t);
            }
          SetDBody(spec, foldexp(t));
        }
        break;
      /*}}}*/
      /*{{{  VALOF*/
      case S_VALOF:
        /* Fold the result list */
        SetVLResultList(tptr, foldexplist(VLResultListOf(tptr)));
        break;
      /*}}}*/
      /*{{{  backend array item, segment item, segstart, elsize*/
      case S_ARRAYITEM:
        /* There will be nothing to fold on the ASBase side */
        if (ASExpOf(tptr) != NULL)
          SetASExp(tptr, foldexp(ASExpOf(tptr)));
        break;
      case S_SEGMENTITEM:
        /* There will be nothing to fold on the SName side */
        if (SSubscriptExpOf(tptr) != NULL)
          SetSSubscriptExp(tptr, foldexp(SSubscriptExpOf(tptr)));
        break;
      case S_ELSIZE:
        {
          treenode *t = gettype(OpOf(tptr));
          if (ARDimOf(t) != (-1))
            tptr = newconstexpnode(S_CONSTEXP, chklocn, tptr, ZERO32, ARDimOf(t));
        }
        break;
      case S_SEGSTART:
        if (isconst(SStartExpOf(OpOf(tptr))))
          tptr = newconstexp(tptr);
        break;
      /*}}}*/
      /*}}}*/
      default:
        badtag(chklocn, (BIT32)TagOf(tptr), "foldexp");
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *foldexplist (tptr)*/
PUBLIC treenode *foldexplist ( treenode *tptr )
{
  treenode *t = tptr;
  while (!EndOfList(t))
    {
      NewItem(foldexp(ThisItem(t)), t);
      t = NextItem(t);
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *foldexpinto (treenode *tptr, int type)*/
/*****************************************************************************
 *
 *  foldexpinto folds the expression 'tptr', and if it is entirely constant,
 *              makes sure that the constant value node is S_CONSTCONSTRUCTOR
 *              if 'type' is S_ARRAY, S_CONSTEXP otherwise.
 *              This routine is used for storing constant RETYPEs in the
 *              type of the lhs, not the rhs.
 *
 *****************************************************************************/
PUBLIC treenode *foldexpinto ( treenode *tptr , int type )
{
  treenode *foldedexp = foldexp(tptr);
  if (TagOf(foldedexp) == S_CONSTEXP && type == S_ARRAY)
    /*{{{  convert constexp to constconstructor*/
    {
      INT32 count = bytesin(gettype(tptr));
      BYTE *cptr = (BYTE *)newvec((size_t)(count + 1));
      foldscalar_into_array(HiValOf(foldedexp), LoValOf(foldedexp), cptr, (int)count);
      foldedexp = newconsttablenode(S_CONSTCONSTRUCTOR, LocnOf(tptr),
                    newwordnode(S_NAME, (char *)cptr, (int)count, NULL),
                    CExpOf(foldedexp));
    }
    /*}}}*/
  else if ((TagOf(foldedexp) == S_STRING ||
            TagOf(foldedexp) == S_CONSTCONSTRUCTOR) && type != S_ARRAY)
    /*{{{  convert constconstructor to constexp*/
    {
      BIT32 lo = ZERO32, hi = ZERO32;
      wordnode *c = CTValOf(foldedexp);
      int count = WLengthOf(c);
      BYTE *cptr = (BYTE *)WNameOf(c);
      foldarray_into_scalar(cptr, count, &hi, &lo);
      foldedexp = newconstexpnode(S_CONSTEXP, LocnOf(tptr), tptr, hi, lo);
    }
    /*}}}*/
  return foldedexp;
}
/*}}}*/

/*{{{  folding protocol instances*/
/*{{{  PUBLIC void foldvariant (vptr)*/
/*****************************************************************************
 *
 *  foldvariant folds a protocol variant, 'vptr'.
 *
 *****************************************************************************/
PUBLIC void foldvariant ( treenode *vptr )
{
  vptr = skipspecifications(vptr);
  if (TagOf(vptr) == S_VARIANT)
    SetVRTaggedList(vptr, foldexplist(VRTaggedListOf(vptr)));
}
/*}}}*/
/*}}}*/
/*{{{  fold tree*/
/* Apply constant folding to a tree.
   Not to any nested procedure/function decls
 */

/*{{{  PRIVATEPARAM int do_foldtree(tptr)*/
/*
 *  Do constant folding on tree tptr
 */
PRIVATEPARAM int do_foldtree ( treenode *t )
{
  switch(TagOf(t))
    {
      /*{{{  spec*/
      case S_RETYPE: case S_VALRETYPE:
        if (isconst(DValOf(t)))
          SetDVal(t, foldexpinto(newconstexp(DValOf(t)),
                                 TagOf(NTypeOf(DNameOf(t)))));
        else
          SetDVal(t, foldexpinto(DValOf(t), TagOf(NTypeOf(DNameOf(t)))));
        break;
      case S_VALABBR: case S_ABBR:
        if (isconst(DValOf(t)))
          SetDVal(t, newconstexp(DValOf(t)));
        else
          SetDVal(t, foldexp(DValOf(t)));
        break;
      /*}}}*/
      /*{{{  process*/
      /*{{{  REPLSEQ REPLPAR REPLIF REPLALT*/
      case S_REPLSEQ: case S_REPLPAR: case S_REPLIF:
      case S_REPLALT: case S_PRIREPLPAR: case S_PRIREPLALT:
      case S_PLACEDREPLPAR: case S_REPLDO:
        SetReplCStartExp(t, foldexp(ReplCStartExpOf(t)));
        SetReplCLengthExp(t, foldexp(ReplCLengthExpOf(t)));
        break;
      /*}}}*/
      /*{{{  WHILE CHOICE SELECTION*/
      case S_WHILE: case S_CHOICE:
        SetCondGuard(t, foldexp(CondGuardOf(t)));
        break;
      case S_SELECTION:
        if (TagOf(CondGuardOf(t)) != S_ELSE)
          SetCondGuard(t, foldexplist(CondGuardOf(t)));
        break;
      /*}}}*/
      /*{{{  ALTERNATIVE*/
      case S_ALTERNATIVE:
        if (AltGuardOf(t) != NULL)
          SetCondGuard(t, foldexp(AltGuardOf(t)));
        break;
      /*}}}*/
      /*{{{  PINSTANCE FINSTANCE*/
      case S_PINSTANCE: case S_FINSTANCE:
        if (IParamListOf(t) != NULL);
          SetIParamList(t, foldexplist(IParamListOf(t)));
        break ;
      /*}}}*/
      /*{{{  CASE_INPUT, DELAYED_INPUT, OUTPUT, INPUT, TAGGED_INPUT, ASS, CASE  break*/
      /* case S_CASE_INPUT: */ /* removed from here for bug 1034 1/11/90 */
      case S_DELAYED_INPUT: case S_OUTPUT: case S_INPUT:
      case S_TAGGED_INPUT: case S_ASS:
        if (TagOf(LHSOf(t)) == S_LIST)
          SetLHS(t, foldexplist(LHSOf(t)));
        else
          SetLHS(t, foldexp(LHSOf(t)));
        if (TagOf(RHSOf(t)) == S_LIST)
          SetRHS(t, foldexplist(RHSOf(t)));
        else
          SetRHS(t, foldexp(RHSOf(t)));
        break;
      case S_CASE:
      case S_CASE_INPUT: /* added for bug 1034 1/11/90 */
        SetLHS(t, foldexp(LHSOf(t)));
        break;
      /*}}}*/
      /*{{{  VARIANT             break*/
      case S_VARIANT:
        SetVRTaggedList(t, foldexplist(VRTaggedListOf(t)));
        break;
      /*}}}*/
      /*{{{  GUYCODE GUYSTEP     return*/
      case S_GUYCODE: case S_GUYSTEP:
        SetRightOp(t, foldexplist(RightOpOf(t)));
        break;
      /*}}}*/
      /*}}}*/
      /*{{{  VALOF*/
      case S_VALOF:
        SetVLResultList(t, foldexplist(VLResultListOf(t)));
        break;
      /*}}}*/
      /*{{{  allocation*/
      case S_PLACE: case S_WSPLACE: case S_VSPLACE:
        if (PlaceExpOf(t) != NULL)
          SetPlaceExp(t, foldexp(PlaceExpOf(t)));
        break;
      #ifdef CONFIG
      case S_PLACEON:
        SetDName(t, foldexplist(DNameOf(t)));
        SetDVal(t, foldexp(DValOf(t)));
        break;
      #endif
      /*}}}*/
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PUBLIC void foldtree(tptr)*/
PUBLIC void foldtree ( treenode *tptr )
{
  prewalkproctree(tptr, do_foldtree);
}
/*}}}*/
/*{{{  PUBLIC int is_evaluable (n)*/
/*****************************************************************************
 *
 *  Test whether an expression can be compile time evaluated.
 *  This is possible if it contains only constants and replicators of
 *  known base and count and no function calls or in-line VALOFs
 *
 *****************************************************************************/
PUBLIC int is_evaluable ( treenode *n )
{
  int res;
  overlaps_are_const = TRUE;
  res = isconst(n);
  overlaps_are_const = FALSE;
  return res;
}
/*}}}*/
/*{{{  PUBLIC INT32 evaluate (n)*/
/*****************************************************************************
 *
 *  Evaluate a compile-time evaluable expression and return result.
 *  The expression is assumed to be an int
 *
 *****************************************************************************/
PUBLIC INT32 evaluate ( treenode *n )
{
  BIT32 low, high;
  foldconstexp(n, &high, &low, CHK_EXP_NOT_CONST);
  return ((INT32)low);
}
/*}}}*/
/*}}}*/
/*}}}*/
