/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two checker   Semantic analyser
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  COMMENT The semantic analyser*/
/*
   The interface to the semantic analyser is via a set of functions
   defined in this module which check various parts of the parse
   tree built up by the parser.
   The semantic analyser makes few modifications to the parse tree,
   but does insert types into operator nodes, and constant values
   into constant expression nodes (subscript lengths in array declarations,
   values of selections in case processes), it also fills in types of
   abbreviations and untyped literals where deducible.
*/
/*}}}*/
/*{{{  COMMENT What the semantic analyser checks*/
/**********************  Start comment out ****************************
|*
The semantic analyser performs the following checks:

1  Assignments are balanced, and the types of the variables match the types of
   the expressions.

2  Abbreviations have only one name and the type of the expression/element
   matches the type specified. If no type is specified, the type of the rhs is
   inserted.

3  Retypes have only one name and the type of the expression/element matches
   the type specified.

4  Function result expression lists match the types of the functions.

5  Channel protocol equivalence is checked in abbreviations and retypes.

6  Array dimensions exist, and are constant expressions.

7  CASE processes: the selector is typechecked, each selection is typechecked
   and constant folded - each must have the same type as the selector,
   all selections are distinct, there is only one ELSE selection.

8  Formal parameter list specifiers are validated - var parameters to functions
   are thrown out.

9  Replicator start and count are type checked to be INT, a replicated PAR
   count is constant-folded.

10 Guards on IF, WHILE, and ALT are type checked to be BOOL.

11 Procedure actual parameters are type-checked against procedure formal
   parameters.

12 VAL TIMERs CHANs and PORTs are thrown out in abbreviations, retypes and
   parameter definitions.

13 Actual var params are checked to be elements.

14 Ports only have simple protocols.

15 Protocols are checked on inputs, outputs, case inputs, tagged inputs.
   Timer inputs and delayed inputs are checked to be INT.

16 Protocol definitions are checked to contain distinct tags, which are also
   distinct from the protocol name.

17 Actual parameters to functions are type-checked against function formal
   parameters.

18 ALT, PAR, !, ? are disallowed in VALOFs.  Write access to non-local
   variables is disallowed within VALOFs, and any specifications
   preceding them.  Calls to non-local procedures is disallowed in
   VALOFs.

19 Placed objects are decls only, the placement address is constant and of
   type INT.

20 The tags in a case input are all distinct.
*|
 **********************   End comment out  ****************************/
/*}}}*/

/*{{{  include files*/
# include <stdlib.h> 
# include <string.h>
# include <stdio.h>
# include "includes.h"
# include "extlib.h"

# include "chkerror.h" /**/
# include "chkdef.h" /**/
# include "chk1def.h" /**/
# include "chk2def.h" /**/
# include "chk4def.h" /**/
# include "instdef.h" /**/
# include "instruct.h" /**/
# include "predefhd.h"
/*}}}*/

/*{{{  definitions*/
#define MAX_TAGS 256  /* Maximum number of tags allowed in a protocol defn */

/* These structures are used when checking that
   cases and variants are  distinct. */
typedef struct
  {
    /*treenode *ca_decl;*/
    SOURCEPOSN ca_locn;
    BIT32 ca_hi, ca_lo;
  } casenode_t;

typedef struct
  {
    treenode *vnt_decl;   /* The variant's declaration */
    treenode *vnt_tag;    /* The tag's namenode */
  } vntnode_t;
/*}}}*/

/*{{{  PRIVATE variables*/
PRIVATE int elseflag;
PRIVATE casenode_t *sroot = NULL;
/*}}}*/

/*{{{  support routines*/
/*{{{  protocol checking*/
/*{{{  checking definitions*/
/*{{{  PRIVATE void cdefsimpleprotocol (pptr)*/
/*****************************************************************************
 *
 *  cdefsimpleprotocol checks the definition of a simple protocol, 'pptr'.
 *
 *****************************************************************************/
PRIVATE void cdefsimpleprotocol ( treenode *pptr )
{
  if (TagOf(pptr) == S_COLON2)
    /* syn ensures that the left-hand side is primitive - so doesn't need
       checking, and that the right-hand side is an array of unknown size -
       we need to check the type of that array only */
    {
      treenode *aptr = RightOpOf(pptr);
      /* Fill in the unknown array dimension */
      SetARDim(aptr, -1);
      /* Check the type of the array */
      ctype (ARTypeOf(aptr));
    }
  else
    ctype (pptr);
}
/*}}}*/
/*{{{  PRIVATE void cdefseqprotocol (pptr)*/
/*****************************************************************************
 *
 *  cdefseqprotocol checks the definition of a sequential protocol, 'pptr'
 *
 *****************************************************************************/
PRIVATE void cdefseqprotocol ( treenode *pptr )
{
  walklist(cdefsimpleprotocol, pptr);
}
/*}}}*/
/*{{{  PRIVATE void checktag (tagptr)*/
/*****************************************************************************
 *
 *  checktag checks that a symbol table entry for a tag, 'tagptr' is
 *           distinct from all other tags in THIS protocol.
 *
 *****************************************************************************/
PRIVATE void checktag ( treenode *tagptr )
{
  treenode *n = findname(NNameOf(tagptr));
  if (n == tagptr)
    /* Check the sequential protocol, if any */
    {
      if (NTypeOf(tagptr) != NULL)
        cdefseqprotocol(NTypeOf(tagptr));
    }
  else
    /*{{{  report the scoping problem*/
    {
      if (TagOf(n) == N_TAGDEF)
        chkreport_s(CHK_INDISTINCT_TAG, chklocn, WNameOf(NNameOf(tagptr)));
      else
        chkreport_s(CHK_TAG_CONFLICT, chklocn, WNameOf(NNameOf(tagptr)));
    }
    /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  checking instances*/
/*{{{  PRIVATE void csimpleprotocol (pptr, iptr, pitem)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  csimpleprotocol checks the usage of a simple protocol.
 *                  'pptr' points to the protocol definition,
 *                  'iptr' points to the protocol use,
 *                  'pitem' is the item number in a sequential protocol.
 *
 *****************************************************************************/
/*}}}*/

/* Protocol definition */
/* Protocol usage */
/* Item number in sequential protocol */
PRIVATE void csimpleprotocol ( treenode *pptr , treenode *iptr , INT32 pitem )
{
  int anyprotocol = (TagOf(pptr) == S_ANY);
  if (TagOf(iptr) == S_COLON2)
    /*{{{  check COLON2*/
    {
      treenode *countexp, *counttype;
      treenode *p, *t;
      if (!anyprotocol && (TagOf(pptr) != S_COLON2))
        chkreport_i(CHK_PTYPE_MISMATCH, chklocn, pitem);
      /*{{{  check left-hand side*/
      if (anyprotocol)
        {
          countexp = LeftOpOf(iptr);
          counttype = typecheck(countexp, S_UNKNOWN);
          if (!isintorbyte(TagOf(counttype)))
            chkreport_i(CHK_PTYPE_MISMATCH, chklocn, pitem);
        }
      else
        {
          p = LeftOpOf(pptr);
          countexp = LeftOpOf(iptr);
          counttype = typecheck (countexp, basetype(p));
          if (!typesequivalent(p, counttype, TRUE))
            chkreport_i(CHK_PTYPE_MISMATCH, chklocn, pitem);
        }
      /*}}}*/
      /*{{{  check right-hand side*/
      p = RightOpOf(pptr);
      if (anyprotocol)
        {
          /* Parser ensures that RHS is an array */
          t = typecheck(RightOpOf(iptr), S_UNKNOWN);
        }
      else
        {
          t = typecheck(RightOpOf(iptr), basetype(p));
          if (!typesequivalent(p, t, TRUE))
            chkreport_i(CHK_PTYPE_MISMATCH, chklocn, pitem);
        }
      /*}}}*/
      /*{{{  check a constant count*/
      if (isconst(countexp))
        {
          BIT32 counthi, countlo;
          int greater;
          foldconstexp(countexp, &counthi, &countlo, CHK_EXP_NOT_CONST);
          if (TagOf(counttype) != S_INT64)
            I32ToI64(&counthi, &countlo, countlo);
          Int64Gt(&greater, ZERO32, ZERO32, counthi, countlo);
          if (greater)
            chkreport_i(CHK_COUNT_OUT_OF_RANGE, chklocn, pitem);
          if (ARDimOf(t) != (-1))  /* We know the array size */
            {
              Int64Gt(&greater, counthi, countlo, ZERO32, ARDimOf(t));
              if (greater)
                chkreport_i(CHK_COUNT_OUT_OF_RANGE, chklocn, pitem);
            }
        }
      /*}}}*/
    }
    /*}}}*/
  else
    {
      if (anyprotocol)
        {
          treenode *type = typecheck(iptr, S_UNKNOWN);
          if (bytesin(type) == 0)
            chkreport(CHK_BAD_ANY_PROT, chklocn);
        }
      else if (!typesequivalent(pptr, typecheck (iptr, basetype(pptr)), TRUE))
        chkreport_i(CHK_PTYPE_MISMATCH, chklocn, pitem);
    }
}
/*}}}*/
/*{{{  PRIVATE void cseqprotocol (pptr, iptr)*/
/*****************************************************************************
 *
 *  cseqprotocol checks the usage of a sequential protocol.
 *               'pptr' points to the protocol definition,
 *               'iptr' points to the protocol instance.
 *
 *****************************************************************************/

/* Protocol definition */
/* Protocol usage */
PRIVATE void cseqprotocol ( treenode *pptr , treenode *iptr )
{
  int pitem = 1;
  int anyprotocol = (TagOf(pptr) == S_ANY);
  while (!EndOfList(iptr) && (anyprotocol || !EndOfList(pptr)))
    /*{{{  check this item, move to next item*/
    {
      if (anyprotocol)
        csimpleprotocol(pptr, ThisItem(iptr), pitem);
      else
        {
          csimpleprotocol (ThisItem(pptr), ThisItem(iptr), pitem);
          pptr = NextItem(pptr);
        }
      iptr = NextItem(iptr);
      pitem++;
    }
    /*}}}*/
  if (!anyprotocol && !EndOfList(pptr))
    chkreport (CHK_TOO_FEW_PITEMS, chklocn);
  else if (!EndOfList(iptr))
    chkreport (CHK_TOO_MANY_PITEMS, chklocn);
}
/*}}}*/
/*{{{  PRIVATE void protocolcheck (protocol, instance)*/
/*****************************************************************************
 *
 *  protocolcheck checks the usage of a protocol.
 *                'protocol' points to the protocol definition,
 *                'instance' points to the protocol instance.
 *
 *****************************************************************************/

/* Tree representing protocol definition */
/* The protocol usage to be checked */
PRIVATE void protocolcheck ( treenode *protocol , treenode *instance )
{
  switch (TagOf(protocol))
    {
      /*{{{  sequential protocol definition and ANY*/
      case N_SPROTDEF:
        cseqprotocol(NTypeOf(protocol), instance);
        break;
      case S_ANY:
        cseqprotocol(protocol, instance);
        break;
      /*}}}*/
      /*{{{  tagged protocol definition*/
      case N_TPROTDEF:
        {
          treenode *instancetag;
          treenode *tagptr;
          treenode *sptr;
          int found;
      
          /*{{{  find a tag in the definition which corresponds to instance tag*/
          
          /*{{{  point instancetag to the first item of the instance list*/
          if (TagOf(instance) == S_LIST)
            instancetag = ThisItem(instance);
          else
            instancetag = instance;
          
          if (TagOf(instancetag) != N_TAGDEF)
            chkreport (CHK_NOT_A_TAG, chklocn);
          /*}}}*/
          
          tagptr = NTypeOf(protocol);
          found = FALSE;
          while (!EndOfList(tagptr) && (found == FALSE))
            {
              if (ThisItem(tagptr) == instancetag)
                {
                  tagptr = ThisItem(tagptr);
                  found = TRUE;
                }
              else
                tagptr = NextItem(tagptr);
            }
          if ((found == FALSE) && (tagptr != instancetag))
            chkreport (CHK_BAD_PTAG, chklocn);
          /*}}}*/
      
          /*{{{  see if there is a sequential protocol following the tag*/
          sptr = NTypeOf(tagptr);
          if (EmptyList(sptr))
            /*{{{  there is no sequential protocol, check none is used*/
            {
              if (!EmptyList(NextItem(instance)))
                chkreport (CHK_TOO_MANY_PITEMS, chklocn);
            }
            /*}}}*/
          else
            /*{{{  there is a sequential protocol check it*/
            cseqprotocol (sptr, NextItem(instance));
            /*}}}*/
          /*}}}*/
          break;
        }
      /*}}}*/
      default:
        /*{{{  simple protocol*/
        {
          if (!EmptyList(NextItem(instance)))
            chkreport (CHK_TOO_MANY_PITEMS, chklocn);
          else
            csimpleprotocol (protocol, ThisItem(instance), ONE32);
        }
        /*}}}*/
    }
}
/*}}}*/
/*}}}*/
/*}}}*/
/*{{{  PRIVATE int validval (tptr)*/
/*****************************************************************************
 *
 *  validval takes a type tree and returns TRUE if it is a legal type for a
 *           VAL abbreviation, retype or parameter.
 *
 *****************************************************************************/
PRIVATE int validval ( treenode *tptr )
{
  /* we can't use basetype for this loop, cos it goes inside PORTs */
  while (TagOf(tptr) == S_ARRAY)
    tptr = ARTypeOf(tptr);

  switch (TagOf(tptr))
    {
      case S_CHAN: case S_PORT: case S_TIMER:
        return FALSE;
      default:
      #ifdef CONFIG
        return !network_datatype(TagOf(tptr));
      #else
        return TRUE;
      #endif
    }
}
/*}}}*/
/*{{{  PRIVATE int iselement (tptr)*/
/*****************************************************************************
 *
 *  iselement takes an expression tree 'tptr' and returns TRUE if it is
 *            an element.
 *            if "modifiable" is set FALSE, tests that the element may be
 *            written to.
 *
 *****************************************************************************/
PRIVATE int iselement ( treenode *tptr, int modifiable )
{
  treenode *nptr = nameof(tptr);
  switch (TagOf(nptr))
    {
      default:
        return (FALSE);
      case N_ABBR:
      case N_RETYPE:
      case N_DECL:
      case N_PARAM:
        /* This modifiablility test added 10/10/90 to prevent chan := chan */
        return (modifiable || validval(NTypeOf(nptr)));
    }
}
/*}}}*/
/*{{{  PRIVATE int istableelement (tptr)*/
/*****************************************************************************
 *
 *  istableelement takes an expression tree 'tptr' and returns TRUE if it is a
 *            (modifiable) element or table of elements.
 *
 *****************************************************************************/
PRIVATE int istableelement ( treenode *tptr, int modifiable )
{
  switch (TagOf(tptr))
    /*{{{  cases*/
    {
      default:
        return iselement(tptr, modifiable);
      case S_CONSTRUCTOR:
        /*{{{  look at list*/
        {
          tptr = OpOf(tptr);
          while (!EndOfList(tptr))
            {
              if (!istableelement(ThisItem(tptr), modifiable))
                return FALSE;
              tptr = NextItem(tptr);
            }
          return TRUE;
        }
        /*}}}*/
    #ifdef ARRAYCONSTRUCTOR
      case S_ARRAYCONSTRUCTOR :
        return istableelement(ACValExpOf(tptr));
    #endif
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void checkvariable (tptr)*/
/*****************************************************************************
 *
 *  checkvariable checks that the tree 'tptr' is writable.
 *
 *****************************************************************************/
PRIVATE void checkvariable ( treenode *tptr )
{
  if (!iselement(tptr, FALSE))
    {
      treenode *nameptr = nameof(tptr);
      switch (TagOf(nameptr))
        {
          case S_STRING:
            chkreport (CHK_BAD_DEST_STRING, chklocn);
            break;
          case S_CONSTCONSTRUCTOR:
          case S_CONSTRUCTOR:
            chkreport (CHK_BAD_DEST_CONSTRUCTOR, chklocn);
            break;
          default:
            chkreport_s(CHK_BAD_DEST, chklocn, WNameOf(NNameOf(nameptr)));
            break;
        }
    }
}
/*}}}*/
/*{{{  PRIVATE void checkprotvariable (tptr)*/
/*****************************************************************************
 *
 *  checkprotvariable checks that the tree 'tptr' is a valid input destination
 *
 *****************************************************************************/
PRIVATE void checkprotvariable ( treenode *tptr )
{
  if (TagOf(tptr) == S_COLON2)
    {
      checkvariable(LeftOpOf(tptr));
      checkvariable(RightOpOf(tptr));
    }
  else
    checkvariable(tptr);
}
/*}}}*/
/*{{{  PRIVATE void checkexpsandtypes(exps, types, e_bad_type, e_too_many_vars, ..*/
/*****************************************************************************
 *
 *  checkexpsandtypes takes a list of expressions 'exps' and a list of
 *                    types 'types', type checks each expression and
 *                    ensures it has the same type as the corresponding
 *                    entry of 'types'.
 *
 *****************************************************************************/
PRIVATE void checkexpsandtypes ( treenode *exps , treenode *types , int e_bad_type , int e_too_many_vars , int e_too_few_vars )
{
  int varno = 1;
  while (!EndOfList(exps) && !EndOfList(types))
    /*{{{  check item on list, move to next item*/
    {
      treenode *type = ThisItem(types);
      if (!typesequivalent(type, typecheck(ThisItem(exps), basetype(type)), TRUE))
        {
          char buf[50];
          sprintf(buf, "%d", varno);
          chkreport_s(e_bad_type, chklocn, buf);
        }
      exps = NextItem(exps);
      types = NextItem(types);
      varno++;
    }
    /*}}}*/
  if (!EndOfList(exps))
    chkreport (e_too_many_vars, chklocn);
  else if (!EndOfList(types))
    chkreport (e_too_few_vars, chklocn);
}
/*}}}*/
/*{{{  PRIVATE void checkasslist(lhs, rhs)*/
/*****************************************************************************
 *
 *  checkasslist type checks the lists of assignments whose sources are 'rhs'
 *               and whose destinations are 'lhs'.
 *
 *****************************************************************************/
PRIVATE void checkasslist ( treenode *lhs , treenode *rhs )
{
  int varno = 1;
  int old = switch_to_temp_workspace();
  while (!EndOfList(lhs) && !EndOfList(rhs))
    {
      char buf[50];
      sprintf(buf, "%d", varno);
      checksame (ThisItem(lhs),ThisItem(rhs), S_UNKNOWN, CHK_INVTYPE_ASS, buf);
      lhs = NextItem(lhs);
      rhs = NextItem(rhs);
      varno++;
    }
  switch_to_prev_workspace(old);
  if (!EndOfList(lhs))
    chkreport (CHK_TOO_MANY_VARS, chklocn);
  else if (!EndOfList(rhs))
    chkreport (CHK_TOO_FEW_VARS, chklocn);
}
/*}}}*/
/*{{{  PRIVATE void cparam (nptr, paramtype)*/
/*****************************************************************************
 *
 *  cparam checks the formal parameter declaration 'nptr'.
 *         paramtype is S_PROC if we are checking a procedure parameter,
 *         otherwise we are checking a function parameter.
 *
 *****************************************************************************/
PRIVATE void cparam ( treenode *nptr , int paramtype )
{
  /*{{{  check it's not a var param to a function*/
  if ((paramtype != S_PROCDEF) && (TagOf(nptr) == N_PARAM))
    chkreport (CHK_FN_VAR_PARAM, chklocn);
  /*}}}*/
  /*{{{  check the specifier*/
  cspecifier(NTypeOf(nptr));
  /*}}}*/
  /*{{{  check a CHAN, PORT or TIMER is not a val param*/
  if ((TagOf(nptr) == N_VALPARAM) && !validval(NTypeOf(nptr)))
    chkreport (CHK_INV_VAL, chklocn);
  /*}}}*/

}
/*}}}*/
/*{{{  PRIVATE int csellist (type, tptr)*/
/*****************************************************************************
 *
 *  csellist type checks and constant folds a list of selections 'tptr' of
 *           type 'type', and returns the number of selections if they are
 *           all valid, or -1 if an error is found.
 *
 *****************************************************************************/

/* Type check and constant fold a list of selections */
PRIVATE int csellist ( int type , treenode *tptr )
{
  int alliswell = TRUE;
  int nselections = 0;
  if (TagOf(tptr) == S_ELSE)
    return(nselections);
  while (!EndOfList(tptr))
    /*{{{  type check and constant fold a selection*/
    {
      treenode *selection = ThisItem(tptr);
      treenode *t = typecheck (selection, type);
    
      if (TagOf(t) == S_UNDECLARED)
        alliswell = FALSE;
      else if (TagOf(t) == type)
        /*{{{  constant fold selection*/
        {
          selection = foldexp(selection);
          NewItem(selection, tptr);
          if (isconst(selection))
            {
              if (type != S_INT64)
                I32ToI64 (HiValAddr(selection), LoValAddr(selection),
                          LoValOf(selection));
            }
          else
            {
              alliswell = FALSE;
              chkreport(CHK_SELN_NOTCONST, chklocn);
            }
          nselections = nselections + 1;
        }
        /*}}}*/
      else
        chkreport (CHK_INVTYPE_SELN, chklocn);
      tptr = NextItem(tptr);
    }
    /*}}}*/
  return(alliswell ? nselections : (-1));
}
/*}}}*/
/*{{{  PRIVATE casenode_t *addcaselist (caseptr, tptr)*/
/*****************************************************************************
 *
 *  addcaselist adds all the selections in the list 'tptr' to the case table
 *              beginning at 'caseptr', updates 'caseptr' and returns it.
 *
 *****************************************************************************/
PRIVATE casenode_t *addcaselist ( casenode_t *caseptr , treenode *tptr )
{
  SOURCEPOSN locn = LocnOf(tptr);
  if (!EmptyList(tptr))
    {
      if (TagOf(tptr) == S_ELSE)
        /*{{{  check its the only one*/
        {
          if (elseflag)
            chkreport (CHK_TOO_MANY_ELSES, chklocn);
          else
            elseflag = TRUE;
        }
        /*}}}*/
      else
        while (!EndOfList(tptr))
          /*{{{  add this selection to casetable, move to next selection*/
          {
            treenode *cptr = ThisItem(tptr);
            caseptr->ca_hi = HiValOf(cptr);
            caseptr->ca_lo = LoValOf(cptr);
            /*caseptr->ca_decl = cptr;*/
            caseptr->ca_locn = locn;
            caseptr++; /* Point to the next casenode structure */
            tptr = NextItem(tptr);
          }
          /*}}}*/
    }
  return(caseptr);
}
/*}}}*/
/*{{{  PRIVATE int compcases (c1, c2)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  compcases takes two pointers to casenodes, 'c1', 'c2', and returns
 *             1 if case c1 > case c2,
 *             0 if case c1 = case c2,
 *            -1 if case c1 < case c2.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE int compcases ( casenode_t *c1 , casenode_t *c2 )
{
  int greater =  FALSE;
  Int64Gt(&greater, c1->ca_hi, c1->ca_lo, c2->ca_hi, c2->ca_lo);
  if (greater)
    return (1);
  else
    {
      int equal;
      Int64Eq(&equal, c1->ca_hi, c1->ca_lo, c2->ca_hi, c2->ca_lo);
      if (equal)
        return(0);
      else
        return(-1);
    }
}
/*}}}*/
/*{{{  PRIVATE void checkvariant (pptr, vptr)*/
/*****************************************************************************
 *
 *  checkvariant checks the variant tree 'vptr' against the channel protocol
 *               'pptr'.
 *
 *****************************************************************************/
PRIVATE void checkvariant ( treenode *pptr , treenode *vptr )
{
  /*{{{  skip over any leading specifications to the real variant*/
  vptr = skipspecifications(vptr);
  /*}}}*/
  chklocn = LocnOf(vptr);

  if (TagOf(vptr) == S_VARIANT)
    {
      protocolcheck (pptr, VRTaggedListOf(vptr));
      walklist(checkprotvariable, NextItem(VRTaggedListOf(vptr)));
    }
  else
    msg_out(SEV_INTERNAL, CHK, CHK_LOST_VARIANT, chklocn);
}
/*}}}*/
/*{{{  PRIVATE int compvariants (v1, v2)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  compvariants takes two pointers to vntnodes, 'v1', 'v2', and returns
 *             1 if case v1 > case v2,
 *             0 if case v1 = case v2,
 *            -1 if case v1 < case v2.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE int compvariants ( vntnode_t *v1 , vntnode_t *v2 )
{
  return ((v1->vnt_tag >  v2->vnt_tag) ? 1 :
          (v1->vnt_tag == v2->vnt_tag) ? 0 : -1);
}
/*}}}*/
/*{{{  PRIVATE void vscope (tptr, scope, locn, e)*/
/*****************************************************************************
 *
 *  vscope checks that the element 'tptr' is not in scope.
 *
 *****************************************************************************/

/* Error number */
PRIVATE void vscope ( treenode *tptr , BIT16 scope , SOURCEPOSN locn , int e )
{
  if (TagOf(tptr) == S_ARRAYSUB || TagOf(tptr) == S_RECORDSUB)
    /*{{{  scope the subscripted element*/
    vscope(ASBaseOf(tptr), scope, locn, e);
    /*}}}*/
  else if (TagOf(tptr) == S_SEGMENT)
    /*{{{  scope the segmented array*/
    vscope(SNameOf(tptr), scope, locn, e);
    /*}}}*/
  else if ((TagOf(tptr) != S_CONSTRUCTOR) && (TagOf(tptr) != S_STRING) &&
           (NScopeOf(tptr) < scope))
    chkreport_s(e, locn, WNameOf(NNameOf(tptr)));
}
/*}}}*/
/*{{{  PRIVATE void checkvalof (tptr, scope, nestedproc, locn)*/
PRIVATE BIT16 chkvalof_scope;
PRIVATE int chkvalof_nestedproc;
PRIVATE SOURCEPOSN chkvalof_locn;
/*{{{  PRIVATEPARAM void do_checkvalof (tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  checkvalof takes a VALOF tree 'tptr' and checks
 *             (i)  that it does not contain ALT, PAR, input or output
 *             (ii) that all variables written to (lhs of assignment,
 *                  rhs of var. abbreviation or retype, var. params) are
 *                  in scope 'scope'.
 *
 *****************************************************************************/
/*{{{  note on VALOF scoping*/
/* The 'scope' parameter is the scope that we are checking; all variables
   assigned to must have a scope number greater than or equal to this value.
   We check that procedures which are called do not access local variables
   by recursively calling checkvalof on the body of the procedure with
   the scope of the procedure.
 */
/*}}}*/
/*}}}*/
PRIVATEPARAM int do_checkvalof ( treenode *tptr )
{
   switch (TagOf(tptr))
     {
       /*{{{  PLACE WSPLACE VSPLACE*/
       case S_PLACE: case S_WSPLACE: case S_VSPLACE:
       #ifdef CONFIG
       case S_PLACEON:
       #endif
         if (chkvalof_nestedproc)
           vscope(DNameOf(tptr), chkvalof_scope, chkvalof_locn, CHK_VALOF_CALL_SCOPE);
         else
           vscope(DNameOf(tptr), chkvalof_scope, LocnOf(tptr), CHK_VALOF_SCOPE);
         break;
       /*}}}*/
       /*{{{  PINSTANCE*/
       case S_PINSTANCE:
         /* Recursively check the procedure body with the procedure scope */
         /* Check that var params are in the VALOF scope */
         {
           treenode *actparamlist = IParamListOf(tptr);
           treenode *iname = INameOf(tptr);
           treenode *fparamlist = NTypeOf(iname);
           if (!separatelycompiled(iname) && TagOf(iname) != N_PREDEFPROC)
             /*{{{  check instanced procedure*/
             {
               BIT16 save_chkvalof_scope = chkvalof_scope;
               int save_chkvalof_nestedproc = chkvalof_nestedproc;
               SOURCEPOSN save_chkvalof_locn = chkvalof_locn;
             /* This breaks if the PROC is declared INSIDE the valof, and accesses
                a 'free' variable which is also declared INSIDE the VALOF.
                We fix it by taking the outermost scope of the two */
             /*chkvalof_scope = NScopeOf(iname);*/
               chkvalof_scope = (BIT16)min(chkvalof_scope, NScopeOf(iname));
               chkvalof_nestedproc = TRUE;
               chkvalof_locn = LocnOf(tptr);
               prewalkproctree(DValOf(NDeclOf(iname)), do_checkvalof);
               chkvalof_scope = save_chkvalof_scope;
               chkvalof_nestedproc = save_chkvalof_nestedproc;
               chkvalof_locn = save_chkvalof_locn;
             }
             /*}}}*/
       
           while (!EndOfList(fparamlist) && (TagOf(fparamlist) != S_UNDECLARED))
             {
               treenode *fparam = ThisItem(fparamlist);
               if (TagOf(fparam) == N_PARAM)
                 /*{{{  check that var param is in scope*/
                 {
                   if (chkvalof_nestedproc)
                     vscope(ThisItem(actparamlist), chkvalof_scope, chkvalof_locn, CHK_VALOF_CALL_SCOPE);
                   else
                     vscope(ThisItem(actparamlist), chkvalof_scope, LocnOf(tptr), CHK_VALOF_SCOPE);
                 }
                 /*}}}*/
               fparamlist = NextItem(fparamlist);
               actparamlist = NextItem(actparamlist);
             }
         }
         break;
       /*}}}*/
       /*{{{  ASS*/
       case S_ASS:
         /* Check that each lhs element is in the VALOF scope */
         {
           treenode *lhs = LHSOf(tptr);
           if (TagOf(lhs) == S_LIST)
             /*{{{  multiple assignment*/
             while (!EndOfList(lhs))
               {
                 if (chkvalof_nestedproc)
                   vscope (ThisItem(lhs), chkvalof_scope, chkvalof_locn, CHK_VALOF_CALL_SCOPE);
                 else
                   vscope (ThisItem(lhs), chkvalof_scope, LocnOf(tptr), CHK_VALOF_SCOPE);
                 lhs = NextItem(lhs);
               }
             /*}}}*/
           else
             {
               if (chkvalof_nestedproc)
                 vscope (lhs, chkvalof_scope, chkvalof_locn, CHK_VALOF_CALL_SCOPE);
               else
                 vscope (lhs, chkvalof_scope, LocnOf(tptr), CHK_VALOF_SCOPE);
             }
         }
         break;
       /*}}}*/
       /*{{{  ABBR RETYPE*/
       case S_ABBR:
       case S_RETYPE:
         if (chkvalof_nestedproc)
           vscope (DValOf(tptr), chkvalof_scope, chkvalof_locn, CHK_VALOF_CALL_SCOPE);
         else
           vscope (DValOf(tptr), chkvalof_scope, LocnOf(tptr), CHK_VALOF_SCOPE);
         break;
       /*}}}*/
       /*{{{  pars, inputs, outputs and alts*/
       case S_INPUT: case S_OUTPUT:
       case S_CASE_INPUT: case S_DELAYED_INPUT: case S_TAGGED_INPUT:
       case S_ALT: case S_REPLALT: case S_PRIALT: case S_PRIREPLALT:
         {
           char name[MAX_ERR_SIZE];
           ftagstring(name, TagOf(tptr));
           if (chkvalof_nestedproc)
             chkreport_s(CHK_BAD_VALOF_CALL, chkvalof_locn, name);
           else
             chkreport_s(CHK_BAD_VALOF, LocnOf(tptr), name);
         }
         return STOP_WALK;
       
       /*
       case S_PAR: case S_REPLPAR:
         if (!networkdesc)
           {
             char name[MAX_ERR_SIZE];
             ftagstring(name, TagOf(tptr));
             if (nestedproc)
               chkreport_s(CHK_BAD_VALOF_CALL, chkvalof_locn, name);
             else
               chkreport_s(CHK_BAD_VALOF, LocnOf(tptr), name);
             return STOP_WALK;
           }
         break;
       */
       /*}}}*/
     }
   return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATE void checkvalof (tptr, scope, nestedproc, locn)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  checkvalof takes a VALOF tree 'tptr' and checks
 *             (i)  that it does not contain ALT, PAR, input or output
 *             (ii) that all variables written to (lhs of assignment,
 *                  rhs of var. abbreviation or retype, var. params) are
 *                  in scope 'scope'.
 *
 *****************************************************************************/
/*{{{  note on VALOF scoping*/
/* The 'scope' parameter is the scope that we are checking; all variables
   assigned to must have a scope number greater than or equal to this value.
   We check that procedures which are called do not access local variables
   by recursively calling checkvalof on the body of the procedure with
   the scope of the procedure.
 */
/*}}}*/
/*}}}*/
PRIVATE void checkvalof ( treenode *tptr , BIT16 scope , int nestedproc , SOURCEPOSN locn )
{
  chkvalof_scope = scope;
  chkvalof_nestedproc = nestedproc;
  chkvalof_locn = locn;
  prewalkproctree(tptr, do_checkvalof);
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE int checkinstruction (tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  checkinstruction takes a GUYCODE or GUYSTEP node and checks that the
 *                   instruction is valid, and inserts the instruction value
 *                   into the node.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void checkinstruction ( treenode *tptr , int guy_not_asm, int *ptrs_allowed, int *element_required )
{
  /* add 1 to skip the leading '.' */
  char *instruction = WNameOf((wordnode *)LeftOpOf(tptr)) + 1;
  int err, required_operands;
  int ival = (int)lookupinstruction(instruction, guy_not_asm,
                                 &err, &required_operands, ptrs_allowed, element_required);
  int operands = listitems(RightOpOf(tptr)); /* number of operands */

  if ((ival & I_PSEUDO_OP) && (ival & INST_MASK) == I_ALIGN)
    chkreport_s(CHK_UNIMPLEMENTED_ASMCODE, chklocn, instruction);
  else if (err == INSTRUCTION_NOT_ENABLED)
    chkreport_s(CHK_DISABLED_GUYCODE, chklocn, instruction);
  else if (err == INSTRUCTION_NOT_VALID)
    chkreport_s(CHK_INVALID_GUYCODE, chklocn, instruction);
  else if (err == INSTRUCTION_NOT_DECLARED)
    chkreport_s(CHK_BAD_GUYCODE, chklocn, instruction);

  else if (((operands < required_operands) && (required_operands >= 0)) ||
           ((operands == 0)                && (required_operands <  0))   )
    chkreport_s(CHK_MISSING_OPERAND, chklocn, instruction);

  else if ((operands > required_operands) && (required_operands >= 0))
    chkreport_s(CHK_EXTRANEOUS_OPERAND, chklocn, instruction);
  else
    SetDOpType(tptr, ival);
  return;
}
/*}}}*/
/*{{{  PUBLIC void checkelement*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  checkelement makes sure the tptr is an element
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void checkelement ( treenode *tptr, treenode *type, int paramno )
{
  if (basetype(type) == S_CHAN && chanaspointer)
    {
      if (istableelement(tptr, TRUE))
        return;
    }
  else if (iselement(tptr, TRUE))
        return;
  if (paramno == 0)
    chkreport(CHK_INV_ABBR, chklocn);
  else
    chkreport_i(CHK_INVVARIABLE_PARAM, chklocn, paramno);
}
/*}}}*/
/*{{{  PRIVATE void checkforopendimslist(treenode *)*/
PRIVATE void checkforopendimslist(treenode *tptr)
/* This is used to check that we don't attempt to use 'open' arrays
   in multiple assignment */
{
  while (!EndOfList(tptr))
    {
      int unknowndims = 0;
      treenode *type = gettype(ThisItem(tptr));
      while (TagOf(type) == S_ARRAY)
        {
          if (ARDimLengthOf(type) == NULL)
            unknowndims++;
          type = ARTypeOf(type);
        }
      if (unknowndims != 0)
        chkreport(CHK_INV_MULTIASSIGN, chklocn);
      tptr = NextItem(tptr);
    }
}
/*}}}*/
/*{{{  PRIVATE void checkprimary*/
PRIVATE void checkprimary(int instruction, int guy_not_asm, int addressof, treenode *tptr, SOURCEPOSN locn)
/* This checks that tptr is a valid primary operand, or op to ADDRESSOF */
{
  while (TRUE)
    /*{{{  switch (TagOf(tptr))*/
    switch(TagOf(tptr))
      {
        default:
          /* Allow expressions involving Special names in ASM primaries only */
          if (!guy_not_asm && !addressof &&
              isscalartype(typeof(tptr)) && wouldbeconst(tptr))
            ; /* skip */
          else
            chkreport(CHK_BAD_GUY_OPERAND, locn);
          return;
        case S_CONSTEXP:
        case N_DECL: case N_REPL: case N_PARAM: case N_VALPARAM:
          return;
        case N_ABBR: case N_VALABBR: case N_RETYPE: case N_VALRETYPE:
        #if 0
          /* this breaks some examples: Eg we should allow
            LDL x, where VAL x IS array[i] :
            But we should disallow:
            LDL  x, where VAL x IS "fred" :
          */
          tptr = DValOf(NDeclOf(tptr)); /* bug 594 2/11/90 */
          break;
        #else
          return;
        #endif
        case S_ARRAYSUB: case S_RECORDSUB:
          /* Variable subscripts are OK in addressof */
          if (addressof)
            return;
          /* Constant subscripts in GUY, none at all in ASM */
          if (!guy_not_asm || !isconst(ASIndexOf(tptr)))
            chkreport(CHK_BAD_GUY_OPERAND, locn);
          tptr = ASBaseOf(tptr);
          break;
        case S_CONSTCONSTRUCTOR:
        case S_STRING: /* added bug 594 2/11/90 */
          if (addressof)
            return;
          if (guy_not_asm &&
              ((instruction == I_LDLP) || (instruction == I_LDNLP)))
            return;
          chkreport(CHK_BAD_GUY_OPERAND, locn);
          return;
      }
    /*}}}*/
}
/*}}}*/
/*}}}*/

/*{{{  functions called from scopeandcheck*/
/*{{{  PUBLIC treenode *caction(tptr)                  actions*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  caction performs semantic checking on action tree 'tptr'
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *caction ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{   do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        chklocn = LocnOf(tptr);
        switch (TagOf(tptr))
          {
            /*{{{  case S_ASS*/
            case S_ASS:
              {
                treenode *lhs = LHSOf(tptr), *rhs = RHSOf(tptr);
                if (TagOf(lhs) == S_LIST)
                  /*{{{  we have a multi-assignment*/
                  {
                    walklist(checkvariable, lhs);
                    if (listitems(rhs) == 1)
                      /*{{{  multi-valued function or valof*/
                      {
                        /* if rhs is a multi-valued function or VALOF, there is only one expression
                           on rhs, otherwise there is a list of expressions */
                        treenode *exp1 = ThisItem(rhs);
                      
                        if (TagOf(exp1) == S_FINSTANCE)
                          /*{{{  rhs is a multi-valued function call*/
                          {
                            treenode *nameptr = INameOf(exp1);
                            treenode *rtypes;
                          
                            checkparams(exp1, S_FUNCTION);
                            /*{{{  skip if FUNCTION is not defined*/
                            if (TagOf(nameptr) &&
                                (TagOf(NTypeOf(nameptr)) == S_UNDECLARED))
                              {
                               memcpy((char *)env, (char *)savedenv, sizeof(env));
                               return NULL;
                              }
                            /*}}}*/
                            rtypes = FnTypeListOf(NTypeOf(nameptr));
                            checkexpsandtypes(lhs, rtypes, CHK_INVTYPE_ASS,
                                              CHK_TOO_MANY_VARS, CHK_TOO_FEW_VARS);
                            /* fold lhs and rhs parameters */
                            SetLHS(tptr, foldexplist(LHSOf(tptr)));
                            SetIParamList(exp1, foldexplist(IParamListOf(exp1)));
                          }
                          /*}}}*/
                        else if (isspecification(exp1) || (TagOf(exp1) == S_VALOF))
                          /*{{{  rhs is a valof*/
                          {
                            /*{{{  skip leading specifications*/
                            exp1 = skipspecifications(exp1);
                            /*}}}*/
                            checkasslist(lhs, VLResultListOf(exp1));
                            /* fold lhs and right hand side result list */
                            SetLHS(tptr, foldexplist(LHSOf(tptr)));
                            SetVLResultList(exp1, foldexplist(VLResultListOf(exp1)));
                          }
                          /*}}}*/
                        else
                          chkreport(CHK_TOO_MANY_VARS, chklocn);
                      }
                      /*}}}*/
                    else
                      /*{{{  list of simple assignments*/
                      {
                        checkasslist(lhs, rhs);
                        SetLHS(tptr, foldexplist(LHSOf(tptr)));
                        SetRHS(tptr, foldexplist(RHSOf(tptr)));
                        checkforopendimslist(lhs);
                        checkforopendimslist(rhs);
                      }
                      /*}}}*/
                  }
                  /*}}}*/
                else
                  {
                    int old;
                    checkvariable(lhs);
                    old = switch_to_temp_workspace();
                    checksame (lhs, rhs, S_UNKNOWN, CHK_INVTYPE_ASS, "1");
                    switch_to_prev_workspace(old);
                    /*{{{  fold the left and right-hand sides*/
                    SetLHS(tptr, foldexp(LHSOf(tptr)));
                    SetRHS(tptr, foldexp(RHSOf(tptr)));
                    /*}}}*/
                  }
              }
              break;
            /*}}}*/
            /*{{{  case S_OUTPUT, S_TAGGED_INPUT*/
            case S_OUTPUT:
            case S_TAGGED_INPUT:
            /* Check that lhs is a channel.
               Check that the rhs conforms to the channel protocol.
            */
              {
                treenode *t;
                /* Type check the channel */
                t = typecheck (LHSOf(tptr), S_UNKNOWN);
            
                if ((TagOf(t) == S_CHAN) || (TagOf(t) == S_PORT))
                  /*{{{  protocol check the right-hand side*/
                  {
                    treenode *protocol = ProtocolOf(t);
                    if (TagOf(tptr) == S_TAGGED_INPUT)
                      /*{{{  check the protocol is a tagged protocol*/
                      if (TagOf(protocol) != N_TPROTDEF)
                        chkreport(CHK_BAD_TAGGED_INPUT_PROTOCOL, chklocn);
                      /*}}}*/
                    protocolcheck (protocol, RHSOf(tptr));
                    /*{{{  check all the input destinations are variables*/
                    if (TagOf(tptr) == S_TAGGED_INPUT)
                      /* But don't check the tag */
                      walklist(checkprotvariable, NextItem(RHSOf(tptr)));
                    /*}}}*/
                    /*{{{  fold the left and right-hand sides*/
                    SetLHS(tptr, foldexp(LHSOf(tptr)));
                    SetRHS(tptr, foldexplist(RHSOf(tptr)));
                    /*}}}*/
                  }
                  /*}}}*/
                else if (TagOf(t) != S_UNDECLARED)
                  chkreport (CHK_NOT_CHANNEL, chklocn);
              }
              break;
            /*}}}*/
            /*{{{  case S_CASE_INPUT*/
            case S_CASE_INPUT:
            /* type check the channel, protocol check each variant on the variant list */
              {
                treenode *variantlist = RHSOf(tptr);
                treenode *pptr;                          /* Pointer to channel protocol */
                treenode *t;
                int nvariants = 0;
            
                /* Type check the channel */
                t = typecheck (LHSOf(tptr), S_UNKNOWN);
                if ((TagOf(t) == S_CHAN) || (TagOf(t) == S_PORT))
                  {
                    pptr = ProtocolOf(t);
                    /*{{{  check the protocol is a tagged protocol*/
                    if (TagOf(pptr) != N_TPROTDEF)
                      chkreport(CHK_BAD_CASE_INPUT_PROTOCOL, chklocn);
                    /*}}}*/
                    /*{{{  check each variant on the variant list*/
                    while (!EndOfList(variantlist))
                      {
                        checkvariant(pptr, ThisItem(variantlist));
                        variantlist = NextItem(variantlist);
                        nvariants++;
                      }
                    /*}}}*/
                    /*{{{  fold the left-hand side*/
                    SetLHS(tptr, foldexp(LHSOf(tptr)));
                    /*}}}*/
                    /*{{{  fold the right-hand side*/
                    walklist(foldvariant, RHSOf(tptr));
                    /*}}}*/
                    /*{{{  check all the tags are distinct*/
                    { /* we add 1 to this to prevent problems with zero byte memalloc */
                      vntnode_t *varianttablebase =
                                    (vntnode_t *)memalloc(sizeof(vntnode_t) * (nvariants+1)),
                                     *variantptr = varianttablebase;
                      /*{{{  walk variant list  putting all variant tags in variant table*/
                      variantlist = RHSOf(tptr);
                      while (!EndOfList(variantlist))
                        {
                          treenode *thisvarianttag;
                          treenode *thisvariant = ThisItem(variantlist);
                          /*{{{  skip leading specifications on thisvariant*/
                          thisvariant = skipspecifications(thisvariant);
                          /*}}}*/
                          thisvarianttag = ThisItem(VRTaggedListOf(thisvariant));
                          if (TagOf(thisvarianttag) == S_CONSTEXP)
                            thisvarianttag = CExpOf(thisvarianttag);
                          variantptr->vnt_decl = thisvariant;
                          variantptr->vnt_tag = thisvarianttag;
                          variantptr++;
                          variantlist = NextItem(variantlist);
                        }
                      /*}}}*/
                      /*{{{  sort variant table*/
                      local_qsort(varianttablebase, nvariants, sizeof(vntnode_t),
                                  (int (*)())compvariants);
                      /*}}}*/
                      /*{{{  check each variant is distinct*/
                      {
                        int i;
                        variantptr = varianttablebase;
                        for (i = 0; i < (nvariants - 1); i++)
                          {
                            if (compvariants(variantptr, variantptr + 1) == 0)
                              {
                                chklocn = LocnOf(tptr);
                                chkreport_s(CHK_MULTIPLE_VARIANT, chklocn,
                                            WNameOf(NNameOf(variantptr->vnt_tag)));
                              }
                            variantptr++;
                          }
                      }
                      /*}}}*/
                      memfree(varianttablebase);
                    }
                    /*}}}*/
                  }
                else if (TagOf(t) != S_UNDECLARED)
                  chkreport (CHK_NOT_CHANNEL, chklocn);
              }
              break;
            /*}}}*/
            /*{{{  case S_INPUT*/
            case S_INPUT:
            /* Check that lhs is a channel or a timer
               Check that the rhs conforms to the channel protocol, or is a time.
            */
              {
                treenode *t = typecheck (LHSOf(tptr), S_UNKNOWN);
            
                if ((TagOf(t) == S_CHAN) || (TagOf(t) == S_PORT))
                  /*{{{  protocol check the channel usage, fold the rhs*/
                  {
                    treenode *protocol = ProtocolOf(t);
                    /*{{{  check the protocol is not a tagged protocol*/
                    if (TagOf(protocol) == N_TPROTDEF)
                      chkreport(CHK_BAD_INPUT_PROTOCOL, chklocn);
                    /*}}}*/
                    protocolcheck (protocol, RHSOf(tptr));
                    walklist(checkprotvariable, RHSOf(tptr));
                    SetRHS(tptr, foldexplist(RHSOf(tptr)));
                  }
                  /*}}}*/
                else if (TagOf(t) == S_TIMER)
                  /*{{{  type check the time, fold the time*/
                  {
                    if (listitems(RHSOf(tptr)) > 1)
                      chkreport (CHK_INV_TIMER_INPUT, chklocn);
                    else
                      {
                        treenode *timeexp = ThisItem(RHSOf(tptr));
                        treenode *timetype = typecheck (timeexp, S_INT);
                        if (!sametype(TagOf(timetype), S_INT))
                          chkreport (CHK_TIME_TYPE_MISMATCH, chklocn);
                        checkvariable(timeexp);
                        NewItem(foldexp(timeexp), RHSOf(tptr));
                      }
                  }
                  /*}}}*/
                else if (TagOf(t) != S_UNDECLARED)
                  chkreport (CHK_NOT_CHANNEL_OR_TIMER, chklocn);
            
                /*{{{  fold the left-hand side*/
                SetLHS(tptr, foldexp(LHSOf(tptr)));
                /*}}}*/
            
              }
              break;
            /*}}}*/
            /*{{{  case S_DELAYED_INPUT*/
            case S_DELAYED_INPUT:
            /* Check that lhs is a timer
               Check that there is only one rhs and it is a time.
            */
              {
                treenode *t = typecheck (LHSOf(tptr), S_UNKNOWN);
            
                if (TagOf(t) == S_TIMER)
                  /*{{{  type check the time*/
                  {
                    treenode *timetype = typecheck (RHSOf(tptr), S_INT);
                    if (!sametype(TagOf(timetype), S_INT))
                      chkreport (CHK_TIME_TYPE_MISMATCH, chklocn);
                    /*{{{  fold the left and right-hand sides*/
                    SetLHS(tptr, foldexp(LHSOf(tptr)));
                    SetRHS(tptr, foldexp(RHSOf(tptr)));
                    /*}}}*/
                  }
                  /*}}}*/
                else if (TagOf(t) != S_UNDECLARED)
                  chkreport (CHK_NOT_TIMER, chklocn);
            
              }
              break;
            /*}}}*/
            default:
              badtag(chklocn, (BIT32)TagOf(tptr), "caction");
          }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked action node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE int illegal_retype(int type)*/
PRIVATE int illegal_retype(int type)
/* returns TRUE if the type is not permitted in a RETYPE */
{
  return ((type == S_TIMER)
#ifdef CONFIG
          || network_datatype(type)
#endif
         );
}
/*}}}*/
/*{{{  PUBLIC treenode *cdeclaration(tptr)             declarations*/
/*****************************************************************************
 *
 *  cdeclaration performs semantic checking on declaration tree 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *cdeclaration ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
      
        chklocn = LocnOf(tptr);
        switch (TagOf(tptr))
          {
            /*{{{  case S_VALABBR S_ABBR                                             ***/
            case S_VALABBR:
            case S_ABBR:
              {
                treenode *t = NTypeOf(DNameOf(tptr));
                int type;
                treenode *abbrtype;
            
                /* We can't plug the result of typecheck in directly as it already
                   exists elsewhere on the tree */
                type = (t == NULL) ? S_UNKNOWN : basetype(t);
                if (DValOf(tptr) == NULL)
                  break;
                abbrtype  = typecheck (DValOf(tptr), type);
                if (t == NULL)
                  SetNType(DNameOf(tptr), copytree(abbrtype));
                else
                  if (spec_into_type(cspecifier(t), abbrtype) != TRUE)
                    chkreport (CHK_ABBR_TYPE_MISMATCH, chklocn);
                /*{{{  check type is legal for a VAL or VAR*/
                if (TagOf(tptr) == S_VALABBR)
                  {
                    if (!validval(NTypeOf(DNameOf(tptr))))
                      chkreport (CHK_INV_VAL, chklocn);
                  }
                else /* if (TagOf(tptr) == S_ABBR) */
                  checkelement(DValOf(tptr), NTypeOf(DNameOf(tptr)), 0);
                /*}}}*/
                /*{{{  fold the initialiser*/
                if (TagOf(abbrtype) != S_UNDECLARED)
                  {
                    /* Force a constant fold here (if the initialiser is a name of an array
                       the constant fold would not be generated) */
                    /* bug 1158 - note that this creates `copies' of tables here;
                       this needs to be done, or something else breaks;
                       the copies are `removed' just as they are generated
                       into the code buffer - CON 14/2/91 */
                    if (isconst(DValOf(tptr)))
                      SetDVal(tptr, newconstexp(DValOf(tptr)));
                    else
                      SetDVal(tptr, foldexp(DValOf(tptr)));
                  }
                /*}}}*/
              }
              break;
            /*}}}*/
            /*{{{  case S_VALRETYPE S_RETYPE*/
            case S_VALRETYPE:
            case S_RETYPE:
              {
                treenode *oldt = NTypeOf(DNameOf(tptr));
                treenode *rhstype;
                if (DValOf(tptr) == NULL)
                  break;
                /* We don't have any idea what the type of the rhs of a RETYPE is, so
                   say S_UNKNOWN */
                rhstype = typecheck (DValOf(tptr), S_UNKNOWN);
            
                if (oldt == NULL)
                  chkreport (CHK_RETYPE_NOTYPE, chklocn);
                if (TagOf(rhstype) == S_UNDECLARED) /* Don't try any checking */
                  break;
                /*{{{  check there is at most one unknown dimension, work it out*/
                /* If the RETYPE specifier has an array with null dimension,
                   we must work out the dimension for ourselves from the
                   size of the rhs. */
                {
                  int unknowndims = 0;
                  treenode *t;
                  /*{{{  count unknown dimensions*/
                  t = oldt;
                  while (TagOf(t) == S_ARRAY)
                    {
                      if (ARDimLengthOf(t) == NULL)
                        unknowndims++;
                      t = ARTypeOf(t);
                    }
                  /*}}}*/
                  if (unknowndims > 1)
                    chkreport (CHK_ADIM_MISSING, chklocn);
                  else if (unknowndims == 1)
                    /*{{{  fill in the missing dimension*/
                    {
                      BIT32 lbytes, rbytes;
                      INT32 nelements = 1;
                      treenode *unknowndimp = NULL;
                      cspecifier(oldt);
                      /*{{{  look for unknown dimensions*/
                      t = oldt;
                      while (TagOf(t) == S_ARRAY)
                        {
                          DEBUG_MSG(("cdeclaration: RETYPE; looking for unknown dim: t is %x\n", t));
                          if (ARDimLengthOf(t) == NULL)
                            unknowndimp = t;
                          else if ((BIT32)ARDimOf(t) > (max_array_size / (BIT32)nelements))
                            chkreport(CHK_ARRAY_SIZE_OVERFLOW, chklocn);
                          else
                            nelements *= ARDimOf(t);
                          t = ARTypeOf(t);
                        }
                      /*}}}*/
                      lbytes = nelements * bytesin(t);
                      rbytes = bytesin(rhstype);
                      if (rbytes == -1)
                        SetARDim(unknowndimp, -1);
                      else if ((rbytes % lbytes) == 0)
                        {
                          BIT32 dim = rbytes / lbytes;
                          SetARDim(unknowndimp, dim);
                          SetARDimLength(unknowndimp, newconstexpnode(S_CONSTEXP, 0, dummyexp_p,
                                                                    ZERO32, dim));
                        }
                      else
                        chkreport (CHK_RETYPE_TYPE_MISMATCH, chklocn);
                    }
                    /*}}}*/
                  else
                    /*{{{  check the RETYPE specifier and then check lh & rh sizes are equal*/
                    {
                      BIT32 rhsize = bytesin(rhstype);
                      ctype (oldt);                      /* Check the type tree is valid */
                      /* Check the left and right-hand sizes are equal */
                      if ((rhsize != -1) &&    /* If we know the right-hand size at compile time */
                          (bytesin(oldt) != rhsize))
                        chkreport (CHK_RETYPE_TYPE_MISMATCH, chklocn);
                    }
                    /*}}}*/
                }
                /*}}}*/
                /*{{{  check type is legal*/
                /* VAL abbreviations may not be PORT, CHAN or TIMER */
                /* We also enforce that a var. RETYPE cannot be a PORT, CHAN or TIMER */
                /* OLD VERSION:
                if (!validval(NTypeOf(DNameOf(tptr))) ||
                    !validval(rhstype))
                  chkreport (CHK_INV_RETYPE, chklocn);
                */
                /* We enforce that CHANs can only be RETYPEd to CHANs etc */
                /* COMMENTED OUT AGAIN. New try below
                if (!validval(NTypeOf(DNameOf(tptr))) ||
                    !validval(rhstype))
                  {
                    if (TagOf(tptr) == S_VALRETYPE)
                      chkreport (CHK_BAD_VALRETYPE, chklocn);
                    /@ TEMPORARY: ALLOW ANY RETYPEs
                    if (basetype(NTypeOf(DNameOf(tptr))) != basetype(rhstype))
                      chkreport (CHK_BAD_CHANRETYPE, chklocn);
                    @/
                  }
                */
                { int lhsbase = basetype(NTypeOf(DNameOf(tptr)));
                  int rhsbase = basetype(rhstype);
                  if (illegal_retype(lhsbase) || illegal_retype(rhsbase))
                    chkreport(CHK_INV_RETYPE, chklocn);
                  if ((rhsbase == S_CHAN) &&
                      (TagOf(tptr) != S_VALRETYPE) && (lhsbase != S_CHAN))
                    chkreport(CHK_BAD_CHANRETYPE, chklocn);
                }
                /*}}}*/
                /*{{{  check rhs is legal for a var*/
                if (TagOf(tptr) == S_RETYPE)
                  /*checkvariable(DValOf(tptr));*/
                  checkelement(DValOf(tptr), rhstype, 0);
                /*}}}*/
                /*{{{  fold the initialiser*/
                /* Force a constant fold here (if the initialiser is a name of an array
                   the constant fold would not be generated) */
                /* bug 1158 - note that this creates `copies' of tables here;
                   this needs to be done, or something else breaks;
                   the copies are `removed' just as they are generated
                   into the code buffer - CON 14/2/91 */
                SetDVal(tptr, foldexpinto(isconst(DValOf(tptr)) ? newconstexp(DValOf(tptr))
                                                                : DValOf(tptr),
                                          TagOf(oldt)));
                /*}}}*/
                break;
              }
            /*}}}*/
            /*{{{  case S_SFUNCDEF S_LFUNCDEF*/
            case S_SFUNCDEF:
            case S_LFUNCDEF:
              /* Parameter specifiers have already been checked */
              /* VALOF contents and scope have already been checked */
              if (!separatelycompiled(DNameOf(tptr)))
                {
                  treenode *valofptr;
                  treenode *typesptr = FnTypeListOf(NTypeOf(DNameOf(tptr)));
                  /*{{{  check that the function result types are all scalar*/
                  {
                    treenode *t = typesptr;
                    while (!EndOfList(t))
                      {
                        treenode *type = ThisItem(t);
                        if (!isscalartype(TagOf(type)))
                          chkreport(CHK_BAD_FUNCTION_TYPE, chklocn);
                        t = NextItem(t);
                      }
                  }
                  /*}}}*/
                  /*{{{  find the valof*/
                  valofptr = DValOf(tptr);
                  while (valofptr != NULL && TagOf(valofptr) != S_VALOF)
                    valofptr = DBodyOf(valofptr);
                  /*}}}*/
                  if (valofptr != NULL)
                    {
                      /*{{{  check the valof result types match the function formal types*/
                      {
                        treenode *resultlist = VLResultListOf(valofptr);
                        chklocn = LocnOf(resultlist);
                        checkexpsandtypes(resultlist, typesptr, CHK_INVTYPE_FRESULT,
                                          CHK_TOOMANYEXPS, CHK_TOOFEWEXPS);
                      }
                      /*}}}*/
                      /*{{{  fold the result list*/
                      SetVLResultList(valofptr, foldexplist(VLResultListOf(valofptr)));
                      /*}}}*/
                    }
                }
              break;
            /*}}}*/
            /*{{{  case S_DECL*/
            case S_DECL:
              /* Check that the type tree is valid */
              {
                treenode *dtptr = DNameOf(tptr);
            
                /* Unfortunately, all the namenodes in the list point to the same
                   type tree, so we only have to check the type tree once. */
            
                if (TagOf(dtptr) == S_LIST)
                  dtptr = ThisItem(dtptr);
            
                if (TagOf(dtptr) != N_DECL)     /* Should never happen if syn is correct */
                  msg_out_i(SEV_INTERNAL, CHK, CHK_INV_DECL_TAG, chklocn, TagOf(dtptr));
                
                ctype(NTypeOf(dtptr));
                break;
              }
            /*}}}*/
            /*{{{  case S_PROCDEF*/
            case S_PROCDEF: /* Parameter specifiers have already been checked */
            CASE_CONFIG_SPEC /* Nothing to do */
              break;
            /*}}}*/
            /*{{{  case S_TPROTDEF*/
            case S_TPROTDEF:
              {
                treenode *taglist = NTypeOf(DNameOf(tptr));
                walklist (checktag, taglist);
                /*{{{  give each tag a value*/
                {
                  int i = 0;
                  while (!EndOfList(taglist) && (i < MAX_TAGS))
                    {
                      SetNTValue(ThisItem(taglist), (BIT32)i);
                      taglist = NextItem(taglist);
                      i = i + 1;
                    }
                  if (!EndOfList(taglist))
                    chkreport_i(CHK_TOO_MANY_TAGS, chklocn, MAX_TAGS);
                }
                /*}}}*/
              }
              break;
            /*}}}*/
            /*{{{  case S_SPROTDEF*/
            case S_SPROTDEF:
              cdefseqprotocol(NTypeOf(DNameOf(tptr)));
              break;
            /*}}}*/
            default:
              badtag(chklocn, (BIT32)TagOf(tptr), "cdeclaration");
          }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked decl node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *cparmlist(tptr, paramtype, locn)     parameters*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  cparmlist performs semantic checking on formal parameter list 'tptr'.
 *            paramtype is S_PROC if we are checking procedure parameters,
 *            otherwise we are checking function parameters.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *cparmlist ( treenode *tptr , int paramtype , SOURCEPOSN locn )
{
  if (!nochecking)
    {
      /*{{{  do checking*/
      jmp_buf savedenv;
      memcpy((char *)savedenv, (char *)env, sizeof(env));
      /*{{{  recover from errors*/
      if (setjmp (env))
        {
         memcpy((char *)env, (char *)savedenv, sizeof(env));
         return NULL;
        }
      /*}}}*/
      {
        treenode *t = tptr;
        chklocn = locn;
        while (!EndOfList(t))
          {
            cparam(ThisItem(t), paramtype);
            t = NextItem(t);
          }
      }
      memcpy((char *)env, (char *)savedenv, sizeof(env));
      /*}}}*/
      DEBUG_MSG(("Checked paramlist node ok near line %d\n",locn));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *crepl(tptr)                    replicated nodes*/
/*****************************************************************************
 *
 *  crepl performs semantic checking on replicator tree 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *crepl ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{  do checking*/
      {
        /*{{{  */
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*}}}*/
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *t;
          chklocn = LocnOf(tptr);
          /*{{{  typecheck and fold start expression*/
          t = typecheck (ReplCStartExpOf(tptr), S_INT);
          if (TagOf(t) != S_UNDECLARED)
            {
              if (!sametype(TagOf(t), S_INT))
              chkreport (CHK_INVTYPE_STARTEXP, chklocn);
              /* Fold start expression */
              SetReplCStartExp(tptr, foldexp(ReplCStartExpOf(tptr)));
            }
          /*}}}*/
          /*{{{  typecheck and fold length expression*/
          t = typecheck(ReplCLengthExpOf(tptr), S_INT);
          if (TagOf(t) != S_UNDECLARED)
            {
              if (!sametype(TagOf(t), S_INT))
                chkreport (CHK_INVTYPE_LENGTHEXP, chklocn);
          
              SetReplCLengthExp(tptr, foldexp(ReplCLengthExpOf(tptr)));
          
              if (TagOf(ReplCLengthExpOf(tptr)) == S_CONSTEXP)  /* if length is constant */
                /*{{{  check length >= 0*/
                {
                  int toosmall;
                  BIT32 *reshi = HiValAddr(ReplCLengthExpOf(tptr)),
                        *reslo = LoValAddr(ReplCLengthExpOf(tptr));
                
                  if (targetintsize != S_INT64)
                    I32ToI64 (reshi, reslo, *reslo);
                  Int64Gt (&toosmall, ZERO32, ZERO32, *reshi, *reslo);
                  if (toosmall)
                    chkreport (CHK_REPL_NEGATIVE, chklocn);
                }
                /*}}}*/
            }
          /*}}}*/
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked repl node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *ccase(tptr)                    case process*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  ccase performs semantic checking on case tree 'tptr'.
 *
 *        1. Type check the selector, and ensure that it is of a legal type.
 *
 *        2. Check that each selection is a constant expression of the same
 *           type as the selector.
 *
 *        3. Check that all selections are distinct and that there is only one
 *           ELSE.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *ccase ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *sptr;
          treenode *t;
          int type;
          int checkdistinct = TRUE;
          int ncases = 0;
          chklocn = LocnOf(tptr);
      
          /*{{{  selector*/
          t = typecheck (LHSOf(tptr), S_UNKNOWN);
          type = TagOf(t);
          /*{{{  check selector type is valid*/
          switch(type)
            {
              default:
                chkreport (CHK_INVTYPE_SELECTOR, chklocn);
                break;
              case S_BOOL: case S_BYTE:
              case S_INT: case S_INT16: case S_INT32: case S_INT64:
                break;
              case S_UNDECLARED:
                memcpy((char *)env, (char *)savedenv, sizeof(env));
                return(tptr);  /* Give up early */
            }
          /*}}}*/
          /*{{{  fold selector*/
          SetLHS(tptr, foldexp(LHSOf(tptr)));
          /*}}}*/
          /*}}}*/
      
          /*{{{  selections*/
          /*{{{  type check and constant fold selections*/
          sptr = RHSOf(tptr);
          while (!EndOfList(sptr))
            {
              treenode *selection = skipspecifications(ThisItem(sptr));
              int j;
              chklocn = LocnOf(selection);
              j = csellist(type, CondGuardOf(selection));
              if (j < 0)
                checkdistinct = FALSE;
              else
                ncases += j;
          
              sptr = NextItem(sptr);
            }
          /*}}}*/
          if (checkdistinct)
            /*{{{  check selections are distinct and there is only one ELSE*/
            {
              /* make this 1 too large in case of zero length list of selections */
              casenode_t *casetablebase =
                         (casenode_t *)memalloc(sizeof(casenode_t) * (ncases + 1));
              casenode_t *caseptr = casetablebase;
              sroot = NULL;
              elseflag = FALSE;
            
              sptr = RHSOf(tptr);
            
              /*{{{  build up the selections in the case table*/
              while (!EndOfList(sptr))
                {
                  treenode *selection = skipspecifications(ThisItem(sptr));
                  chklocn = LocnOf(selection);
                  caseptr = addcaselist(caseptr, CondGuardOf(selection));
                  sptr = NextItem(sptr);
                }
              /*}}}*/
              /*{{{  sort the case table*/
              local_qsort(casetablebase, ncases, sizeof(casenode_t),
                          (int (*)())compcases);
              /*}}}*/
              /*{{{  check all selections are distinct*/
              {
                int i;
                caseptr = casetablebase;
                for (i = 0; i < (ncases - 1); i++)
                  {
                    if (compcases(caseptr, caseptr + 1) == 0)
                      {
                        /*chklocn = LocnOf(CExpOf(caseptr->ca_decl));*/
                        chklocn = caseptr->ca_locn;
                        chkreport(CHK_MULTIPLE_CASE, chklocn /*, caseptr->ca_lo*/);
                      }
                    caseptr++;
                  }
              }
              /*}}}*/
              /* If we have an error we aren't going to reach here, so the
                 case table won't be freed up.
                 I don't worry about this, on the grounds that we're never going
                 to reach the backend anyway. */
              memfree(casetablebase);
            }
            /*}}}*/
          /*}}}*/
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked case node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *ccond(tptr)                    if and while*/
/*****************************************************************************
 *
 *  ccond performs semantic checking on guard tree 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *ccond ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          chklocn = LocnOf(tptr);

          if (!sametype(TagOf(typecheck(CondGuardOf(tptr), S_BOOL)), S_BOOL))
            chkreport (CHK_INVTYPE_GUARD, chklocn);
          SetCondGuard(tptr, foldexp(CondGuardOf(tptr)));
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked conditional node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *calt(tptr)                     alternative*/
/*****************************************************************************
 *
 *  calt performs semantic checking on alternative tree 'tptr'.
 *
 *****************************************************************************/
/* Type check the guard, make sure it is of type BOOL */
/* The input has already been checked, but we must disallow PORT inputs */
PUBLIC treenode *calt ( treenode *tptr )
{
  if (!nochecking)
    /*{{{  do the checking*/
    {
      jmp_buf savedenv;
      memcpy((char *)savedenv, (char *)env, sizeof(env));
      /*{{{  recover from errors*/
      if (setjmp (env))
        {
         memcpy((char *)env, (char *)savedenv, sizeof(env));
         return NULL;
        }
      /*}}}*/
      {
        chklocn = LocnOf(tptr);
    
        /*{{{  type check and fold the guard*/
        if (AltGuardOf(tptr) != NULL)
          {
            if (!sametype(TagOf(typecheck(AltGuardOf(tptr), S_BOOL)), S_BOOL))
              chkreport (CHK_INVTYPE_GUARD, chklocn);
            SetAltGuard(tptr, foldexp(AltGuardOf(tptr)));
          }
        /*}}}*/
        /*{{{  check that the input is not a PORT input*/
        {
          treenode *inputptr = AltInputOf(tptr);
          if ((inputptr != NULL) && (TagOf(inputptr) != S_SKIP))
            {
              if (basetype(gettype(LHSOf(inputptr))) == S_PORT)
                chkreport(CHK_ALT_INPUT_PORT, chklocn);
            }
        }
        /*}}}*/
      }
      memcpy((char *)env, (char *)savedenv, sizeof(env));
    }
    /*}}}*/

  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *cprocessor(tptr)*/
/*****************************************************************************
 *
 *  cprocessor performs semantic checking on processor statement 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *cprocessor ( treenode *tptr )
{
  jmp_buf savedenv;

  if (nochecking) return tptr;

  memcpy((char *)savedenv, (char *)env, sizeof(env));
  if (setjmp (env))
    tptr = NULL;
  else
  {
    treenode *t;
#ifdef CONFIG
    int ptype = S_NODE;
#else
    int ptype = S_INT;
#endif
    chklocn = LocnOf(tptr);
    t = typecheck (ProcessorExpOf(tptr), ptype);
    if (!sametype(TagOf(t), ptype))
      chkreport_s(CHK_INVTYPE_PROCEXP, chklocn, tagstring(ptype));
    SetProcessorExp(tptr, foldexp(ProcessorExpOf(tptr)));

#if 0  /*ifndef CONFIG*/
    ptype = typeofprocessor(WNameOf(ProcessorTypeOf(tptr)));
    if (setprocessor(WNameOf(ProcessorTypeOf(tptr))) == 0)
      {
        chkreport_s(CHK_INVPROCTYPE, chklocn,
                    WNameOf(ProcessorTypeOf(tptr)));
        tptr = NULL;
      }
#endif
    DEBUG_MSG(("Checked config node ok near line %d\n", LocnOf(tptr)));
  }

  memcpy((char *)env, (char *)savedenv, sizeof(env));
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *callocation(tptr)*/
/*****************************************************************************
 *
 *  callocation performs semantic checking on allocation tree 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *callocation ( treenode *tptr )
{
  if (!nochecking)
    {
      jmp_buf savedenv;
      memcpy((char *)savedenv, (char *)env, sizeof(env));
      /*{{{  recover from errors*/
      if (setjmp (env))
        {
         memcpy((char *)env, (char *)savedenv, sizeof(env));
         return NULL;
        }
      /*}}}*/
      chklocn = LocnOf(tptr);
      switch (TagOf(tptr))
        {
          /*{{{  S_PLACE                     break*/
          case S_PLACE:
            /* check lhs is variable, channel, port, timer, or array */
            /* check rhs is of type INT */
            {
              treenode *t;
              treenode *pname = DNameOf(tptr);
              if (TagOf(pname) != N_DECL)
                chkreport_s(CHK_BAD_PLACE_NAME, chklocn, WNameOf(NNameOf(pname)));
              if(NModeOf(pname) != NM_DEFAULT)
                chkreport_s(CHK_NAME_MULTIPLY_PLACED, chklocn,
                            WNameOf(NNameOf(pname)));
          
              t = typecheck (DValOf(tptr), S_INT);
          
              if (TagOf(t) == S_INT)
                /*{{{  fold the place expression, insert it into tree and symbol table*/
                {
                  INT32 placeaddr;
                  SetDVal(tptr, newconstexp(DValOf(tptr)));
                  placeaddr = LoValOf(DValOf(tptr));
                  SetNMode(pname, NM_PLACED);
                  SetNVOffset(pname, placeaddr);
                }
                /*}}}*/
              else if (TagOf(t) != S_UNDECLARED)
                chkreport (CHK_INVTYPE_PLACEMENT, chklocn);
            }
            break;
          /*}}}*/
          /*{{{  S_WSPLACE S_VSPLACE         break*/
          case S_WSPLACE: case S_VSPLACE:
            /* Check placed object is a variable array */
            {
              treenode *pname = DNameOf(tptr);
              /*{{{  check its a variable*/
              if (TagOf(pname) != N_DECL)
                chkreport_s(CHK_BAD_PLACE_NAME, chklocn, WNameOf(NNameOf(pname)));
              /*}}}*/
              /*{{{  check its an array if PLACEd IN*/
              if ((DValOf(tptr) == NULL) &&  (typeof(pname) != S_ARRAY))
                chkreport_s(CHK_BAD_WSPLACE, chklocn, WNameOf(NNameOf(pname)));
              /*}}}*/
              /*{{{  check it hasn't already been placed*/
              if(NModeOf(pname) != NM_DEFAULT)
                chkreport_s(CHK_NAME_MULTIPLY_PLACED, chklocn,
                            WNameOf(NNameOf(pname)));
              /*}}}*/
          
              if (TagOf(tptr) == S_WSPLACE)
                {
                  if (DValOf(tptr) != NULL)
                    /*{{{  it's at a specific position*/
                    {
                      treenode *t = typecheck (DValOf(tptr), S_INT);
                    
                      if (TagOf(t) == S_INT)
                       /*{{{  fold the place expression*/
                       {
                         INT32 placeaddr;
                         SetDVal(tptr, newconstexp(DValOf(tptr)));
                         placeaddr = LoValOf(DValOf(tptr));
                         SetNMode(pname, NM_WSPLACED);
                         SetNVOffset(pname, placeaddr);
                       }
                       /*}}}*/
                      else if (TagOf(t) != S_UNDECLARED)
                        chkreport (CHK_INVTYPE_PLACEMENT, chklocn);
                    }
                    /*}}}*/
                  else
                    SetNMode(pname, NM_WORKSPACE);
                }
              /* else TagOf(tptr) == S_VSPLACE */
            #if 1 /*SHARED_VARS*/
              /* shared variables are marked EXPERIMENTALLY by a S_VSPLACE
                with a non-NULL expression field:
                CON 5/2/91
              */
              else if (DValOf(tptr) != NULL)
                SetNVShared(pname, TRUE);
            #endif
              else
                SetNMode(pname, NM_VECSPACE);
            }
            break;
          /*}}}*/
#ifdef CONFIG
          /*{{{  S_PLACEON:*/
          case S_PLACEON:
            {
              treenode *t;
              treenode *source = DNameOf(tptr);
              while (!EndOfList(source))
                {
                  t = typecheck(ThisItem(source), S_CHAN);
                  if (!sametype(TagOf(t), S_CHAN))
                    chkreport_s(CHK_INV_MAPPING_LHS, LocnOf(tptr), tagstring(S_CHAN));
                  NewItem(foldexp(ThisItem(source)), source);
                  source = NextItem(source);
                }
              t = typecheck(DValOf(tptr), S_ARC);
              if (!sametype(TagOf(t), S_ARC))
                chkreport(CHK_INV_PLACEON_RHS, LocnOf(tptr));
              SetDVal(tptr, foldexp(DValOf(tptr)));
            }
            break;
          /*}}}*/
#endif
        }
      memcpy((char *)env, (char *)savedenv, sizeof(env));
      DEBUG_MSG(("Checked placement node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *cinstance(tptr)                procedure instance*/
/*****************************************************************************
 *
 *  cinstance performs semantic checking on instance tree 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *cinstance ( treenode *tptr )
{
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        {
          treenode *name = INameOf(tptr);
          chklocn = LocnOf(tptr);
          checkparams (tptr, S_PROC);
          SetIParamList(tptr, foldexplist(IParamListOf(tptr)));

          if ((TagOf(name) == N_PREDEFPROC) && (NModeOf(name) == PD_ASSERT))
            {
              treenode *param = ThisItem(IParamListOf(tptr));
              if (isconst(param) && (LoValOf(param) == 0))
                chkreport(CHK_INV_ASSERT, chklocn);
            }
        }
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked instance node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *cvalof(tptr, scope)            valof*/
/*****************************************************************************
 *
 *  cvalof performs semantic checking on valof tree 'tptr'.
 *
 *****************************************************************************/

/*{{{  comment*/
/* Check that the VALOF (and immediately preceding specifications)
does not contain ALT, PAR, input or output
Check that all writeable variables (lhs of assignment, rhs of
var abbreviation or retype, and var params) are within the VALOF scope.
*/
/*{{{  COMMENT */
/**********************  Start comment out ****************************
|*{{{  *|
|* VALOF scoping is enforced in a rather obscure way: when cvalof is called
all names declared within and immediately before the VALOF (ie.
writeable variables) have been
descoped.  We go through the valof looking up all writeable variables
on the scope stack: if any are found they must be out of the VALOF
scope.
*|
|*}}}*|
**********************   End comment out  ****************************/
/*}}}*/
/* When cvalof is called all names declared within and immediately before
the VALOF (ie. writeable variables) have been
descoped. We check that all variables assigned within the VALOF have
a scope greater than or equal to the current scope.
*/
/*}}}*/
PUBLIC treenode *cvalof ( treenode *tptr , BIT16 scope )
{
  if (!nochecking)
    {
      /*{{{  do the checking*/
      {
        jmp_buf savedenv;
        memcpy((char *)savedenv, (char *)env, sizeof(env));
        /*{{{  recover from errors*/
        if (setjmp (env))
          {
           memcpy((char *)env, (char *)savedenv, sizeof(env));
           return NULL;
          }
        /*}}}*/
        chklocn = LocnOf(tptr);
        checkvalof (tptr, scope, FALSE, (SOURCEPOSN)0);
        memcpy((char *)env, (char *)savedenv, sizeof(env));
      }
      /*}}}*/
      DEBUG_MSG(("Checked valof node ok near line %d\n", LocnOf(tptr)));
    }
  return (tptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *cguy_or_asm(tptr, guy_not_asm)           guycode*/
/*****************************************************************************
 *
 *  cguy_or_asm performs semantic checking on guycode tree tptr
 *
 *****************************************************************************/
PUBLIC treenode *cguy_or_asm ( treenode *tptr , int guy_not_asm )
{
  if (!nochecking)
    /*{{{  do the checking*/
    {
      jmp_buf savedenv;
      memcpy((char *)savedenv, (char *)env, sizeof(env));
      /*{{{  recover from errors*/
      if (setjmp (env))
        {
         memcpy((char *)env, (char *)savedenv, sizeof(env));
         return NULL;
        }
      /*}}}*/
      chklocn = LocnOf(tptr);
      switch (TagOf(tptr))
        {
          default:
            break;
          /*{{{  S_GUYCODE*/
          case S_GUYCODE:
            {
              treenode *operand = RightOpOf(tptr);
              int element_required, ptrs_allowed;
          
              checkinstruction(tptr, guy_not_asm, &ptrs_allowed, &element_required);
          
              while ((operand != NULL) && (TagOf(operand) == S_LIST))
                /*{{{  type check and constant fold the list of operands*/
                {
                  treenode *thisnode = ThisItem(operand);
                  treenode *thisop;
                  int addressof = TagOf(thisnode) == S_ADDRESSOF;
                  /*{{{  check for ADDRESSOF*/
                  if (addressof)
                    {
                      if (!ptrs_allowed)
                        chkreport(CHK_BAD_GUY_OPERAND, LocnOf(tptr));
                      thisop = OpOf(thisnode);
                    }
                  else
                    thisop = thisnode;
                  /*}}}*/
                  if ((thisop != NULL) && (TagOf(thisop) != N_LABELDEF))
                  {
                    int instruction = DOpTypeOf(tptr) & INST_MASK;
                    typecheck(thisop, S_UNKNOWN);
                    /*{{{  constant fold the expression*/
                    thisop = foldexp(thisop);
                    if (addressof)
                      SetOp(thisnode, thisop);  /* replace the target of addressof*/
                    else
                      NewItem(thisop, operand); /* replace the whole operand */
                    /*}}}*/
                    if (guy_not_asm)        /* do all the old style checks */
                      checkprimary(instruction, guy_not_asm, FALSE, thisop, LocnOf(tptr));
                    else
                      /*{{{  do the ASM checks*/
                      {
                        int bytes = bytesinscalar(TagOf(gettype(thisop)));
                      
                        /* Check for stores to expressions */
                        if (element_required && !iselement(thisop, FALSE))
                          chkreport(CHK_ASM_BAD_STORE, LocnOf(tptr));
                      
                        /* Operands to primaries, or of ADDRESSOF, must be constant or name*/
                        if ((DOpTypeOf(tptr) & I_PRIMARY) || addressof)
                          checkprimary(instruction, guy_not_asm, addressof, thisop, LocnOf(tptr));
                            
                        /* Check that loads and stores fit in a word */
                        if ((ptrs_allowed || element_required) &&
                            (!addressof) &&
                            ((bytes < 0) || (bytes > bytesperword)))
                          chkreport(CHK_ASM_TOO_BIG, LocnOf(tptr));
                      
                        /* Check that the operand to BYTE or WORD is constant */
                        if ((instruction == I_BYTE) || (instruction == I_WORD))
                          {
                            if (!isconst(thisop))
                              chkreport(CHK_EXP_NOT_CONST, LocnOf(tptr));
                            if (basetype(gettype(thisop)) !=
                                (instruction == I_BYTE ? S_BYTE : S_INT))
                              chkreport(CHK_TYPE_MISMATCH, LocnOf(tptr));
                          }
                      }
                      /*}}}*/
                  }
                  operand = NextItem(operand);
                }
                /*}}}*/
            }
            break;
          /*}}}*/
          /*{{{  S_GUYSTEP  -- only for GUY code*/
          case S_GUYSTEP:
            {
              int dummy;
              checkinstruction(tptr, guy_not_asm, &dummy, &dummy);
            }
            break;
          /*}}}*/
        }
      memcpy((char *)env, (char *)savedenv, sizeof(env));
    }
    /*}}}*/
  return (tptr);
}
/*}}}*/
/*}}}*/

