/****************************************************************************
 *
 *  Occam two checker          Type checking
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "extlib.h"

# include "lex1def.h" 
# include "chkerror.h" 
# include "predefhd.h"
# include "syn1def.h" 
# include "chkdef.h" 
# include "chk1def.h" 
# include "chk2def.h" 
# include "chk4def.h" 

#define TYPE_COERCION FALSE
/*}}}*/

/*{{{  global variables*/
/*PUBLIC int targetintsizeval;*/
PUBLIC SOURCEPOSN chklocn;
PUBLIC treenode *dummyexp_p;
PUBLIC treenode *undeclaredp;

PRIVATE treenode *boolnodeptr, *bytenodeptr, *intnodeptr, *int16nodeptr,
         *int32nodeptr, *int64nodeptr, /* *realnodeptr,*/
         *real32nodeptr, *real64nodeptr, /* *stringnodeptr,*/
         *channodeptr /*, *unknownnodeptr*/;

#ifdef CONFIG
PRIVATE treenode *nodenodeptr, *arcnodeptr, *edgenodeptr;
#endif

PUBLIC BIT32 max_array_size;
/* Negative sizes totally confuse the workspace allocator */
#define MAX_16BIT_ARRAY_SIZE     ((BIT32)0x7fff)
#define MAX_32BIT_ARRAY_SIZE ((BIT32)0x7fffffff)
/*}}}*/

/*{{{  PUBLIC void chkinit ()*/
PUBLIC void chkinit ( void )
{
  boolnodeptr    = newleafnode (S_BOOL,   NOPOSN);
  bytenodeptr    = newleafnode (S_BYTE,   NOPOSN);
  intnodeptr     = newleafnode (S_INT,    NOPOSN);
  int16nodeptr   = newleafnode (S_INT16,  NOPOSN);
  int32nodeptr   = newleafnode (S_INT32,  NOPOSN);
  int64nodeptr   = newleafnode (S_INT64,  NOPOSN);
  real32nodeptr  = newleafnode (S_REAL32, NOPOSN);
  real64nodeptr  = newleafnode (S_REAL64, NOPOSN);
/*stringnodeptr  = newleafnode (S_STRING, NOPOSN);*/
#ifdef CONFIG
  nodenodeptr    = newleafnode (S_NODE,   NOPOSN);
  arcnodeptr     = newleafnode (S_ARC,    NOPOSN);
  edgenodeptr    = newleafnode (S_EDGE,   NOPOSN);
#endif
/*unknownnodeptr = newleafnode (S_UNKNOWN,   NOPOSN);*/
  channodeptr    = newchannode (S_CHAN, NOPOSN, newleafnode(S_ANY, NOPOSN));

  dummyexp_p     = newleafnode (S_DUMMYEXP,  NOPOSN);
  undeclaredp    = newleafnode (S_UNDECLARED,NOPOSN);

  max_array_size = (targetintsize == S_INT16) ? MAX_16BIT_ARRAY_SIZE : MAX_32BIT_ARRAY_SIZE;
  /*
  switch (targetintsize)
    {
      case S_INT16: targetintsizeval = S_INT16VAL; break;
      case S_INT32: targetintsizeval = S_INT32VAL; break;
      case S_INT64: targetintsizeval = S_INT64VAL; break;
    }
  */
  chklocn = NOPOSN;
}
/*}}}*/

/*{{{  routines*/
/*{{{  support routines*/
/*{{{  PUBLIC int min (x, y)*/
PUBLIC INT32 min ( INT32 x , INT32 y )
{
  return (x < y ? x : y);
}
/*}}}*/
/*{{{  PUBLIC int max (x, y)*/
PUBLIC INT32 max ( INT32 x , INT32 y )
{
  return (x > y ? x : y);
}
/*}}}*/
/*{{{  PUBLIC int isint (t)*/
PUBLIC int isint ( int t )
{
  return (t == S_INT || t == S_INT16 || t == S_INT32 || t == S_INT64);
}
/*}}}*/
/*{{{  PUBLIC int isintorbyte (t)*/
PUBLIC int isintorbyte ( int t )
{
  return (t==S_INT || t==S_INT16 || t==S_INT32 || t==S_INT64 ||t==S_BYTE);
}
/*}}}*/
/*{{{  PUBLIC int isreal (t)*/
PUBLIC int isreal ( int t )
{
  return (t == S_REAL32 || t == S_REAL64);
}
/*}}}*/
/*{{{  PUBLIC int isscalartype (t)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  isscalartype returns TRUE if t is a scalar type, FALSE otherwise.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC int isscalartype ( int t )
{
  switch(t)
    /*{{{  cases*/
    {
      default:        return(FALSE);
      case S_BOOL:   case S_BYTE:
      case S_INT:    case S_INT16:  case S_INT32: case S_INT64:
      case S_REAL32: case S_REAL64:
      case S_CHAN:   case S_PORT:
      CASE_CONFIG_TYPE
      case S_TIMER:   return(TRUE);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int littag (t)*/
PUBLIC int littag ( int t )
{
  switch (t)
    /*{{{  cases*/
    {
      case S_BYTE:   return (S_BYTELIT);
      case S_INT:    return (S_INTLIT);
      case S_INT16:  return (S_INT16LIT);
      case S_INT32:  return (S_INT32LIT);
      case S_INT64:  return (S_INT64LIT);
      case S_REAL32: return (S_REAL32LIT);
      case S_REAL64: return (S_REAL64LIT);
      default:       return (S_UNKNOWN);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE treenode *litnodeptr (t)*/
PRIVATE treenode *litnodeptr ( int t )
{
  switch (t)
    /*{{{  cases*/
    {
      case S_TRUE:
      case S_FALSE:     return boolnodeptr;
      case S_BYTELIT:   return bytenodeptr;
      case S_INTLIT:    return intnodeptr;
      case S_INT16LIT:  return int16nodeptr;
      case S_INT32LIT:  return int32nodeptr;
      case S_INT64LIT:  return int64nodeptr;
      case S_REAL32LIT: return real32nodeptr;
      case S_REAL64LIT: return real64nodeptr;
      default:          return NULL;
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC treenode *typenodeptr (t)*/
PUBLIC treenode *typenodeptr ( int t )
{
  switch (t)
    /*{{{  cases*/
    {
      case S_BOOL:       return boolnodeptr;
      case S_BYTE:       return bytenodeptr;
      case S_INT:        return intnodeptr;
      case S_INT16:      return int16nodeptr;
      case S_INT32:      return int32nodeptr;
      case S_INT64:      return int64nodeptr;
      case S_REAL32:     return real32nodeptr;
      case S_REAL64:     return real64nodeptr;
      case S_CHAN:       return channodeptr;
      case S_UNDECLARED: return undeclaredp;
    #ifdef CONFIG
      case S_NODE:       return nodenodeptr;
      case S_ARC:        return arcnodeptr;
      case S_EDGE:       return edgenodeptr;
    #endif
      default:           return NULL;
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int bytesinscalar(t)*/
PUBLIC int bytesinscalar ( int t )
{
  switch (t)
    /*{{{  cases*/
    {
      case S_BOOL:
      case S_BYTE:   return 1;
      case S_INT16:  return 2;
      case S_INT32:
      case S_REAL32: return 4;
      case S_INT64:
      case S_REAL64: return 8;
      case S_INT:
      case S_CHAN:   return bytesinscalar(targetintsize);
      case S_PORT:   chkreport_i(CHK_UNKNOWN_TYPE, chklocn, t); return(-1);
      case S_TIMER:  return 0;  /* Added 23/4/90 by CO'N for bug 287 */
      default:       return(-1);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC INT32 bytesin (t)*/
PUBLIC INT32 bytesin ( treenode *t )
{
  switch (TagOf(t))
    /*{{{  cases*/
    {
      default:       return bytesinscalar(TagOf(t));
      case S_PORT:   return bytesin(ProtocolOf(t));
      case S_ARRAY:
        {
          INT32 b, d = ARDimOf(t);
          if (d == (-1))
            return (-1);
          b = bytesin(ARTypeOf(t));
          if (b == (-1))
            return (-1);
          if ((d != 0) && ((BIT32)b > (max_array_size / (BIT32)d)))
            chkreport(CHK_ARRAY_SIZE_OVERFLOW, LocnOf(t));
            /*chkreport(CHK_ARRAY_SIZE_OVERFLOW, chklocn, ZERO32);*/
          return ((INT32)((BIT32)d * (BIT32)b));
        }
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC BIT32 wordsin (tptr)*/
PUBLIC BIT32 wordsin ( treenode *tptr )
{
  INT32 b = bytesin(tptr);
  if (b <= 0)
    return(0);
  else
    return (((b - 1) / bytesperword) + 1);       /* round up to nearest word */
}
/*}}}*/
/*{{{  PUBLIC INT32 elementsin (t)*/
PUBLIC INT32 elementsin ( treenode *t )
{
  switch(TagOf(t))
    {
      case S_ARRAY:
        {
          INT32 e = elementsin(ARTypeOf(t));
          return ((ARDimOf(t) == (-1)) || (e == (-1))) ? (-1) : (ARDimOf(t) * e);
        }
      case S_PORT:
        return elementsin(ProtocolOf(t));
      default:
        return 1;
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *basetype_tree (tptr)*/
PUBLIC treenode *basetype_tree ( treenode *tptr )
{
  while (TRUE)
    {
      if (TagOf(tptr) == S_ARRAY)
        tptr = ARTypeOf(tptr);
      else if (TagOf(tptr) == S_PORT)
        tptr = ProtocolOf(tptr);
      else
        return tptr;
    }
}
/*}}}*/
/*{{{  PUBLIC int basetype (tptr)*/
PUBLIC int basetype ( treenode *tptr )
{
  return TagOf(basetype_tree(tptr));
}
/*}}}*/
/*{{{  PUBLIC int isspecification (tptr)*/
PUBLIC int isspecification ( treenode *tptr )
{
  return ((tptr != NULL) && (nodetypeoftag(TagOf(tptr)) == DECLNODE));
}
/*}}}*/
/*{{{  PUBLIC treenode *nameof(tptr)*/
/*****************************************************************************
 *
 *  return the underlying name of element 'tptr'
 *
 *****************************************************************************/
PUBLIC treenode *nameof ( treenode *tptr )
{
  while (TRUE)
    switch (nodetypeoftag(TagOf(tptr)))
      {
        case ARRAYSUBNODE: tptr = ASBaseOf(tptr);  break;
        case SEGMENTNODE:  tptr = SNameOf(tptr);   break;
        default:           return(tptr);
      }
}
/*}}}*/
/*{{{  PUBLIC treenode *basedecl(tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  basedecl returns the base declaration of the element 'tptr',
 *           ie. the subscripted array, the segmented array, or the
 *           abbreviated variable.  It cannot, however, dealias through
 *           formal parameters.
 *           Value abbreviations and retypes are not dealiased, as that
 *           might take us through to an expression.
 *           So the declaration returned is an
 *           N_DECL, N_REPL, N_PARAM, N_VALPARAM, N_VALABBR, N_VALRETYPE
 *           namenode, or a S_FNFORMALRESULT hidden parameter node.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *basedecl ( treenode *tptr )
{
  while (TRUE)
    switch (TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  ARRAYSUB ARRAYITEM                   move to the subscripted array*/
        case S_ARRAYSUB: case S_RECORDSUB:
        case S_ARRAYITEM:
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        /*{{{  SEGMENT  SEGMENTITEM                 move to the segmented array*/
        case S_SEGMENT:
        case S_SEGMENTITEM:
          tptr = SNameOf(tptr);
          break;
        /*}}}*/
        /*{{{  ABBR RETYPE                          move to the aliased variable*/
        case N_ABBR:
        case N_RETYPE:
          /* If name is a channel constructor, return name */
          /* This fixes bug 273, but would be unnecessary if we didn't call
             basedecl from transformalt */
          if (TagOf(DValOf(NDeclOf(tptr))) == S_CONSTRUCTOR)
            return (tptr);  /* can only be a channel constructor */
          tptr = DValOf(NDeclOf(tptr));
          break;
        /*}}}*/
        /*{{{  DECL REPL PARAM VALPARAM FNFORMALRESULT VALABBR VALRETYPE*/
        case N_DECL: case N_REPL:
        case N_PARAM: case N_VALPARAM:
        case N_VALABBR: case N_VALRETYPE: case N_FIELD:
        case T_TEMP: case T_PREEVALTEMP:
        case S_FNFORMALRESULT:
          return (tptr);
        /*}}}*/
        default:                 /* for safety's sake */
          badtag(LocnOf(tptr), (BIT32)TagOf(tptr), "basedecl");
      }
      /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int sametype(tag1, tag2)*/
/*****************************************************************************
 *
 *  sametype returns TRUE if the type tags 'tag1' and 'tag2' match: it
 *           matches an undeclared type with anything.
 *
 *****************************************************************************/
PUBLIC int sametype ( int tag1 , int tag2 )
  {
    return(tag1 == tag2 || tag1 == S_UNDECLARED || tag2 == S_UNDECLARED);
  }
/*}}}*/
/*{{{  PUBLIC treenode *skipspecifications(tptr)*/
/*****************************************************************************
 *
 *  skipspecifications takes a tree and walks down from the root until
 *                     it finds a node which isn't a specification, and
 *                     returns a pointer to this node.
 *
 *****************************************************************************/
PUBLIC treenode *skipspecifications ( treenode *tptr )
  {
    while (isspecification(tptr))
      tptr = DBodyOf(tptr);
    return (tptr);
  }
/*}}}*/
/*{{{  PUBLIC int separatelycompiled (nptr)*/
/*****************************************************************************
 *
 *  separatelycompiled returns TRUE if the namenode nptr represents an
 *                     item which has been separately compiled.
 *
 *****************************************************************************/
PUBLIC int separatelycompiled ( treenode *nptr )
{
  switch(TagOf(nptr))
    {
      case     N_SCPROCDEF: case     N_SCFUNCDEF:
      case    N_LIBPROCDEF: case    N_LIBFUNCDEF:
      case N_STDLIBPROCDEF: case N_STDLIBFUNCDEF:   return TRUE;
      default:                                      return FALSE;
    }
}
/*}}}*/
/*{{{  PUBLIC int issimple (tptr)*/
/*****************************************************************************
 *
 *  int issimple returns TRUE if the element tptr is simple, ie. not
 *               subscripted or segmented.
 *
 *****************************************************************************/
PUBLIC int issimple ( treenode *tptr )
{
  switch (TagOf(tptr))
    /*{{{  cases*/
    {
      case N_VALABBR: case N_ABBR:
      case N_VALRETYPE: case N_RETYPE:
      case N_VALPARAM: case N_PARAM:
      case N_DECL: case N_REPL:
      case T_TEMP: case T_PREEVALTEMP:
      /* A function result pointer is effectively an abbreviation */
      case S_FNFORMALRESULT:
        return (TRUE);
      default:
        return (FALSE);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int inline (nptr)*/
/*****************************************************************************
 *
 * inline returns TRUE if the routine 'nptr' is to be expanded inline
 *
 *****************************************************************************/
PUBLIC int inline ( treenode *nptr )
{
  int ntag = TagOf(nptr);
  return ntag == N_INLINEFUNCDEF || ntag == N_INLINEPROCDEF;
}
/*}}}*/
/*{{{  PUBLIC treenode *dimexpof(tptr, dimension)*/
/*****************************************************************************
 *
 *  dimexpof returns the dimension tree for the dimension'th dimension
 *           of element tptr.
 *           The first dimension is dimension 0.
 *
 *****************************************************************************/
PUBLIC treenode *dimexpof ( treenode *tptr , int dimension )
{
  while (TRUE)
    switch (TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  ARRAYSUB ARRAYITEM       break*/
        case S_ARRAYITEM:
        case S_ARRAYSUB:
          dimension++;
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        /*{{{  SEGMENT SEGMENTITEM      break / return*/
        case S_SEGMENT:
        case S_SEGMENTITEM:
          /* The segment length may already have been evaluated into a temp,
             in which case dimexp will be a temp node */
          if (dimension == 0)
            return SLengthExpOf(tptr);
          else
            tptr = SNameOf(tptr);
          break;
        /*}}}*/
        /*{{{  CONSTCONSTRUCTOR         break*/
        case S_CONSTCONSTRUCTOR:
          tptr = CTExpOf(tptr);
          break;
        /*}}}*/
        /*{{{  CONSTRUCTOR              break / return*/
        case S_CONSTRUCTOR:
          if (dimension > 0)
            {
              tptr = ThisItem(OpOf(tptr));
              dimension--;
            }
          else
            return(newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32,
                                                  listitems(OpOf(tptr))));
          break;
        /*}}}*/
        /*{{{  name                             return*/
        case N_VALABBR: case N_ABBR:
        case N_VALRETYPE: case N_RETYPE:
        case N_DECL:
        case N_VALPARAM: case N_PARAM:
        case T_TEMP: case T_PREEVALTEMP:
        case S_RECORDSUB:
          tptr = NTypeOf((TagOf(tptr) == S_RECORDSUB) ? ASIndexOf(tptr) : tptr);
          while (dimension > 0)
            {
              tptr = ARTypeOf(tptr);
              dimension--;
            }
          return(ARDimLengthOf(tptr));
        /*}}}*/
        /*{{{  string*/
        case S_STRING:
          return(newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32,
                                 WLengthOf(CTValOf(tptr))));
        /*}}}*/
        default:
          badtag(chklocn, (BIT32)TagOf(tptr), "dimexpof");
      }
      /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC treenode *skipevals (treenode *tptr)*/
/*****************************************************************************
 *
 *  skipevals skips any leading 'eval' nodes on tree 'tptr'.
 *
 *****************************************************************************/
PUBLIC treenode *skipevals ( treenode *tptr )
{
  while (TagOf(tptr) == S_EVAL) tptr = RightOpOf(tptr);
  return tptr;
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE treenode *arraytype (tptr, subscripts, str)*/
/* Name of the element we are subscripting. */
/* Move down array type tree 'tptr' 'subscripts' times.  If there aren't enough
dimensions in the array report error.
Return a tree representing the type subscripted 'subscripts' times.
*/
PRIVATE treenode *arraytype ( treenode *tptr , int subscripts , char *str )
{
  while (subscripts > 0)
    { 
      if (TagOf(tptr) == S_ARRAY)
        {
          subscripts--;
          tptr = ARTypeOf(tptr);
        }
      else
        chkreport_s(CHK_INV_SUBSCRIPT, chklocn, str);
    }
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE int protocolsequal (p1, p2)*/
/*
This was identical to protocolsequivalent, except for the treatment
of ANY. This has now been merged, parameterising protocolsequivalent.
CON 1/2/91
*/
/*}}}*/
/*{{{  PRIVATE int protocolsequivalent (fprot, aprot)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  protocolsequivalent checks that the two protocols 'fprot' and 'aprot' are
 *                      equivalent. 'fprot' is the formal protcol (ie. protocol
 *                      of formal parameter, or lhs of abbreviation) , and
 *                      'aprot' is the actual protocol.
 *                      We enforce name equivalence rather than structural
 *                      equivalence.
 *                      if 'match_any' is TRUE,
 *                      an actual ANY protocol will match any formal protocol.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE int protocolsequivalent (treenode *fprot, treenode *aprot, const int match_any)
{
  if (match_any && TagOf(aprot) == S_ANY) return TRUE;

  while (TagOf(fprot) == TagOf(aprot))
    switch (TagOf(fprot))
      /*{{{  cases*/
      {
        /*{{{  anarchic protocol*/
        case S_ANY: return TRUE;
        /*}}}*/
        /*{{{  named protocol*/
        case N_TPROTDEF: case N_SPROTDEF:
        case N_DECL: /* undeclared protocol name */
          return (fprot == aprot);
        /*}}}*/
        /*{{{  simple protocol*/
        case S_BOOL:   case S_BYTE:
        case S_INT:    case S_INT16:  case S_INT32: case S_INT64:
        case S_REAL32: case S_REAL64:
          return TRUE;
        case S_COLON2:
          /*{{{  if left tags match, move pointers to right-hand nodes*/
          if (TagOf(LeftOpOf(fprot)) == TagOf(LeftOpOf(aprot)))
            {
              fprot = RightOpOf(fprot);
              aprot = RightOpOf(aprot);
            }
          else
            return FALSE;
          break;
          /*}}}*/
        /*{{{  case S_ARRAY*/
        case S_ARRAY:
          {
            /* Check dimension lengths are the same */
            BIT32 l1 = ARDimOf(fprot), l2 = ARDimOf(aprot);
        
            if ((l1 == l2) || (l1 == -1) || (l2 == -1))
              {
                fprot = ARTypeOf(fprot); aprot = ARTypeOf(aprot);
              }
            else
              return FALSE;
            break;
          }
        /*}}}*/
        /*}}}*/
        default:
          chkreport_i(CHK_UNKNOWN_TYPE, chklocn, TagOf(fprot));
      }
      /*}}}*/
  return FALSE;
}
/*}}}*/
/*{{{  PUBLIC int typesequal (t1, t2)*/
/*
This was identical to typesequivalent, except for the way it treated
CHAN OF ANY protocols. This has now been parameterised.
CON 1/2/91
*/
/*}}}*/
/*{{{  PUBLIC int typesequivalent (atype, ftype)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  typesequivalent checks that the two types 'ftype' and 'atype' are
 *                  equivalent. 'ftype' is normally the formal type (ie. type
 *                  of formal parameter, or lhs of abbreviation) , and 'atype'
 *                  is the actual type, but this routine is also called
 *                  to match assignments, etc. (was 'typesequal').
 *                  match_any will be TRUE if processing an argument list or
 *                  abbreviation, or FALSE otherwise.
 *                  Return TRUE if they are equivalent, FALSE otherwise.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC int typesequivalent (treenode *atype, treenode *ftype, const int match_any)
{
  if (TagOf(atype) ==  S_UNDECLARED || TagOf(ftype) == S_UNDECLARED)
    return TRUE;
  while (TRUE)
    {
      if (TagOf(atype) == TagOf(ftype))
        switch (TagOf(atype))
          /*{{{  cases*/
          {
            /*{{{  primitive types*/
            case S_BOOL: case S_BYTE:
            case S_INT: case S_INT16: case S_INT32: case S_INT64: case S_UINTLIT:
            case S_REAL32: case S_REAL64: case S_UREALLIT:
            case S_TIMER:
            CASE_CONFIG_TYPE
              return (TRUE);
            /*}}}*/
            /*{{{  chan and port*/
            case S_CHAN: case S_PORT:
              return protocolsequivalent(ProtocolOf(atype), ProtocolOf(ftype), match_any);
            /*}}}*/
            /*{{{  S_ARRAY*/
            case S_ARRAY:
              {
                /* Check dimensions are the same */
                const BIT32 l1 = ARDimOf(atype);
                const BIT32 l2 = ARDimOf(ftype);
                if ((l1 == l2) || (l1 == -1) || (l2 == -1))
                  {
                    atype = ARTypeOf(atype); ftype = ARTypeOf(ftype);
                  }
                else
                  return FALSE;
                break;
              }
            /*}}}*/
            default:
              chkreport_i(CHK_UNKNOWN_TYPE, chklocn, TagOf(atype));
          }
          /*}}}*/
      else
        /*{{{  check for wild-card types - S_UINTLIT, S_UREALLIT*/
        {
          const int tag1 = TagOf(atype);
          const int tag2 = TagOf(ftype);
          return (((tag1 == S_UINTLIT)  && isint(tag2))  ||
                  ((tag1 == S_UREALLIT) && isreal(tag2)) ||
                  ((tag2 == S_UINTLIT)  && isint(tag1))  ||
                  ((tag2 == S_UREALLIT) && isreal(tag1)));
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC int spec_into_type (specptr, typeptr)*/
/*{{{  comment*/
/* Check that the types represented by the two trees specptr and typeptr are
the same.  If there are any missing array dimensions in the specifier tree,
copy them from the type tree.
The type tree may contain wildcard types (integer or real literals) - we
must check that the specifier tree fully defines these.

Return TRUE if the trees represent equivalent types.
*/
/*}}}*/
PUBLIC int spec_into_type ( treenode *specptr , treenode *typeptr )
{
  while (TRUE)
    /*{{{  loop body*/
    {
      int spectag = TagOf(specptr), typetag = TagOf(typeptr);
      if (spectag == typetag)
        /*{{{  switch on the tag*/
        switch (spectag)
          /*{{{  cases*/
          {
            /*{{{  primitive types S_UNDECLARED*/
            case S_BOOL: case S_BYTE:
            case S_INT: case S_INT16: case S_INT32: case S_INT64: case S_REAL32:
            case S_REAL64: case S_TIMER:
            case S_UNDECLARED:
            CASE_CONFIG_TYPE
              return (TRUE);
            /*}}}*/
            /*{{{  chan*/
            case S_CHAN: case S_PORT:
              return protocolsequivalent(ProtocolOf(specptr), ProtocolOf(typeptr), TRUE);
            /*}}}*/
            /*{{{  S_ARRAY*/
            case S_ARRAY:
              if (ARDimLengthOf(specptr) == NULL)
                /*{{{  fill in array dimension in the specifier*/
                {
                  treenode *lengthexp = ARDimLengthOf(typeptr);
                  BIT32 dim = ARDimOf(typeptr);
                  if ((lengthexp != NULL) && (TagOf(lengthexp) == S_CONSTEXP))
                    lengthexp = newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, dim);
                #if 0
                  else
                    lengthexp = copytree(lengthexp);
                  /* bug 878 - This should theoretically be inserted to ensure
                     that the tree remains a tree.
                     However, this might break the backend, so I'm currently
                     not enabling it.
                     CON - 28/1/91
                  */
                #endif
                  SetARDim(specptr, dim);
                  SetARDimLength(specptr, lengthexp);
                }
                /*}}}*/
              else
                /*{{{  check array lengths are the same*/
                if (ARDimLengthOf(typeptr) != NULL && ARDimOf(specptr) != ARDimOf(typeptr))
                  return (FALSE);
                /*}}}*/
            
              specptr = ARTypeOf(specptr); typeptr = ARTypeOf(typeptr);
              break;
            /*}}}*/
            default:
              chkreport(CHK_INVTYPE_SPEC, chklocn);
          }
          /*}}}*/
        /*}}}*/
      else
        /*{{{  check wildcard types*/
        /* We might still have a wildcard type in the type tree, if so
           we must have a proper type in the specifier in order to resolve it.
        */
        if (((typetag == S_UINTLIT)  && isint(spectag)) ||
            ((typetag == S_UREALLIT) && isreal(spectag)))
          return (TRUE);
        /*}}}*/
      else
        /*{{{  allow through UNDECLARED types*/
        if ((spectag == S_UNDECLARED) || (typetag == S_UNDECLARED))
          return(TRUE);
        /*}}}*/
      else
        return(FALSE);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE treenode *checkdop (tptr, type)*/
PRIVATE treenode *checkdop ( treenode *tptr , int type )
/* This returns a tree in the 'current' workspace (normally temp) */
{
  char buf[MAX_ERR_SIZE];
  ftagstring(buf, TagOf(tptr));
  return checksame(LeftOpOf(tptr), RightOpOf(tptr),
                    type, CHK_TYPES_DIFF, buf);
}
/*}}}*/
/*{{{  PRIVATE int typeknown (tptr)*/
PRIVATE int typeknown ( treenode *tptr )
{
  if (tptr == NULL)
    return FALSE; /* bug 1120 25/1/91 */
  while (TRUE)
    switch (TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  NEG BITNOT UMINUS*/
        case S_NEG: case S_BITNOT: case S_UMINUS:
          tptr = OpOf(tptr);
          break;
        /*}}}*/
        /*{{{  dyadic op*/
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_LSHIFT: case S_RSHIFT:
        case S_PLUS: case S_MINUS: case S_TIMES:
          if (typeknown(LeftOpOf(tptr)))
            return (TRUE);
          else
            tptr = RightOpOf(tptr);
          break;
        /*}}}*/
        /*{{{  ARRAYSUB*/
        case S_ARRAYSUB:
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        /*{{{  VALOF*/
        case S_VALOF:
          tptr = VLResultListOf(tptr);
          /* If it's a multi-valued valof we must know its type from the
             element on the lhs of the assignment. */
          if (listitems(tptr) > 1)
            return(TRUE);
          else
            tptr = ThisItem(tptr);
          break;
        /*}}}*/
        /*{{{  SEGMENT*/
        case S_SEGMENT:
          tptr = SNameOf(tptr);
          break;
        /*}}}*/
        /*{{{  CONSTRUCTOR*/
        case S_CONSTRUCTOR:
          /* Just test the first element, all the others must be of the same
             type */
          tptr = ThisItem(OpOf(tptr));
          break;
        /*}}}*/
        #ifdef ARRAYCONSTRUCTOR
        /*{{{  ARRAYCONSTRUCTOR*/
        case S_ARRAYCONSTRUCTOR:
          /* Just test the first element, all the others must be of the same
             type */
          tptr = ACValExpOf(tptr);
          break;
        /*}}}*/
        #endif
        /*{{{  UINTLIT UREALLIT*/
        case S_UINTLIT:
        case S_UREALLIT:
          return (FALSE);
        /*}}}*/
        default:
          return (TRUE);
      }
      /*}}}*/
}
/*}}}*/
/*{{{  type checking of expressions*/
/*{{{  PRIVATE treenode *typecheck_main (tptr, type)*/
/*{{{  comment*/
/* Check that the types of expression operands are consistent.
Place type of operands in field of operator nodes.
tptr points to the root of the expression tree.
The value returned represents the type of the expression tree.
Insert the types of untyped literals.
*/
/* N.B. This function returns a pointer to a type tree which already
exists, so the return value should only be used for  comparsion and
not for inserting elsewhere on the tree.
To insert a type tree elsewhere, first copy it using function
'copytree'.
*/
/* type is a type tag used to resolve the type of
unttyped integer and real literals where possible.
N.B. TYPE COERCION IS CURRENTLY DISABLED
If it takes an incompatible type an error will
be reported.
If type is UNKNOWN and we find an untyped integer literal we assume the type
INT.
If type is UNKNOWN and we find an untyped real literal we report an error.
*/
/*}}}*/
PRIVATE treenode *typecheck_main ( treenode *tptr , int type )
{
  treenode *t;                                /* Type of the expression tree */

  if (tptr == NULL)
    return undeclaredp; /* bug 838 20/12/90 */

  switch (TagOf(tptr))
    {
      /*{{{   node type*/
      /*{{{  monadic operators*/
      /*{{{  case S_NEG*/
      case S_NEG:
        t = typecheck_main (OpOf(tptr), type);
        switch (TagOf(t))
          {
            case S_INT:    case S_INT16: case S_INT32: case S_INT64:
            case S_REAL32: case S_REAL64:
            case S_UNDECLARED:
              SetMOpType(tptr, TagOf(t));
              break;
            default:
              chk_invtype (chklocn, S_NEG);
          }
        return (t);
      /*}}}*/
      /*{{{  case S_BITNOT S_UMINUS*/
      case S_BITNOT: case S_UMINUS:
        t = typecheck_main (OpOf(tptr), type);
        switch (TagOf(t))
          {
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_UNDECLARED:
              SetMOpType(tptr, TagOf(t));
              break;
            default:
              chk_invtype (chklocn, TagOf(tptr));
          }
        return (t);
      /*}}}*/
      /*{{{  case S_NOT*/
      case S_NOT:
        t = typecheck_main (OpOf(tptr), type);
        if (TagOf(t) == S_BOOL)
          SetMOpType(tptr, S_BOOL);
        else if (TagOf(t) == S_UNDECLARED)
          SetMOpType(tptr, S_UNDECLARED);
        else
          chk_invtype (chklocn, S_NOT);
        return (t);
      /*}}}*/
      /*{{{  case S_SIZE*/
      case S_SIZE:
        {
          treenode *t = typecheck_main (OpOf(tptr), S_UNKNOWN);
          if (TagOf(t) == S_ARRAY)
            /* The operand must be an array */
            {
              SetMOpType(tptr, S_ARRAY);
              return (intnodeptr);
            }
          else if (TagOf(t) == S_UNDECLARED)
            {
              SetMOpType(tptr, S_UNDECLARED);
              return (intnodeptr);
            }
          else
            chk_invtype (chklocn, S_SIZE);
        }
      /*}}}*/
      /*}}}*/
      /*{{{  arithmetic operators*/
      /*{{{  case S_ADD S_SUBTRACT S_MULT S_DIV S_REM*/
      case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        t = checkdop (tptr, type);
        switch (TagOf(t))
          {
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_REAL32: case S_REAL64:
            case S_UREALLIT: case S_UINTLIT:
            case S_UNDECLARED:
              SetDOpType(tptr,TagOf(t));
              break;
            default:
              chk_invtype (chklocn, TagOf(tptr));
          }
        return (t);
      /*}}}*/
      /*}}}*/
      /*{{{  bitwise operators*/
      /*{{{  case S_BITAND S_BITOR S_XOR and modulo operators*/
      case S_BITAND: case S_BITOR: case S_XOR:
      case S_PLUS: case S_MINUS: case S_TIMES:
        t = checkdop (tptr, type);
        switch (TagOf(t))
          {
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_UINTLIT:
            case S_UNDECLARED:
              SetDOpType(tptr, TagOf(t));
              break;
            default:
              chk_invtype (chklocn, TagOf(tptr));
          }
        return (t);
      /*}}}*/
      /*{{{  case S_LSHIFT S_RSHIFT*/
      case S_LSHIFT: case S_RSHIFT:
        {
          treenode *rt = typecheck_main(RightOpOf(tptr), S_INT);
          t = typecheck_main (LeftOpOf(tptr), type);
          if (TagOf(rt) == S_INT)
            /*{{{  check type of operand*/
            switch (TagOf(t))
              {
                case S_INT: case S_INT16: case S_INT32: case S_INT64:
                case S_UINTLIT:
                case S_UNDECLARED:
                  SetDOpType(tptr, TagOf(t));
                  break;
                default:
                  chk_invtype (chklocn, TagOf(tptr));
              }
            /*}}}*/
          else
            chkreport (CHK_INVCOUNT, chklocn);
          return (t);
        }
      /*}}}*/
      /*}}}*/
      /*{{{  logical operators*/
      /*{{{  case S_AND S_OR*/
      case S_AND: case S_OR:
        t = checkdop (tptr, type);
        if (TagOf(t) == S_UNDECLARED)
          t = boolnodeptr;
        if (TagOf(t) == S_BOOL)
          SetDOpType(tptr, S_BOOL);
        else
          chk_invtype (chklocn, TagOf(tptr));
        return (t);
      /*}}}*/
      /*}}}*/
      /*{{{  relational operators*/
      /*{{{  case S_EQ S_NE S_LS S_LE S_GR S_GE*/
      case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE:
        t = checkdop (tptr, S_UNKNOWN);
        /*{{{  check type is correct*/
        switch (TagOf(t))
          {
            case S_BOOL:
              switch (TagOf(tptr))
                {
                  case S_EQ: case S_NE:
                    SetDOpType(tptr, TagOf(t));
                    break;
                  default:
                    chk_invtype (chklocn, TagOf(tptr));
                }
              break;
            case S_BYTE: case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_REAL32: case S_REAL64:
            case S_UNDECLARED:
              SetDOpType(tptr, TagOf(t));
              break;
            default:
              chk_invtype (chklocn, TagOf(tptr));
          }
        /*}}}*/
        return (boolnodeptr);
      /*}}}*/
      /*{{{  case S_AFTER*/
      case S_AFTER:
        t = checkdop (tptr, S_UNKNOWN);
        switch (TagOf(t))
          {
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_UNDECLARED:
              SetDOpType(tptr, TagOf(t));
              break;
            default:
              chk_invtype (chklocn, S_AFTER);
          }
        return (boolnodeptr);
      /*}}}*/
      /*}}}*/
      /*{{{  mostpos, mostneg*/
      /*{{{  case S_MOSTPOS S_MOSTNEG*/
      case S_MOSTPOS: case S_MOSTNEG:
        t = OpOf(tptr);
        switch (TagOf(t))
          {
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
              SetMOpType(tptr, TagOf(t));
              return (t);
            default:
              chk_invtype (chklocn, TagOf(tptr));
          }
      /*}}}*/
      /*}}}*/
      /*{{{  conversions*/
      /*{{{  case S_EXACT*/
      case S_EXACT:
        {
          int ct = MOpTypeOf(tptr);             /* The type to convert to */
          int tt;                               /* The source type */
          treenode *converttype = typenodeptr(ct);
      
          t = typecheck_main (OpOf(tptr), S_UNKNOWN);
          tt = TagOf(t);
      
          if (tt == S_UNDECLARED)
            return(converttype);
          if (((tt == S_BOOL) || (tt == S_BYTE) || isint (tt)) &&
              ((ct == S_BOOL) || (ct == S_BYTE) || isint (ct)))
            return (converttype);
          if ((tt == S_REAL32) && isreal(ct))
            return (converttype);
          if ((tt == S_REAL64) && (ct == S_REAL64))
            return (converttype);
          {
            char type[MAX_ERR_SIZE];
            ftagstring(type, ct);
            chkreport_s(CHK_INV_EXACT, chklocn, type);
          }
        }
      /*}}}*/
      /*{{{  case S_ROUND, case S_TRUNC*/
      case S_ROUND: case S_TRUNC:
        {
          int ct = MOpTypeOf(tptr);             /* The type to convert to */
          int tt;                               /* The source type */
          treenode *converttype = typenodeptr(ct);
      
          t = typecheck_main (OpOf(tptr), S_UNKNOWN);
          tt = TagOf(t);
          if (tt == S_UNDECLARED)
            return(converttype);
          if ( (isint (tt) && isreal (ct)) ||
               (isreal (tt) && isint (ct)) ||
               (isreal (tt) && isreal (ct)) )
            return (converttype);
          chkreport(CHK_INV_ROUND_TRUNC, chklocn);
        }
      /*}}}*/
      /*}}}*/
      /*{{{        literals*/
      case S_TRUE:      case S_FALSE:    case S_BYTELIT:
      case S_INTLIT:    case S_INT16LIT: case S_INT32LIT: case S_INT64LIT:
      case S_REAL32LIT: case S_REAL64LIT:
        return(litnodeptr(TagOf(tptr)));
      case S_ASMNAME:
        return intnodeptr;
      /*{{{  S_STRING*/
      case S_STRING:
        {
          BIT32 dim = WLengthOf(CTValOf(tptr));
          treenode *aptr = newarraynode(S_ARRAY, 0,
                             newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, dim),
                                         bytenodeptr);
          SetARDim(aptr, dim);
          return (aptr);
        }
      /*}}}*/
      /*{{{  S_UBYTELIT*/
      case S_UBYTELIT:
      #if TYPE_COERCION
        if (isintorbyte(type))
          SetTag(tptr, littag(type));
        else
          if (type == S_UNKNOWN)
            SetTag(tptr, S_BYTELIT);
          else
            chkreport (CHK_TYPE_MISMATCH, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #else
        if ((type == S_UNKNOWN) || (type == S_BYTE))
            SetTag(tptr, S_BYTELIT);
        else
          chkreport (CHK_TYPE_MISMATCH, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #endif
      /*}}}*/
      /*{{{  S_UINTLIT*/
      case S_UINTLIT:
      #if TYPE_COERCION
        if (isintorbyte(type))
          SetTag(tptr, littag(type));
        else
          if (type == S_UNKNOWN)
            SetTag(tptr, S_INTLIT);
          else
            chkreport (CHK_TYPE_MISMATCH, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #else
        if ((type == S_UNKNOWN) || (type == S_INT))
          SetTag(tptr, S_INTLIT);
        else
          chkreport (CHK_TYPE_MISMATCH, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #endif
      /*}}}*/
      /*{{{  S_UREALLIT*/
      case S_UREALLIT:
      #if TYPE_COERCION
        if (isreal(type))
          SetTag(tptr, littag(type));
        else
          if (type == S_UNKNOWN)
            chkreport (CHK_UNRESOLVED_REALTYPE, chklocn);
          else
            chkreport (CHK_TYPE_MISMATCH, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #else
        chkreport (CHK_UNRESOLVED_REALTYPE, chklocn);
        return (litnodeptr(TagOf(tptr)));
      #endif
      /*}}}*/
      /*}}}*/
      /*{{{  name*/
      case N_VALABBR: case N_ABBR:
      case N_VALRETYPE: case N_RETYPE:
      case N_DECL: case N_REPL:
      case N_PARAM: case N_VALPARAM:
        return (NTypeOf(tptr));
      case N_TAGDEF:
        return bytenodeptr;
      /* All the following are invalid */
      case N_TPROTDEF: case N_SPROTDEF:
      case N_PROCDEF:  case N_SCPROCDEF:
      case N_LIBPROCDEF: case N_INLINEPROCDEF: case N_PREDEFPROC:
      CASE_CONFIG_NAME
        chkreport (CHK_TYPE_MISMATCH, chklocn);
        return undeclaredp;
      case N_SFUNCDEF:   case N_LFUNCDEF:      case N_SCFUNCDEF:
      case N_LIBFUNCDEF: case N_INLINEFUNCDEF: case N_PREDEFFUNCTION:
        /* added for bug 1115 29/1/91 */
        chkreport_s(CHK_TOO_FEW_ACTUALS,chklocn, WNameOf(NNameOf(tptr)));
        return undeclaredp;
      case N_FIELD:
        chkreport_s(CHK_LONELY_FIELD, chklocn, WNameOf(NNameOf(tptr)));
        return undeclaredp;
      /*}}}*/
      /*{{{  function instance*/
      case S_FINSTANCE:
        {
          treenode *nptr = INameOf(tptr);
          treenode *p;
          checkparams (tptr, S_FUNCTION);    /* Check actual parameter types  */
          /*{{{  check to see if declared*/
          if ((TagOf(nptr) == N_DECL) &&
              (TagOf(NTypeOf(nptr)) == S_UNDECLARED))
            return (NTypeOf(nptr));
          /*}}}*/
          p = FnTypeListOf(NTypeOf(nptr));   /* point p to the result types   */
          if (listitems(p) > 1)              /* It is a multi-valued function */
            chkreport (CHK_INV_FUNC_LIST, chklocn);
          else
            return (ThisItem(p));
        }
      /*}}}*/
      #ifdef CONFIG
      /*{{{  RECORDSUB*/
      case S_RECORDSUB:
        if (TagOf(ASIndexOf(tptr)) == N_FIELD)
          return (NTypeOf(ASIndexOf(tptr)));
        else
          return undeclaredp;
      /*}}}*/
      #endif
      /*{{{  subscript*/
      case S_ARRAYSUB:
        {
          treenode *nptr = tptr;
          treenode *st;
          int subscripts = 0;
          char *str;
          while (TagOf(nptr) == S_ARRAYSUB)
            {
              subscripts++;
              /*{{{  check subscript expression is of type INT*/
              st = typecheck_main (ASIndexOf(nptr), S_INT);
              if ((TagOf(st) != S_UNDECLARED) && (TagOf(st) != S_INT))
                chkreport (CHK_ASUB_NOT_INT, chklocn);
              /*}}}*/
              nptr = ASBaseOf(nptr);
            }

          st = typecheck_main(nptr, type);
          if (TagOf(st) == S_UNDECLARED)
            return (st);

          switch (TagOf(nptr))
            {
              /*{{{  case S_SEGMENT*/
              case S_SEGMENT:
                {
                  treenode *name = nameof(nptr);
                  
                  if (TagOf(name) == S_CONSTRUCTOR || TagOf(name) == S_STRING
                  #ifdef ARRAYCONSTRUCTOR
                      || TagOf(name) == S_ARRAYCONSTRUCTOR
                  #endif
                     )
                    str = "table";
                  else
                    str = WNameOf(NNameOf(name));
                }
                break;
              /*}}}*/
              /*{{{  case N_DECL, N_ABBR, etc.*/
              case N_DECL: case N_ABBR: case N_VALABBR:
              case N_RETYPE: case N_VALRETYPE: case N_PARAM: case N_VALPARAM: case N_REPL:
              case N_FIELD:
                str = WNameOf(NNameOf(nptr));
                break;
              /*}}}*/
              /*{{{  case S_CONSTRUCTOR case S_STRING case S_ARRAYCONSTRUCTOR*/
              case S_STRING:
                str = "string";
                break;
              case S_CONSTRUCTOR:
              #ifdef ARRAYCONSTRUCTOR
              case S_ARRAYCONSTRUCTOR:
              #endif
                str = "table";
                break;
              /*}}}*/
              #ifdef CONFIG
              case S_RECORDSUB:
                str = WNameOf(NNameOf(ASIndexOf(nptr)));
                break;
              #endif
              default:
                /*badtag(chklocn /@LocnOf(nptr)@/, (BIT32)TagOf(nptr), "typecheck_main");*/
                chkreport (CHK_NAMENOTARRAY, chklocn); /* bug 1088 21/12/90 */
            }
          return arraytype(st, subscripts, str);
        }
      /*}}}*/
      /*{{{  valof*/
      case S_VALOF:
        {
          treenode *nptr = VLResultListOf(tptr);
          if (listitems(nptr) > 1)              /* It is a multivalued VALOF */
            chkreport (CHK_INV_VALOF_LIST, chklocn);
          else
            return(typecheck_main(ThisItem(nptr), type));
        }
      /*}}}*/
      /*{{{  segment*/
      case S_SEGMENT:
        {
          treenode *aptr = typecheck_main (SNameOf(tptr), type),
                   *sptr = typecheck_main (SStartExpOf(tptr), S_INT),
                   *lptr = typecheck_main (SLengthExpOf(tptr), S_INT),
                   *bptr;
      
          /*{{{  check the segmented operand is an array*/
          if (TagOf(aptr) == S_UNDECLARED)
            return(aptr);
          if (TagOf(aptr) != S_ARRAY)
            chkreport (CHK_INV_SEGMENT_OPERAND, chklocn);
          /*}}}*/
          /*{{{  typecheck the start and length expressions, make sure they are INT*/
          if ((TagOf(sptr) != S_UNDECLARED) && (TagOf(sptr) != S_INT))
            chkreport (CHK_ASUB_NOT_INT, chklocn);
          if ((TagOf(lptr) != S_UNDECLARED) && (TagOf(lptr) != S_INT))
            chkreport (CHK_ASUB_NOT_INT, chklocn);
          /*}}}*/
      
          bptr = newarraynode (S_ARRAY, 0, NULL, ARTypeOf(aptr));
          if (isconst(SLengthExpOf(tptr)))
            /*{{{  fold the segment length expression and store in our new array node*/
            {
              if (TagOf(SLengthExpOf(tptr)) != S_CONSTEXP)    /* it's not already folded */
                /*{{{  fold it*/
                {
                  int old = switch_to_real_workspace ();
                  SetSLengthExp(tptr, newconstexp(SLengthExpOf(tptr)));
                  switch_to_prev_workspace (old);
                }
                /*}}}*/
              SetARDim(bptr, LoValOf(SLengthExpOf(tptr)));
            }
            /*}}}*/
          else
            SetARDim(bptr, -1);                              /* Match anything */
          SetARDimLength(bptr, SLengthExpOf(tptr));
          return (bptr);
        }
      /*}}}*/
      /*{{{  constructor*/
      case S_CONSTRUCTOR:
        {
          treenode *aptr = NULL, *ctype = NULL;
          treenode *cptr = OpOf(tptr);
          BIT32 clength = ZERO32;
          while (!EndOfList(cptr))
            {
              treenode *t = typecheck_main(ThisItem(cptr), type);
              if (TagOf(t) == S_UNDECLARED)
                { if (aptr == NULL) aptr = t; }
              if (elementsin(t) == (-1))
                {
                  /* N.B. As an implementation restriction we don't allow any
                     open dimensions in the type of a constructor,
                     and therefore in the components of it. */
                  chkreport(CHK_UNKNOWN_CONSTRUCTOR_SIZE, chklocn);
                  { if (aptr == NULL) aptr = t; }
                }
              else if (ctype == NULL) /* first element */
                ctype = t;
              else if (!typesequivalent(ctype, t, FALSE))
                chkreport (CHK_CONSTRUCTOR_TYPE_MISMATCH, chklocn);

              cptr = NextItem(cptr);
              clength++;
            }
          if (aptr == NULL) /* everything ok so far */
            /*{{{  fabricate an array node*/
            {
              aptr = newarraynode (S_ARRAY, 0,
                                 newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, clength),
                                   ctype);
              SetARDim(aptr, clength);
            }
            /*}}}*/
          return (aptr);
        }
      /*}}}*/
      #if 0
      /*{{{  structconstructor*/
      case S_STRUCTCONSTRUCTOR:
        return (typenodeptr(MOpTypeOf(tptr)));
      /*}}}*/
      #endif
      #ifdef ARRAYCONSTRUCTOR
      /*{{{  arrayconstructor*/
      case S_ARRAYCONSTRUCTOR:
        {
          /* N.B. As an implementation restriction we don't allow any
             open dimensions in the type of a constructor,
             and therefore in the components of it. */
      
          treenode *aptr, *bptr, *st;
      
          /*{{{  typecheck the start and length expressions, make sure they are INT*/
          st = typecheck_main (ACStartExpOf(tptr), S_INT);
          if (TagOf(st) == S_UNDECLARED) st = intnodeptr;
          if (TagOf(st) != S_INT) chkreport (CHK_ASUB_NOT_INT, chklocn);
          st = typecheck_main (ACLengthExpOf(tptr), S_INT);
          if (TagOf(st) == S_UNDECLARED) st = intnodeptr;
          if (TagOf(st) != S_INT) chkreport (CHK_ASUB_NOT_INT, chklocn);
          /*}}}*/
          /*{{{  check type of value expression*/
          aptr = typecheck_main (ACValExpOf(tptr), type);
          if (TagOf(aptr) == S_UNDECLARED) return(aptr);
          /*}}}*/
      
          bptr = newarraynode (S_ARRAY, 0, NULL, aptr);
      
          if (isconst(ACLengthExpOf(tptr)) && (elementsin(ACValExpOf(tptr)) != (-1)))
            /*{{{  fold the segment length expression and store in our new array node*/
            {
              if (TagOf(ACLengthExpOf(tptr)) != S_CONSTEXP)    /* it's not already folded */
                {
                  int old = switch_to_real_workspace ();
                  SetACLengthExp(tptr, newconstexp(ACLengthExpOf(tptr)));
                  switch_to_prev_workspace (old);
                }
              SetARDim(bptr, LoValOf(ACLengthExpOf(tptr)));
            }
            /*}}}*/
          else
            {
              chkreport(CHK_UNKNOWN_CONSTRUCTOR_SIZE, chklocn);
              SetARDim(bptr, -1);
            }
          SetARDimLength(bptr, ACLengthExpOf(tptr));
          return (bptr);
        }
      /*}}}*/
      #endif
      #ifdef CONDEXP
      /*{{{  conditional expression*/
      case S_CONDEXP:
        {
          char buf[MAX_ERR_SIZE];
          t = typecheck_main (CondExpGuardOf(tptr), S_BOOL);
          if (TagOf(t) == S_UNDECLARED)
            t = boolnodeptr;
          if (TagOf(t) != S_BOOL)
            chkreport (CHK_CONDEXP_NOT_BOOL, chklocn);
          ftagstring(buf, S_CONDEXP);
          t = checksame(CondExpTrueOf(tptr), CondExpFalseOf(tptr),
                            type, CHK_TYPES_DIFF, buf);
          /*{{{  check type is legal*/
          switch (TagOf(t))
            {
              case S_BOOL: case S_BYTE:
              case S_INT: case S_INT16: case S_INT32: case S_INT64:
              case S_REAL32: case S_REAL64 :
              case S_UNDECLARED:
              CASE_CONFIG_TYPE
                SetCondExpType(tptr, TagOf(t));
                break;
              default:
                chk_invtype(chklocn, TagOf(tptr));
            }
          /*}}}*/
          return (t);
        }
      /*}}}*/
      #endif
      /*{{{  specification*/
      case S_VALABBR: case S_ABBR:
      case S_VALRETYPE: case S_RETYPE:
      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
      case S_TPROTDEF: case S_SPROTDEF:
      case S_DECL:
      case S_PLACE: case S_VSPLACE: case S_WSPLACE:
      #ifdef CONFIG
      case S_PLACEON:
      #endif
        tptr = skipspecifications(tptr);
        /*{{{  now we must have a VALOF*/
        if (TagOf(tptr) == S_VALOF)
          {
            treenode *nptr = VLResultListOf(tptr);
            if (listitems(nptr) > 1)   /* throw out multi-valued valof */
              chkreport (CHK_INV_VALOF_LIST, chklocn);
            else
              return(typecheck_main(ThisItem(nptr), type));
          }
        else
          msg_out_i(SEV_INTERNAL, CHK, CHK_LOST_VALOF, chklocn, TagOf(tptr));
        /*}}}*/
      /*}}}*/
      /*}}}*/
      default:
        badtag(chklocn, (BIT32)TagOf(tptr), "typecheck_main");
        return(NULL);
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *typecheck (tptr, type)*/
PUBLIC treenode *typecheck ( treenode *tptr , int type )
{
  treenode *t;

  /* Generate any temporary nodes in the temporary free space */
  int old = switch_to_temp_workspace ();

  t = typecheck_main (tptr, type);

  /* Switch back to real workspace */
  switch_to_prev_workspace (old);

  return (t);
}
/*}}}*/
/*{{{  PUBLIC treenode *checksame (t1, t2, type, e, s)*/
/* Type checks t1 and t2,
and then checks that their types are equal,
and returns the common type (in 'current' workspace (normally temp)).
type is a type tag used to resolve untyped integer and real literals
where possible.
*/
PUBLIC treenode *checksame ( treenode *t1, treenode *t2, int type,
                             int e, char *s)
{
  treenode *l;
  if ((type == S_UNKNOWN) && (typeknown(t1) != TRUE))
    /*{{{  swap t1 and t2*/
    {
      treenode *temp = t1;
      t1 = t2;
      t2 = temp;
    }
    /*}}}*/
  l = typecheck_main (t1, type);
  if (TagOf(l) == S_UNDECLARED)
      {
        typecheck(t2, type);
        return l;
      }
  else
    /*{{{  check right is the same as the left*/
    {
      if (type == S_UNKNOWN)
        type = basetype(l);
      if (typesequivalent(l, typecheck_main (t2, type), FALSE))
        return (l);
      else
        {
          chkreport_s(e, chklocn, s);
          return(NULL);
        }
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC treenode *gettype_main (tptr)*/
/*{{{  comment*/
/* Assumes that the tree has already been type-checked. */
/* N.B. This function returns a pointer to a type tree which already
exists, so the return value should only be used for  comparsion and
not for inserting elsewhere on the tree.
To insert a type tree elsewhere, first copy it using function
'copytree'.
*/
/*}}}*/
PUBLIC treenode *gettype_main ( treenode *tptr )
{
  switch (TagOf(tptr))
    {
      /*{{{  node type*/
      /*{{{  monadic operators*/
      /*{{{  NEG BITNOT NOT UMINUS*/
      case S_NEG:
      case S_BITNOT:
      case S_NOT:
      case S_UMINUS:
        return (typenodeptr(MOpTypeOf(tptr)));
      /*}}}*/
      /*{{{  SIZE ELSIZE SEGSTART*/
      case S_ADDRESSOF:
      case S_SIZE:
      case S_ELSIZE:
      case S_SEGSTART:
      case S_DUMMYEXP:
        return (intnodeptr);
      /*}}}*/
      /*}}}*/
      /*{{{  dyadic operators*/
      case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
      case S_BITAND: case S_BITOR: case S_XOR:
      case S_LSHIFT: case S_RSHIFT:
      case S_AND: case S_OR:
      case S_PLUS: case S_MINUS: case S_TIMES:
      case S_CSUB0: case S_CCNT1:
      case S_OVERLAPCHECK: case S_EVAL:
        return (typenodeptr(DOpTypeOf(tptr)));
      /*}}}*/
      /*{{{  relational operators*/
      case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
        return (boolnodeptr);
      /*}}}*/
      /*{{{  mostpos, mostneg*/
      case S_MOSTPOS:
      case S_MOSTNEG:
        return (OpOf(tptr));
      /*}}}*/
      /*{{{  conversions*/
      case S_EXACT:
      case S_ROUND:
      case S_TRUNC:
        return (typenodeptr(MOpTypeOf(tptr)));
      /*}}}*/
      /*{{{  constant expression*/
      case S_CONSTEXP:
        return (gettype_main(CExpOf(tptr)));
      /*}}}*/
      /*{{{  literals*/
      case S_TRUE:
      case S_FALSE:
      case S_BYTELIT:
      case S_INTLIT:
      case S_INT16LIT:
      case S_INT32LIT:
      case S_INT64LIT:
      case S_REAL32LIT:
      case S_REAL64LIT:
        return (litnodeptr(TagOf(tptr)));
      case S_STRING:
        {
          BIT32 dim = WLengthOf(CTValOf(tptr));
          treenode *aptr =
            newarraynode (S_ARRAY, 0,
                          newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, dim),
                          bytenodeptr);
          SetARDim(aptr, dim);
          return (aptr);
        }
      case S_ASMNAME:
        return (intnodeptr);
      /*}}}*/
      /*{{{  name temporary*/
      case N_VALABBR:
      case N_ABBR:
      case N_VALRETYPE:
      case N_RETYPE:
      case N_DECL:
      case N_PARAM:
      case N_VALPARAM:
      case N_REPL: case N_FIELD:
      case T_TEMP:
      case T_PREEVALTEMP:
        return (NTypeOf(tptr));
      case N_TAGDEF:
        return(bytenodeptr);
      /*}}}*/
      /*{{{  function instance*/
      case S_FINSTANCE:
        {
          treenode *nptr = INameOf(tptr);
          treenode *p;
          if (TagOf(nptr) == N_DECL)  /* undeclared - bug 557 23/7/90 */
            return undeclaredp;
          p = FnTypeListOf(NTypeOf(nptr));/* point p to the result types */
          if (listitems(p) == 1)
            p = ThisItem(p);
          return (p);
        }
      /*}}}*/
      /*{{{  subscript*/
      #ifdef CONFIG
      case S_RECORDSUB:
        return NTypeOf(ASIndexOf(tptr));
      #endif
      case S_ARRAYSUB: case S_ARRAYITEM:
        {
          int subscripts = 0;
          while (TRUE)
            {
              switch (TagOf(tptr))
                {
                  /*{{{  case S_ARRAYSUB*/
                  case S_ARRAYSUB:
                  case S_ARRAYITEM:
                    subscripts++;
                    tptr = ASBaseOf(tptr);
                    break;
                  /*}}}*/
                  /*{{{  case S_SEGMENT S_SEGMENTITEM*/
                  case S_SEGMENT:
                  case S_SEGMENTITEM:
                    if (subscripts == 0)
                      return(gettype_main(tptr));
                    tptr = SNameOf(tptr);
                    break;
                  /*}}}*/
                  /*{{{  case S_CONSTRUCTOR*/
                  case S_CONSTRUCTOR:
                    if (subscripts == 0)
                      return(gettype_main(tptr));
                    else if (subscripts == 1)
                      return(gettype_main(ThisItem(OpOf(tptr))));
                    else
                      {
                        tptr = ThisItem(OpOf(tptr));
                        subscripts--;
                      }
                    break;
                  /*}}}*/
                  #ifdef ARRAYCONSTRUCTOR
                  /*{{{  case S_ARRAYCONSTRUCTOR*/
                  case S_ARRAYCONSTRUCTOR:
                    if (subscripts == 0)
                      return(gettype_main(tptr));
                    else if (subscripts == 1)
                      return(gettype_main(ACValExpOf(tptr)));
                    else
                      {
                        tptr = ACValExpOf(tptr);
                        subscripts--;
                      }
                    break;
                  /*}}}*/
                  #endif
                  /*{{{  case S_CONSTCONSTRUCTOR*/
                  case S_CONSTCONSTRUCTOR:
                    tptr = CTExpOf(tptr);
                    break;
                  /*}}}*/
                  /*{{{  case S_STRING*/
                  case S_STRING:
                    return arraytype(gettype_main(tptr), subscripts, "string");
                  /*}}}*/
                  /*{{{  name temporary*/
                  case N_DECL:
                  case N_ABBR: case N_VALABBR:
                  case N_RETYPE: case N_VALRETYPE:
                  case N_PARAM: case N_VALPARAM:
                  case N_REPL:
                  case T_TEMP: case T_PREEVALTEMP:
                  case N_FIELD:
                    return(arraytype(NTypeOf(tptr), subscripts, WNameOf(NNameOf(tptr))));
                  /*}}}*/
                  #ifdef CONFIG
                  case S_RECORDSUB:
                    return arraytype(gettype_main(tptr), subscripts, WNameOf(NNameOf(ASIndexOf(tptr))));
                  #endif
                  default:
                    badtag(chklocn, (BIT32)TagOf(tptr), "gettype_main");
                }
            }
        }
      /*}}}*/
      /*{{{  segment*/
      case S_SEGMENT:
      case S_SEGMENTITEM:
        {
          treenode *slength = newmopnode (S_ELSIZE, 0, tptr, 0);
          treenode *aptr = gettype_main (SNameOf(tptr)),
                   *bptr = newarraynode (S_ARRAY, 0, slength, ARTypeOf(aptr));
          SetARDim(bptr, isconst(SLengthExpOf(tptr)) ?
                         LoValOf(SLengthExpOf(tptr)) : (-1));
          return (bptr);
        }
      /*}}}*/
      /*{{{  constructor*/
      case S_CONSTRUCTOR:
        {
          treenode *ctype, *aptr;
          BIT32 dim;
          tptr = OpOf(tptr);
          dim = listitems(tptr);
          ctype = gettype_main(ThisItem(tptr));
          aptr = newarraynode (S_ARRAY, 0,
                             newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, dim),
                               ctype);
          SetARDim(aptr, dim);
          return (aptr);
        }
      /*}}}*/
      #ifdef ARRAYCONSTRUCTOR
      /*{{{  arrayconstructor*/
      case S_ARRAYCONSTRUCTOR:
        {
          treenode *ctype, *aptr;
          BIT32 dim;
          dim = isconst(ACLengthExpOf(tptr)) ? LoValOf(ACLengthExpOf(tptr)) : (-1);
          ctype = gettype_main(ACValExpOf(tptr));
          aptr = newarraynode (S_ARRAY, 0,
                             newconstexpnode(S_CONSTEXP, 0, dummyexp_p, ZERO32, dim),
                               ctype);
          SetARDim(aptr, dim);
          return (aptr);
        }
      /*}}}*/
      #endif
      /*{{{  constant constructor*/
      case S_CONSTCONSTRUCTOR:
        return (gettype_main(CTExpOf(tptr)));
      /*}}}*/
      #if 0
      /*{{{  structconstructor*/
      case S_STRUCTCONSTRUCTOR:
        return (typenodeptr(MOpTypeOf(tptr)));
      /*}}}*/
      #endif
      #ifdef CONDEXP
      /*{{{  conditional expression*/
      case S_CONDEXP:
        return (typenodeptr(CondExpTypeOf(tptr)));
      /*}}}*/
      #endif
      /*{{{  specification*/
      case S_VALABBR: case S_VALRETYPE:
      case S_TPROTDEF: case S_SPROTDEF: case S_DECL:
      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
      case S_ABBR: case S_RETYPE:
      case S_VALOF:
        while (TagOf(tptr) != S_VALOF)
          tptr = DBodyOf(tptr);
        return(gettype_main(ThisItem(VLResultListOf(tptr))));
      /*}}}*/
      /*{{{  backend parameters*/
      case S_HIDDEN_PARAM:
      case S_PARAM_STATICLINK:
      case S_PARAM_VSP:
      case S_FNACTUALRESULT:
        return intnodeptr;
      case S_FNFORMALRESULT:
          return(HExpOf(tptr));
      /*}}}*/
      /*}}}*/
      default:
        badtag(chklocn, (BIT32)TagOf(tptr), "gettype_main");
        return(NULL);
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *gettype (tptr)*/
PUBLIC treenode *gettype ( treenode *tptr )
{
  treenode *t;

  /* Generate any temporary nodes in the temporary free space */
  int old = switch_to_temp_workspace ();

  t = gettype_main(tptr);

  /* Switch back to real workspace */
  switch_to_prev_workspace (old);

  return (t);
}
/*}}}*/
/*{{{  PUBLIC int typeof (tptr)*/
/* Cut down version of gettype which only returns scalar types or S_ARRAY */
/* Assumes that all type checking has been done - uses the type field     */
/* of operator nodes                                                      */
PUBLIC int typeof ( treenode *tptr )
{
  while (TRUE)
    switch (TagOf(tptr))
      {
        /*{{{  cases*/
        /*{{{  monadic operator*/
        case S_MOSTPOS: case S_MOSTNEG: case S_NEG:
        case S_BITNOT: case S_UMINUS: case S_NOT:
        case S_EXACT: case S_ROUND: case S_TRUNC:
          return (MOpTypeOf(tptr));
        case S_ADDRESSOF: case S_SIZE: case S_ELSIZE: case S_SEGSTART:
          return S_INT;
        /*}}}*/
        /*{{{  dyadic operator*/
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_LSHIFT: case S_RSHIFT:
        case S_AND: case S_OR:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_OVERLAPCHECK: case S_EVAL:
          return (DOpTypeOf(tptr));
        case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
          return S_BOOL;
        case S_CSUB0: case S_CCNT1:
          return S_INT;
        /*}}}*/
        /*{{{  literal*/
        case S_TRUE: case S_FALSE: return (S_BOOL);
        case S_BYTELIT:            return (S_BYTE);
        case S_ASMNAME:
        case S_INTLIT:             return (S_INT);
        case S_INT16LIT:           return (S_INT16);
        case S_INT32LIT:           return (S_INT32);
        case S_INT64LIT:           return (S_INT64);
        case S_REAL32LIT:          return (S_REAL32);
        case S_REAL64LIT:          return (S_REAL64);
        case S_STRING:
        case S_CONSTRUCTOR:
        case S_CONSTCONSTRUCTOR:
        #ifdef ARRAYCONSTRUCTOR
        case S_ARRAYCONSTRUCTOR:
        #endif
          return (S_ARRAY);
        /*}}}*/
        /*{{{  name temporary function result*/
        case N_VALABBR: case N_ABBR: case N_VALRETYPE: case N_RETYPE:
        case N_DECL: case N_PARAM: case N_VALPARAM: case N_FIELD:
        case N_REPL: case T_TEMP: case T_PREEVALTEMP:
          return (TagOf(NTypeOf(tptr)));
        case N_TAGDEF:
          return(S_BYTE);
        case S_FNFORMALRESULT:
          return(TagOf(HExpOf(tptr)));
        /*}}}*/
        /*{{{  function instance*/
        case S_FINSTANCE:
          /* return tag of result type (assume it's not multi-valued) */
          {
            treenode *ftypelist = FnTypeListOf(NTypeOf(INameOf(tptr)));
            return(TagOf(ThisItem(ftypelist)));
          }
        /*}}}*/
        /*{{{  subscript*/
        #ifdef CONFIG
        case S_RECORDSUB:
          return TagOf(NTypeOf(ASIndexOf(tptr)));
        #endif
        case S_ARRAYSUB: case S_ARRAYITEM:
          {
            int subscripts = 0;
            int descending = TRUE;
            /*{{{  work down to the name*/
            while (descending)
              {
                switch(TagOf(tptr))
                  {
                    default:
                      badtag(chklocn, (BIT32)TagOf(tptr), "typeof");
                      break;
                    /*{{{  name temporary*/
                    case N_DECL:
                    case N_ABBR: case N_VALABBR:
                    case N_RETYPE: case N_VALRETYPE:
                    case N_PARAM: case N_VALPARAM:
                    case N_REPL:
                    case T_TEMP: case T_PREEVALTEMP:
                      descending = FALSE;
                      break;
                    /*}}}*/
                    /*{{{  S_ARRAYSUB*/
                    case S_ARRAYSUB:
                    case S_ARRAYITEM:
                      subscripts++;
                      tptr = ASBaseOf(tptr);
                      break;
                    /*}}}*/
                    #ifdef CONFIG
                    /*{{{  record sub*/
                    case S_RECORDSUB:
                      tptr = ASBaseOf(tptr);
                      break;
                    /*}}}*/
                    #endif
                    /*{{{  S_SEGMENT S_SEGMENTITEM*/
                    case S_SEGMENT:
                    case S_SEGMENTITEM:
                      if (subscripts == 0)
                        return(S_ARRAY);
                      else
                        tptr = SNameOf(tptr);
                      break;
                    /*}}}*/
                    /*{{{  S_STRING*/
                    case S_STRING:
                      if (subscripts == 0)
                        return S_ARRAY;
                      else
                        return S_BYTE;
                    /*}}}*/
                    /*{{{  S_CONSTRUCTOR*/
                    case S_CONSTRUCTOR:
                      if (subscripts == 0)
                        return(S_ARRAY);
                      else if (subscripts == 1)
                        return(typeof(ThisItem(OpOf(tptr))));
                      else
                        {
                          subscripts--;
                          tptr = ThisItem(OpOf(tptr));
                        }
                      break;
                    /*}}}*/
                    #ifdef ARRAYCONSTRUCTOR
                    /*{{{  S_ARRAYCONSTRUCTOR*/
                    case S_ARRAYCONSTRUCTOR:
                      if (subscripts == 0)
                        return(S_ARRAY);
                      else if (subscripts == 1)
                        return(typeof(ACValExpOf(tptr)));
                      else
                        {
                          subscripts--;
                          tptr = ACValExpOf(tptr);
                        }
                      break;
                    /*}}}*/
                    #endif
                    /*{{{  S_CONSTCONSTRUCTOR*/
                    case S_CONSTCONSTRUCTOR:
                      tptr = CTExpOf(tptr);
                      break;
                    /*}}}*/
                  }
              }
            /*}}}*/
            return(TagOf(arraytype(NTypeOf(tptr), subscripts,
                                   WNameOf(NNameOf(tptr)))));
          }
        /*}}}*/
        /*{{{  segment*/
        case S_SEGMENT: case S_SEGMENTITEM:
          return(S_ARRAY);
        /*}}}*/
        /*{{{  valof, specification - must be preceding a VALOF*/
        case S_VALABBR: case S_VALRETYPE: case S_TPROTDEF: case S_SPROTDEF:
        case S_DECL: case S_SFUNCDEF: case S_LFUNCDEF: case S_ABBR: case S_RETYPE:
        case S_VALOF:
          while (TagOf(tptr) != S_VALOF)
            tptr = DBodyOf(tptr);
          tptr = ThisItem(VLResultListOf(tptr));
          break;
        /*}}}*/
        /*{{{  constant expression*/
        case S_CONSTEXP:
          tptr = CExpOf(tptr);
          break;
        /*}}}*/
        #if 0
        /*{{{  structconstructor*/
        case S_STRUCTCONSTRUCTOR:
          return (MOpTypeOf(tptr));
        /*}}}*/
        #endif
        /*{{{  backend parameters, bits and pieces*/
        case S_HIDDEN_PARAM: case S_PARAM_STATICLINK: case S_PARAM_VSP:
        case S_DUMMYEXP:
          return S_INT;
        case S_FNACTUALRESULT:
          /*assert(FALSE);*/ return S_UNKNOWN;
        /*}}}*/
        /*}}}*/
      default:
        badtag(chklocn, (BIT32)TagOf(tptr), "typeof");
      }
}
/*}}}*/
/*{{{  PUBLIC treenode *nametype(tptr)*/
/*****************************************************************************
 *
 * nametype returns the type of the underlying name in an element
 *
 *****************************************************************************/
PUBLIC treenode *nametype ( treenode *tptr )
{
  while (TRUE)
    {
      switch(TagOf(tptr))
        {
          /*{{{  subscript*/
          case S_ARRAYSUB: case S_ARRAYITEM: case S_RECORDSUB:
            tptr = ASBaseOf(tptr); break;
          /*}}}*/
          /*{{{  segment*/
          case S_SEGMENT: case S_SEGMENTITEM:  tptr = SNameOf(tptr);  break;
          /*}}}*/
          /*{{{  name*/
          case N_VALABBR: case N_ABBR: case N_VALRETYPE: case N_RETYPE:
          case N_DECL: case N_VALPARAM: case N_PARAM:
          case N_REPL:     return(NTypeOf(tptr));
          /*}}}*/
          /*{{{  constructor string*/
          case S_CONSTRUCTOR: case S_STRING:
          #ifdef ARRAYCONSTRUCTOR
          case S_ARRAYCONSTRUCTOR:
          #endif
            return(gettype(tptr));
          /*}}}*/
          default:
            badtag(chklocn, (BIT32)TagOf(tptr), "nametype");
        }
    }
}
/*}}}*/
/*}}}*/
/*{{{  PUBLIC void checkparams (tptr, mode)*/
/*****************************************************************************
 *
 *  checkparams checks that the actual parameter types of procedure or
 *              function instance 'tptr' match the formal parameter types
 *              of the function/procedure.  'mode' is set to S_PROC if we
 *              are checking a procedure, 'S_FUNCTION' if we are checking
 *              a function.
 *
 *****************************************************************************/

/* S_PROC or S_FUNCTION */
PUBLIC void checkparams ( treenode *tptr , int mode )
{
  treenode *pname = INameOf(tptr);
  /*{{{  ignore undeclared functions and procedures*/
  if ((TagOf(pname) == N_DECL) && (TagOf(NTypeOf(pname)) == S_UNDECLARED))
    /*{{{  check the parameter types*/
    {
      treenode *aparamptr = IParamListOf(tptr);
    
      while (!EndOfList(aparamptr))
        /*{{{  check left-hand side, move to right hand side*/
        {
          typecheck(ThisItem(aparamptr), S_UNKNOWN);
          aparamptr = NextItem(aparamptr);
        }
        /*}}}*/
      return;
    }
    /*}}}*/
  /*}}}*/
  if (mode == S_PROC)
    /*{{{  checking a procedure instance and the namenode isn't a procedure*/
      switch(TagOf(pname))
        {
          default:
            chkreport_s(CHK_NAME_NOT_PROC, chklocn,
                        WNameOf(NNameOf((pname))));
            break;
          case N_PROCDEF: case N_SCPROCDEF:
          case N_LIBPROCDEF: case N_STDLIBPROCDEF: case N_PREDEFPROC:
          case N_INLINEPROCDEF:
            break;
        }
    /*}}}*/
  else
    /*{{{  checking a function instance and the namenode isn't a function*/
    switch(TagOf(pname))
      {
        default:
          chkreport_s(CHK_NAME_NOT_FUNCTION, chklocn,
                      WNameOf(NNameOf(pname)));
          break;
        case N_SFUNCDEF: case N_LFUNCDEF:
        case N_SCFUNCDEF: case N_LIBFUNCDEF: case N_STDLIBFUNCDEF:
        case N_PREDEFFUNCTION: case N_INLINEFUNCDEF:
          break;
      }
    /*}}}*/
  /*{{{  check the parameter types*/
  {
    treenode *fparamptr = NTypeOf(pname),
             *aparamptr = IParamListOf(tptr);
    int paramno = 1;
    int nfparams, naparams;
    if (mode == S_FUNCTION)
      fparamptr = FnParamsOf(fparamptr);
    nfparams = listitems(fparamptr);
    naparams = listitems(aparamptr);
    if (naparams > nfparams)
      chkreport_s(CHK_TOO_MANY_ACTUALS, chklocn, WNameOf(NNameOf(pname)));
    else if (naparams < nfparams)
      chkreport_s(CHK_TOO_FEW_ACTUALS, chklocn, WNameOf(NNameOf(pname)));
    else
      while (!EndOfList(fparamptr) && !EndOfList(aparamptr))
        /*{{{  check left-hand side, move to right hand side*/
        {
          treenode *f = ThisItem(fparamptr);
          treenode *a = ThisItem(aparamptr);
          treenode *atype = typecheck(a, basetype(NTypeOf(f)));
        
          if (typesequivalent(NTypeOf(f), atype, TRUE))
            /*{{{  check a var param is a variable*/
            {
              if (TagOf(f) == N_PARAM)
                checkelement(a, NTypeOf(f), paramno);
            }
            /*}}}*/
          else if
            /*{{{  we are checking some special predefined routines*/
            /* For the predefined procedures LOAD.INPUT.CHANNEL, LOAD.OUTPUT.CHANNEL,
               LOAD.INPUT.CHANNEL.VECTOR and LOAD.OUPUT.CHANNEL.VECTOR, we allow
               any protocol on the channel parameter */
            (TagOf(pname) == N_PREDEFPROC)
              {
                int ok = FALSE;
                const int pdno = NModeOf(pname);
                if ((pdno == PD_LOADINPUTCHANNEL || pdno == PD_LOADINPUTCHANNELVECTOR ||
                     pdno == PD_LOADOUTPUTCHANNEL || pdno == PD_LOADOUTPUTCHANNELVECTOR) &&
                    basetype(NTypeOf(f)) == S_CHAN)
                  /*{{{  fudge the type, and see if it matches*/
                  {
                    treenode *fudgedtype, *t;
                    const int old = switch_to_temp_workspace();
                    fudgedtype = copytree(atype);
                    t = fudgedtype;
                    while (TagOf(t) == S_ARRAY) t = ARTypeOf(t);
                    if (TagOf(t) == S_CHAN)
                      SetProtocol(t, newleafnode(S_ANY, LocnOf(t)));
                    ok = typesequivalent(NTypeOf(f), fudgedtype, TRUE);
                    switch_to_prev_workspace(old);
                  }
                  /*}}}*/
                if (!ok)
                  msg_out_is(SEV_ERR_JMP, CHK, CHK_INVTYPE_PARAM, chklocn,
                             paramno, WNameOf(NNameOf(pname)));
              }
            /*}}}*/
          else
            msg_out_is(SEV_ERR_JMP, CHK, CHK_INVTYPE_PARAM, chklocn,
                       paramno, WNameOf(NNameOf(pname)));
          /*{{{  move to right-hand side*/
          fparamptr = NextItem(fparamptr);
          aparamptr = NextItem(aparamptr);
          paramno++;
          /*}}}*/
        }
        /*}}}*/
  }
  /*}}}*/
}
/*}}}*/
/*{{{  checking types and specifiers*/
/*{{{  PRIVATE treenode *ctypeorspec (tptr, typetree)*/

/*{{{  comment*/
/* Check a type or specifier tree :
For a type tree, check array dimensions are exist and are
integer constant, fold them, and store value in array node.

For a specifier tree, if array dimensions exist do the above.
Check channel protocol names correspond to protocol definitions.

For channel simple protocols, check array dimensions as above
(apart from the first one after '::', which must be null).
Returns the checked type tree.

For a specifier tree, throw out PORTs.
*/
/*}}}*/
PRIVATE treenode *ctypeorspec ( treenode *tptr , int typetree )
{
  treenode *typeptr = tptr;

  while (TRUE)
    switch (TagOf(typeptr))
      {
        /*{{{  BOOL, BYTE, INT, INTn, REALn, TIMER, NODE UNDECLARED*/
        case S_BOOL: case S_BYTE:
        case S_INT:  case S_INT16: case S_INT32: case S_INT64:
        case S_REAL32: case S_REAL64: case S_TIMER: case S_UNDECLARED:
        CASE_CONFIG_TYPE
          return (tptr);
        /*}}}*/
        /*{{{  ARRAY*/
        case S_ARRAY:
          {
            treenode *atype;
            if (ARDimLengthOf(typeptr) != NULL)
              /*{{{  check array dimension*/
              {
                if (TagOf(ARDimLengthOf(typeptr)) != S_CONSTEXP)
                  /*{{{  check the array dimension type and value*/
                  {
                    atype = typecheck (ARDimLengthOf(typeptr), S_INT);
                    if (TagOf(atype) == S_INT)
                      {
                        BIT32 reslo, reshi;
                        treenode *dimptr = ARDimLengthOf(typeptr);
                        foldconstexp (dimptr, &reshi, &reslo, CHK_ADIM_NOT_CONST);
                        /*{{{  check 0 < array dimension*/
                        {
                          int toosmall;
                        
                          if (targetintsize != S_INT64)
                            I32ToI64 (&reshi, &reslo, reslo);
                          Int64Gt (&toosmall, ZERO32, ONE32, reshi, reslo);
                          if (toosmall)
                            chkreport (CHK_ADIM_NEGATIVE, chklocn);
                          if (checkbounds (S_INT64, targetintsize, reshi, reslo) != 0)
                            chkreport (CHK_ADIM_TOOBIG, chklocn);
                        }
                        /*}}}*/
                        SetARDimLength(typeptr,
                                     newconstexpnode(S_CONSTEXP, chklocn, dimptr, reshi, reslo));
                        SetARDim(typeptr, reslo);
                      }
                    else if (TagOf(atype) == S_UNDECLARED)
                      SetARDim(typeptr, MOSTPOS_INT32);
                    else
                      chkreport (CHK_ADIM_NOT_INT, chklocn);
                  }
                  /*}}}*/
                /* else
                     This tree is shared with another parameter which has already been
                     checked, so we don't have to do any work here */
              }
              /*}}}*/
            else
              {
                if (typetree)
                  /*{{{  null dimensions are illegal on type trees*/
                  chkreport (CHK_ADIM_MISSING, chklocn);
                  /*}}}*/
                else
                  SetARDim(typeptr, -1);                 /* Set dimension to match any */
              }
            typeptr = ARTypeOf(typeptr);                  /* Move down to array type */
            break;
          }
        /*}}}*/
        /*{{{  CHAN*/
        case S_CHAN:
          {
            treenode *pptr = ProtocolOf(typeptr);
        
            switch (TagOf(pptr))
              {
                /*{{{  S_COLON2*/
                case S_COLON2:
                  /* syn ensures that the left-hand type is valid, and that the right-hand
                     type is a NULL array. */
                  {
                    treenode *aptr = RightOpOf(pptr);
                    SetARDim(aptr, -1);                     /* Set dimension length to unknown */
                    typetree = TRUE;              /* The simple protocol tree is a type tree */
                    typeptr = ARTypeOf(aptr);
                    break;
                  }
                /*}}}*/
                /*{{{  S_ARRAY*/
                case S_ARRAY:
                  typeptr = pptr;
                  typetree = TRUE;     /* The simple protocol is a type, not a specifier */
                  break;
                /*}}}*/
                /*{{{  S_ANY name stype*/
                case S_ANY:
                case N_SPROTDEF: case N_TPROTDEF: case S_BOOL: case S_BYTE:
                case S_INT: case S_INT16: case S_INT32: case S_INT64:
                case S_REAL32: case S_REAL64:
                case S_UNDECLARED:
                  return(tptr);
                case N_DECL: /* An undeclared protocol name */
                  return undeclaredp;
                /*}}}*/
                default:
                  chkreport (CHK_INV_PROT, chklocn);
              }
            break;
          }
        /*}}}*/
        /*{{{  PORT*/
        case S_PORT:
          {
            treenode *pptr = ProtocolOf(typeptr);
            switch (TagOf(pptr))
              {
                /*{{{  S_ARRAY*/
                case S_ARRAY:
                  typeptr = pptr;
                  typetree = TRUE;         /* The simple protocol is a type, not a specifier */
                  break;
                /*}}}*/
                /*{{{  stype*/
                case S_BOOL: case S_BYTE:
                case S_INT: case S_INT16: case S_INT32: case S_INT64:
                case S_REAL32: case S_REAL64:
                  return(tptr);
                /*}}}*/
                default:
                  chkreport (CHK_INV_PORT_PROT, chklocn);
              }
            break;
          }
        /*}}}*/
        default:
          badtag(chklocn, (BIT32)TagOf(typeptr), "ctypeorspec");
      }
}
/*}}}*/
/*{{{  PUBLIC treenode *ctype (tptr)*/
PUBLIC treenode *ctype ( treenode *tptr )
{
  return (ctypeorspec (tptr, TRUE));
}
/*}}}*/
/*{{{  PUBLIC treenode *cspecifier (tptr)*/
PUBLIC treenode *cspecifier ( treenode *tptr )
{
  return (ctypeorspec (tptr, FALSE));
}
/*}}}*/
/*}}}*/
/*}}}*/

