/*#define DEBUG*/
/*****************************************************************************
 *
 *  tran1 - tree transformer
 *
 *****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1990
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "extlib.h"

# include "generror.h"
# include "genhdr.h"
# include "trandef.h"
# include "predefhd.h"

# include "syndef.h"
# include "syn1def.h"
# include "chkdef.h"
# include "bind3def.h"
# include "gen1def.h"
# include "gen2def.h"
# include "gen8def.h"
/*}}}*/

/*{{{  constants*/
/* This is used to optimise VALOFs which expand into constants. */
#define OPTIMISE_VALOFS TRUE
/*}}}*/

/*{{{  PRIVATE variables*/
PRIVATE int inline_expansions;
PRIVATE int optimise_valof = TRUE;
PRIVATE int overlap_checks;
/*}}}*/

/*{{{  forward declarations*/
PRIVATE void trans        PARMS((treenode **tptr));
PRIVATE void transexp     PARMS((treenode **tptr));
PRIVATE void sub_transexp PARMS((treenode **tptr));
/*}}}*/

/*{{{  private*/
/*{{{  PRIVATE treenode *add_to(t, doptag, newnode)*/
/*****************************************************************************
 *
 *  add_to combines a new node, 'newnode', with an expression tree 't', using
 *         dyadic operator 'doptag'.  If the expression tree is NULL, then
 *         just the new node is returned.
 *
 *****************************************************************************/
PRIVATE treenode *add_to ( treenode *t , int doptag , treenode *newnode )
{
  if (t != NULL)
    newnode = newdopnode(doptag, NOPOSN, t, newnode, S_INT);
  return(newnode);
}
/*}}}*/
/*{{{  PUBLIC treenode *scaletreeof (type, basetype)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  scaletreeof takes a type tree 'type' and returns an expression tree
 *              which generates the number of elements of the same size as
 *              basetype (where basetype is a scalar type) in 'type'.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *scaletreeof ( treenode *type , int basetype )
{
  INT32 constantpart = ONE32;
  treenode *t = NULL;
  while (TagOf(type) == S_ARRAY)
    {
      if (ARDimOf(type) == (-1))
        /*{{{  hidden dimension*/
        t = add_to(t, S_TIMES, ARDimLengthOf(type));
        /*}}}*/
      else
        /*{{{  constant dimension*/
        constantpart *= ARDimOf(type);
        /*}}}*/
      type = ARTypeOf(type);
    }
  constantpart *= (bytesin(type) / bytesinscalar(basetype));
  if ((constantpart != ONE32) || (t == NULL))
    /*{{{  multiply t by the constant part*/
    t = add_to(t, S_TIMES, newconstant(constantpart));
    /*}}}*/
  return(t);
}
/*}}}*/
/*{{{  PRIVATE int issubscript*/
PRIVATE int issubscript (int tag)
/* returns TRUE if S_ARRAYSUB or S_ARRAYITEM,
           FALSE if S_SEGMENT or S_SEGMENTITEM
*/
{
  return (tag == S_ARRAYSUB || tag == S_ARRAYITEM);
}
/*}}}*/
/*{{{  PRIVATE int can_use_virtual_origin*/
PRIVATE int can_use_virtual_origin(treenode *subscript)
/* This returns TRUE if the subscript expression is of the form
   expression + constant, expression - constant, or constant + expression.
   This means that we can 'fold' out the constant into the array base address
   calculation; this is known as using a virtual origin for the array.
   CON 9/2/91.

   We do this when rangechecking is off, so that we needn't worry
   about its effects on subscript checking, etc.

   NOTE: If we determine that the result is TRUE, we `normalise' the subscript
   tree to make it expression + constant.
   (IE - we move the constant to the rhs, and make it always an addition).
*/
{
  if (NEED_ERRORS)
    ; /* skip */
  else if (TagOf(subscript) == S_ADD)
    {
      treenode *l;
      treenode *r = RightOpOf(subscript);
      if (isconst(r))
        return TRUE;
      l = LeftOpOf(subscript);
      if (isconst(l))
        {
          /* first we `normalise' the subscript tree, by moving the constant
             to the right hand side
          */
          SetRightOp(subscript, l);
          SetLeftOp (subscript, r);
          return TRUE;
        }
    }
  else if (TagOf(subscript) == S_SUBTRACT)
    {
      treenode *r = RightOpOf(subscript);
      if (isconst(r))
        {
          /* we normalise the rhs to be an addition of the negated value */
          /* We rely here on the fact that a subscript expression
             MUST be an integer, so we needn't bother with worrying
             about the high part of the constant.
          */
          SetTag(subscript, S_ADD);
          SetLoVal(r, -LoValOf(r));
          return TRUE;
        }
    }
  return FALSE;
}
/*}}}*/
/*{{{  PRIVATE treenode *transformsubscript (...)*/
/*****************************************************************************
 *
 *  transformsubscript takes a subscript tree 'tptr'
 *                     and transforms it into something more suitable
 *                     for code generation.
 *                     offset is a constant offset from the value of the
 *                     subscript tree to the actual item.
 *                     If rangechecking is TRUE, subscript checks are inserted
 *                     into the tree as required.
 *
 *                     New version which realises that we don't actually
 *                     need useoffset, as long as we fold the constant offset
 *                     into the expression when we hit a hidden dimension.
 *
 *                     Converts ARRAYITEMs back into ARRAYSUBs, and
 *                     SEGMENTITEMs back into SEGMENTs.
 *
 *****************************************************************************/
PRIVATE treenode *transformsubscript ( treenode *tptr , BIT32 *offset , int rangechecking )
{
  treenode *t = NULL; /* result tree */
  treenode *base = issubscript(TagOf(tptr)) ? ASBaseOf(tptr) : SNameOf(tptr);
  const int basetag = TagOf(base);
  treenode *dim = dimexpof(base, 0);
  const int dimisconst = isconst(dim);
  DEBUG_MSG(("transformsubscript: subscript?%d\n", issubscript(TagOf(tptr))));
  if (!dimisconst)
  /*shouldn't that be copytree(base) ? */ /* CON - 24/1/91 */
  /* NO - this breaks all sorts of stuff. We capitulate, and state that
     S_ELSIZE nodes are special and do not obey these rules! - CON 7/2/91 bug 1142 */
    dim = newmopnode(S_ELSIZE, LocnOf(tptr), base, S_INT);
#if 0
  else /* bug 1107 14/1/91 - must create a copy of the dim tree */
    {
      /* Technically, we create a copy of a constant dimension here.
         However, we don't really want to do this unless we're going to use
         this tree node.
         So as an optimisation, this `conversion' is delayed until we know a
         little more about what is happening.
         Note that if !dimisconst, then we've created a new dim node anyway.
         CON 14/1/91.
      */
      /*dim = copytree(dim);*/
      dim = newconstant(LoValOf(dim));
    }
#endif

  /*{{{  transform the base*/
  if
    /*{{{  base is subscripted, transform it*/
    (issubscript(basetag))
      {
        DEBUG_MSG(("transformsubscript: recursing because base was a subscript\n"));
        t = transformsubscript(base, offset, rangechecking);
        /*{{{  multiply up by this dimension*/
        {
          if (dimisconst)
            /*{{{  constant dimension*/
            {
              *offset *= LoValOf(dim);
              if (t != NULL)
                /*t = newdopnode(S_TIMES, NOPOSN, t, dim, S_INT);*/
                /* bug 1107 14/1/91 */
                t = newdopnode(S_TIMES, NOPOSN, t, newconstant(LoValOf(dim)), S_INT);
            }
            /*}}}*/
          else
            /*{{{  hidden dimension*/
            {
              /* We have to give up trying to calculate a constant offset,
                 and add our current offset into the subscript expression. */
              if (*offset != ZERO32)
                {
                  treenode *offsetnode = newconstexpnode(S_CONSTEXP, LocnOf(tptr), dummyexp_p,
                                                         ZERO32, *offset);
                  t = add_to(t, S_ADD, offsetnode);
                  *offset = ZERO32;
                }
            
              if (t != NULL)
                t = newdopnode(S_TIMES, NOPOSN, t, dim, S_INT);
            }
            /*}}}*/
        }
        /*}}}*/
      }
    /*}}}*/
  else if ((basetag == S_SEGMENT) || (basetag == S_SEGMENTITEM))
    /*{{{  base is segmented, transform it*/
      {
        DEBUG_MSG(("transformsubscript: recursing because base was a SEGMENT\n"));
        t = transformsubscript(base, offset, rangechecking);
        /* Don't multiply it up as we are not changing dimension number. */
      }
    /*}}}*/
  else
    {
      /* base is a name */
      *offset = ZERO32;
    }
  /*}}}*/
  /*{{{  transform this node*/
  if
    /*{{{  TagOf(tptr) == S_ARRAYSUB or S_ARRAYITEM*/
    (issubscript(TagOf(tptr)))
      {
        treenode *r = ASIndexOf(tptr);
        SetTag(tptr, S_ARRAYSUB);  /* convert ARRAYITEMs back to ARRAYSUBs */
        SetASExp(tptr, NULL);
        SetASOffset(tptr, 0);
        if
          /*{{{  right is constant, and dimisconst, offset += right*/
          /* rangechecking test added 24/1/91 for bug 1125 */
          /*(isconst(r) && dimisconst)*/
          (isconst(r) && (dimisconst || !rangechecking))
            *offset += LoValOf(r); /* subscript is constant */
          /*}}}*/
      #if 1
        else if (can_use_virtual_origin(r)) /* Optimise a[i+1] */
          {
            *offset += LoValOf(RightOpOf(r)); /* add the constant */
            t = add_to(t, S_ADD, copytree(LeftOpOf(r)));
          }
      #endif
        else
          /*{{{  t = checked right  or t = (ADD, t, checked right)*/
          {
            if (rangechecking && (!isconst(skipevals(r)) || !dimisconst))
              /*{{{  r = checked right*/
              {
                int no_check_needed = FALSE;

                /* shouldn't we have this here? - CON 24/1/91 */ /* YES - bug 1142 5/2/91 */
                r = copytree(r); /* to be added into check exp */

                /* Optimisation for subscripting by replicators: */
                if (dimisconst && TagOf(skipevals(r)) == N_REPL)
                  {
                    treenode *repl  = NDeclOf(skipevals(r));
                    treenode *start = ReplCStartExpOf(repl);
                    treenode *len   = ReplCLengthExpOf(repl);
                    DEBUG_MSG(("transformsubscript: checking for subscript by constant replicator\n"));
                    no_check_needed = isconst(start) && isconst(len) &&
                                      ((INT32)LoValOf(start) >= 0) &&  /* not negative */
                                      /* (we know that dim>=0 and len>=0) */
                                      (LoValOf(start)                <= LoValOf(dim)) &&
                                      /* This line prevents the addition overflowing: */
                                      (LoValOf(len)                  <= LoValOf(dim)) &&
                                      (LoValOf(start) + LoValOf(len) <= LoValOf(dim));
                  }
                if (!no_check_needed)
                  {
                    /* bug 1107 - if dimisconst, we must make dim into
                       a copy of the dim tree */
                    if (dimisconst)
                      dim = newconstant(LoValOf(dim)); /* bug 1107 14/1/91 */
                    r = newdopnode(S_CSUB0, NOPOSN, r, dim, S_INT);
                  }
              }
              /*}}}*/
          
            t = add_to(t, S_ADD, r);
          }
          /*}}}*/
      }
    /*}}}*/
  else
    /*{{{  TagOf(tptr) == S_SEGMENT or S_SEGMENTITEM*/
    {
      SetTag(tptr, S_SEGMENT);  /* convert SEGMENTITEMs back to SEGMENTs */
      /*{{{  create a check tree, put it in the check field, or the start exp*/
      {
        treenode *startexp = skipevals(SStartExpOf(tptr));
        treenode *lengthexp = SLengthExpOf(tptr);
        SOURCEPOSN locn = LocnOf(tptr);
      
        if (rangechecking &&
            (!isconst(startexp) || !isconst(lengthexp) || !dimisconst))
          /*{{{  what the tree looks like*/
          /*
              size is   PLUS    (may be constant)
                        /  \
                      dim   1
          
              base is SEGSTART    (may be constant)
                        |
                      segment
          
              count is ELSIZE    (may be constant)
                         |
                       segment
          
              Basic tree is
                              CSUB0
                             /     \
                         count    MINUS
                                 /     \
                               size   CSUB0
                                      /   \
                                    base  size
          
              if size is not known at compile time,
          
                 if base is constant 0          if count is constant 0
          
                         CSUB0                          CSUB0
                         /   \                          /   \
                      count size                    base    size
          
          
              if count is constant,
          
                         CSUB0
                         /   \
                      MINUS  base
                      /   \
                   size  count
          
              if size is known at compile time and base is constant,
          
                         CSUB0
                         /   \
                      MINUS  count
                      /   \
                   size  base
          
            'segment' is the original segment. Note that this introduces a cycle
              into the tree, but only via the special ELSIZE and SEGSTART nodes.
          */
          /*}}}*/
          {
            treenode *checkexp = NULL;
            int sizeknown, countknown, baseknown ;
            treenode *size, *count;
            /*{{{  set up sizeknown, size*/
            if (dimisconst)
              {
                sizeknown = TRUE;
                size = newconstant(LoValOf(dim) + 1);
              }
            else
              {
                sizeknown = FALSE;
                size = newdopnode(S_PLUS, locn, dim, newconstant(1), S_INT);
              }
            /*}}}*/
            /*{{{  set up countknown, count*/
            if (isconst(lengthexp))
              {
                countknown = TRUE;
                count = newconstant(LoValOf(lengthexp));
              }
            else
              {
                countknown = FALSE;
                /* shouldn't we use copytree(tptr) instead of tptr here? CON 24/1/91 */
                /* NO - this breaks all sorts of stuff. We capitulate, and state that
                   S_ELSIZE nodes are special and do not obey these rules! - CON 7/2/91 bug 1142 */
                count = newmopnode(S_ELSIZE, locn, tptr, S_INT);
              }
            /*}}}*/
            /*{{{  set up baseknown*/
            baseknown = isconst(startexp);
            /*}}}*/
            if (!sizeknown && baseknown && LoValOf(base) == ZERO32)
              /*{{{  checkexp := count CSUB0 size*/
              checkexp = newdopnode(S_CSUB0, locn, count, size, S_INT);
              /*}}}*/
            else if (countknown)
              /*{{{  we can put the check expression in the start field*/
              {
                /* Put these checks into the start expression, so that we get better
                   code sequences */
                treenode *base = SStartExpOf(tptr);
                if (!sizeknown && LoValOf(count) == ZERO32)
                  /*{{{  checkexp := base CSUB0 size*/
                    checkexp = newdopnode(S_CSUB0, locn, base, size, S_INT);
                  /*}}}*/
                else if (sizeknown)
                  /*{{{  checkexp := base CSUB0 (size MINUS count)*/
                  {
                    SetLoVal(size, LoValOf(size) - LoValOf(count));
                    checkexp = newdopnode(S_CSUB0, locn, base, size, S_INT);
                  }
                  /*}}}*/
                else
                  /*{{{  checkexp := base CSUB0 (size MINUS count)*/
                  {
                    checkexp = newdopnode(S_MINUS, locn, size, count,    S_INT);
                    checkexp = newdopnode(S_CSUB0, locn, base, checkexp, S_INT);
                  }
                  /*}}}*/
                SetSStartExp(tptr, checkexp);
                checkexp = NULL;
              }
              /*}}}*/
            else if (sizeknown && baseknown)
              /*{{{  checkexp := count CSUB0 (size MINUS base)*/
              {
                SetLoVal(size, LoValOf(size) - LoValOf(startexp));
                checkexp = newdopnode(S_CSUB0, locn, count, size, S_INT);
              }
              /*}}}*/
            else
              /*{{{  checkexp := count CSUB0 (size MINUS (base CSUB0 size))*/
              {
                treenode *base;
                /*{{{  set up base*/
                if (isconst(startexp))
                  base = newconstexpnode(S_CONSTEXP, locn, dummyexp_p,
                                         ZERO32, LoValOf(startexp));
                else
                  /* shouldn't we use copytree(tptr) instead of tptr here? CON 24/1/91 */
                  /* NO - this breaks all sorts of stuff. We capitulate, and state that
                     S_SEGSTART nodes are special and do not obey these rules! - CON 7/2/91 bug 1142 */
                  base = newmopnode(S_SEGSTART, locn, tptr, S_INT);
                /*}}}*/
                checkexp = newdopnode(S_CSUB0, locn, base,  size,     S_INT);
                checkexp = newdopnode(S_MINUS, locn, size,  checkexp, S_INT);
                checkexp = newdopnode(S_CSUB0, locn, count, checkexp, S_INT);
              }
              /*}}}*/
            SetSCheckExp(tptr, checkexp);
          }
      }
      /*}}}*/
      if
        /*{{{  start exp is constant, offset += startexp*/
        (isconst(SStartExpOf(tptr)))                  /* subscript is constant */
          *offset += LoValOf(SStartExpOf(tptr));
        /*}}}*/
      else
        /* shouldn't we use copytree(tptr) instead of tptr here? CON 24/1/91 */
        /* NO - this breaks all sorts of stuff. We capitulate, and state that
           S_SEGSTART nodes are special and do not obey these rules! - CON 7/2/91 bug 1142 */
        t = add_to(t, S_ADD, newmopnode(S_SEGSTART, LocnOf(tptr), tptr, S_INT));
    }
    /*}}}*/
  /*}}}*/
  return(t);
}
/*}}}*/
/*{{{  PRIVATE treenode *scalesubscript (type, t, offset)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  scalesubscript takes a transformed subscript expression 't', together
 *                 with its constant offset '*offset' and scales it by
 *                 the size of an element of type 'type'.  Updates *offset,
 *                 and returns the scaled tree.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE treenode *scalesubscript ( treenode *type , treenode *t , BIT32 *offset )
{
  INT32 scale = elementsin(type);
  const int b = bytesinscalar(basetype(type));
  const int subscripttype = (b > bytesperword && haswsubdb) ? S_INT64 :
                            (b >= bytesperword)             ? S_INT   : S_BYTE;
  /* Note that the offset scale is different from the subscript scale
     when we use wsubdb (ie. subscripttype == S_INT64) */
  const int offsettype = (subscripttype == S_INT64) ? S_INT : subscripttype;

  DEBUG_MSG(("scalesubscript: scale is %d\n", scale));
  if (scale == (-1))
    /*{{{  scale contains hidden dimensions*/
    {
      if (*offset != ZERO32)
        /*{{{  add offset into subscript expression before we scale it*/
        {
          t = add_to(t, S_ADD, newconstant(*offset));
          *offset = ZERO32;
        }
        /*}}}*/
      if (t != NULL)
        t = add_to(t, S_TIMES, scaletreeof(type, subscripttype));
    }
    /*}}}*/
  else
    /*{{{  constant scale*/
    {
      const INT32 offsetscale = scale * (b / bytesinscalar(offsettype));
      scale *= b / bytesinscalar(subscripttype);
      if ((scale != 1) && (t != NULL))
        t = add_to(t, S_TIMES, newconstant(scale));
      *offset *= offsetscale;
    }
    /*}}}*/
  if (t != NULL)
    t = foldexp(t);
  return(t);
}
/*}}}*/
/*}}}*/

/*{{{  PUBLIC treenode *unknowndimsof (type)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  unknowndimsof
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *unknowndimsof ( treenode *type )
{
  treenode *t = NULL;
  while (TagOf(type) == S_ARRAY)
    {
      if (ARDimOf(type) == (-1))
        t = add_to(t, S_TIMES, ARDimLengthOf(type));
      type = ARTypeOf(type);
    }
  return(t);
}
/*}}}*/
/*{{{  PUBLIC treenode *transformelement (tptr, rangechecking)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  transformelement  creates a subscript expression tree which includes
 *                    this node and any nested ARRAYSUB and SEGMENT nodes,
 *                    (or nested ARRAYITEM or SEGMENTITEM nodes, which are
 *                     converted back into their respective ARRAYSUB etc)
 *                    converts this node to an ARRAYITEM node if it was an
 *                    ARRAYSUB node, or a SEGMENTITEM node if it was a SEGMENT
 *                    node, and inserts the subscript expression into it.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC treenode *transformelement ( treenode *tptr , int rangechecking )
{
  BIT32 offset = ZERO32;
  treenode *subscriptexp;
  int t;
  
  DEBUG_MSG(("transformelement: %s\n", issubscript(TagOf(tptr)) ? "ARRAYSUB" : "SEGMENT"));
  t = typeof(tptr);
  /*{{{  build the subscript expression tree*/
  /*subscriptexp = transformsubscript(tptr, &offset, &useoffset,rangechecking);*/
  subscriptexp = transformsubscript(tptr, &offset, rangechecking);
  /*}}}*/
  if (TagOf(tptr) == S_ARRAYSUB)
    /*{{{  make ARRAYITEM node*/
    {
      DEBUG_MSG(("transformelement: converting into ARRAYITEM node\n"));
      subscriptexp = scalesubscript(gettype(tptr), subscriptexp, &offset);
      SetTag(tptr, S_ARRAYITEM);
      /* leave the ASBase and ASIndex fields unchanged */
      SetASExp(tptr, subscriptexp);
      SetASOffset(tptr, offset);
    }
    /*}}}*/
  else
    /*{{{  make SEGMENTITEM node*/
    {
      DEBUG_MSG(("transformelement: converting into SEGMENTITEM node\n"));
      subscriptexp = scalesubscript(ARTypeOf(gettype(tptr)),subscriptexp, &offset);
      SetTag(tptr, S_SEGMENTITEM);
      SetSSubscriptExp(tptr, subscriptexp);
      SetSOffset(tptr, offset);
    }
    /*}}}*/

  /*{{{  debugging code*/
  /*
  fprintf(outfile, "\nSubscript/segment transformed to : \n");
  printtree(0, tptr);
  fputc('\n', outfile);
  */
  /*}}}*/
  return (tptr);
}
/*}}}*/
/*{{{  PRIVATE treenode *buildoverlapcheck (treenode *tptr)*/
/*****************************************************************************
 *
 *  buildoverlapcheck takes an overlapcheck tree 'tptr' and returns an
 *                    expression tree which will perform the overlap check.
 *
 *****************************************************************************/
PRIVATE treenode *buildoverlapcheck ( treenode *tptr )
{
  treenode *lh_subscripts = LeftOpOf(tptr);
  treenode *rh_subscripts = RightOpOf(tptr);
  SOURCEPOSN locn = LocnOf(tptr);
  treenode *result = NULL;
  while (!EndOfList(lh_subscripts) && !EndOfList(rh_subscripts))
    /*{{{  add this check to result*/
    {
      treenode *lhs = ThisItem(lh_subscripts);
      treenode *rhs = ThisItem(rh_subscripts);
      treenode *checktree = NULL;
      if (TagOf(lhs) == S_FOR || TagOf(rhs) == S_FOR)
        /*{{{  checktree =  full check*/
        /*{{{  COMMENT what the full check tree looks like*/
        /**********************  Start comment out ****************************
        The full check is
        
                            BITAND
                         /         \
                      GR             GR
                     /  \           /  \
                   ADD  base2     ADD  base1
                 /     \        /     \
              base1  count1  base2  count2
        If this is TRUE, we have an overlap.
         **********************   End comment out  ****************************/
        /*}}}*/
        {
          treenode *base1, *count1, *base2, *count2;
          treenode *c1, *c2;
          /*{{{  set up base1, count1, base2, count2*/
          if (TagOf(lhs) == S_FOR)
            { base1 = LeftOpOf(lhs); count1 = RightOpOf(lhs); }
          else
            { base1 = lhs;           count1 = newconstant(1); }
          if (TagOf(rhs) == S_FOR)
            { base2 = LeftOpOf(rhs); count2 = RightOpOf(rhs); }
          else
            { base2 = rhs;           count2 = newconstant(1); }
          /*}}}*/
          c1 = newdopnode(S_ADD, locn, base1, count1,          S_INT);
          c1 = newdopnode(S_GR,  locn, c1,    copytree(base2), S_INT);
          c2 = newdopnode(S_ADD, locn, base2, count2,          S_INT);
          c2 = newdopnode(S_GR,  locn, c2,    copytree(base1), S_INT);

          /* bug 1117 29/1/91 - one half of these may be constant TRUE.
             if so, we must constant fold it out:
             (if it were constant FALSE, it would not have been generated
              by the alias checker)
             (similarly, they cannot both be constant TRUE)
          */
          c1 = foldexp(c1); c2 = foldexp(c2);
          if      (isconst(c1)) /* must be constant TRUE */
            checktree = c2;
          else if (isconst(c2)) /* must be constant TRUE */
            checktree = c1;
          else
            checktree = newdopnode(S_BITAND, locn, c1, c2, S_INT);
        }
        /*}}}*/
    #if 0  /* Constant FALSE possibilities are now removed in the frontend */
      else if (isconst(lhs) && isconst(rhs))
        /* don't need a check here */
        /* If they are different, we don't need a check at all */
        {
          if (LoValOf(lhs) != LoValOf(rhs))
            return dummyexp_p;
        }
    #endif
      else
        /*{{{  checktree = simple check*/
        checktree = newdopnode(S_EQ, locn, lhs, rhs, S_INT);
        /*}}}*/
      if (checktree != NULL)
        {
        /* 1/11/90 Added for bug 1032 to prevent an alias check of
           a[i][0] and a[j][0] inserting an AND TRUE for the 0 against 0 check.
           Theoretically we should also check that the result of folding the
           checktree is TRUE, but we've already removed all FALSE checks
           in the frontend.
        */
          checktree = foldexp(checktree);
          if ((TagOf(checktree) == S_CONSTEXP) /*&& (LoValOf(checktree) != 0)*/)
            checktree = NULL; /* remove this check */
        }
      if (checktree != NULL)
        /*{{{  AND checktree into result*/
        {
          if (result == NULL)
            result = checktree;
          else
            result = newdopnode(S_BITAND, locn, result, checktree, S_INT);
        }
        /*}}}*/
      lh_subscripts = NextItem(lh_subscripts);
      rh_subscripts = NextItem(rh_subscripts);
    }
    /*}}}*/
  /* Now we don't need to bother folding the whole thing, because we've
     folded each check individually - 1/11/90 bug 1032 */
  result = newdopnode(S_CSUB0, locn, result, newconstant(1), S_INT);
  /*result = foldexp(result);*/
  /* fprintf(outfile, "\nOverlap tree generated :\n");
     printtree(0, result); */
  return result;
}
/*}}}*/
/*{{{  PRIVATE void overlapwarnings()*/
PRIVATE void overlapwarnings (SOURCEPOSN locn)
{
  if ((overlap_checks > 0) && (warning_flags & WARNING_OVERLAPS))
    {
      genlocn = locn;
      if (overlap_checks > 1)
        genwarn_i(GEN_RUN_TIME_OVERLAP_CHECKS, overlap_checks);
      else
        genwarn(GEN_RUN_TIME_OVERLAP_CHECK);
    }
}
/*}}}*/ 
/*{{{  PRIVATE void transsubscripts (treenode *tptr)*/
/*****************************************************************************
 *
 *  transsubscripts performs trans on segments and subscripts.
 *                  For segments, temporaries are inserted where needed on
 *                  the start and length expressions.
 *                  No tree transformation is done, this is performed by
 *                  trans on the top-level subscript only.
 *
 *****************************************************************************/
PRIVATE void transsubscripts ( treenode *tptr )
{
  while (TRUE)
    switch(nodetypeoftag(TagOf(tptr)))
      {
        /*{{{  SEGMENT SEGMENTITEM*/
        case SEGMENTNODE:
          {
            int saved_optimise_valof;
            DEBUG_MSG(("transsubscripts: SEGMENTNODE. Now calling trans on SStartExp\n"));
            sub_transexp(SStartExpAddr(tptr));
            DEBUG_MSG(("transsubscripts: SEGMENTNODE. Now calling trans on SLengthExp\n"));
            saved_optimise_valof = optimise_valof;
            optimise_valof = FALSE; /* Fix for bug 634 */
            sub_transexp(SLengthExpAddr(tptr));
            optimise_valof = saved_optimise_valof;
            if (TagOf(tptr) == S_SEGMENTITEM)  /* Added by CO'N 6/4/90 */
              {
                DEBUG_MSG(("transsubscripts: SEGMENTITEM. Now calling trans on SSubscriptExp and SCheckExp\n"));
                sub_transexp(SSubscriptExpAddr(tptr));
                if (SCheckExpOf(tptr) != NULL)
                  sub_transexp(SCheckExpAddr(tptr));
              }
            DEBUG_MSG(("transsubscripts: SEGMENTNODE. Recursing on SNameOf\n"));
            transsubscripts(SNameOf(tptr));
            /*{{{  put temporaries on start and length expressions if necessary*/
            {
              treenode *lengthexp = SLengthExpOf(tptr);
              treenode *startexp = SStartExpOf(tptr);
              treenode *s = skipevals(startexp);
              int need_check = (SSubscriptExpOf(tptr) != NULL) && (SCheckExpOf(tptr) != NULL);
              if (!isconst(s) && !issimplelocal(s) && need_check)
                /* we would evaluate it twice - once in the range check and once in the
                   base, otherwise */
                {
                  DEBUG_MSG(("transsubscripts: creating temp for startexp\n"));
                  SetSStartExp(tptr, newtempnode(T_TEMP, startexp, NM_WORKSPACE));
                }
              /* we nearly always convert to a temporary here, incase the
                 value is required as a range check later;
                 that's why we ignore the 'need_check' test: */
              if (!isconst(lengthexp) && !issimplelocal(lengthexp) /*&& need_check*/)
                {
                  DEBUG_MSG(("transsubscripts: creating temp for lengthexp\n"));
                  SetSLengthExp(tptr, newtempnode(T_TEMP, lengthexp, NM_WORKSPACE));
                }
            }
            /*}}}*/
          }
          return;
        /*}}}*/
        /*{{{  ARRAYSUB*/
        case ARRAYSUBNODE:
          DEBUG_MSG(("transsubscripts: ARRAYSUBNODE. recursing on ASExpOf\n"));
          sub_transexp(ASExpAddr(tptr));
          DEBUG_MSG(("transsubscripts: ARRAYSUBNODE. looping down ASBaseOf\n"));
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        /*{{{  CONSTRUCTOR*/
        case MOPNODE:
          if (TagOf(tptr) == S_CONSTRUCTOR)
            sub_transexp(OpAddr(tptr));
          return;
        /*}}}*/
        default:
          return;
      }
}
/*}}}*/
/*{{{  PRIVATE treenode ** transspecs(treenode **tptr)*/
PRIVATE treenode ** transspecs(treenode **tptr)
{
  while (isspecification(*tptr))
    {
      treenode *const t = *tptr;
      genlocn = LocnOf(t);
      switch (TagOf(t))
        {
          default:
            badtag(genlocn, TagOf(t), "transspecs");
            break;
          /*{{{  routine specification*/
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
            if (!inline(DNameOf(t)))  /* Added 29/5/90 by CO'N for bug 315 */
              {
                treenode *n = DNameOf(t);
                lexlevel++;  /* Formal parameters are at a higher lexical level */
                augmentformals(n);
                if (!separatelycompiled(n))
                  trans(DValAddr(t));
                lexlevel--;
              }
            break;
          /*}}}*/
          /*{{{  other specification*/
          case S_DECL: /* we have to set the access mode */
            { /* moved from the mapper 15/2/91 for bug 1156 */
              treenode *nptr = DNameOf(t);
              if (TagOf(nptr) == S_LIST)
                for (; !EndOfList(nptr); nptr = NextItem(nptr))
                  (void)isinvectorspace(ThisItem(nptr)); /* does SetNMode */
              else
                  (void)isinvectorspace(nptr); /* does SetNMode */
            }
            break;
          case S_TPROTDEF: case S_SPROTDEF:
          case S_WSPLACE: case S_VSPLACE:
          case S_LABELDEF:
            break;
          case S_VALABBR: case S_VALRETYPE: case S_ABBR: case S_RETYPE:
            {
              treenode **rhsaddr = DValAddr(t);
              treenode *nptr = DNameOf(t);
              const int tag = TagOf(nptr);
              SOURCEPOSN thislocn = genlocn;
              const int wasconstant = isconst(*rhsaddr); /* bug 1040 13/11/90 */
              overlap_checks = 0;
              transexp(rhsaddr);
              overlapwarnings(thislocn);
              /*{{{  Mode must be set*/
              /* bug 1156 - 15/2/91 
                 We have to set the access mode of the variable */
              switch(abbrevmode(t))
                {
                  case AM_CONST:
                    /* Though it isn't really, just give it a valid mode */
                    SetNMode(nptr, NM_WORKSPACE);
                    break;
                  case AM_ISRHS:
                    SetNMode(nptr, NModeOf(nameof(*rhsaddr)));
                    break;
                  case AM_PTR:
                    {
                    #if 0 /* removed 19/9/90 */
                      treenode *rhsname = nameof(rhs);
                      if (TagOf(rhsname) != S_CONSTCONSTRUCTOR && TagOf(rhsname) != S_STRING &&
                          TagOf(rhsname) != S_CONSTRUCTOR      && isinvectorspace(rhsname))
                        SetNMode(nptr, NM_VECSPACE);
                      else
                    #endif
                        SetNMode(nptr, NM_POINTER);
                    }
                    break;
                  default: /* case AM_VAL: */
                    (void)isinvectorspace(nptr); /* does SetNMode */
                    break;
                }
              /*}}}*/
              /*{{{  fill in the unknown array dimensions*/
              {
                treenode *type = NTypeOf(nptr);
                if ((tag == N_VALABBR) || (tag == N_ABBR))
                  /*{{{  copy dimensions from the type of the right-hand side*/
                  {
                    int dimension = 0;
                    for(; TagOf(type) == S_ARRAY; type = ARTypeOf(type))
                      {
                        if (ARDimOf(type) == (-1))
                          SetARDimLength(type, dimexpof(*rhsaddr, dimension));
                        dimension++;
                      }
                  }
                  /*}}}*/
                else
                  /*{{{  there is at most one hidden dimension in a retype*/
                  {
                    for (; TagOf(type) == S_ARRAY; type = ARTypeOf(type))
                      {
                        if (ARDimOf(type) == (-1))
                          /* Make up a temporary for the open dimension */
                          {
                            treenode *temp = newtempnode(T_TEMP, dummyexp_p, NM_WORKSPACE);
                            DEBUG_MSG(("transspecs: creating temp dimension at lexlevel %d\n", lexlevel));
                            SetARDimLength(type, temp);
                          }
                      }
                  }
                  /*}}}*/
              }
              /*}}}*/
              if (!wasconstant && isconst(*rhsaddr)) /* bug 1040 13/11/90 */
                {
                  /* We keep on getting bugs because of non-constants in the
                     front-end being turned into constants in the back end
                     by INLINING functions.
                     This makes sure that we don't - bug 1040 13/11/90 */
                  *rhsaddr = newvalofnode(S_VALOF, thislocn,
                                          newleafnode(S_SKIP, thislocn),
                                          newlistnode(S_LIST, thislocn, *rhsaddr, NULL));
                }
            }
            break;
          /*}}}*/
          /*{{{  PLACE*/
          case S_PLACE:
            transexp(DValAddr(t));
            break;
          /*}}}*/
        }
      tptr = DBodyAddr(t);
    }
  return tptr;
}
/*}}}*/
/*{{{  PRIVATE void sub_transexp(treenode **tptr)*/
PRIVATE void sub_transexp (treenode **tptr)
/* This is trans for an expression. */
{
  while (*tptr != NULL)
    {
      treenode *t;
      treenode **original_tptr = tptr; /* used to optimise valofs */
      tptr = transspecs(tptr);
      t = *tptr;
      if (t == NULL) return;
      genlocn = LocnOf(t);
      switch(nodetypeoftag(TagOf(t)))
        /*{{{  cases*/
        {
          default:
            badtag(genlocn, TagOf(t), "sub_transexp");
            break;
          /*{{{  LIST*/
          case LISTNODE:
            sub_transexp(ThisItemAddr(t));
            tptr = NextItemAddr(t);
            break;
          /*}}}*/
          /*{{{  monadic and conversion*/
          case MOPNODE:
            switch (TagOf(t))
              {
                default:
                  tptr = OpAddr(t);
                  break;
                case S_ELSIZE: case S_SEGSTART:
                  return;
                case S_EXACT: case S_ROUND: case S_TRUNC: /* Added 20/6/90 for bug 334 */
                  {
                    int sourcetype = typeof(OpOf(t));
                    int desttype   = MOpTypeOf(t);
                    if (((desttype   == S_INT) ? targetintsize : desttype  ) ==
                        ((sourcetype == S_INT) ? targetintsize : sourcetype))
                      *tptr = OpOf(t);
                    else
                      tptr = OpAddr(t);
                  }
                  break;
              }
            break;
          /*}}}*/
          /*{{{  dyadic*/
          case DOPNODE:
            switch(TagOf(t))
              {
                default:
                  sub_transexp(LeftOpAddr(t));
                  tptr = RightOpAddr(t);
                  break;
                case S_LSHIFT: case S_RSHIFT:
                  if ((errormode & ERRORMODE_SHIFTCHECK) && !isconst(RightOpOf(t)))
                    /*{{{  add tree to range check the shift count*/
                    {
                      INT32 shiftrange = (8 * bytesinscalar(DOpTypeOf(t))) + 1;
                      SetRightOp(*tptr, newdopnode(S_CSUB0, 0, RightOpOf(t),
                                                   newconstexpnode(S_CONSTEXP, 0, dummyexp_p,
                                                                   ZERO32, shiftrange),
                                        S_INT));
                    }
                    /*}}}*/
                  sub_transexp(LeftOpAddr(t));
                  tptr = RightOpAddr(t);
                  break;
                case S_EVAL:
                  if (TagOf(LeftOpOf(t)) == S_OVERLAPCHECK)
                    {
                      SetLeftOp(*tptr, buildoverlapcheck(LeftOpOf(t)));
                      if (TagOf(LeftOpOf(t)) != S_DUMMYEXP)
                        overlap_checks++;
                    }
                  if (TagOf(LeftOpOf(t)) == S_DUMMYEXP)
                    *tptr = RightOpOf(t);
                  else
                    {
                      sub_transexp(LeftOpAddr(t));
                      tptr = RightOpAddr(t);
                    }
                  break;
              }
            break;
          /*{{{  function instance*/
          case INSTANCENODE:
            trans(tptr);  /* use the same code as for procedure instances */
            return;
          /*}}}*/
          /*{{{  element*/
          case ARRAYSUBNODE:
          case SEGMENTNODE:
            *tptr = transformelement(t, RANGECHECKING);
            transsubscripts(*tptr);
            return;
          /*}}}*/
          /*{{{  constants and dummy expression*/
          case NAMENODE: case WORDNODE: case CONSTEXPNODE:
          case CONSTTABLENODE:
          case LEAFNODE: /* S_DUMMYEXP should be the only one */
            return;
          /*}}}*/
          /*{{{  valof*/
          case VALOFNODE: /* this must be an inline valof */
            trans(VLBodyAddr(t));
            transexp(VLResultListAddr(t));
          #if OPTIMISE_VALOFS
          /*We optimise away specifications which have been left in front of
            valofs which can be turned into constants:
            (This can particularly occur with INLINE FUNCTIONs).
            Eg:
            z := increment(1)
            which might be expanded into:
            z := (VAL y IS 2 :
                  VALOF
                    SKIP
                    RESULT 2
                 )
            which is now turned into:
            z := 2
          */
            if (optimise_valof) /* we turn this off while inside segment lengths */
            {
              treenode *result = VLResultListOf(t);
              /* This only works for single valued functions, so we
                 might aswell optimise it just for that case */
            #if 0  /* 20/9/90 */
              int constant = TRUE;
              int results = 0;
              while (constant && !EndOfList(result))
                {
                  constant = constant && isconst(ThisItem(result));
                  result = NextItem(result);
                  results++;
                }
              if (constant && (results == 1))
                *original_tptr = ThisItem(VLResultListOf(t));
            #else
              if ((listitems(result) == 1) && isconst(ThisItem(result)))
                *original_tptr = ThisItem(result);
            #endif
            }
          #endif
            return;
          /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PRIVATE void transexp(treenode **tptr)*/
PRIVATE void transexp (treenode **tptr)
/* This simply calls sub_transexp, and folds the resulting expression tree,
   incase any functions have been in-lined, which provide opportunities for
   constant propagation.
*/
{
  int temp = inline_expansions;
  inline_expansions = FALSE;
  sub_transexp(tptr);
  if (inline_expansions)
    {
      if (TagOf(*tptr) == S_LIST)
        *tptr = foldexplist(*tptr);
      else
        *tptr = foldexp(*tptr);
    }
  inline_expansions = temp;
}
/*}}}*/
/*{{{  PRIVATE void trans(treenode **tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  trans takes a complete SC tree and performs the following transformations:
 *        1.  ARRAYSUB trees are expanded to ARRAYITEM trees
 *        2.  SEGMENT trees are expanded to SEGMENTITEM trees
 *        3.  Temporaries are inserted (but not scoped) on segment start and
 *            length trees if required
 *        4.  Hidden dimensions are inserted in all type trees.
 *        5.  Formal parameter lists are augmented with hidden dimensions
 *            and result pointers.
 *        6.  Checks are added to shift operators if required.
 *        7.  Inline routine calls are expanded out.
 *        8.  All variables have their SetNMode field set correctly.
 *
 *  N.B. lexlevel must have the correct value upon calling 'trans', and it is
 *       altered within trans.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void trans ( treenode **tptr )
{
  while (*tptr != NULL)
    {
      treenode *t;
      tptr = transspecs(tptr);
      t = *tptr;
      if (t == NULL) return;
      genlocn = LocnOf(t);
      switch(nodetypeoftag(TagOf(t)))
        /*{{{  cases*/
        {
          default:
            badtag(genlocn, TagOf(t), "trans");
            break;
          /*{{{  SEQ IF ALT PAR PRIALT PRIPAR GUY ASM*/
          case CNODE:
            tptr = CBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  REPLSEQ REPLIF REPLALT REPLPAR PRIREPLALT PRIREPLPAR*/
          case REPLCNODE:
            transexp(ReplCStartExpAddr(t));
            transexp(ReplCLengthExpAddr(t));
            if (parrepl(TagOf(t)))
              {
                lexlevel++;
                DEBUG_MSG(("trans: raising lex level as we enter a repl par\n"));
                trans(ReplCBodyAddr(t));
                lexlevel--;
                DEBUG_MSG(("trans: lowering lex level as we leave a repl par\n"));
                return;
              }
            else
              tptr = ReplCBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  WHILE CHOICE*/
          case CONDNODE:
            transexp(CondGuardAddr(t));
            tptr = CondBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  VARIANT*/
          case VARIANTNODE:
            transexp(VRTaggedListAddr(t));
            tptr = VRBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  ALTERNATIVE*/
          case ALTNODE:
            {
              /* modified 2/11/90 to create a copy of the channel expression
                 so that it isn't mapped twice; once when enabling, and
                 once when the input is actually performed */
              treenode *input = skipspecifications(AltInputOf(t));
              switch(inputtypeof(input))
                {
                  case INP_INPUT: case INP_CASE_INPUT: case INP_TAGGED_INPUT:
                    SetAltChanExp(*tptr, copytree(LHSOf(input)));
                    break;
                  default:
                    break; /* leave the channel expression as NULL */
                }
              if (AltGuardOf(t) == NULL)
                SetAltGuard(*tptr, trueguard);
              transexp(AltGuardAddr(t));
              transexp(AltChanExpAddr(t));
              trans(AltInputAddr(t));
              tptr = AltBodyAddr(t);
            }
            break;
          /*}}}*/
          /*{{{  LIST*/
          case LISTNODE:
            trans(ThisItemAddr(t));
            tptr = NextItemAddr(t);
            break;
          /*}}}*/
          /*{{{  SKIP STOP END*/
          case LEAFNODE:
            return;
          /*}}}*/
          /*{{{  configuration*/
          /*{{{  PROCESSOR*/
          case PROCESSORNODE:
            tptr = ProcessorBodyAddr(t);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  instance*/
          /*{{{  PINSTANCE FINSTANCE*/
          case INSTANCENODE:
            {
              treenode *name = INameOf(t);
              const SOURCEPOSN locn = genlocn;
              const int saved_checks = overlap_checks;
              /* we have to call trans on actual parameters first, so that
                 we correctly expand actual parameters which are themselves
                 instances of inlined functions - bug 1004 4/10/90 */
              overlap_checks = 0;
              transexp(IParamListAddr(t)); /* do params first */
              overlapwarnings(locn); /* bug 1118 25/1/91 */
              overlap_checks = saved_checks;
              if (inline(name))
                /*{{{  expand out the inline routine definition*/
                {
                  treenode *inlinetree = NULL;
                  treenode **inlinetreeptr = &inlinetree;
                  treenode *fparams = NParamListOf(name);
                  treenode *aparams = IParamListOf(t);
                  int inside_fn = (TagOf(t) == S_FINSTANCE);
                  marknametrans();
                  while (!EndOfList(fparams))
                    /*{{{  make up an abbreviation for the formal parameter*/
                    {
                      treenode *aparam = ThisItem(aparams);
                      treenode *fparam = ThisItem(fparams);
                      if (TagOf(aparam) == S_CONSTEXP || issimple(aparam))
                        addnametrans(fparam, aparam);
                      else
                        {
                          int abbrtag = (TagOf(fparam) == N_PARAM ? S_ABBR : S_VALABBR);
                          treenode *fabbrev = declare(abbrtag, genlocn,
                                                      copytree(NTypeOf(fparam)),
                                                      NNameOf(fparam), aparam);
                          addnametrans(fparam, DNameOf(fabbrev));
                          *inlinetreeptr = fabbrev; inlinetreeptr = DBodyAddr(fabbrev);
                        }
                    
                      fparams = NextItem(fparams);
                      aparams = NextItem(aparams);
                    }
                    /*}}}*/
                  /* we pass in the location of the instance, which makes
                     the whole copied tree look like it all appeared
                     on the same line of source as the instance.
                     CON 9/10/90 */
                  *inlinetreeptr = transcopytree(DValOf(NDeclOf(name)),
                                                 LocnOf(t));
                  freenametrans();
                  foldtree(inlinetree); /* fold procedure instance */
                  /*debug
                  fprintf(outfile, "Inline routine instance expanded to :\n");
                  printtree(0, inlinetree);
                  */
                  *tptr = inlinetree;
                  inline_expansions = TRUE;
                  if (inside_fn)
                    sub_transexp(tptr);
                  else
                    trans(tptr);
                }
                /*}}}*/
              else
                {
                  /* transexp(IParamListAddr(t)); */ /* done already */
                  if ((TagOf(t) == S_FINSTANCE) &&
                      (TagOf(name) == N_PREDEFFUNCTION))
                    {
                      switch (NModeOf(name))
                        {
                          case PD_ASHIFTLEFT: case PD_ASHIFTRIGHT:
                          case PD_ROTATELEFT: case PD_ROTATERIGHT:
                          case PD_BITREVNBITS:
                            {
                              /* insert a range check on the 'places' (second) parameter */
                              treenode *param2 = NextItem(IParamListOf(t));
                              if ((errormode & ERRORMODE_SHIFTCHECK) && (!isconst(ThisItem(param2))))
                                NewItem(newdopnode(S_CSUB0, 0, ThisItem(param2),
                                        newconstexpnode(S_CONSTEXP, 0, dummyexp_p,
                                                        ZERO32, (8 * bytesinscalar(S_INT)) + 1), S_INT),
                                        param2);
                            }
                            break;
                          default:
                            break;
                        }
                    }
                }
            }
            return;
          /*}}}*/
          /*}}}*/
          /*{{{  action*/
          case ACTIONNODE:
            switch(TagOf(t))
              {
                case S_CASE: case S_CASE_INPUT:
                  transexp(LHSAddr(t));

                  if ((TagOf(t) == S_CASE) /* added for bug 1052 - 23/11/90 */
                      && isshortint(typeof(LHSOf(t))))
                    {
                      SetLHS(*tptr, newmopnode(S_EXACT, LocnOf(t),
                                               LHSOf(t), S_INT));
                      /* really we should constant fold here, but nobody uses
                         constants in CASE expressions! */
                      /* Note too that we don't need to change the type of
                         all the selection values - they fall out in the wash */
                    }
                  tptr = RHSAddr(t);
                  break;
                default:
                  {
                    SOURCEPOSN thislocn = genlocn;
                    overlap_checks = 0;
                    transexp(LHSAddr(t));
                    transexp(RHSAddr(t));
                    overlapwarnings(thislocn);
                  }
                  return;
              }
            break;
          /*}}}*/
          /*{{{  valof*/
          case VALOFNODE:  /* This must be a function body */
            trans(VLBodyAddr(t));
            transexp(VLResultListAddr(t));
            return;
          /*}}}*/
          /*{{{  GUYCODE GUYSTEP*/
          case DOPNODE:
            if (TagOf(t) == S_GUYCODE)
              transexp(RightOpAddr(t));
            return;
          /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC void transmain(treenode *tptr)*/
PUBLIC void transmain ( treenode *tptr )
{
  int temp = lexlevel;
  lexlevel = initlexlevel;
  trans(&tptr);
  lexlevel = temp;
}
/*}}}*/
