
/******************************************************************************
*
*  srcout - source code and assembly interspersed listing - /zo option
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lex1def.h"
# include "generror.h"
# include "lexconst.h"
# include "gen1def.h"
# include "debugdef.h"
# include "codehdr.h"
# include "srcoutde.h"
# include "popen.h"
/*}}}*/

#ifdef TDS
/*{{{  public routines*/
/*{{{  file handling*/
/*{{{  PUBLIC void init_source_code_output()*/
PUBLIC void init_source_code_output ( void )
{
}
/*}}}*/
/*{{{  PUBLIC void end_source_code_output()*/
PUBLIC void end_source_code_output ( void )
{
}
/*}}}*/
/*}}}*/
/*{{{  source code handling*/
/*{{{  PUBLIC void so_endofproc()*/
PUBLIC void so_endofproc ( void )
{
  ;
}
/*}}}*/
/*{{{  PUBLIC void so_endofwhile()*/
PUBLIC void so_endofwhile ( void )
{
  ;
}
/*}}}*/
/*{{{  PUBLIC void so_endofpar(branch, end)*/
PUBLIC void so_endofpar ( int branch , int end )
{
  ;
}
/*}}}*/
/*{{{  PUBLIC void so_stop()*/
PUBLIC void so_stop ( void )
{
  ;
}
/*}}}*/
/*{{{  PUBLIC void output_source_code(index, debug_type)*/
/*{{{  comment*/
/******************************************************************************
*
*  output_source_code(index)
*  Output a chunk of source code to the output file.
*  Andy Whitlow 15.11.88
*
******************************************************************************/
/*}}}*/
PUBLIC void output_source_code ( int index , int debug_type )
{
  ;
}
/*}}}*/
/*}}}*/
/*}}}*/
#else
/*{{{  definitions*/
#define SO_SPACE 0
#define SO_PROC 1
#define SO_COLON 2
#define SO_DECL 3
#define SO_COMMENT 4
#define SO_INCLUDE 5
#define SO_WHILE 6
#define SO_PAR 7
#define SO_PROTOCOL 8
#define SO_OPR 9        /* operator - word type */
#define SO_OTHER 10     /* anything other than those listed */
#define SO_ALT 11
#define SO_SPECALT 12   /* special ALT type */
#define SO_PRIPAR 13
#define SO_PRI 14
#define SO_CASE 15
#define SO_SPECCASE 16  /* special CASE type */
#define SO_NULL 17
/*}}}*/
/*{{{  local variables*/
/*{{{  comment*/
/******************************************************************************
*
*  source code output locals - These locals are used by the source code output
*  code which produces an interleaved listing of source code and disassembly.
*  Andy Whitlow 20.12.88.
*
******************************************************************************/
/*}}}*/
/*{{{  structures*/
/*{{{  line cell*/
struct line_cell  /* used to store all information required about a line */
{
  /* char          *sline;  */
  char             sline[LINEMAX];  /* the line itself                     */
  int              line_number;     /* line number in file                 */
  int              v_line_number;   /* virtual line number                 */
  int              type;            /* line type                           */
  int              file_num;        /* number of file from which line came */
  struct line_cell *next;           /* link to next cell in list           */
};
/*}}}*/
/*{{{  item cell*/
struct item_cell /* used to construct lists or stacks consisting of integers */
{
  int              item;   /* the value to be stored             */
  struct item_cell *next;  /* link to next cell in list or stack */
};
/*}}}*/
/*}}}*/
/*{{{  stacks*/
/*{{{  file stack*/
struct file_stack_cell  /* used to store the context of files */
{
  FILE                   *saved_sfptr;             /* file pointer           */
  char                   saved_proc_name[LINEMAX]; /* name of current routine*/
  char                   saved_saved_line[LINEMAX];/* contents of line buffer*/
  int                    saved_line_number;        /* current line number    */
  int                    saved_filenumber;         /* current file number    */
/*   struct line_cell       *saved_saved_list; */
  struct file_stack_cell *next;                    /* link to next stack cell*/
};
PRIVATE struct file_stack_cell *file_stack;        /* file stack pointer */
/*}}}*/
/*{{{  while/par stack*/
PRIVATE struct item_cell *pw_stack; /* stack pointer for PAR and WHILE numbers */
PRIVATE int curr_while_number;      /* current WHILE identification number     */
PRIVATE int curr_par_number;        /* current PAR identification number       */
/*}}}*/
/*}}}*/
/*{{{  lists and list pointers*/
PRIVATE struct line_cell *lines_list;       /* list of source lines          */
PRIVATE struct line_cell *output_ptr;       /* pointer to current line       */
PRIVATE struct item_cell *block_line_list;  /* list of code block boundaries */
/*}}}*/
/*{{{  character string storage*/
PRIVATE char curr_proc_name[MAXNAMELENGTH]; /* current routine name          */
PRIVATE char saved_line[LINEMAX];           /* saved line buffer             */
/*}}}*/
/*{{{  file variables*/
PRIVATE FILE *sfptr;                        /* source file pointer           */
PRIVATE int continuation;                   /* line continuation flag        */
PRIVATE int filenumber;                     /* file counter                  */
PRIVATE int curr_filenumber;                /* id of current file            */
PRIVATE int prev_filenumber;                /* id of previous file           */
PRIVATE int end_of_file;                    /* EOF flag                      */
/*}}}*/
/*{{{  ALT variables*/
PRIVATE int altflag;                        /* ALT construct flag            */
PRIVATE int done_enb;                       /* ALT enable counter            */
PRIVATE int done_disb;                      /* ALT disable counter           */
PRIVATE int altcount;                       /* ALT sub process count         */
/*}}}*/
/*{{{  sequential code block watermarks*/
PRIVATE int low_block_line;                 /* code block low watermark      */
PRIVATE int high_block_line;                /* code block high watermark     */
/*}}}*/
/*{{{  line number counters*/
PRIVATE int curr_line_number;               /* current line number in file   */
PRIVATE int v_linenumber;                   /* virtual line number counter   */
/*}}}*/
/*}}}*/

/*{{{  private routines*/
/*{{{  forward definition*/
PRIVATE void process_read_line PARMS((char *line, int type));
/*}}}*/
/*{{{  line handling*/
/*{{{  PRIVATE void read_line(line)*/
/*{{{  comment*/
/***************************************************************************
*
*  read_line(line)
*  Read a line into the line buffer.
*  The line is taken from the saved_line buffer if its is not empty,
*  otherwise it is read from the source file.
*  Sets 'end_of_file' to TRUE if EOF encountered.
*  Andy Whitlow 20.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE void read_line ( char *line )
{
  if (*saved_line)
  {
    /*{{{  retrieve saved line*/
    strcpy(line, saved_line);
    *saved_line = '\0';
    /*}}}*/
  }
  else
  {
    /*{{{  get line from file*/
    fgets(line, LINEMAX - 1, sfptr);
    /*{{{  replace end of line character with end of string character*/
    {
      char *c;
      for (c = line; (*c != '\r') && (*c != '\n') && (*c != '\0'); c++)
        ;
      *c = '\0';
    }
    /*}}}*/
    if (feof(sfptr))
      end_of_file = TRUE;
    /*{{{  COMMENT */
    /**********************  Start comment out ****************************
    @*{{{  *@
    while ((c = fgetc(sfptr)) != '\r' && c != EOF)
      *line++ = c;
    *line = '\0';
    if (c != EOF)
    {
      if ((c = fgetc(sfptr)) != '\n')
        ungetc(c, sfptr);
    }
    else
    {
      end_of_file = TRUE;
      return;
    }
    @*}}}*@
     **********************   End comment out  ****************************/
    /*}}}*/
    /*}}}*/
  }
  curr_line_number++;
}
/*}}}*/
/*{{{  PRIVATE void put_line_back(line)*/
/*{{{  comment*/
/***************************************************************************
*
*  put_line_back(line)
*  store line in the saved line buffer.
*  In some instance a line will be read from a file before it is required
*  this line can be stored in the saved line buffer by using this function.
*  read line will read a line from this buffer before trying to read from the
*  file.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE void put_line_back ( char *line )
{
  strcpy(saved_line, line);
  curr_line_number--;
}
/*}}}*/
/*{{{  PRIVATE int is_before_comment(line, word)*/
/*{{{  comment*/
/***************************************************************************
*
*  is_before_comment(line, word)
*  Returns TRUE if the string in 'word' appears in 'line' before an occurence
*  of the comment symbol, i.e. '--'.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE int is_before_comment ( char *line , char *word )
{
  int i, length;

  length = strlen(line) - strlen(word);
  for (i=0; i<=length; i++)
    if (!strncmp(&line[i], "--", 2))
      return(FALSE);
    else if (!strncmp(&line[i], word, strlen(word)))
      return(TRUE);
  return(FALSE);
}
/*}}}*/
/*{{{  PRIVATE char *get_next_word(buffer, word)*/
/*{{{  comment*/
/****************************************************************************
*
*  get_next_word(buffer, word)
*  copies the first word found in buffer to word. Returns a pointer to the
*  position in buffer directly after the found word.
*  returns NULL if end-of-line encountered or if it finds a comment.
*  Andy Whitlow 15.11.88
*
*****************************************************************************/
/*}}}*/
PRIVATE char *get_next_word ( char *buffer , char *word )
{
  /*{{{  skip leading spaces*/
  while (*buffer == ' ')
    buffer++;
  /*}}}*/
  /*{{{  return if line is a comment line*/
  if (!strncmp(buffer, "--", 2))
  {
    strcpy(word, "--");
    return(NULL);
  }
  /*}}}*/
  /*{{{  return if begins with (*/
  if (*buffer == '(')
  {
    *word = '(';
    return(buffer + 1);
  }
  /*}}}*/
  /*{{{  copy next word from buffer to word*/
  while (*buffer != ' ' && *buffer != '\0' &&
         *buffer != '(' && strncmp(buffer, "--", 2))
    *word++ = *buffer++;
  *word = '\0';
  /*}}}*/
  if (*buffer == '\0' || !strncmp(buffer, "--", 2))
    return(NULL);
  else if (*buffer == '(')
    return(buffer + 1); /* skip '(' otherwise might enter infinite loop */
  else
    return(buffer);
}
/*}}}*/
/*{{{  PRIVATE void get_last_word(line, word)*/
/*{{{  comment*/
/***************************************************************************
*
*  get_last_word(line, word)
*  Finds the last word in 'line' and stores it in 'word'.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE void get_last_word ( char *line , char *word )
{
  char prev_word[LINEMAX], *lptr;

  lptr = line;
  while ((lptr = get_next_word(lptr, word)) != NULL)
    strcpy(prev_word, word);
  /*{{{  comment*/
  /* if the line has trailing spaces it is possible that 'word' could contain
     only '\0', also, if the line ends with a trailing comment 'word' will
     contain the comment symbol, '--'. In these cases 'prev_word' contains the
     last word on the line */
  /*}}}*/
  if (*word == '\0' || !strcmp(word, "--"))
    strcpy(word, prev_word);
}
/*}}}*/
/*{{{  PRIVATE int get_word_type(word)*/
/*{{{  comment*/
/****************************************************************************
*
*  get_word_type(word)
*  Returns the line type corresponding to the word held in 'word'.
*  Andy Whitlow 21.12.88
*
*****************************************************************************/
/*}}}*/
PRIVATE int get_word_type ( char *word )
{
  char *wordptr = word;

  switch(*wordptr++)
  {
    /*{{{  cases*/
    /*{{{  A*/
    case 'A':
      /*{{{  AND AFTER*/
      if (!strcmp(wordptr, "ND") || !strcmp(wordptr, "FTER"))
        return(SO_OPR);
      /*}}}*/
      /*{{{  ALT*/
      if (!strcmp(wordptr, "LT"))
        return(SO_ALT);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  B*/
    case 'B':
      /*{{{  BOOL BYTE*/
      if (!strcmp(wordptr, "OOL") || !strcmp(wordptr, "YTE"))
        return(SO_DECL);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  C*/
    case 'C':
      /*{{{  CHAN*/
      if (!strcmp(wordptr, "HAN"))
        return(SO_DECL);
      /*}}}*/
      /*{{{  CASE*/
      if (!strcmp(wordptr, "ASE"))
        return(SO_CASE);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  F*/
    case 'F':
      /*{{{  FROM FOR*/
      if (!strcmp(wordptr, "OR") || !strcmp(wordptr, "ROM"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  I*/
    case 'I':
      /*{{{  IS*/
      if (!strcmp(wordptr, "S"))
        return(SO_OPR);
      /*}}}*/
      /*{{{  INT INT16 INT32 INT64*/
      if (!strcmp(wordptr, "NT") || !strcmp(wordptr, "NT16") ||
          !strcmp(wordptr, "NT32") || !strcmp(wordptr, "NT64"))
        return(SO_DECL);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  M*/
    case 'M':
      /*{{{  MINUS*/
      if (!strcmp(wordptr, "INUS"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  N*/
    case 'N':
      /*{{{  NOT*/
      if (!strcmp(wordptr, "OT"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  O*/
    case 'O':
      /*{{{  OR*/
      if (!strcmp(wordptr, "R"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  P*/
    case 'P':
      /*{{{  PROC*/
      if (!strcmp(wordptr, "ROC"))
        return(SO_PROC);
      /*}}}*/
      /*{{{  PROTOCOL*/
      if (!strcmp(wordptr, "ROTOCOL"))
        return(SO_PROTOCOL);
      /*}}}*/
      /*{{{  PAR*/
      if (!strcmp(wordptr, "AR"))
        return(SO_PAR);
      /*}}}*/
      /*{{{  PLACE*/
      if (!strcmp(wordptr, "LACE"))
        return(SO_DECL);
      /*}}}*/
      /*{{{  PLUS*/
      if (!strcmp(wordptr, "LUS"))
        return(SO_OPR);
      /*}}}*/
      /*{{{  PRI*/
      if (!strcmp(wordptr, "RI"))
        return(SO_PRI);
      /*}}}*/
      /*{{{  PORT*/
      if (!strcmp(wordptr, "ORT"))
        return(SO_DECL);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  R*/
    case 'R':
      /*{{{  REM RETYPES*/
      if (!strcmp(wordptr, "EM") || !strcmp(wordptr, "ETYPES"))
        return(SO_OPR);
      /*}}}*/
      /*{{{  REAL32 REAL64*/
      if (!strcmp(wordptr, "EAL32") || !strcmp(wordptr, "EAL64"))
        return(SO_DECL);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  S*/
    case 'S':
      /*{{{  SIZE*/
      if (!strcmp(wordptr, "IZE"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  T*/
    case 'T':
      /*{{{  TIMER*/
      if (!strcmp(wordptr, "IMER"))
        return(SO_DECL);
      /*}}}*/
      /*{{{  TIMES*/
      if (!strcmp(wordptr, "IMES"))
        return(SO_OPR);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  V*/
    case 'V':
      /*{{{  VAL*/
      if (!strcmp(wordptr, "AL"))
        return(SO_DECL);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  W*/
    case 'W':
      /*{{{  WHILE*/
      if (!strcmp(wordptr, "HILE"))
        return(SO_WHILE);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  #*/
    case '#':
      /*{{{  #INCLUDE*/
      if (!strcmp(wordptr, "INCLUDE"))
      {
        filenumber++;
        return(SO_INCLUDE);
      }
      /*}}}*/
      /*{{{  #USE #SC #IMPORT*/
      if (!strcmp(wordptr, "USE") || !strcmp(wordptr, "SC") ||
          !strcmp(wordptr,  "IMPORT"));
      {
        filenumber++;
        return(SO_OTHER);
      }
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  (*/
    case '(':
      return(SO_OTHER);
      break;
    /*}}}*/
    /*{{{  default*/
    default:
      return(SO_OTHER);
      break;
    /*}}}*/
    /*}}}*/
  }
  return(SO_OTHER);
}
/*}}}*/
/*{{{  PRIVATE int is_break_char(c)*/
/*{{{  comment*/
/****************************************************************************
*
*  is_break_char(c)
*  Returns TRUE if the character 'c' is a char after which a line can be
*  broken.
*  Andy Whitlow 21.12.88
*
*****************************************************************************/
/*}}}*/
PRIVATE int is_break_char ( int c )
{
  switch(c)
  {
    /*{{{  line break characters*/
    case '+':
    case '-':
    case '*':
    case '/':
    case ',':
    case ';':
    case '=':
    case '\\':
    case '<':
    case '>':
    case '~':
      return(TRUE);
    /*}}}*/
    /*{{{  non line break characters*/
    default:
      return(FALSE);
    /*}}}*/
  }
}
/*}}}*/
/*{{{  PRIVATE int continuationlines(line, type)*/
/*{{{  comment*/
/******************************************************************************
*
*  continuationlines(line, type)
*  Returns true if source on line continues on further lines.
*  If line is a blank or a comment continuationlines returns the current value
*  of the continuation flag. This handles the fact that blanks and comments can
*  be interspersed with continuing lines.
*  Andy Whitlow 29.11.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int continuationlines ( char *line , int type )
{
  char word[LINEMAX];

  /*{{{  handle blank lines and comments*/
  if (type == SO_SPACE || type == SO_COMMENT)
    return(continuation);
  /*}}}*/
  /*{{{  get last word before comment*/
  get_last_word(line, word);
  /*}}}*/
  /*{{{  return truth value for last word*/
  if (is_break_char(word[strlen(word) - 1]) || get_word_type(word) == SO_OPR)
    continuation = TRUE;
  else
    continuation = FALSE;
  return(continuation);
  /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE int get_line(line)*/
/*{{{  comment*/
/****************************************************************************
*
*  get_line(line)
*  get a line from the source file into line. Return an integer type which
*  represents the type of line, PROC, :, blank, DECL etc.
*  Andy Whitlow 15.11.88
*
*****************************************************************************/
/*}}}*/
PRIVATE int get_line ( char *line )
{
  int type;
  char firstword[LINEMAX], *lptr = line;

  read_line(line);
  lptr = get_next_word(line, firstword);
  /*{{{  handle comment*/
  if (lptr == NULL && !strncmp(firstword, "--", 2)) /* line is a comment line */
    return(SO_COMMENT);
  /*}}}*/
  /*{{{  blank line*/
  if (*firstword == '\0')
    return(SO_SPACE);
  /*}}}*/
  /*{{{  colon*/
  if (*firstword == ':')
    return(SO_COLON);
  /*}}}*/
  type = get_word_type(firstword);
  /*{{{  catch any occurences of SO_OPR*/
  if (type == SO_OPR)
    return(SO_OTHER);
  /*}}}*/
  /*{{{  PRI*/
  if (type == SO_PRI)
  {
    get_next_word(lptr, firstword);
    type = get_word_type(firstword);
    if (type == SO_PAR)
      return(SO_PRIPAR);    /* mark PRI PAR, PRI ALT will be typed as SO_ALT */
    return(type);
  }
  /*}}}*/
  /*{{{  PROC FUNCTION*/
  if (type == SO_PROC || is_before_comment(line, "FUNCTION"))
  {
    /*{{{  get PROC or FUNCTION name*/
    firstword[0] = '\0';
    lptr = line;
    while (strncmp(firstword, "PROC", 4) && strncmp(firstword, "FUNCTION", 8))
      lptr = get_next_word(lptr, firstword);
    lptr = get_next_word(lptr, curr_proc_name);
    /*}}}*/
    return(SO_PROC);
  }
  /*}}}*/
  /*{{{  array declaration*/
  if  (*firstword == '[' && is_before_comment(line, "]") &&
       !is_before_comment(line, ":="))
    return(SO_DECL);
  /*}}}*/
  return(type);
}
/*}}}*/
/*{{{  PRIVATE int indent_count(line)*/
/*{{{  comment*/
/****************************************************************************
*
*  indent_count(line)
*  Returns the level of indentation of line.
*  Andy Whitlow 21.12.88
*
*****************************************************************************/
/*}}}*/
PRIVATE int indent_count ( char *line )
{
  int count = 0;

  while (*line++ == ' ')
    count++;
  return(count);
}
/*}}}*/
/*}}}*/
/*{{{  file handling*/
/*{{{  PRIVATE void open_source_file(fname)*/
/*{{{  comment*/
/******************************************************************************
*
*  open_source_file(fname)
*  Open the source file with name 'fname'. Use path searching.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void open_source_file ( char *fname )
{
  char full_name[MAX_FILENAME_LENGTH];

  sfptr = popen_read(fname, pathname, &full_name[0], POPEN_MODE_TEXT);
}
/*}}}*/
/*{{{  PRIVATE void push_file()*/
/*{{{  comment*/
/*****************************************************************************
*
*  push_file()
*  Pushes the current file context onto the stack and opens the new file.
*  Andy Whitlow 29.11.88.
*
******************************************************************************/
/*}}}*/
PRIVATE void push_file ( void )
{
  struct file_stack_cell *fscptr;

  /*{{{  add current file context to stack*/
  fscptr = (struct file_stack_cell *) newvec(sizeof(*fscptr));
  fscptr->saved_sfptr = sfptr;
  strcpy(fscptr->saved_proc_name, curr_proc_name);
  strcpy(fscptr->saved_saved_line, saved_line);
  fscptr->saved_line_number = curr_line_number;
  fscptr->saved_filenumber = curr_filenumber;
  fscptr->next = file_stack;
  file_stack = fscptr;
  /*}}}*/
  /*{{{  open new file*/
  open_source_file(lookupfilename(filenumber));
  /*}}}*/
  /*{{{  initialise statics*/
  *saved_line = '\0';
  curr_line_number = 0;
  curr_filenumber = filenumber;
  /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE int pop_file()*/
/*{{{  comment*/
/*****************************************************************************
*
*  pop_file()
*  Closes the current file. Restores the file context on the top of the stack.
*  If the file stack is empty an error message is output.
*  Andy Whitlow 29.11.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int pop_file ( void )
{
  fclose(sfptr);
  if (file_stack != NULL)
  {
    /*{{{  restore old file context*/
    sfptr = file_stack->saved_sfptr;
    strcpy(curr_proc_name, file_stack->saved_proc_name);
    strcpy(saved_line, file_stack->saved_saved_line);
    curr_line_number = file_stack->saved_line_number;
    curr_filenumber = file_stack->saved_filenumber;
    file_stack = file_stack->next;
    end_of_file = FALSE;
    return 0;
    /*}}}*/
  }
  else
    return 1;
}
/*}}}*/
/*{{{  PRIVATE struct line_cell *set_up_cell(...)*/
/*{{{  comment*/
/*****************************************************************************
*
*  set_up_cell(ptr, line, type, linenumber)
*  Sets up a proc stack sub cell with the passed parameters.
*  Recursively calls itself to handle any continuation lines.
*  Returns a pointer to the cell created. i.e. the pointer will point to the
*  first cell in a chain of lines for any continuation lines found.
*  Andy Whitlow 29.11.88.
*
******************************************************************************/
/*}}}*/
PRIVATE struct line_cell *set_up_cell ( char *line , int type , int linenumber , int main_type )
{
  struct line_cell *npsscptr;

  /*{{{  build new cell*/
  npsscptr = (struct line_cell *) newvec(sizeof(*npsscptr));
  npsscptr->next = NULL;
  npsscptr->type = type;
  npsscptr->line_number = linenumber;
  npsscptr->v_line_number = v_linenumber;
  npsscptr->file_num = curr_filenumber;
  strcpy(npsscptr->sline, line);
  /*}}}*/
  v_linenumber++;
  /*{{{  handle any continuation line*/
  if (continuationlines(line, type))
  {
    type = get_line(line);
    if (end_of_file)
      generr(GEN_UNEXPECTED_EOF);
    if (type != SO_COMMENT && type != SO_SPACE)
      /*{{{  set up another stack sub cell*/
      npsscptr->next = set_up_cell(line, type, curr_line_number, main_type);
      /*}}}*/
  }
  /*}}}*/
  return(npsscptr);  /* return pointer to this cell */
}
/*}}}*/
/*{{{  PRIVATE int is_short_prot_or_fn()*/
/*{{{  comment*/
/***************************************************************************
*
*  is_short_prot_or_fn()
*  returns TRUE if the last lines added to the lines_list represent a short
*  protocol or function definition. This is determined by finding the last line
*  added to the list ends in a colon.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE int is_short_prot_or_fn ( void )
{
  char word[LINEMAX];
  struct line_cell *lptr = lines_list;

  /*{{{  find last line on lines list - i.e. last line of prot or fn header*/
  while (lptr->next != NULL)
    lptr = lptr->next;
  /*}}}*/
  /*{{{  get last word on line before comment*/
  get_last_word(lptr->sline, word);
  /*}}}*/
  /*{{{  return if last character is a colon - i.e. not a variant protocol*/
  if (word[strlen(word) - 1] == ':')
    return(TRUE);
  else
    return(FALSE);
  /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void add_to_block_line_list()*/
/*{{{  comment*/
/***************************************************************************
*
*  add_to_block_line_list()
*  Add the current value of v_linenumber to the block_line_list.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE void add_to_block_line_list ( void )
{
  struct item_cell *new;

  new = (struct item_cell *) newvec(sizeof(*new));
  new->item = v_linenumber;
  new->next = block_line_list;
  block_line_list = new;
}
/*}}}*/
/*{{{  PRIVATE void add_to_lines_list(line, type)*/
/*{{{  comment*/
/***************************************************************************
*
*  add_to_lines_list(line, type)
*  Add the 'line' with type 'type' to lines list along with all its attributes.
*  Andy Whitlow 21.12.88
*
****************************************************************************/
/*}}}*/
PRIVATE void add_to_lines_list ( char *line , int type )
{
  struct line_cell *lineptr = lines_list;

  if (lines_list == NULL)
    /*{{{  first lines to be added to lines list*/
    lines_list = set_up_cell(line, type, curr_line_number, type);
    /*}}}*/
  else
  {
    /*{{{  append lines to end of lines list*/
    /*{{{  find end of lines list*/
    while (lineptr->next != NULL)
      lineptr = lineptr->next;
    /*}}}*/
    lineptr->next = set_up_cell(line, type, curr_line_number, type);
    /*}}}*/
  }
}
/*}}}*/
/*{{{  PRIVATE void read_routine(line)*/
/*{{{  comment*/
/******************************************************************************
*
*  read_routine()
*  Read source for a PROC or FUNCTION into lines list.
*  Add the line numbers for the routine header, the first line after the header
*  and the last line plus one to the block line list.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void read_routine ( char *line )
{
  int type, done_first_line = FALSE;

  add_to_block_line_list();  /* procedure header is a sequential block */
  add_to_lines_list(line, SO_PROC); /* add routine header to lines list */
  /*{{{  return if found a short function declaration*/
  if (is_short_prot_or_fn())
  {
    /*{{{  add line number of line after short function to block line list*/
    get_line(line);
    add_to_block_line_list();
    put_line_back(line);
    /*}}}*/
    return;
  }
  /*}}}*/
  type = get_line(line);
  while (type != SO_COLON)
  {
    /*{{{  mark first line of procedure*/
    if (!done_first_line && type != SO_PROC && type != SO_SPACE &&
         type != SO_COMMENT)
    {
      /*{{{  set up block marker for first line of routine*/
      add_to_block_line_list();
      done_first_line = TRUE;
      /*}}}*/
    }
    /*}}}*/
    process_read_line(line, type);
    if (end_of_file && pop_file())
      generr(GEN_UNEXPECTED_EOF);
    type = get_line(line);
  }
  add_to_block_line_list();
}
/*}}}*/
/*{{{  PRIVATE void read_protocol(line)*/
/*{{{  comment*/
/******************************************************************************
*
*  read_protocol()
*  Read source for a PROTOCOL definition into lines list.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void read_protocol ( char *line )
{
  int type;

  /*{{{  comment*/
  /* a protocol header is given type SO_DECL so a declaration message
     can be output for it */
  /*}}}*/
  add_to_lines_list(line, SO_DECL); /* add protocol header to lines list */
  /*{{{  return if not a variant protocol declaration*/
  if (is_short_prot_or_fn())
    return;
  /*}}}*/
  type = get_line(line);
  while (type != SO_COLON)
  {
    /*{{{  add line of variant protocol to lines list with type SO_PROTOCOL*/
    add_to_lines_list(line, SO_PROTOCOL);
    if (end_of_file && pop_file())
      generr(GEN_UNEXPECTED_EOF);
    type = get_line(line);
    /*}}}*/
  }
  add_to_lines_list(line, SO_PROTOCOL); /* add final colon to lines list */
}
/*}}}*/
/*{{{  PRIVATE void read_case_or_pripar(line, type)*/
/*{{{  comment*/
/******************************************************************************
*
*  read_case_or_pripar()
*  Read source for a CASE or PRIPAR construct into lines list.
*  Add the line numbers for all CASE selections and PRIPAR subprocesses and the
*  last line plus one to the block line list.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void read_case_or_pripar ( char *line , int type )
{
  int blockindent, this_indent;

  blockindent = indent_count(line);
  add_to_lines_list(line, type);
  type = get_line(line);
  while ((this_indent = indent_count(line)) > blockindent)
  {
    /*{{{  read a line from input*/
    if (this_indent == blockindent + 2)
    {
      /*{{{  found a CASE selection or DECL or start of PRIPAR subprocess*/
      add_to_block_line_list();
      add_to_lines_list(line, type);
      /*{{{  if line just read was DECL read all further DECLs and selection line*/
      if (type == SO_DECL)
      {
        /*{{{  line was DECL so read all further DECLs and selection line*/
        type = get_line(line);
        while (type == SO_DECL)
        {
          /*{{{  add all further DECLs to lines list*/
          add_to_lines_list(line, type);
          if (end_of_file && pop_file())
            generr(GEN_UNEXPECTED_EOF);
          type = get_line(line);
          /*}}}*/
        }
        /*{{{  add selection line to lines list*/
        add_to_lines_list(line, type);
        if (end_of_file && pop_file())
          generr(GEN_UNEXPECTED_EOF);
        /*}}}*/
        /*}}}*/
      }
      /*}}}*/
      /*}}}*/
    }
    else
    {
      /*{{{  found a line other than above*/
      process_read_line(line, type);
      if (end_of_file && pop_file())
        generr(GEN_UNEXPECTED_EOF);
      /*}}}*/
    }
    type = get_line(line);
    /*}}}*/
  }
  /*{{{  add line number of line after CASE or PRIPAR to block line list*/
  add_to_block_line_list();
  /*}}}*/
  put_line_back(line);
}
/*}}}*/
/*{{{  PRIVATE void process_read_line(line, type)*/
/*{{{  comment*/
/******************************************************************************
*
*  process_read_line()
*  Perform some action depending on the type of the line just read.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void process_read_line ( char *line , int type )
{
  switch(type)
  {
    /*{{{  PROC*/
    case SO_PROC:
      read_routine(line);
      break;
    /*}}}*/
    /*{{{  PRIPAR CASE*/
    case SO_PRIPAR:
    case SO_CASE:
      read_case_or_pripar(line, type);
      break;
    /*}}}*/
    /*{{{  INCLUDE*/
    case SO_INCLUDE:
      push_file();
      break;
    /*}}}*/
    /*{{{  OTHER WHILE PAR DECL PROTOCOL ALT*/
    case SO_OTHER:
    case SO_WHILE:
    case SO_PAR:
    case SO_DECL:
    case SO_ALT:
       add_to_lines_list(line, type);
       break;
    /*}}}*/
    /*{{{  PROTOCOL*/
    case SO_PROTOCOL:
       read_protocol(line);
       break;
    /*}}}*/
    /*{{{  default*/
    default:
       break;
    /*}}}*/
  }
}
/*}}}*/
/*{{{  PRIVATE void check_file_entry(file_num)*/
/*{{{  comment*/
/******************************************************************************
*
*  check_file_entry(file_num)
*  Checks file_num against the file number of the current file. If it is
*  different a message indicating a new file entry is output.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void check_file_entry ( int file_num )
{
  if (file_num != prev_filenumber)
  {
    fprintf(outfile, "\n**** [ File \"%s\" ]",
            lookupfilename(file_num));
    prev_filenumber = file_num;
  }
}
/*}}}*/
/*{{{  PRIVATE void build_source_code_lists()*/
/*{{{  comment*/
/******************************************************************************
*
*  build_source_code_lists(file_num)
*  Main routine to build the source code lists for use in the source output
*  code.
*  Andy Whitlow 21.12.88
*
******************************************************************************/
/*}}}*/
PRIVATE void build_source_code_lists ( void )
{
  struct item_cell *bllptr, *prev_bllptr;
  int type, prev_block_line, going = TRUE;
  char line[LINEMAX];

  type = get_line(line);
  while (going)
  {
    /*{{{  read initial lines to first routine declaration*/
    if (end_of_file && pop_file())
      going = FALSE;
    else
      {
        process_read_line(line, type);
        type = get_line(line);
      }
    /*}}}*/
  }
  /*{{{  preprocess block line list to remove any NULL blocks*/
  bllptr = block_line_list;
  prev_block_line = bllptr->item;
  prev_bllptr = bllptr;
  bllptr = bllptr->next;
  while (bllptr != NULL)
  {
    /*{{{  process current block*/
    if (bllptr->item == prev_block_line)
    {
      /*{{{  remove duplicate line number from list*/
      prev_bllptr->next = bllptr->next;
      /*}}}*/
    }
    prev_block_line = bllptr->item;
    prev_bllptr = bllptr;
    bllptr = bllptr->next;
    /*}}}*/
  }
  /*}}}*/
  /*{{{  debug*/
  /* {
    struct line_cell *lptr = lines_list;
  
    while (lptr != NULL)
    {
      printf("\nLL : line : %s : line : %d : file : %d : v_line : %d",
              lptr->sline, lptr->line_number, lptr->file_num,
              lptr->v_line_number);
      lptr = lptr->next;
    }
  } */
  /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  stack handling*/
/*{{{  PRIVATE void push_pw(number)*/
/*{{{  comment*/
/*****************************************************************************
*
*  push_pw(number)
*  Pushes number onto the PAR/WHILE stack.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE void push_pw ( int number )
{
  struct item_cell *new;

  new = (struct item_cell *) newvec(sizeof(*new));
  new->item = number;
  new->next = pw_stack;
  pw_stack = new;
}
/*}}}*/
/*{{{  PRIVATE int pop_pw()*/
/*{{{  comment*/
/*****************************************************************************
*
*  pop_pw()
*  Returns the number on the top of the PAR/WHILE stack.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int pop_pw ( void )
{
  int ret_item;

  ret_item = pw_stack->item;
  pw_stack = pw_stack->next;
  return(ret_item);
}
/*}}}*/
/*}}}*/
/*{{{  source code handling*/
/*{{{  PRIVATE int convert_to_virtual_line(line_num, file_id)*/
/*{{{  comment*/
/*****************************************************************************
*
*  convert_to_virtual_line(line_num, file_id)
*  Because a piece of source code can extend over a number of files a virtual
*  line number system is used to distinguish each line. This function takes a
*  lines position in a file and its file identifier and returns the
*  corresponding virtual line number.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int convert_to_virtual_line ( int line_num , int file_id )
{
  struct line_cell *llptr = lines_list;
  while (llptr != NULL &&
         (llptr->line_number != line_num || llptr->file_num != file_id))
    llptr = llptr->next;
#if 0
  if (llptr != NULL)
    return(llptr->v_line_number);
  else
    geninternal(GEN_LINE_NOT_IN_LIST, line_num, file_id);
  return (0); /* Not reached */
#else
  assert(llptr != NULL);
  return(llptr->v_line_number);
#endif
}
/*}}}*/
/*{{{  PRIVATE struct line_cell *get_pointer_to_line(line_num)*/
/*{{{  comment*/
/*****************************************************************************
*
*  get_pointer_to_line(lnum)
*  Returns a pointer to the line_cell in the lines_list which corresponds to
*  the virtual line number 'lnum'.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE struct line_cell *get_pointer_to_line ( int lnum )
{
  struct line_cell *linesptr = lines_list;

  while (linesptr->v_line_number != lnum)
    linesptr = linesptr->next;
  return(linesptr);
}
/*}}}*/
/*{{{  PRIVATE struct line_cell *output_line_plus_continuation(lineptr)*/
/*{{{  comment*/
/*****************************************************************************
*
*  output_line_plus_continuation(lineptr)
*  Outputs the line pointed to by lineptr and all its continuation lines.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE struct line_cell *output_line_plus_continuation ( struct line_cell *lineptr )
{
  while (continuationlines(lineptr->sline, lineptr->type))
  {
    check_file_entry(lineptr->file_num);
    fprintf(outfile, "\nLine %4d: %s", lineptr->line_number, lineptr->sline);
    lineptr = lineptr->next;
  }
  check_file_entry(lineptr->file_num);
  fprintf(outfile, "\nLine %4d: %s", lineptr->line_number, lineptr->sline);
  return(lineptr->next);
}
/*}}}*/
/*{{{  PRIVATE int output_alt_message(string, lnum, counter)*/
/*{{{  comment*/
/*****************************************************************************
*
*  output_alt_message(string, line_num, counter)
*  Outputs the messages associated with the enabling and disabling of ALTs.
*  'string' contains either 'En' or 'Dis' depending on the operation being
*  performed. 'line_num' is the virtual line number of the ALT guard being
*  processed. 'counter' is a pointer to the relevant enable or disable counter.
*  This function also handles ALT replicators which cause a different message
*  to be output.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int output_alt_message ( char *string , int lnum , int counter )
{
  struct line_cell *altptr;

  altptr = get_pointer_to_line(lnum);
  if (altptr->type == SO_ALT || altptr->type == SO_SPECALT)
    /*{{{  found ALT*/
    fprintf(outfile,"\n**** [ set up %sabling replicator for following code ]",
            string);
    /*}}}*/
  else
  {
    /*{{{  found guard*/
    fprintf(outfile,"\n**** [ %sable following guard ]", string);
    counter++;
    /*}}}*/
  }
  /*{{{  output source lines*/
  output_line_plus_continuation(altptr);
  /*}}}*/
  return(counter);
}
/*}}}*/
/*{{{  PRIVATE int get_altcount(altptr, indent)*/
/*{{{  comment*/
/*****************************************************************************
*
*  get_alt_count(altptr, indent)
*  'altptr' points to the position in the line list of the line after the ALT
*  statement. 'indent' is the level of indentation of this line. This function
*  returns the number of guards that exist within the ALT. This is achieved by
*  counting all lines with an indent of 'indent + 2' which are not DECLs. It
*  also recurses into any ALTs nested at this level.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE int get_altcount ( struct line_cell *altptr , int indent )
{
  int local_indent, local_count;

  local_count = 0;
  while (altptr != NULL &&
         (local_indent = indent_count(altptr->sline)) > indent)
  {
    /*{{{  process line*/
    if (local_indent == indent + 2 && altptr->type != SO_DECL)
    {
      if (altptr->type == SO_ALT) /* ALTs are present on guard lines */
      {
        /*{{{  process nested ALTs*/
        altptr->type = SO_SPECALT;
        /*{{{  skip lines of ALT statement*/
        while (continuationlines(altptr->sline, altptr->type))
          altptr = altptr->next;
        altptr = altptr->next;
        /*}}}*/
        local_count = local_count + get_altcount(altptr, local_indent);
        /*}}}*/
      }
      else
      {
        /*{{{  increment ALT guard count*/
        while (continuationlines(altptr->sline, altptr->type))
          altptr = altptr->next;
        local_count++;
        /*}}}*/
      }
    }
    altptr = altptr->next;
    /*}}}*/
  }
  return(local_count);
}
/*}}}*/
/*{{{  PRIVATE struct line_cell *process_output_lines(lineptr)*/
/*{{{  comment*/
/*****************************************************************************
*
*  process_output_lines(lineptr)
*  Perform some action on a line pointed to by lineptr.
*  Andy Whitlow 21.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE struct line_cell *process_output_lines ( struct line_cell *lineptr )
{
  switch (lineptr->type)
  {
    /*{{{  cases*/
    /*{{{  DECL*/
    case SO_DECL :
      fprintf(outfile, "\n**** [ declaration local to %s ]", curr_proc_name);
      lineptr = output_line_plus_continuation(lineptr);
      /*{{{  output all lines of variant protocol definition*/
      while (lineptr->type == SO_PROTOCOL)
        lineptr = output_line_plus_continuation(lineptr);
      /*}}}*/
      break;
    /*}}}*/
    /*{{{  WHILE*/
    case SO_WHILE :
      fprintf(outfile, "\n**** [ WHILE number %d ]", curr_while_number);
      lineptr = output_line_plus_continuation(lineptr);
      push_pw(curr_while_number);
      curr_while_number++;
      break;
    /*}}}*/
    /*{{{  PAR PRIPAR*/
    case SO_PAR :
    case SO_PRIPAR :
      fprintf(outfile, "\n**** [ PAR number %d ]", curr_par_number);
      lineptr = output_line_plus_continuation(lineptr);
      push_pw(curr_par_number);
      curr_par_number++;
      break;
    /*}}}*/
    /*{{{  OTHER continuation SPECALT PROC*/
    case SO_OTHER :
    case SO_PROC :
    case SO_SPECALT :
      lineptr = output_line_plus_continuation(lineptr);
      break;
    /*}}}*/
    /*{{{  CASE*/
    case SO_CASE :
      lineptr->type = SO_SPECCASE; /* change type to flag that its been output */
      lineptr = output_line_plus_continuation(lineptr);
      break;
    /*}}}*/
    /*{{{  ALT*/
    case SO_ALT:
    {
      int altindent;
    
      altflag = TRUE;
      done_enb = 0;
      done_disb = 0;
      altindent = indent_count(lineptr->sline);
      lineptr = output_line_plus_continuation(lineptr);
      altcount = get_altcount(lineptr, altindent);
    }
    break;
    /*}}}*/
    /*{{{  default*/
    default:
      break;
    /*}}}*/
    /*}}}*/
  }
  return(lineptr);
}
/*}}}*/
/*}}}*/
/*}}}*/

/*{{{  public routines*/
/*{{{  file handling*/
/*{{{  PUBLIC void init_source_code_output()*/
/*{{{  comment*/
/****************************************************************************
*
*  init_source_code_output
*  set up variables required if source code output has been selected.
*  i.e. if /zo option was given.
*  Andy Whitlow 15.11.88
*
*****************************************************************************/
/*}}}*/
PUBLIC void init_source_code_output ( void )
{
  /*{{{  initialise statics*/
  /*{{{  stacks*/
  file_stack = NULL;
  pw_stack = NULL;
  /*}}}*/
  /*{{{  saved line buffer*/
  *saved_line = '\0';
  /*}}}*/
  /*{{{  line numbers*/
  curr_line_number = 0;
  curr_while_number = 1;
  curr_par_number = 1;
  /*}}}*/
  /*{{{  line continuation*/
  continuation = FALSE;
  /*}}}*/
  /*{{{  files*/
  filenumber = 0;
  curr_filenumber = 0;
  prev_filenumber = -1;
  end_of_file = FALSE;
  /*}}}*/
  /*{{{  ALT*/
  altflag = FALSE;
  altcount = 0;
  /*}}}*/
  /*{{{  block*/
  lines_list = NULL;
  block_line_list = NULL;
  high_block_line = -1;
  v_linenumber = 1;
  /*}}}*/
  /*}}}*/
  set_up_cell("", SO_NULL, 0, SO_NULL);
  open_source_file(lookupfilename(0));
  build_source_code_lists();
}
/*}}}*/
/*{{{  PUBLIC void end_source_code_output()*/
/*{{{  comment*/
/******************************************************************************
*
*  end_source_code_output()
*  Close source file.
*  Andy Whitlow 15.11.88
*
******************************************************************************/
/*}}}*/
PUBLIC void end_source_code_output ( void )
{
  fclose(sfptr);
}
/*}}}*/
/*}}}*/
/*{{{  source code handling*/
/*{{{  PUBLIC void so_endofproc()*/
PUBLIC void so_endofproc ( void )
{
  fprintf(outfile, "\n**** [ Return from routine %s ]", curr_proc_name);
}
/*}}}*/
/*{{{  PUBLIC void so_endofwhile()*/
PUBLIC void so_endofwhile ( void )
{
  fprintf(outfile, "\n**** [ loop to start of WHILE number %d ]", pop_pw());
}
/*}}}*/
/*{{{  PUBLIC void so_endofpar(branch, end)*/
PUBLIC void so_endofpar ( int branch , int end )
{
  int local_par;

  local_par = pop_pw();
  fprintf(outfile, "\n**** [ end of branch %d of PAR number %d ]",
          branch, local_par);
  if (!end)
    push_pw(local_par);
}
/*}}}*/
/*{{{  PUBLIC void so_stop()*/
PUBLIC void so_stop ( void )
{
  fprintf(outfile, "\n**** [ default STOP process ]");
}
/*}}}*/
/*{{{  PUBLIC void output_source_code(index, debug_type)*/
/*{{{  comment*/
/******************************************************************************
*
*  output_source_code(index)
*  Output a chunk of source code to the output file.
*  Andy Whitlow 15.11.88
*
******************************************************************************/
/*}}}*/
PUBLIC void output_source_code ( int index , int debug_type )
{
  int new_line_number, new_file_number;
  struct line_cell *ptr;

  /*{{{  get line number*/
  new_line_number = get_from_linemark_list(index);
  new_file_number = get_file_number(index);
  new_line_number = convert_to_virtual_line(new_line_number, new_file_number);
  
  if (new_line_number == 0 && new_file_number == 0)
    return;
  /*}}}*/
  /*{{{  validate locate info*/
  ptr = get_pointer_to_line(new_line_number);
  if (debug_type == C_ADDRESSFIX)
  {
    /*{{{  check that line referred to is routine, if not return*/
    if (ptr->type != SO_PROC)
      /*{{{  addressfix does not correspond to routine so return*/
      return;
      /*}}}*/
    else
    {
      /*{{{  get new current routine name*/
      char word[LINEMAX], *lptr;
      
      lptr = ptr->sline;
      word[0] = '\0';
      while (strncmp(word, "PROC", 4) && strncmp(word, "FUNCTION",8))
        lptr = get_next_word(lptr, word);
      lptr = get_next_word(lptr, curr_proc_name);
      /*}}}*/
    }
    /*}}}*/
  }
  else /* debug_type is C_LOCATE */
  {
    /*{{{  ignore codemarks referring to short function declarations*/
    if (ptr->type == SO_PROC)
      return;
    /*}}}*/
  }
  /*}}}*/
  /*{{{  ALT processing*/
  if (altflag && done_enb < altcount)
  {
    /*{{{  do alt enable line*/
    done_enb = output_alt_message("En", new_line_number, done_enb);
    return;
    /*}}}*/
  }
  if (altflag && done_disb < altcount)
  {
    /*{{{  do alt disable line*/
    done_disb = output_alt_message("Dis", new_line_number, done_disb);
    return;
    /*}}}*/
  }
  if (altflag)
    /*{{{  end of alt*/
    altflag = FALSE;
    /*}}}*/
  /*}}}*/
  /*{{{  block processing*/
  if (new_line_number >= high_block_line || new_line_number <= low_block_line)
  {
    /*{{{  locate info is outside present block*/
    struct item_cell *bllptr = block_line_list;
    
    /*{{{  get first block line watermarks*/
    high_block_line = bllptr->item;
    bllptr = bllptr->next;
    low_block_line = bllptr->item;
    /*}}}*/
    while (new_line_number < low_block_line)
    {
      /*{{{  get next block line watermarks*/
      high_block_line = bllptr->item;
      bllptr = bllptr->next;
      low_block_line = bllptr->item;
      /*}}}*/
    }
    output_ptr = get_pointer_to_line(low_block_line);
    /*}}}*/
  }
  else
  {
    /*{{{  handle implicit stops in CASE*/
    struct line_cell *llptr;
    
    llptr = get_pointer_to_line(new_line_number);
    if (llptr->type == SO_SPECCASE)
    {
      fprintf(outfile, "\n**** [ for following CASE ]");
      output_line_plus_continuation(llptr);
      return;
    }
    /*}}}*/
  }
  /*}}}*/
  /*{{{  output all lines to current line*/
  while (output_ptr != NULL && output_ptr->v_line_number <= new_line_number)
    output_ptr = process_output_lines(output_ptr);
  /*}}}*/
}
/*}}}*/
/*}}}*/
/*}}}*/
#endif
