/****************************************************************************
 *
 *  Occam two compiler harness
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include <ctype.h>
#ifdef ANSI
# include <stdlib.h>
#endif
#ifdef IMS
# include <host.h> /* used for determining escape_char */
# include <misc.h> /* for setting special error actions */
#endif
# include "includes.h"
# include "synerror.h"
# include "genhdr.h"
# include "lexconst.h"
# include "lex1def.h"
# include "lexdef.h"
# include "syndef.h"
# include "chkdef.h"
# include "usedef.h"
# include "instdef.h"
# include "deschdr.h"
# include "desc1def.h"
# include "miscdef.h"
# include "tcoff.h"
# include "trandef.h"
# include "bind1def.h"
# include "bind3def.h"
# include "gen1def.h"
# include "gen8def.h"
# include "code1def.h"
# include "debugdef.h"
#if defined(CONFIG)
# include "confdef.h"
#endif
#if defined(SUN) || defined(GNU)
/*# include <sys/types.h>*/
/*# include <sys/time.h>*/
/*# include <sys/resource.h>*/
/*void getrusage(int who, struct rusage *rusage);*/
char *sbrk(int incr);
#endif
/*}}}*/

/*{{{  definitions*/
/*{{{  processor attributes*/
/*{{{  ptypes maps command line string to processor type*/
PRIVATE struct proctype_struct
  {
    char *pstring;      /* string representing processor type 'ptype' */
    BIT32 pinstr;       /* processor instruction attributes */
    BIT32 pattr;        /* other processor attributes */
    char pobjextchar;   /* character in LFF object file extension for 'ptype' */
    char plfflibchar;   /* character in LFF extlib file extension for 'ptype' */
    char ptcofflibchar; /* character in TCOFF extlib file extension for 'ptype' */
    int  plinks;        /* number of links */
    int  pconfig_type;  /* processor type for configurer */
    int  pclass;        /* TRUE if the type is a class (not permitted configuring */
  } ptypes[] =
  {
    { "TA",   TA_INSTR,   TA_ATTRIB,   'a', 'b', 'a', 4, -6, 1 },
    { "TB",   TB_INSTR,   TB_ATTRIB,   'b', 'b', 'a', 4, -7, 1 },
  /*{ "TC",   TC_INSTR,   TC_ATTRIB,   'c', 'b', 'a', 4 },*/
    { "T2",   T212_INSTR, T212_ATTRIB, '2', '2', '2', 4, -2, 1 },
    { "T212", T212_INSTR, T212_ATTRIB, '2', '2', '2', 4, -2, 0 },
    { "T222", T222_INSTR, T222_ATTRIB, '2', '2', '2', 4, -5, 0 },
    { "T225", T225_INSTR, T225_ATTRIB, '3', '2', '2', 4, 40, 0 },
    { "T3",   T225_INSTR, T225_ATTRIB, '3', '2', '2', 4, 40, 1 },
    { "M212", T212_INSTR, T212_ATTRIB, '2', '2', '2', 2, -1, 0 },
    { "T4",   T414_INSTR, T414_ATTRIB, '4', 'b', 'a', 4, -3, 1 },
    { "T400", T400_INSTR, T400_ATTRIB, '5', 'b', 'a', 2, 50, 0 },
    { "T414", T414_INSTR, T414_ATTRIB, '4', 'b', 'a', 4, -3, 0 },
    { "T425", T425_INSTR, T425_ATTRIB, '5', 'b', 'a', 4, 00, 0 },
    { "T5",   T425_INSTR, T425_ATTRIB, '5', 'b', 'a', 4, 00, 1 },
    { "T8",   T800_INSTR, T800_ATTRIB, '8', '8', '8', 4, -4, 1 },
    { "T800", T800_INSTR, T800_ATTRIB, '8', '8', '8', 4, -4, 0 },
    { "T801", T801_INSTR, T801_ATTRIB, '9', '8', '8', 4, 20, 0 },
    { "T805", T805_INSTR, T805_ATTRIB, '9', '8', '8', 4, 10, 0 },
    { "T9",   T805_INSTR, T805_ATTRIB, '9', '8', '8', 4, 10, 1 },
    { "H1",   T805_INSTR, T805_ATTRIB, 'h', '8', '8', 4, 10, 0 },
    { "H1L",  T805_INSTR, T805_ATTRIB, 'h', '8', '8', 4, 10, 0 }
  };
#define DEFAULT_PROCESSOR_TYPE "T414"
#define UNKNOWN_PROCESSOR_TYPE 0  /* dissimilar to TA_INSTR etc */
/*}}}*/
/*{{{  errmodes maps command line string to error mode*/
#define HALT_MODE       ( ERRORMODE_HALT \
                        | ERRORMODE_RANGECHECK \
                        | ERRORMODE_CONVERSIONCHECK \
                        | ERRORMODE_NEED_ERRORS \
                        | ERRORMODE_STOP_IS_SETERR )
#define STOP_MODE       ( ERRORMODE_STOP \
                        | ERRORMODE_RANGECHECK \
                        | ERRORMODE_CONVERSIONCHECK \
                        | ERRORMODE_NEED_ERRORS \
                        | ERRORMODE_TIMESLICECHECK \
                        | ERRORMODE_NEED_STOPERR \
                        | ERRORMODE_STOP_IS_STOPP )
#define UNIVERSAL_MODE  ( ERRORMODE_UNIVERSAL \
                        | ERRORMODE_RANGECHECK \
                        | ERRORMODE_CONVERSIONCHECK \
                        | ERRORMODE_NEED_ERRORS \
                        | ERRORMODE_TIMESLICECHECK \
                        | ERRORMODE_NEED_STOPERR \
                        | ERRORMODE_STOP_IS_SETERR \
                        | ERRORMODE_STOP_IS_STOPP )

PRIVATE struct errmode_struct
  {
    int emode;        /* Mode used internally to this compiler */
    int eattr;        /* Bits to OR into the TCOFF attributes */
    char emodechar;   /* Command line character */
    char eobjextchar; /* Default object file extension */
    char eextlibchar; /* Default standard library extension */
  } errmodes[] =
  {
    { HALT_MODE,      ATTRIB_HALT,      'H', 'h', 'h' },
    { STOP_MODE,      ATTRIB_STOP,      'S', 's', 's' },
  /*{ REDUCED_MODE,   ?,                'U', 'u', 'u' },*/
    { UNIVERSAL_MODE, ATTRIB_UNIVERSAL, 'X', 'x', 'x' }
  };
#define DEFAULT_ERROR_MODE "H"
#define UNKNOWN_ERROR_MODE 0 /* dissimilar to HALT_MODE etc */
/*}}}*/
/*}}}*/
/* These are chosen at run-time: */
/*#ifndef CONFIG*/
#define TCOFF_LIB_NAMES
/*#endif*/

#ifdef TCOFF_LIB_NAMES
  #define TCOFF_EXTLIB_FILENAME "occam?.lib"
#else
  #define TCOFF_EXTLIB_FILENAME "occam??.lib"
#endif

# define LFF_EXTLIB_FILENAME   "occambh.lib"
# define LFF_VLIB_FILENAME     "virtual.lib"

#ifdef CONFIG
  #define SOURCEFILE_EXTENSION  ".pgm"
  /*#define DEFAULT_OBJ_EXTENSION ".lib"*/
  /*#define DEFAULT_OBJ_EXTENSION ".clu"*/
  #define CFB_FILE_EXTENSION    ".cfb"
  #define DEFAULT_OBJ_EXTENSION ".clu"

/*#define TCOFF_VLIB_FILENAME   "virtuall.lib"*/
  #define TCOFF_VLIB_FILENAME   "occonfio.lib"

#else
  #define SOURCEFILE_EXTENSION  ".occ"
  #define DEFAULT_OBJ_EXTENSION ".tco"

  #define TCOFF_VLIB_FILENAME   "virtual.lib"
#endif

#define DEFAULT_PATHNAME "ISEARCH"
/*}}}*/
/*{{{  global variables*/
PUBLIC FILE *infile, *outfile, *errfile, *objfile;
PRIVATE char rootfilename[MAX_FILENAME_LENGTH];
PUBLIC char sourcefilename[MAX_FILENAME_LENGTH];
PUBLIC char objfilename[MAX_FILENAME_LENGTH];
/*PUBLIC char extlibfilename[13];*/ /* 7 or 8 chars, plus a dot, plus 3 suffix, plus nul */
PUBLIC char extlibfilename[MAX_FILENAME_LENGTH];
PRIVATE int extlibfilename_overruled = FALSE;
/*PUBLIC char vlibsfilename[13];*/ /* 8 chars, plus a dot, plus 3 suffix, plus nul */
PUBLIC char vlibsfilename[MAX_FILENAME_LENGTH];

/* Processor attributes */
PUBLIC BIT32 processortype = UNKNOWN_PROCESSOR_TYPE;
PUBLIC BIT32 processorattr;
PUBLIC int targetintsize, bytesperword, real32isaword;
PUBLIC int fpinline, hasdup, haswsubdb, fpsupport, graphicsmove,
           hascrc, hasbitops, fracmul, debugsupport, haspop,
           fptesterr, hastimerdis, hasldmemstart, haslddevid;
PUBLIC int H1_instr, H1L_process = FALSE;

PUBLIC int errormode = UNKNOWN_ERROR_MODE;

PUBLIC int onlylex         = FALSE;
PUBLIC int nochecking      = FALSE;
PUBLIC int checkalias      = TRUE;
PUBLIC int checkusage      = TRUE;
#ifdef CONFIG
/*PUBLIC int compilemode     = COMP_PROGRAM;*/ /* never used */
PUBLIC int stdlibsenabled  = FALSE; /* Whether to allow access to standard libs */
#else
/*PUBLIC int compilemode     = COMP_SC;*/ /* never used */
PUBLIC int stdlibsenabled  = TRUE; /* Whether to allow access to standard libs */
#endif
PUBLIC int vlibsenabled    = TRUE; /* Whether to allow access to virtual libs */
PUBLIC int vsenabled       = TRUE;
PUBLIC int guyinserts      = FALSE;
PUBLIC int information     = FALSE;
PUBLIC int disassemble     = FALSE;
PUBLIC int assembly_output = FALSE;
PUBLIC int diagnostics     = FALSE;
PUBLIC int debuguse        = FALSE;
PUBLIC int brieferrors     = FALSE;

PUBLIC int debugoutput     = TRUE; /* Whether to insert debug info into object file */
PUBLIC int minimal_debugoutput = FALSE; /* Just enough for backtracing - only valid if*/
                                        /* debugoutput is TRUE                        */
#ifdef CONFIG
PUBLIC int symbolic_debugoutput = FALSE;
#else
PUBLIC int symbolic_debugoutput = TRUE;
#endif

PUBLIC int source_output   = FALSE; /* This forces full debug output (to screen)     */

PUBLIC int iobycall        = TRUE;
PUBLIC int chanaspointer   = TRUE;
PUBLIC int prtree          = FALSE;
PUBLIC int testflag        = FALSE; /* a generic test flag for debugging */

PUBLIC int allowpredefs    = TRUE; /* Whether to allow predefines */
PUBLIC int hidelibnames    = TRUE; /* Use REAL32OP% etc rather than REAL32OP */
PUBLIC int libpatchsize    = 0;    /* How many bytes for library patch */
                                   /* Any number except 8, overridden by setprocessor */
PUBLIC int code_style_flags = CODE_STYLE_DEFAULT; /* Flags for variations in code style */
PUBLIC int warning_flags    = WARNING_DEFAULT;    /* Which warnings are enabled */

PUBLIC char pathname[MAX_FILENAME_LENGTH];      /* ISEARCH or ZI option */

PUBLIC char *compilersuffix = NULL;
PUBLIC char *predefsuffix   = NULL;
PUBLIC char *vlibsuffix     = NULL;
/*}}}*/
/*{{{  local variables*/
/* We allow for the fact that we may add default suffixes to any option
   to create a filename */
#define MAX_OPTION_LENGTH (MAX_FILENAME_LENGTH - 5)

PRIVATE char objectfileext[5] = DEFAULT_OBJ_EXTENSION;

#define MAX_LIBNAMESSUFFIX 10
PRIVATE char compilersuffixstring[MAX_LIBNAMESSUFFIX];
PRIVATE char   predefsuffixstring[MAX_LIBNAMESSUFFIX];
PRIVATE char     vlibsuffixstring[MAX_LIBNAMESSUFFIX];

PRIVATE int generatecode = TRUE;
PRIVATE int call_check   = TRUE;
PRIVATE int argsdone;
PRIVATE char *cptr;
PRIVATE char *optstring, *optparam;
PRIVATE char escape_char;

PRIVATE int zinfo = FALSE;  /* Whether to display z option help page */
PRIVATE int stop_after_trans = FALSE;
PRIVATE int stop_after_map   = FALSE;
PRIVATE BIT32 processorattr_errmode;
PRIVATE int flagasiobycall = TRUE; /* so that -zv doesn't set the flag */

PRIVATE int errormodes_off_mask = 0;

#ifdef IMS
  PRIVATE int repeat_loop = FALSE;  /* Used for option 'XM' repeating */
  PRIVATE int load_only   = FALSE;  /* ditto */

  #define HOSTED FALSE
#else
  #define HOSTED TRUE
#endif

#if defined(SUN) || defined(GNU)
PRIVATE char *original_sbrk;
PRIVATE int memstats        = FALSE;
#endif
/*}}}*/

/*{{{  PUBLIC void end_compiler*/
void end_compiler ( int result )
{
  lexfinish(); /* used to output some debugging messages */
  vtifinish(); /* ditto */
#ifdef IMS
  if (repeat_loop)
    exit_repeat(result);
#endif
  exit(result);
}
/*}}}*/
/*{{{  PRIVATE void harnesserror (char *s, BIT32 p1, BIT32 p2)*/
PRIVATE void harnesserror ( char *s , BIT32 p1 , BIT32 p2 )
{
  fprintf(errfile, "Fatal-%s- ", compilername);
  fprintf(errfile, s, p1, p2);
  fprintf(errfile, "\n");
  end_compiler(EXIT_FAILURE);
}
/*}}}*/
/*{{{  PRIVATE int eqstrprefix(char *s1, char *s2)*/
PRIVATE int eqstrprefix(char *s1, char *s2)
/* checks that the two strings match as far as possible, and that any
   overlap is purely spaces */
{
  /* Check that the first characters match */
  while (((*s2) != '\0') && ((*s1) != '\0'))
    if ((*s2++) != (*s1++))
      return FALSE;

  /* Now check that any trailing chars are spaces */
  while ((*s1) != '\0')
    if ((*s1++) != ' ')
      return FALSE;
  while ((*s2) != '\0')
    if ((*s2++) != ' ')
      return FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  PUBLIC BIT32 setprocessor(s)*/
/* set processor dependant attributes */
/* pass in name of processor, in upper case */
/* This is also called by the configuration stuff */
PUBLIC BIT32 setprocessor ( char *s )
{
  int i;
  for (i = 0; i < (sizeof(ptypes) / sizeof(struct proctype_struct)); i++)
    if (eqstrprefix(s, ptypes[i].pstring))
      {
        BIT32 instr = ptypes[i].pinstr;
        BIT32 attr  = ptypes[i].pattr;
        processortype = instr;
        processorattr = attr;

        H1_instr      = (s[0] == 'H');
        H1L_process   = H1_instr && (s[2] == 'L'); /* must be exactly 1 or 0 */
        fpinline      = (instr & INSTR_FPU_CORE) != 0;
        hasdup        = (instr & INSTR_DUP) != 0;
        haswsubdb     = (instr & INSTR_WSUBDB) != 0;
        fpsupport     = (instr & INSTR_FP_SUPPORT) != 0 /*|| H1_instr*/;
        graphicsmove  = (instr & INSTR_MOVE2D) != 0;
        hascrc        = (instr & INSTR_CRC) != 0;
        hasbitops     = (instr & INSTR_BITOPS) != 0;
        fracmul       = (instr & INSTR_FMUL) != 0;
        debugsupport  = (instr & INSTR_DEBUG_SUPPORT) != 0;
        hastimerdis   = (instr & INSTR_TIMER_DISABLE) != 0;
        haslddevid    = (instr & INSTR_LDDEVID) != 0;
        haspop        = (instr & INSTR_POP) != 0;
        fptesterr     = (instr & INSTR_FPTESTERR) != 0;
        hasldmemstart = (instr & INSTR_LDMEMSTARTVAL) != 0;

        bytesperword  = (attr  & ATTRIB_WORD_16) ? 2       : 4;
        targetintsize = (attr  & ATTRIB_WORD_16) ? S_INT16 : S_INT32;
        libpatchsize  = (libpatchsize == 8)      ? 8       :
                        (attr  & ATTRIB_WORD_16) ? 4       : 6 ;
        real32isaword = (attr  & ATTRIB_WORD_32) != 0;

        if (!tcoff_obj_format) /* Set default output to .t4h etc */
          objectfileext[2]= ptypes[i].pobjextchar;
        if (!extlibfilename_overruled)
          {
          #ifdef TCOFF_LIB_NAMES
            extlibfilename[5] = ptypes[i].ptcofflibchar;
          #else
            extlibfilename[5] = ptypes[i].plfflibchar;
          #endif
          }
#ifdef CONFIG
        no_of_links       = ptypes[i].plinks;
        config_target_type= ptypes[i].pconfig_type;
        processor_class   = ptypes[i].pclass;
#endif
        if (H1L_process)
          code_style_flags |= CODE_STYLE_ALT_PRI_PAR;

        return (processortype);
      }
  return (ZERO32);
}
/*}}}*/
/*{{{  PUBLIC void setprcessorattr(void)*/
/* Add error mode and iocall to processor attributes */
PUBLIC void setprocessorattr(void)
{
  processorattr = (processorattr & ~(ATTRIB_IO_MASK | ATTRIB_ERROR_MASK))
                | processorattr_errmode
                | ((flagasiobycall) ? ATTRIB_CALL_IO : ATTRIB_INSTR_IO);
}
/*}}}*/
/*{{{  PUBLIC void setprocessordefault()*/
PUBLIC void setprocessordefault(void)
{
  setprocessor(DEFAULT_PROCESSOR_TYPE);
  setprocessorattr();
}
/*}}}*/
/*{{{  PRIVATE void seterrormode*/
PRIVATE void seterrormode(void)
{
  errormode &= ~errormodes_off_mask;
}
/*}}}*/
/*{{{  PUBLIC BIT32 typeofprocessor ( char *name )*/
/* Return processor type from processor string */
PUBLIC BIT32 typeofprocessor ( char *name )
{
  int i;
  for (i = 0; i < (sizeof(ptypes) / sizeof(struct proctype_struct)); i++)
    if (strcmp(name, ptypes[i].pstring) == 0)
      return (ptypes[i].pinstr);
  return (ZERO32);
}
/*}}}*/
/*{{{  PUBLIC char *processorstring*/
PUBLIC char *processorstring ( BIT32 p, BIT32 a)
{
  int i;
  int mask = ATTRIB_WORD_MASK | ATTRIB_MEMSTART_MASK;
  a &= mask;
  for (i = 0; i < (sizeof(ptypes) / sizeof(struct proctype_struct)); i++)
    if ((ptypes[i].pinstr == p) && ((ptypes[i].pattr & mask) == a))
      return (ptypes[i].pstring);
  return NULL ;
}
/*}}}*/
/*{{{  option setting*/
/*{{{  functions to set options*/
/*{{{  PRIVATE int optseterrormode()*/
PRIVATE int optseterrormode ( void )
{
  int i;
  for (i = 0; i < (sizeof(errmodes) / sizeof(struct errmode_struct)); i++)
    if (optstring[0] == errmodes[i].emodechar)
      {
        if ((errormode != UNKNOWN_ERROR_MODE) && (errormode != errmodes[i].emode))
          harnesserror("Duplicate error modes on command line", 0, 0);
        errormode             = errmodes[i].emode;
        processorattr_errmode = errmodes[i].eattr;
        if (!tcoff_obj_format)
          objectfileext[3]    = errmodes[i].eobjextchar;
#ifndef TCOFF_LIB_NAMES
        if (!extlibfilename_overruled)
          extlibfilename[6]   = errmodes[i].eextlibchar;
#endif
        return TRUE;
      }
  return FALSE;
}
/*}}}*/
#ifndef CONFIG
/*{{{  optprocessor*/
PRIVATE int optprocessor ( void )
{
  BIT32 old = processortype;
  BIT32 new = setprocessor(optstring);
  if ((old != UNKNOWN_PROCESSOR_TYPE) && (old != new))
    harnesserror("Duplicate processor types on command line", 0, 0);
  return (new != ZERO32);
}
/*}}}*/
/*{{{  optnostdlibs ()*/
PRIVATE int optnostdlibs ( void )
{
  stdlibsenabled = FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  optnoalias*/
PRIVATE int optnoalias ( void )
{
  checkalias = FALSE;
  checkusage = FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  optnousage*/
PRIVATE int optnousage ( void )
{
  checkusage = FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  optnodebugoutput*/
PRIVATE int optnodebugoutput ( void )
{
  /* debugoutput = FALSE; */  /* Disable all debug output         */
  minimal_debugoutput = TRUE; /* Just create minimal debug output */
  return TRUE;
}
/*}}}*/
#endif
/*{{{  optnoassert*/
PRIVATE int optnoassert(void)
{
  ignore_assertions = TRUE;
  return TRUE;
}
/*}}}*/
/*{{{  optnointeractive ()*/
PRIVATE int optnointeractive ( void )
{
  /*chanaspointer = FALSE;*/   /* leave this always TRUE */
  iobycall = FALSE;
  flagasiobycall = FALSE;
#ifdef CONFIG
  config_postmortem = TRUE;
#endif
  return TRUE;
}
/*}}}*/
/*{{{  optbrieferrors*/
PRIVATE int optbrieferrors ( void )
{
  return brieferrors = TRUE;
}
/*}}}*/
/*{{{  optcheckonly*/
PRIVATE int optcheckonly ( void )
{
  generatecode = FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  optfullguys*/
PRIVATE int optfullguys ( void )
{
  guyinserts = FULLGUYS;
  return TRUE;
}
/*}}}*/
/*{{{  optseqguys*/
PRIVATE int optseqguys ( void )
{
  guyinserts = SEQUENTIALGUYS;
  return TRUE;
}
/*}}}*/
/*{{{  optinformation*/
PRIVATE int optinformation ( void )
{
  return information = TRUE;
}
/*}}}*/
#if 0
/*{{{  optmakelib*/
PRIVATE int optmakelib ( void )
{
  compilemode = COMP_LIB;
  return TRUE;
}
/*}}}*/
#endif /* 0 */
/*{{{  PRIVATE int optcodesize*/
PRIVATE int optcodesize(void)
{
  if (strlen(optparam) > 0)
    {
      req_code_size = atoi(optparam);
      return TRUE;
    }
  else
    {
      harnesserror("Missing code size", 0, 0);
      return FALSE;
    }
}
/*}}}*/
/*{{{  PRIVATE int optzsuffix*/
PRIVATE int optzsuffix(void)
{
  if (strlen(optparam) < MAX_LIBNAMESSUFFIX)
    {
      if (optstring[2] == 'I') /* ZLIS */
        {
          strcpy(vlibsuffixstring, optparam);
          vlibsuffix = &vlibsuffixstring[0];
        }
      else if (optstring[3] == 'P') /* ZLCP */
        {
          strcpy(predefsuffixstring, optparam);
          predefsuffix = &predefsuffixstring[0];
        }
      else /* if (optstring[3] == 'S') */ /* ZLCS */
        {
          strcpy(compilersuffixstring, optparam);
          compilersuffix = &compilersuffixstring[0];
        }
      return TRUE;
    }
  else
    {
      harnesserror("Suffix too long", 0, 0);
      return FALSE;
    }
}
/*}}}*/
/*{{{  PRIVATE int set_filename*/
PRIVATE int set_filename(char *target_filename, char *err_string)
{
  if (strlen(optparam) > 0)
    {
      /* This can't overflow cos we checked the size of optparam */
      strcpy(target_filename, optparam);
      return TRUE;
    }
  else
    {
      harnesserror("Missing %s file name", (BIT32)err_string, 0);
      return FALSE;
    }
}
/*}}}*/
/*{{{  optobjfilename*/
PRIVATE int optobjfilename ( void )
{
#ifdef CONFIG
  return set_filename(cfbfilename, "object");
#else
  return set_filename(objfilename, "object");
#endif
}
/*}}}*/
/*{{{  PRIVATE optoutputfilename*/
PRIVATE int optoutputfilename ( void )
{
  if (strlen(optparam) > 0)
    {
      if ((outfile = fopen(optparam, "w")) == NULL)
        {
          harnesserror("Cannot open output file (%s)", (BIT32)optparam, 0);
          return FALSE;
        }
      errfile = outfile;
      return TRUE;
    }
  else
    {
      harnesserror("Missing output file name", 0, 0);
      return FALSE;
    }
}
/*}}}*/
/*{{{  PRIVATE int optextlibfilename*/
PRIVATE int optextlibfilename(void)
{
  extlibfilename_overruled = TRUE;
  return set_filename(extlibfilename, "compiler library");
}
/*}}}*/
/*{{{  PRIVATE int optvlibsfilename*/
PRIVATE int optvlibsfilename(void)
{
  return set_filename(vlibsfilename, "io library");
}
/*}}}*/
/*{{{  optnovecspace*/
PRIVATE int optnovecspace ( void )
{
  vsenabled = FALSE;
  return TRUE;
}
/*}}}*/
/*{{{  optnorangecheck*/
PRIVATE int optnorangecheck ( void )
{
  errormodes_off_mask |= ERRORMODE_RANGECHECK;
  return TRUE;
}
/*}}}*/
/*{{{  optnoanycheck*/
PRIVATE int optnoanycheck ( void )
{
  errormodes_off_mask |=  ERRORMODE_RANGECHECK
                        | ERRORMODE_CONVERSIONCHECK
                        | ERRORMODE_NEED_ERRORS
                        | ERRORMODE_TIMESLICECHECK
                        | ERRORMODE_NEED_STOPERR  ;
  return TRUE;
}
/*}}}*/
#ifdef CONFIG
/*{{{  Rom options*/
PRIVATE int optrom (void)
{
  config_rom = TRUE;
  config_rom_in_rom = (optstring[1] == 'O');
  return TRUE;
}
/*}}}*/
/*{{{  reordering*/
PRIVATE int optreorder(void)
{
  config_reorderable = TRUE;
  return TRUE;
}
/*}}}*/
#endif
/*{{{  Wn options and NWn options*/
PRIVATE int optwarnings ( void )
{
  int i = (optstring[0] == 'Z') ? 1 : 0;  /* Z prefix shift the rest by 1 */
  switch ((optstring[i] == 'W') ? optstring[i+1] : optstring[i+2])
    {
    /*case 'C' : warning_flags &= ~WARNING_CSE;                           break;*/
      case 'D' : warning_flags |= (WARNING_DESCOPED_N | WARNING_DESCOPED_P); break;
      case 'F' : warning_flags &= ~WARNING_DESCOPED_P;                    break;
      case 'O' : warning_flags |=  WARNING_OVERLAPS;                      break;
      case 'U' : warning_flags &= ~(WARNING_UNUSED_V | WARNING_UNUSED_R); break;
      case 'P' : warning_flags &= ~WARNING_UNUSED_P;                      break;
    }
  return TRUE;
}
/*}}}*/
/*{{{  Z options */
PRIVATE int optzed ( void )
{
  switch (optstring[1]) /* optstring[0]=='Z' */
    {
      default  : zinfo = TRUE;                                          break;
      case 'B' : assembly_output = TRUE;                                break;
      case 'D' : disassemble = diagnostics = TRUE;                      break;
      case 'E' : hidelibnames = FALSE;                                  break;
      case 'H' : object_file_flags |= OBJ_FILE_OCCAM_HARNESS;           break;
      case 'L' : onlylex = TRUE;                                        break;
#if defined(SUN) || defined(GNU)
      case 'M' : memstats = TRUE;                                       break;
#endif
      case 'V' : iobycall = FALSE; /* Leave 'flagasiobycall' alone */   break;
      case 'X' : libpatchsize = 8;                                      break;
      case 'Z' : code_style_flags |= CODE_STYLE_ALT_PRI_PAR;            break;
      case 'A' :
        switch(optstring[2])
          {
            default  : assembly_output = diagnostics = TRUE;            break;
            case 'N' : alloc_strategy |= ALLOC_NODIVBYSIZE;             break;
            case 'S' : alloc_strategy |= ALLOC_BYSCOPE;                 break;
          }
        break;
#ifdef CONFIG
      case 'C' :
        switch(optstring[2])
          {
            case 'A' : config_readable_names = TRUE;                    break;
          /*case 'B' : config_backend = TRUE;                           break;*/
            case 'D' : config_backend_dump = TRUE;                      break;
            case 'F' : config_backend = FALSE;                          break;
            case 'H' : config_hw_dump = TRUE;                           break;
            case 'I' : config_info = TRUE;                              break;
            case 'O' : config_overlay = FALSE;                          break;
            case 'P' : config_postmortem = FALSE;                       break;
            case 'M' : config_reclaim_mem = FALSE;                      break;
            case 'N' : config_code = FALSE;                             break;
            case 'R' : config_profile = TRUE;                           break;
            case 'S' : config_srcout = TRUE;                            break;
            case 'Y' : symbolic_debugoutput = TRUE;                     break;
          }
        break;
#endif
      case 'O' :
        switch(optstring[2])
          {
            default: minimal_debugoutput = FALSE; /* need full debug info */
                 source_output = assembly_output = debugoutput = TRUE;  break;
            case 'E' : object_file_flags |= OBJ_FILE_NO_EXPORT_ORIGINS; break;
            case 'I' : object_file_flags |= OBJ_FILE_NO_IMPORT_ORIGINS; break;
            case 'L' : object_file_flags |= OBJ_FILE_PRI_TEXT_SECTION;  break;
            case 'R' : object_file_flags |= OBJ_FILE_READ_INCOMPATIBLE_LIBS; break;
            case 'V' : object_file_flags |= OBJ_FILE_NO_HASH_VERSION;   break;
          }
        break;
      case 'N' :
        switch(optstring[2])
          {
            case 'D' : debugoutput = FALSE;                             break;
            case 'E' : lexer_ignore_comments = TRUE;                    break;
            case 'J' : code_style_flags |= CODE_STYLE_CJ_NOT_J;         break;
            case 'I' : allow_inlines = FALSE;                           break;
            case 'O' : alloc_strategy |= ALLOC_NOOPERANDS;              break;
            case 'P' : allowpredefs = FALSE;                            break;
            case 'V' : alloc_strategy |= ALLOC_NOVSOPT;                 break;
          }
        break;
      case 'S' :
        switch(optstring[2])
          {
            case 'C' : call_check = FALSE; (void) optcheckonly();       break;
            case 'M' : stop_after_map   = TRUE;                         break;
            case 'P' : call_check = FALSE; nochecking = TRUE;
                       (void) optcheckonly();                           break;
            case 'T' : stop_after_trans = TRUE;                         break;
            case 'U' : debuguse = TRUE; (void) optcheckonly();          break;
          }
        break;
      case 'T' :
        switch(optstring[2])
          {
            default  : prtree = TRUE;                                   break;
            case 'S' : testflag = TRUE;                                 break;
          }
        break;
      case 'W' :
        switch(optstring[2])
          {
            default  : chanaspointer = FALSE;                           break;
            case 'A' : warn_on_usage_error = TRUE;                      break;
            case 'C' : warn_comment_indent = TRUE;                      break;
          }
        break;
    }
  return TRUE;
}
/*}}}*/
/*{{{  optpathname*/
PRIVATE int optpathname (void)
{
  return set_filename(pathname, "(env var name)");
}
/*}}}*/
/*{{{  optexecute ()*/
PRIVATE int optexecute ( void )
{
#ifdef IMS
  repeat_loop = (optstring[1] == 'M');

  #ifdef CONFIG
    /* This is used because the backend can abort directly if it gets a
       fatal error */
    config_repeat = repeat_loop;
  #endif
#endif
  return TRUE;
}
/*}}}*/
/*{{{  optloadonly ()*/
PRIVATE int optloadonly ( void )
{
#ifdef IMS
  load_only = TRUE;
#endif
  return TRUE;
}
/*}}}*/
/*}}}*/
/*{{{  option table*/
PRIVATE struct optionstruct
  {
    char *optionstring;
    int (*optionaction)(void);
    int hiddenopt;
    int optionparam;
    char *optiondescription;
  } cloptions[] =
  {
/*{{{  a - e*/
#ifndef CONFIG
  {"A",    optnoalias,         FALSE, FALSE, "disable alias checking"},
#endif
  {"B",    optbrieferrors,     FALSE, FALSE, "display brief error messages"},
  {"C",    optcheckonly,       FALSE, FALSE, "check only"},
#ifndef CONFIG
  {"D",    optnodebugoutput,   FALSE, FALSE, "minimum debugging data"},
  {"E",    optnostdlibs,       FALSE, FALSE, "disable compiler libraries"},
#endif
/*}}}*/
/*{{{  g - s*/
  {"G",    optseqguys,         FALSE, FALSE, "sequential code insertion"},
  {"H",    optseterrormode,    FALSE, FALSE, "halt error mode (default)"},
#ifndef CONFIG
  {"H1",   optprocessor,       TRUE,  FALSE, "experimental H1 G-process"},
  {"H1L",  optprocessor,       TRUE,  FALSE, "experimental H1 L-process"},
#endif
  {"I",    optinformation,     FALSE, FALSE, "output information"},
  {"K",    optnorangecheck,    FALSE, FALSE, "disable range checking"},
/*{"KA",   optnoanycheck,      FALSE, FALSE, "disable all error checking"},*/
/*{"L",    optmakelib,         FALSE, FALSE, "create a library"}, */
  {"L",    optloadonly,        HOSTED,FALSE, "load compiler and do nothing"},
#ifndef CONFIG
  {"M212", optprocessor,       TRUE,  FALSE, "target to M212"},
  {"N",    optnousage,         FALSE, FALSE, "disable usage checking"},
#endif
  {"NA",   optnoassert,        FALSE, FALSE, "disable run-time ASSERT checks"},
  {"NWP",  optwarnings,        FALSE, FALSE, "no unused parameter warnings"},
  {"NWU",  optwarnings,        FALSE, FALSE, "no unused name warnings"},
  {"O",    optobjfilename,     FALSE, TRUE,  "specify output file"},
#if 0
  {"Q",    optprogramblocksize, FALSE, TRUE, "set block size"},
#endif
  {"R",    optoutputfilename,  FALSE, TRUE,  "redirect screen output to file"},
#ifdef CONFIG
  {"RA",   optrom,             FALSE, FALSE, "configure for ROM running in RAM"},
  {"RE",   optreorder,         FALSE, FALSE, "enable memory layout reordering"},
  {"RO",   optrom,             FALSE, FALSE, "configure for ROM running in ROM"},
#endif
  {"S",    optseterrormode,    FALSE, FALSE, "stop error mode"},
/*}}}*/
/*{{{  t*/
#ifndef CONFIG
  {"TA",   optprocessor,       FALSE, FALSE, "target to TA processor class"},
  {"TB",   optprocessor,       FALSE, FALSE, "target to TB processor class"},
/*{"TC",   optprocessor,       FALSE, FALSE, "target to TC processor class"},*/
  {"T2",   optprocessor,       TRUE,  FALSE, "target to T212"},
  {"T212", optprocessor,       TRUE,  FALSE, "target to T212"},
  {"T222", optprocessor,       TRUE,  FALSE, "target to T222"},
  {"T225", optprocessor,       TRUE,  FALSE, "target to T225"},
  {"T3",   optprocessor,       TRUE,  FALSE, "target to T225"},
  {"T4",   optprocessor,       TRUE,  FALSE, "target to T414"},
  {"T400", optprocessor,       TRUE,  FALSE, "target to T400"},
  {"T414", optprocessor,       TRUE,  FALSE, "target to T414 (default)"},
  {"T425", optprocessor,       TRUE,  FALSE, "target to T425"},
  {"T5",   optprocessor,       TRUE,  FALSE, "target to T400 / T425"},
  {"T8",   optprocessor,       TRUE,  FALSE, "target to T800"},
  {"T800", optprocessor,       TRUE,  FALSE, "target to T800"},
  {"T801", optprocessor,       TRUE,  FALSE, "target to T801"},
  {"T805", optprocessor,       TRUE,  FALSE, "target to T805"},
  {"T9",   optprocessor,       TRUE,  FALSE, "target to T801 / T805"},
#endif
/*}}}*/
/*{{{  u - y*/
/*{"U",    optseterrormode,    FALSE, FALSE, "undefined error mode"},*/
  {"U",    optnoanycheck,      FALSE, FALSE, "disable run-time error checking"},
  {"V",    optnovecspace,      FALSE, FALSE, "disable separate vector space"},
  {"W",    optfullguys,        FALSE, FALSE, "full code insertion"},
/*{"WC",   optwarnings,        TRUE,  FALSE, "no common subexp elim warnings"},*/
  {"WD",   optwarnings,        FALSE, FALSE, "provide descoped name warnings"},
  {"WO",   optwarnings,        FALSE, FALSE, "provide overlap check warnings"},
  {"X",    optseterrormode,    FALSE, FALSE, "universal error mode"},
  {"XO",   optexecute,         HOSTED,FALSE, "execute once only"},
  {"XM",   optexecute,         HOSTED,FALSE, "execute many times"},
  {"Y",    optnointeractive,   FALSE, FALSE, "disable interactive debugging"},
/*}}}*/
/*{{{      z*/
  /* Hidden options marked as 'used' are _required_ by some other tools,
     so cannot be arbitrarily changed.
     Others might have been used too.
  */
  {"Z",    optzed,             TRUE,  FALSE, "display hidden option info"},
  {"ZA",   optzed,             TRUE,  FALSE, "assembler & diagnostics"},
  {"ZAND", optzed,             TRUE,  FALSE, "don't allocate divided by size"},
  {"ZAS",  optzed,             TRUE,  FALSE, "allocate vars in scope order"},
  {"ZB",   optzed,             TRUE,  FALSE, "assembler output only"},
#ifdef CONFIG
  {"ZCA",  optzed,             TRUE,  FALSE, "readable formal param names"},
/*{"ZCB",  optzed,             TRUE,  FALSE, "configuration backend"},*/
  {"ZCD",  optzed,             TRUE,  FALSE, "configuration backend dump"},
  {"ZCF",  optzed,             TRUE,  FALSE, "no configuration backend"},
  {"ZCH",  optzed,             TRUE,  FALSE, "configuration hardware dump"},
  {"ZCI",  optzed,             TRUE,  FALSE, "configuration diagnostics"},
  {"ZCO",  optzed,             TRUE,  FALSE, "don't overlay startup code"},
  {"ZCP",  optzed,             TRUE,  FALSE, "disable collector patching"},
  {"ZCM",  optzed,             TRUE,  FALSE, "don't reclaim memory"},
  {"ZCN",  optzed,             TRUE,  FALSE, "no configuration code"},
  {"ZCR",  optzed,             TRUE,  FALSE, "turn on profiling"},
  {"ZCS",  optzed,             TRUE,  FALSE, "output configuration source"},
  {"ZCY",  optzed,             TRUE,  FALSE, "output symbolic debug info"},
#endif
  {"ZD",   optzed,             TRUE,  FALSE, "disassemble after crunch"},
  {"ZE",   optzed,             TRUE,  FALSE, "visible compiler library names"},
  {"ZH",   optzed, /* used */  TRUE,  FALSE, "mark output as an occam harness"}, /* used for C run-time harness */
  {"ZI",   optpathname,        TRUE,  TRUE,  "change default pathname"},
  {"ZL",   optzed,             TRUE,  FALSE, "display lex output"},
  {"ZLC",  optextlibfilename,  TRUE,  TRUE,  "change compiler library"},         /* used to build FORTRAN compiler libs */
  {"ZLCP", optzsuffix,/*used*/ TRUE,  TRUE,  "specify compiler predef suffix"},  /* used to build FORTRAN compiler libs */
  {"ZLCS", optzsuffix,/*used*/ TRUE,  TRUE,  "specify compiler library suffix"}, /* used to build FORTRAN compiler libs */
  {"ZLI",  optvlibsfilename,   TRUE,  TRUE,  "change compiler io library"},      /* used to build FORTRAN compiler libs */
  {"ZLIS", optzsuffix,/*used*/ TRUE,  TRUE,  "specify io library suffix"},       /* used to build FORTRAN compiler libs */
#if defined(SUN) || defined(GNU)
  {"ZMEM", optzed,             TRUE,  FALSE, "display memory (sbrk) statistics"},
#endif
  {"ZNEC", optzed,             TRUE,  FALSE, "no comment indentation errors"},
  {"ZND",  optzed,             TRUE,  FALSE, "no debug information at all"},
  {"ZNI",  optzed,             TRUE,  FALSE, "disable INLINE"},
  {"ZNJ",  optzed, /* used */  TRUE,  FALSE, "use cj not j (mostly)"}, /* used in insight kernel */
  {"ZNO",  optzed,             TRUE,  FALSE, "no disassembly operands"},
  {"ZNP",  optzed,             TRUE,  FALSE, "no predefines allowed"},
  {"ZNV",  optzed,             TRUE,  FALSE, "don't opt when vsoffset is zero"},
  {"ZNWF", optwarnings,        TRUE,  FALSE, "no formal param descope warnings"},
#ifndef CONFIG
  {"ZO",   optzed,             TRUE,  FALSE, "display source / disassembly"},
#endif
  {"ZOE",  optzed,             TRUE,  FALSE, "disable export of origin symbol"},
  {"ZOI",  optzed,             TRUE,  FALSE, "disable imported origin checks"},
  {"ZOL",  optzed,             TRUE,  FALSE, "prioritise linkage"},
  {"ZOR",  optzed,             TRUE,  FALSE, "read in incompatible lib entries"},
  {"ZOS",  optcodesize,        TRUE,  TRUE,  "set object buffer size (in K)"},
  {"ZOV",  optzed,             TRUE,  FALSE, "don't hash version into origin"},
  {"ZSC",  optzed,             TRUE,  FALSE, "stop after type check"},
  {"ZSM",  optzed,             TRUE,  FALSE, "stop after mapping"},
  {"ZSP",  optzed,             TRUE,  FALSE, "stop after parse"},
  {"ZST",  optzed,             TRUE,  FALSE, "stop after trans"},
  {"ZSU",  optzed,             TRUE,  FALSE, "stop after & debug usage check"},
  {"ZT",   optzed,             TRUE,  FALSE, "print tree"},
  {"ZTST", optzed,             TRUE,  FALSE, "generic test flag"},
  {"ZV",   optzed, /* used */  TRUE,  FALSE, "do i/o not by call"}, /* used to build virtual.lib */
  {"ZW",   optzed, /* used */  TRUE,  FALSE, "do channels not by pointer"}, /* used to build debugger & bootstraps */
  {"ZWAU", optzed,             TRUE,  FALSE, "warn on alias and usage checks"},
  {"ZWC",  optzed,             TRUE,  FALSE, "warn on comment indentation"},
  {"ZX",   optzed,             TRUE,  FALSE, "8 byte libpatch"},
  {"ZZ",   optzed,             TRUE,  FALSE, "use ALT for PRI PAR"}
/*}}}*/
  };
/*}}}*/
/*}}}*/
/*{{{  PUBLIC int process_option ()*/
/*  This is called with an #OPTION string */
PUBLIC int process_option ( char *s )
{
  int ok = TRUE;
  while (TRUE)
    {
      switch (islower(*s) ? toupper(*s) : *s)
        {
          default:    synerr_i(SYN_BAD_OPTION, flocn, *s);
                      ok = FALSE;
                      break;
          case ' ':   break;
          case '\0':  return ok;
#ifndef CONFIG
          case 'A':   (void) optnoalias();    break;
          case 'E':   (void) optnostdlibs();  break;
#endif
          case 'G':   (void) optseqguys();    break;
          case 'K':   (void) optnorangecheck();  seterrormode();     break;
#ifndef CONFIG
          case 'N':   (void) optnousage();    break;
#endif
          case 'U':   (void) optnoanycheck();    seterrormode();     break;
          case 'V':   (void) optnovecspace(); break;
          case 'W':   (void) optfullguys();   break;
          case 'Y':   (void) optnointeractive(); setprocessorattr(); break;
        }
      s++;
    }
}
/*}}}*/
/*{{{  PRIVATE int matchoption (char *optstr, struct optionstruct *options, int )*/
PRIVATE int matchoption ( char *optstr , struct optionstruct *options , int noptions )
{
  /* This returns the option number, if
     1) The option doesn't require a parameter, and is an exact match, or,
     2) The option does require a parameter, and the first n chars match.
  */
  /* Note that we go 'backwards' through the list of options, just incase
     a possible option which requires a parameter would hide another option
     starting with the same prefix.
     (Assuming that the table is in alphabetical order).
  */

  int i;
  for (i = noptions - 1; i >= 0; i--)
    {
      char *s = options[i].optionstring;
      if (((options[i].optionparam) && (strncmp(optstr, s, strlen(s)) == 0) ) ||
          (strcmp(optstr, s) == 0) )
            return(i);
    }
  return (-1);
}
/*}}}*/
/*{{{  PRIVATE int getnextstring (char *s)*/
PRIVATE int getnextstring (int argc, char *argv[], char *s)
{
  int len = 0;
  char *start;
  if (*cptr == '\0')
    {
      if (argsdone < argc)
        cptr = argv[argsdone++];
      else
        return FALSE;
    }
  start = cptr;
  do
    {
      len ++;
      if (len >= MAX_OPTION_LENGTH)
        harnesserror("Command line option is too long, max %d chars",
                     MAX_OPTION_LENGTH, ZERO32);
      *s++ = *cptr++;
    }
  /*while (*cptr != '\0' && *cptr != escape_char);*/
  while (*cptr != '\0');
  *s = '\0';

  return TRUE;
}
/*}}}*/
/*{{{  PRIVATE int parse_command_line (struct optionstruct *options, int noption)*/
PRIVATE int parse_command_line (int argc, char *argv[],
                                struct optionstruct *options , int noptions )
{
  if (argc > 1)
    {
      char local_optstring[MAX_OPTION_LENGTH];
      char local_optparam[MAX_OPTION_LENGTH];  /* its parameter */
      cptr = argv[1];
      argsdone = 2;
      while (getnextstring(argc, argv, local_optstring))
        /*{{{  parse a string*/
        {
          if (local_optstring[0] == escape_char)
            {
              int p;
              strupr(local_optstring);
              p = matchoption(&(local_optstring[1]), options, noptions);
              if (p >= 0)
                {
                  local_optparam[0] = '\0'; /*strcpy(local_optparam, "");*/
                  if (options[p].optionparam)
                    {
                      /* check for merged option and its parameter */
                      int len_got = strlen(local_optstring);
                      int len_req = strlen(options[p].optionstring);

                      /* if merged, point at start of un-uppercased parameter */
                      if (len_req < len_got)
                        cptr = argv[argsdone-1] + 1 + len_req; /* 1 for escape character */

                      getnextstring(argc, argv, local_optparam);
                    }
                  optstring = &(local_optstring[1]);
                  optparam  = &(local_optparam[0]);
                  if (!(*options[p].optionaction)())
                    return FALSE;
                }
              else
                {
                  harnesserror("Invalid command line option (%s)",(BIT32)(local_optstring + 1), 0);
                  return FALSE;
                }
            }
          else if (sourcefilename[0] == '\0')
            /* This can't overflow cos we've checked optstring already */
            strcpy(sourcefilename, local_optstring);
          else
            {
              harnesserror("Invalid command line option (%s)", (BIT32)local_optstring, 0);
              return FALSE;
            }
        }
        /*}}}*/
    }
  return TRUE;
}
/*}}}*/
/*{{{  PRIVATE void host_setup (void) */
PRIVATE void host_setup ( void )
{
#ifdef IMS
  int host, os, board;
  host_info(&host, &os, &board);
  switch (os)
    {
      case _IMS_OS_VMS : escape_char = '/'; errfile = stderr; return;
      case _IMS_OS_DOS : escape_char = '/'; errfile = stdout; return;
      default          : escape_char = '-'; errfile = stderr; return;
    }
  (void) set_abort_action(ABORT_HALT);
#else
# if defined(DEC) /* VMS */
    escape_char = '/'; errfile = stderr;
# elif defined(MSC)
    escape_char = '/'; errfile = stdout;
# else
    escape_char = '-'; errfile = stderr;
# endif
#endif
}
/*}}}*/
/*{{{  find_extension*/
PRIVATE char *find_extension ( char *filename, int extension )
/* if extension is TRUE, returns the address of the dot,
   if extension is FALSE, returns the address of the last char of the directory,
   returns NULL if an extension, or a directory, was never found.
*/
{
  int i = strlen(filename);
  while (i > 0)
    {
      switch(filename[i])
        /*{{{  see if we have hit extension*/
        {
          case '.' : 
            if (extension) return &(filename[i]);
            break;
          case '[': case ']': case '/': case '\\': case ':': case '<': case '>':
            if (!extension) return &(filename[i]);
            return (NULL);
          default  : break;
        }
        /*}}}*/
      i--;
    }
  return (NULL);
}
/*}}}*/
/*{{{  setup_filenames*/
PRIVATE int setup_filenames ( void )
{
  int equal_names;
  char *extension_start = find_extension(sourcefilename, TRUE);
  int root_length = (extension_start == NULL) ?
                     strlen(sourcefilename)   :
                     extension_start - sourcefilename;
  strncpy(rootfilename, sourcefilename, root_length);
  rootfilename[root_length] = '\0';
  if (extension_start == NULL) /* tack on default extension to source file */
    strcat(sourcefilename, SOURCEFILE_EXTENSION);

#ifdef CONFIG
  if (cfbfilename[0] == '\0')
    {
      char *dir_end = find_extension(rootfilename, FALSE);
      strcpy(cfbfilename, (dir_end == NULL) ? rootfilename : (dir_end + 1));
      strcat(cfbfilename, CFB_FILE_EXTENSION);
    }
  { /* make objfilename same as cfbfilename but with different suffix */
    char *suff_start = find_extension(cfbfilename, TRUE);
    int root_length = (suff_start == NULL) ? strlen(cfbfilename)
                                           : suff_start - cfbfilename;
    strncpy(objfilename, cfbfilename, root_length);
    objfilename[root_length] = '\0';
    strcat(objfilename, objectfileext);
  }
  equal_names = (strcmp(cfbfilename, sourcefilename) == 0) ||
                (strcmp(objfilename, sourcefilename) == 0);
#else
  if (objfilename[0] == '\0')
    {
      char *dir_end = find_extension(rootfilename, FALSE);
      strcpy(objfilename, (dir_end == NULL) ? rootfilename : (dir_end + 1));
      strcat(objfilename, objectfileext);
    }
  equal_names = strcmp(objfilename, sourcefilename) == 0;
#endif
  if (equal_names)
    {
      harnesserror("Output file is the same as the input file", ZERO32, ZERO32);
      return (FALSE);
    }
  return (TRUE);
}
/*}}}*/
/*{{{  PUBLIC void add_default_extension(filename)*/
/*****************************************************************************
 *  add_default_extension takes the source string, and if it doesn't have an
 *                extension, adds the same extension as the object file
 *****************************************************************************/
PUBLIC void add_default_extension ( char *filename )
{
  if (find_extension(filename, TRUE) == NULL)
    {
      char *default_start = find_extension(objfilename, TRUE);
      if (default_start != NULL)
        strcat(filename, tcoff_obj_format ? default_start : objectfileext);
    }
}
/*}}}*/
/*{{{  PRIVATE void display_one_option (option, add_newline)*/
PRIVATE void display_one_option ( struct optionstruct *option , int add_newline )
{
    int len;
    fputc(' ', outfile);

    fputs(option->optionstring, outfile);
    for (len = strlen(option->optionstring); len < 5; len++)
      fputc(' ', outfile);

    fputs(option->optiondescription, outfile);
    if (add_newline)
      fputc('\n', outfile);
    else
      for (len = strlen(option->optiondescription); len < 33; len++)
        fputc(' ', outfile);
}
/*}}}*/
/*{{{  PRIVATE void display_help_page ()*/
PRIVATE void display_help_page ( void )
{
    int i, col = 0;

    fprintf(outfile, "%s : occam 2 toolset %s\n", compilername, compilerfunction);
    fprintf(outfile, "%s\n", C_VERSION);
  /*fprintf(outfile, "Compatibility: %s\n", C_COMPATIBILITY);*/
    fprintf(outfile, "(c) Copyright INMOS Limited 1988, 1989, 1990, 1991\n\n");
    fprintf(outfile, "Usage: %s filename { %coption }\n\n", compilername,
            escape_char);
    fprintf(outfile, "Options:\n");

    for (i = 0; i < sizeof(cloptions) / sizeof(cloptions[0]); i++)
      if (zinfo || (!(cloptions[i].hiddenopt)))
        { 
          display_one_option (&cloptions[i], col % 2);
          col++;
        }
    if (col % 2) fputc('\n', outfile);
#ifndef CONFIG
    if (!zinfo)
      fprintf(outfile, "Also full target processor name required (default is %s)\n",
                       DEFAULT_PROCESSOR_TYPE);
#endif
}
/*}}}*/
#if !defined(FRONTEND) && (defined(OC) || defined(CONFIG))
/*{{{  PUBLIC void allocateworkspace(treeroot)*/
/* Compile the code for treeroot */
PUBLIC void allocateworkspace ( treenode *treeroot )
{
  transmain(treeroot); /* Transform tree */
  freeup_temp_workspace();
#ifndef CONFIG
  if (information && comp_error == 0)
    fprintf(outfile,
      "Syntax tree transformed ok, now occupies %ld bytes\n", tablesize());
#endif
  if (diagnostics && prtree)
    printtree(0, treeroot);
  if (stop_after_trans)
    return;
#ifndef CONFIG
  if (information && comp_error == 0)
    fputs("Allocating workspace\n", outfile);
#endif
  mapmain(treeroot); /* Allocate workspace */
  if (diagnostics && prtree)
    printtree(0, treeroot);
}
/*}}}*/
/*{{{  PUBLIC void codegenerate(open_file, treeroot)*/
/* Compile the code for treeroot */
PUBLIC void codegenerate (int open_file, treenode *treeroot )
{
  if (!assembly_output && !disassemble)
    {
      if (open_file)
        objfile = open_object_file(objfilename);
      write_id(sourcefilename, FALSE /*compilemode == COMP_LIB*/);
    }
  if (debugoutput)
    {
    #ifndef CONFIG
      if (information)
        fputs("Writing debug information\n", outfile);
    #endif
      debugmain(treeroot);
    }
#ifndef CONFIG
  if (information && comp_error == 0)
    fputs("Generating code\n", outfile);
#endif
  tmain(treeroot);   /* Generate code */
  if (assembly_output || disassemble)
    fputc('\n', outfile);
  else if (open_file)
    close_object_file(objfile, objfilename);
#ifndef CONFIG
  if (information && comp_error == 0)
    fputs("Code generated ok\n", outfile);
#endif
  freeup_temp_workspace();
}
/*}}}*/
#endif
/*{{{  PUBLIC void print_memstats*/
PUBLIC void print_memstats(void)
{
#if defined(SUN) || defined(GNU)
  if (memstats)
    fprintf(outfile, "Memory usage: (current mem %ld) (sbrk %d)\n",
                     tablesize(), sbrk(0) - original_sbrk);
#endif
}
/*}}}*/

/*{{{  main*/
int main ( int argc , char *argv [])
{
  treenode *treeroot;
#if defined(SUN) || defined(GNU)
  original_sbrk = sbrk(0);
#endif
  outfile = stdout; /* can't do this as a static initialisation on the VAX */
  /*{{{  set version*/
  host_setup();
  if (!setup_version_string())
    {
      harnesserror("Version string overflow", 0, 0);
      end_compiler(EXIT_FAILURE);
    }
  /*}}}*/
  /*{{{  read in the arguments and go*/
  {
    /*{{{  set default arguments*/
    /* errfile is setup inside host_setup */
    
    object_file_flags = OBJ_FILE_DEFAULT;
    
    strcpy(pathname, DEFAULT_PATHNAME);
    
    strcpy(sourcefilename, "");
    strcpy(objfilename,    "");
  #ifdef CONFIG
    strcpy(cfbfilename,    "");
  #endif
    strcpy(extlibfilename,
      tcoff_obj_format ? TCOFF_EXTLIB_FILENAME : LFF_EXTLIB_FILENAME);
    strcpy(vlibsfilename,
      tcoff_obj_format ? TCOFF_VLIB_FILENAME   : LFF_VLIB_FILENAME) ;
    /*}}}*/

    if (parse_command_line(argc, argv, cloptions, sizeof(cloptions) / sizeof(cloptions[0])))
      /*{{{   start up the compiler*/
      {
        if (processortype == UNKNOWN_PROCESSOR_TYPE)
          setprocessor(DEFAULT_PROCESSOR_TYPE);
        if (errormode     == UNKNOWN_ERROR_MODE)
          {
            optstring = DEFAULT_ERROR_MODE;
            optseterrormode();
          }

        if (information || assembly_output || disassemble)
           fprintf(outfile, "%s%s\n", (assembly_output ? "-- " : ""), C_VERSION);
        #ifdef IMS
          if (load_only) end_compiler(EXIT_SUCCESS);
        #endif
        if ((sourcefilename[0] != '\0') && setup_filenames())
          /* we had a source file specified */
          {
            setprocessorattr();
            seterrormode();
            /*{{{  compile*/
            {
              if (outfile != NULL)
                {
                  feinit ();
                  comp_error = setjmp (env);
                  if (comp_error == 0)
                    {
                    #if !defined(BACKEND)
                      if (onlylex)
                        /*{{{  print the output of the lexer only*/
                        {
                          while ((symb != S_END) && (comp_error == 0))
                            { printlex (); nextsymb(); }
                          /* The first S_END means the end of the predefined names */
                          if (comp_error == 0)
                            {
                              if (open_file(sourcefilename,  LEX_SOURCE, 0))
                                {
                                  printlex (); nextsymb();
                                  while ((symb != S_END) && (comp_error == 0))
                                    { printlex (); nextsymb(); }
                                }
                              else
                                harnesserror("Cannot open source file", 0, 0);
                            }
                        }
                        /*}}}*/
                      else
                    #endif
                        /*{{{  parse and possibly code generate*/
                        {
                          if (information || assembly_output || disassemble)
                            /*{{{  say what we are doing*/
                            {
                              if (assembly_output) fputs("-- ", outfile);
                              #ifdef CONFIG
                                fputs("Configuring (", outfile);
                              #else
                                fputs(generatecode ? "Compiling" : "Checking", outfile);
                                fprintf(outfile, " (%s,", processorstring(processortype, processorattr));
                              #endif
                              fputs(errormode & ERRORMODE_HALT ?    "HALT" :
                                    errormode & ERRORMODE_STOP ?    "STOP" :
                                    errormode & ERRORMODE_REDUCED ? "REDUCED" : "UNIVERSAL", outfile);
                              fputs(") \"", outfile);
                              fputs(sourcefilename, outfile);
                              fputc('\"', outfile);
                              if (generatecode && !assembly_output && !disassemble)
                                {
                                  #ifdef CONFIG
                                    fprintf(outfile, " to \"%s\" and \"%s\"", cfbfilename, objfilename);
                                  #else
                                    fprintf(outfile, " to \"%s\"", objfilename);
                                  #endif
                                }
                              fputc('\n', outfile);
                            }
                            /*}}}*/
                        #if defined(BACKEND)
                          treeroot = readtree(sourcefilename);
                        #else
                          /*{{{  parse and scope*/
                          treeroot = rprogram (rscunit, scopeandcheck_main, sourcefilename);
                          freeup_temp_workspace();
                          /*}}}*/
                          if (information && comp_error == 0)
                            fprintf (outfile, "Syntax tree occupies %ld bytes\n", tablesize());
                          if (comp_error == 0 && call_check)
                            alias_and_usage_check(treeroot);
                          freeup_temp_workspace();
                          if (information)
                            /*{{{  info*/
                            {
                              if (comp_error == 0)
                                fputs("No errors found in source\n", outfile);
                              fprintf (outfile,
                              "Read %d lines of source, syntax tree occupies %ld bytes\n",
                                 totallines, tablesize());
                            }
                            /*}}}*/
                        #endif
                        #if defined(OC) || defined(CONFIG)
                          if (generatecode && (comp_error == 0))
                            {
                              #if defined(FRONTEND)
                                savetree(treeroot, objfilename);
                              #else
                              #ifdef CONFIG
                                config(treeroot);
                              #else
                              /*{{{  generate code*/
                              /*if ((compilemode == COMP_SC) || (compilemode == COMP_LIB))*/
                                {
                                  beinit (0);
                                  allocateworkspace(treeroot);
                                  freeup_temp_workspace();
                                  if (!stop_after_trans && !stop_after_map)
                                    codegenerate(TRUE, treeroot);
                                  freeup_temp_workspace();
                                }
                              /*}}}*/
                              #endif
                              #endif
                            }
                          else
                            if (prtree) printtree (0, treeroot);
                        #endif
                        }
                        /*}}}*/
                    }
                  if (comp_error == 0)
                    end_compiler(EXIT_SUCCESS);
                }
              else
                harnesserror("Cannot open output file", 0, 0);
            }
            /*}}}*/
          }
        /*{{{  otherwise help page, etc*/
        else if (sourcefilename[0] == '\0')
          {
            if (!information)
              display_help_page();
            end_compiler(EXIT_SUCCESS);
          }
        /*}}}*/
      }
      /*}}}*/
  }
  /*}}}*/

  end_compiler(EXIT_FAILURE);
  return (0); /* not reached */
}
/*}}}*/
