/*#define DEBUG*/
/*****************************************************************************
 *
 *  debug walks the parse tree generating debug information into the debug file
 *
 *****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
 *
 *  occam 2 compiler
 *
 *  copyright Inmos Limited 1987
 *
 *
 *****************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdlib.h>
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "extlib.h"

# include "gen2def.h"
# include "lexconst.h"
# include "lex1def.h"
# include "desc1def.h"
# include "chkdef.h"
# include "genhdr.h"
# include "debugdef.h"
# include "debughdr.h"
# include "gen1def.h"
# include "bind2def.h"
# include "generror.h"
/*}}}*/

/*{{{  COMMENT format of debug information*/
/**********************  Start comment out ****************************
@*{{{  format of debug information*@
@*{{{  Symbolic information*@
version      index versionnumber

startfold    index
endfold      index
linemark     index line

beginscope   index
endscope     index

workspace    index segtype thisoffset prevoffset
endseg       index

variable     index vartype access dims {dims access offset } offset name
procedure    index proctype dummy workspace vectorspace name
constant     index vartype hi lo name

filename     index filename
@*}}}*@
@*{{{  Code information*@
codemark     index ptr address
addressfix   index ptr address
libpatch     index address
@*}}}*@
@*}}}*@
 **********************   End comment out  ****************************/
/*}}}*/

/*{{{  constants*/
#define       MAX_DEBUG_STRING_SIZE  255 /* Size of one record */
#define   LFF_MAX_DEBUG_STRING_SIZE  255 /* Size of one LFF debug record */
#define TCOFF_MAX_DEBUG_STRING_SIZE 4096 /* Size of one TCOFF debug record */
#define C_PREFIX 0x80
#define BUF_KEEP 0
#define BUF_THROW 1
#define INDEX_HASH_TABLE_SIZE 256
#define HASH_MASK 255

/*{{{  source code output*/
#define FILE_START 0
#define FILE_END 1
/*}}}*/

#define DEBUG_VERSION (RTL_LINEMARK_BIT | RTL_OCCAM_BITS | RTL_REPLPAR_BIT | \
                       RTL_CODEADDR_BIT | RTL_PROCEDURE_BIT | RTL_ORDER_BIT )
/* The RTL_COMPACTED_BIT is or-ed in if minimal_debugoutput is TRUE */
/* The RTL_VIRTUAL_LINK_BIT is or-ed in if iobycall is TRUE */

#ifdef DEBUG
  #define debug_diagnostics diagnostics /* attach it to the main diagnostics flag */
#else
  #define debug_diagnostics FALSE
#endif

/*}}}*/
/*{{{  PRIVATE variables*/
PRIVATE INT32 enclosing_this_offset;
PRIVATE INT32 workspace_adjust;
PRIVATE int debuginfo_index;
PRIVATE int filenum;  /* The current file */
PRIVATE int nextfile; /* The next one to look at */
PRIVATE int debug_to_file;
PRIVATE int debug_buffer_size;
PRIVATE char *keep  = NULL;  /* keep buffer */
PRIVATE char *throw = NULL;  /* throw buffer */
PRIVATE char *kp;                           /* keep buffer pointer */
PRIVATE char *tp;                           /* throw buffer pointer */
/*{{{  index hash table*/
typedef struct index_hash_cell
  {
    treenode *             address;
    int                    index;
    struct index_hash_cell *next;
  } index_hash_t;

/* note that this will be initialised to NULLs */
PRIVATE index_hash_t *index_hash_table[INDEX_HASH_TABLE_SIZE];
/*}}}*/
/*{{{  source code output*/
/*{{{  linemark list*/
/* needed when producing source output */
typedef struct linemark_cell
  {
    int                  index;
    int                  linenumber;
    struct linemark_cell *next;
  } linemark_cell_t;

PRIVATE linemark_cell_t *linemark_list = NULL;
/*}}}*/
/*{{{  file list*/
/* needed when producing source output */
typedef struct file_cell
  {
    int                  tag;
    int                  fileid;
    int                  index;
    struct file_cell *next;
  } file_cell_t;

PRIVATE file_cell_t *file_list = NULL;
/*}}}*/
/*}}}*/
/*}}}*/
/*{{{  forward declarations*/
PRIVATE void gendebug ( treenode *tptr );
/*}}}*/
/*{{{  PRIVATE routines*/
/*{{{  low-level debug buffer writing*/
/*{{{  PRIVATE char *add_to_debug_buffer(string, length, type)*/
/*{{{  comment*/
/*************************************************************************
*
*  appends the debug record held in 'string' with length 'length' to
*  the debug buffer given by 'type'.
*
**************************************************************************/
/*}}}*/
PRIVATE char *add_to_debug_buffer ( char *string , int length , int type )
{
  char *buffer;
  char *bp;
  int i;
  /*{{{  set up local pointers*/
  buffer = (type == BUF_KEEP) ? keep : throw;
  bp = (type == BUF_KEEP) ? kp : tp;
  /*}}}*/
  if (length > debug_buffer_size - (bp - buffer))
    /*{{{  buffer full so output*/
    {
      write_debug_string(buffer, bp - buffer);
      bp = (type == BUF_KEEP) ? keep : throw;
    }
    /*}}}*/
  for (i=0; i < length; i++)
    *bp++ = string[i];
  return(bp);
}
/*}}}*/
/*}}}*/

/*{{{  index hash table private routines*/
/*{{{  PRIVATE hash_function(address)*/
/*{{{  comment*/
/******************************************************************************
*
*  hash_function(address)
*  address = address of a tree node.
*  The hashing algorithm is performed on the address of a tree node. The
*  tree node pointer is cast into a BIT32 type and the following actions are
*  taken. The address word is shifted two bits right thereby removing the two
*  least significant bits. The resulting address word is then ANDed with 255
*  to produce an eight bit number which is used to index the hash table.
*  Andy Whitlow 19.10.88
*
*  Modified to shift by 5 instead; 9/10/90 CON
*
******************************************************************************/
/*}}}*/
#if 0
  PRIVATE BIT32 hash_function( treenode *address )
  {
    return((((BIT32)address) >> 5) & HASH_MASK);
  }
#else
  #define hash_function(address) ((((BIT32)(address)) >> 5) & HASH_MASK)
#endif
/*}}}*/
/*{{{  PRIVATE void add_to_index_table(address, index)*/
/*{{{  comment*/
/******************************************************************************
*
*     add_to_index_table(address, index)
*
*     address = address of treenode, index = debug record index for that node
*     makes a new entry in the index hash table by performing a hashing
*     function on address.
*     Andy Whitlow 19.10.88.
*
******************************************************************************/
/*}}}*/

PRIVATE void add_to_index_table ( treenode *address , int index )
{
  index_hash_t *hptr;
  BIT32 hash_value = hash_function(address);

#ifdef DEBUG
  BIT32 bit_address = (BIT32)address;
  if ( (bit_address & 3) == 0 )
    DEBUG_MSG(("add_to_index_table: treenode: %lX, tag: %s, hash: %ld\n",
               (BIT32)address, itagstring(TagOf(address)), hash_value));
  else
    DEBUG_MSG(("add_to_index_table: treenode: %lX, tag: %s, hash: %ld (offset)\n",
               (BIT32)address, itagstring(TagOf((treenode *)(bit_address & ~3))), hash_value));
#endif
  /* If that address is already there, ignore it */
  for (hptr = index_hash_table[hash_value]; hptr != NULL; hptr = hptr->next)
    if (hptr->address == address)
      return;

  hptr = (index_hash_t *) newvec(sizeof(*hptr));
  hptr->next = index_hash_table[hash_value];
  index_hash_table[hash_value] = hptr;
  hptr->address = address;
  hptr->index = index;
}
/*}}}*/
/*}}}*/
/*{{{  source code output*/
/*{{{  linemark list private routine*/
/*{{{  PRIVATE void add_to_linemark_list(index, lineno)*/
/*{{{  comment*/
/******************************************************************************
*
*     add_to_linemark_or_libpatch_list(index, lineno, type)
*
*     index = linemark debug record index,
*     lineno = corresponding line number
*     type = either LINEMARK or LIBPATCH
*     Adds a new linemark cell to the linemark list.
*     Andy Whitlow 16.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE void add_to_linemark_list ( int index , int lineno )
{
  linemark_cell_t *lptr;

  lptr = (linemark_cell_t *) newvec(sizeof(*lptr));
  lptr->index = index;
  lptr->linenumber = lineno;
  lptr->next = linemark_list;
  linemark_list = lptr;
}
/*}}}*/
/*}}}*/
/*{{{  file list private routine*/
/*{{{  PRIVATE void add_to_file_list(this_index, this_tag, this_fileid)*/
/*{{{  comment*/
/******************************************************************************
*
*     add_to_file_list(this_index, this_tag, this_fileid)
*
*     this_index = index of endfold or startfold mark.
*     this_tag = either FILE_START or FILE_END.
*     this_fileid = file number of corresponding file.
*     Adds a new file cell to the file list.
*     Andy Whitlow 15.12.88.
*
******************************************************************************/
/*}}}*/
PRIVATE void add_to_file_list ( int this_index , int this_fileid , int this_tag )
{
  file_cell_t *new, *fptr = file_list;

  /*{{{  create new cell*/
  new = (file_cell_t *) newvec(sizeof(*new));
  new->index = this_index;
  new->fileid = this_fileid;
  new->tag = this_tag;
  new->next = NULL;
  /*}}}*/
  if (fptr == NULL)
    file_list = new;
  else
  {
    /*{{{  append cell to list*/
    while (fptr->next != NULL)
      fptr = fptr->next;
    fptr->next = new;
    /*}}}*/
  }
}
/*}}}*/
/*}}}*/
/*}}}*/

/*{{{  support*/
/*{{{  PRIVATE INT32 vartypeof(t)*/
/*****************************************************************************
 *
 *  vartypeof takes a base type t and returns a string representing it.
 *
 *****************************************************************************/
PRIVATE INT32 vartypeof ( int t )
{
  switch(t)
    {
      case S_CHAN:   return(RTL_CHANNEL);
      case S_TIMER:  return(RTL_TIMER);
      case S_BOOL:   return(RTL_BOOLEAN);
      case S_BYTE:   return(RTL_BYTE);
      case S_INT:    return(RTL_INTEGER);
      case S_INT16:  return(RTL_INT16);
      case S_INT32:  return(RTL_INT32);
      case S_INT64:  return(RTL_INT64);
      case S_REAL32: return(RTL_REAL32);
      case S_REAL64: return(RTL_REAL64);
      case S_PORT:   return(RTL_PORT);
    }
  return (ZERO32);  /* never reached */
}
/*}}}*/
/*{{{  PRIVATE char *vartypestr(t)*/
/*****************************************************************************
 *
 *  vartypestr takes a type t and returns a string representing it.
 *
 *****************************************************************************/
PRIVATE char *vartypestr ( INT32 t )
{
  switch(t)
    {
      case RTL_CHANNEL:  return("channel");
      case RTL_TIMER:    return("timer");
      case RTL_BOOLEAN:  return("Boolean");
      case RTL_BYTE:     return("byte");
      case RTL_INTEGER:  return("integer");
      case RTL_INT16:    return("int16");
      case RTL_INT32:    return("int32");
      case RTL_INT64:    return("int64");
      case RTL_REAL32:   return("real32");
      case RTL_REAL64:   return("real64");
      case RTL_PORT:     return("port");
      case RTL_PROTOCOL: return("protocol");
      case RTL_CASETAG:  return("tag");
      case RTL_CHANPOINTER: return("channelptr");
      default : return ("???");
    }
}
/*}}}*/
/*{{{  PRIVATE char *segtypestr(t)*/
/*****************************************************************************
 *
 *  segtypestr takes a segment type t and returns a string representing it.
 *
 *****************************************************************************/
PRIVATE char *segtypestr ( INT32 t )
{
  switch(t)
    {
      case RTL_SEGTYPE_PROC:   return("proc");
      case RTL_SEGTYPE_REPL:   return("repl");
      case RTL_SEGTYPE_PAR:    return("par");
      case RTL_SEGTYPE_PRIPAR: return("pripar");
      case RTL_SEGTYPE_SEQ:    return("seq");
      default:                 return("???");
    }
}
/*}}}*/
/*{{{  PRIVATE char *accessstr(t)*/
/*****************************************************************************
 *
 *  accessstr takes an access mode t and returns a string representing it.
 *
 *****************************************************************************/
PRIVATE char *accessstr ( INT32 t )
{
  switch(t)
    {
      case RTL_CONST:     return("const");
      case RTL_PLACED:    return("placed");
      case RTL_BYREF:     return("byref");
      case RTL_BYVAL:     return("byval");
      case RTL_OPTIMISED: return("opt");
      case RTL_VOIDACCESS: return("voidaccess");
      default:            return("???");
    }
}
/*}}}*/
/*{{{  PRIVATE char *proctypestr(t)*/
/*****************************************************************************
 *
 *  proctypestr takes a proc type t and returns a string representing it.
 *
 *****************************************************************************/
PRIVATE char *proctypestr ( INT32 t )
{
  switch(t)
    {
      case RTL_PROCFLAG:        return("procflag");
      case RTL_FUNCTIONFLAG:    return("functionflag");
      case RTL_SCPROCFLAG:      return("scprocflag");
      case RTL_SCFUNCTIONFLAG:  return("scfunctionflag");
      case RTL_LIBPROCFLAG:     return("libprocflag");
      case RTL_LIBFUNCTIONFLAG: return("libfunctionflag");
      default:                  return("???");
    }
}
/*}}}*/
/*{{{  PRIVATE int isaparentof(f1, f2)*/
/*****************************************************************************
 *
 *  isaparentof returns TRUE if f1 is a parent file of f2.
 *
 *****************************************************************************/
PRIVATE int isaparentof ( int f1 , int f2 )
{
  while (f2 != (-1))
    {
      if (f2 == f1) return TRUE;
      f2 = parentoffile(f2);
    }
  return(FALSE);
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void gendebugstartfold(n)*/
/*****************************************************************************
 *
 *  gendebugstartfold generates a startfold record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugstartfold ( int n )
{
  if (minimal_debugoutput)
    return;
  if (debug_diagnostics)
    {
      fprintf(outfile, "%4d Startfold\n", debuginfo_index);
      debuginfo_index++;
      fprintf(outfile, "%4d Filename %s\n", debuginfo_index, lookupfilename(n));
    }
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
      *b++ = RTL_STARTFILE;
      b = addbuf_3L_num(b, debuginfo_index);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    
      debuginfo_index++;
      b = buffer;
      *b++ = RTL_FILENAME;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_str(b, lookupfilename(n));
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  else if (source_output)
    add_to_file_list(debuginfo_index, n, FILE_START);
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugendfold()*/
/*****************************************************************************
 *
 *  gendebugendfold generates an endfold record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugendfold ( void )
{
  if (minimal_debugoutput)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Endfold\n", debuginfo_index);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_ENDFILE;
      b = addbuf_3L_num(b, debuginfo_index);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  else if (source_output)
    add_to_file_list(debuginfo_index, 0, FILE_END);
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebuglinemark(line)*/
/*****************************************************************************
 *
 *  gendebuglinemark generates a linemark record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebuglinemark ( int line )
{
  if (minimal_debugoutput)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Line mark : %d\n", debuginfo_index, line);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_LINEMARK;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, line);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  else if (source_output)
    add_to_linemark_list(debuginfo_index, line);
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void genstartfolds*/
/*****************************************************************************
 *
 *  genstartfolds generates a list of startfolds
 *
 *****************************************************************************/
PRIVATE void genstartfolds ( int startfile, int endfile )
{
  DEBUG_MSG(("genstartfolds: startfile %d, endfile %d\n", startfile, endfile));
  /*{{{  move endfile out until it is start file*/
  {
    int filestack[FILESMAX], linestack[FILESMAX];
    int files = 0;
    int i;
    while (endfile != startfile)
      {
        filestack[files] = endfile;
        linestack[files] = parentposnoffile(endfile);
        endfile = parentoffile(endfile);
        files++;
      }
    for (i = files - 1; i >= 0; i--)
      {
        gendebuglinemark (linestack[i]);
        gendebugstartfold(filestack[i]);
      }
  }
  /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void gendebugfoldmarks(startfile, endfile)*/
/*****************************************************************************
 *
 *  gendebugfoldmarks generates the appropriate sequence of start folds and
 *                  end folds to move from 'startfile' to 'endfile'.
 *
 *****************************************************************************/
PRIVATE void gendebugfoldmarks ( int startfile , int endfile )
{
  DEBUG_MSG(("gendebugfoldmarks: startfile %d, endfile %d, nextfile %d\n",
             startfile, endfile, nextfile));
#if 0  /* This is the version that misses empty files */
  /*{{{  move start file out until it is a parent of endfile*/
  while (!isaparentof(startfile, endfile))
    {
      gendebugendfold();
      startfile = parentoffile(startfile);
    }
  /*}}}*/
  genstartfolds(startfile, endfile);
#else
  while (startfile != endfile)
    if ((endfile >= nextfile) && (nextfile < numberoffiles()) &&
        (isaparentof(startfile, nextfile)))
      {
        if (filemodeoffile(nextfile) == LEX_SOURCE)
          {
            genstartfolds(startfile, nextfile);
            startfile = nextfile;
          }
        else
          {
            DEBUG_MSG(("gendebugfoldmarks: ignoring non-source file %d\n", nextfile));
          }  
        nextfile++;
      }
    else if (isaparentof(startfile, endfile))
      {
        genstartfolds(startfile, endfile);
        startfile = endfile;
      }
    else
      {
        DEBUG_MSG(("gendebugfoldmarks: endfold for %d\n", startfile));
        gendebugendfold();
        startfile = parentoffile(startfile);
      }
#endif
  filenum = endfile;
}
/*}}}*/
/*{{{  PRIVATE void gendebuglocn(locn)*/
/*****************************************************************************
 *
 *  gendebuglocn generates a linemark record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebuglocn ( SOURCEPOSN locn, treenode *address )
{
  int locnfile = FileNumOf(locn),
      locnline = FileLineOf(locn);

  /*add_to_index_table(address, minimal_debugoutput ? debuginfo_index - 1
                                                  : debuginfo_index);*/
  if (locnfile != filenum)
    gendebugfoldmarks(filenum, locnfile);

  /* this moved to below the foldmark stuff - bug 1140 - 4/2/91 */
  add_to_index_table(address, minimal_debugoutput ? debuginfo_index - 1
                                                  : debuginfo_index);
  gendebuglinemark(locnline);
}
/*}}}*/
/*{{{  PRIVATE void gendebugscope()*/
/*****************************************************************************
 *
 *  gendebugscope generates a scope record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugscope ( void )
{
  if (source_output || minimal_debugoutput || !symbolic_debugoutput)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Scope\n", debuginfo_index);
  else if (debug_to_file)
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;

      *b++ = RTL_BEGSCOPE;
      b = addbuf_3L_num(b, debuginfo_index);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugendscope()*/
/*****************************************************************************
 *
 *  gendebugendscope generates an endscope record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugendscope ( void )
{
  if (source_output || minimal_debugoutput || !symbolic_debugoutput)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Endscope\n", debuginfo_index);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_ENDSCOPE;
      b = addbuf_3L_num(b, debuginfo_index);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugvar(nptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  gendebugvar generates a variable record to the debug file
 *              Does not cope with constant tables properly
 *              Altered to handle placed variables 9.9.88
 *
 * variable     index vartype access dims { dims access offset } offset name
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void gendebugvar ( treenode *nptr )
{
  treenode *tptr = NDeclOf(nptr);
  treenode *type = NTypeOf(nptr);
  treenode *t;
  int dims = 0;
  INT32 vartype;
  INT32 access;
  BIT32 output_offset;

  if (source_output || minimal_debugoutput || !symbolic_debugoutput)
    return;
  /*{{{  set up vartype, access and dims*/
  if ((TagOf(nptr) == N_SPROTDEF) || (TagOf(nptr) == N_TPROTDEF))
    {
      vartype = RTL_PROTOCOL; access = RTL_CONST; /* dims = 0; */
    }
  else if (TagOf(nptr) == N_TAGDEF)
    {
      vartype = RTL_CASETAG; access = RTL_CONST; /* dims = 0; */
    }
  else
    {
      for(t = type; TagOf(t) == S_ARRAY; t = ARTypeOf(t))
        dims++;
      vartype = vartypeof(TagOf(t));
  #if defined(TCOFF) /*|| !defined(CONFIG)*/
      if (chanaspointer && (vartype == RTL_CHANNEL) && !issimplechan(nptr))
        vartype = RTL_CHANPOINTER;
  #endif
      /*{{{  access*/
      access = ((tptr != NULL) && (TagOf(tptr) == S_VALABBR || TagOf(tptr) == S_VALRETYPE) &&
                                              isconst(DValOf(tptr))) ? RTL_CONST  :
                                                      isplaced(nptr) ? RTL_PLACED :
                                                     ispointer(nptr) ? RTL_BYREF  :
                                                                       RTL_BYVAL  ;
      /*}}}*/
    }
  /*}}}*/
  /*{{{  calculate offset*/
  if (access == RTL_PLACED)
    {
      int error;
      BIT32 offhi;
      BIT32 mostneg;
  
      I32ToI64(&offhi, &output_offset, NVOffsetOf(nptr));
      mostneg = (targetintsize == S_INT32) ? MOSTNEG_INT32 : MOSTNEG_INT16;
      Int64Sub(&error, &offhi, &output_offset, offhi, output_offset,
               0xffffffffl, mostneg);
      Int64Div(&error, &offhi, &output_offset, offhi, output_offset,
               ZERO32, bytesperword);
    }
  else if ((vartype == RTL_TIMER) || (vartype == RTL_CASETAG) ||
           (vartype == RTL_PROTOCOL))
    {
      access = RTL_VOIDACCESS;
      output_offset = (vartype == RTL_CASETAG) ? NTValueOf(nptr) : 0;
    }
  else if ((NVUseCountOf(nptr) == 0) &&
           (TagOf(nptr) != N_PARAM) && (TagOf(nptr) != N_VALPARAM))
    {
      /*output_offset = MOSTNEG_INT32;*/ /* Old method */
      access = RTL_OPTIMISED;
      output_offset = 0;
    }
  else if (access == RTL_CONST)
    output_offset = 0;  /* Will be filled in later by an addressfix */
  else
    output_offset = NVOffsetOf(nptr) + workspace_adjust;
  /*}}}*/
  if (debug_to_file || debug_diagnostics)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;

      if (debug_diagnostics)
        {
          fprintf(outfile, "%4d Variable : %s %s ", debuginfo_index,
              vartypestr(vartype), accessstr(access));
          if (dims > 0)
            fprintf(outfile, "dims:%d ", dims);
        }
      else
        {
          *b++ = RTL_VARIABLE;
          b = addbuf_3L_num(b, debuginfo_index);
          b = addbuf_3L_num(b, vartype);
          b = addbuf_3L_num(b, access);
          b = addbuf_3L_num(b, dims);
        }
      /*{{{  write the dimensions*/
      {
        int i;
        for (i = 0; i < dims; i++)
          {
            treenode *dimexp = ARDimLengthOf(type);
            INT32 dimaccess, dimoffset;
            if (isconst(dimexp))
              { dimaccess = RTL_CONST; dimoffset = ARDimOf(type); }
            else if (NLexLevelOf(nptr) == NLexLevelOf(dimexp))
              { dimaccess = RTL_BYVAL; dimoffset = NVOffsetOf(dimexp) + workspace_adjust; }
            else  /* No current way of describing this */
              { dimaccess = RTL_VOIDACCESS; dimoffset = 0;}
            if (debug_diagnostics)
              fprintf(outfile, "%d:%s:%ld ", i, accessstr(dimaccess), dimoffset);
            else
              {
                b = addbuf_3L_num(b, dimaccess);
                b = addbuf_3L_num(b, dimoffset);
              }
            type = ARTypeOf(type);
          }
      }
      /*}}}*/
      if (debug_diagnostics)
        {
          fprintf(outfile, "%ld %s\n", output_offset, WNameOf(NNameOf(nptr)));
        }
      else
        {
          b = addbuf_3L_num(b, output_offset);
          b = addbuf_3L_str(b, WNameOf(NNameOf(nptr)));
          tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
        }
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugconstant(tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  gendebugconstant generates a constant record to the debug file
 *
 * constant     index vartype hi lo name
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void gendebugconstant ( treenode *tptr )
{
  treenode *nptr = DNameOf(tptr);
  INT32 type = basetype(NTypeOf(nptr));
  INT32 vartype = vartypeof(type);
  treenode *cptr = DValOf(tptr);
  BIT32 hival = HiValOf(cptr);
  if (source_output || minimal_debugoutput || !symbolic_debugoutput)
    return;
  if (bytesinscalar(type) <= 4)  /* set high word to zero if not required */
    hival = 0;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Constant : %s %ld %ld %s\n", debuginfo_index,
            vartypestr(vartype), (long)hival, (long)LoValOf(cptr),
            WNameOf(NNameOf(nptr)));
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_CONSTANT;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, vartype);
      b = addbuf_3L_num(b, hival);
      b = addbuf_3L_num(b, LoValOf(cptr));
      b = addbuf_3L_str(b, WNameOf(NNameOf(nptr)));
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugworkspace(segtype, thisoffset, prevoffset)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  gendebugworkspace generates a workspace record to the debug file
 *
 * workspace    index segtype thisoffset prevoffset
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void gendebugworkspace ( INT32 segtype , INT32 thisoffset , INT32 prevoffset )
{
  if (source_output)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Workspace : %s %ld %ld\n", debuginfo_index,
            segtypestr(segtype), (long)thisoffset, (long)prevoffset);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_WORKSPACE;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, segtype);
      b = addbuf_3L_num(b, thisoffset);
      b = addbuf_3L_num(b, prevoffset);
      kp = add_to_debug_buffer(buffer, b - buffer, BUF_KEEP);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugendseg()*/
/*****************************************************************************
 *
 *  gendebugendseg generates an endseg record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugendseg ( void )
{
  if (source_output)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Endseg\n", debuginfo_index);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
      *b++ = RTL_ENDSEG;
      b = addbuf_3L_num(b, debuginfo_index);
      kp = add_to_debug_buffer(buffer, b - buffer, BUF_KEEP);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugversion()*/
/*****************************************************************************
 *
 *  gendebugversion generates a version record to the debug file
 *
 *****************************************************************************/
PRIVATE void gendebugversion ( void )
{
  int debug_version = DEBUG_VERSION;
  if (minimal_debugoutput) debug_version |= RTL_COMPACTED_BIT;
  if (iobycall)            debug_version |= RTL_VIRTUAL_LINK_BIT;
  if (source_output)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Version : %d\n", debuginfo_index, debug_version);
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_VERSION;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, debug_version);
      write_debug_string(buffer, b - buffer);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE void gendebugproc(nptr)*/
/*****************************************************************************
 *
 *  gendebugproc generates a proedure record to the debug file
 *
 * procedure    index proctype dummy workspace vectorspace name
 *
 *****************************************************************************/
PRIVATE void gendebugproc ( treenode *nptr )
{
  char buffer[MAX_DEBUG_STRING_SIZE];
  char *b = buffer;
  int proctype;
  INT32 wsusage, vsusage;
  char *name = WNameOf(NNameOf(nptr));
  if (source_output || minimal_debugoutput || !symbolic_debugoutput)
    return;
  if (separatelycompiled(nptr) && !compiledforcorrectproc(nptr))
    return;
  /*{{{  set up proctype*/
  switch(TagOf(nptr))
    {
      case N_PROCDEF:    proctype = RTL_PROCFLAG;        break;
      case N_SFUNCDEF:
      case N_LFUNCDEF:   proctype = RTL_FUNCTIONFLAG;    break;
      case N_SCPROCDEF:  proctype = RTL_SCPROCFLAG;      break;
      case N_SCFUNCDEF:  proctype = RTL_SCFUNCTIONFLAG;  break;
      case N_LIBPROCDEF: proctype = RTL_LIBPROCFLAG;     break;
      case N_LIBFUNCDEF: proctype = RTL_LIBFUNCTIONFLAG; break;
      default:           proctype = 0;                   break;
    }
  /*}}}*/
  getprocwsandvs(nptr, &wsusage, &vsusage);
  if (debug_diagnostics)
    /*{{{  write text to output file*/
    fprintf(outfile, "%4d Procedure : %s dummy %ld %ld %s\n",
            debuginfo_index, proctypestr(proctype), wsusage, vsusage, name);
    /*}}}*/
  else if (debug_to_file)
    /*{{{  write record to object file*/
    {
      *b++ = RTL_PROCEDURE;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, proctype);
      b = addbuf_3L_num(b, 0);  /* address - filled in later by addressfix */
      b = addbuf_3L_num(b, wsusage);
      b = addbuf_3L_num(b, vsusage);
      b = addbuf_3L_str(b, name);
      
      /* write_debug_string(buffer, b - buffer); */
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PRIVATE treenode *gendebugspecs(tptr)*/
PRIVATE treenode *gendebugspecs(treenode *tptr)
{
  while (isspecification(tptr))
    {
      /*{{{  set line mark if required*/
      if (!separatelycompiled(DNameOf(tptr)))
        gendebuglocn(LocnOf(tptr), tptr);
      /*}}}*/
      switch(TagOf(tptr))
        {
          /*{{{  decl*/
          case S_DECL:
            {
              treenode *n = DNameOf(tptr);
              if (TagOf(n) == S_LIST)
                while (!EndOfList(n))
                  {
                    /* add_to_index_table( ThisItem(n), debuginfo_index); */
                    gendebugvar(ThisItem(n));
                    n = NextItem(n);
                  }
              else
                {
                  /* add_to_index_table( n, debuginfo_index); */
                  gendebugvar(n);
                }
            }
            break;
          /*}}}*/
          /*{{{  abbrev retype*/
          case S_VALABBR:
          case S_ABBR:
          case S_VALRETYPE: case S_RETYPE:
            if ( /* ((TagOf(tptr) == S_VALABBR) || (TagOf(tptr) == S_VALRETYPE)) && */
                (TagOf(DValOf(tptr)) == S_CONSTEXP))
              {
                /* add_to_index_table( DNameOf(tptr), debuginfo_index);*/
                gendebugconstant(tptr);
              }
            else
              {
                gendebug(DValOf(tptr));
                add_to_index_table( DNameOf(tptr), debuginfo_index);
                gendebugvar(DNameOf(tptr));
              }
            break;
          /*}}}*/
          /*{{{  procedure function*/
          case S_PROCDEF:
          case S_SFUNCDEF: case S_LFUNCDEF:
            {
              INT32 savedoffset = enclosing_this_offset;
              INT32 savedworkspace = workspace_adjust;
              treenode *nptr = DNameOf(tptr);
              if (!inline(nptr))
                {
                  add_to_index_table( nptr, debuginfo_index);
                  gendebugproc(nptr);
                  if (!separatelycompiled(nptr))
                #if 0
                /* This was removed by CO'N to change the debug info */
                /* It is now done by a PROC w/s record with value 0 */
                /* followed by a SEQuential w/s record with the correct ws size */
                /* This makes it easier to set a breakpoint at procedure entry */
                    /*{{{  do procedure parameters*/
                    {
                      gendebugworkspace(RTL_SEGTYPE_PROC, NPMaxwspOf(nptr), ZERO32);
                      enclosing_this_offset = NPMaxwspOf(nptr);
                      workspace_adjust = ZERO32;
                      /*{{{  generate information for the parameters*/
                      {
                        treenode *n = NTypeOf(nptr);
                        if (TagOf(nptr) != N_PROCDEF)
                          n = FnParamsOf(n);
                        while (!EndOfList(n))
                          {
                            treenode *thisparam = ThisItem(n);
                            if (!ishiddenparam(thisparam))
                              {
                                /* add_to_index_table( thisparam,
                                   debuginfo_index); */
                                gendebugvar(thisparam);
                              }
                            n = NextItem(n);
                          }
                      }
                      /*}}}*/
                      gendebug(DValOf(tptr));/* Do the routine body */
                      gendebugendseg();
                      enclosing_this_offset = savedoffset;
                      workspace_adjust = savedworkspace;
                    }
                    /*}}}*/
                  #else
                    /*{{{  do parameters and body*/
                    {
                      gendebugworkspace(RTL_SEGTYPE_PROC, ZERO32, ZERO32);
                      enclosing_this_offset = NPMaxwspOf(nptr);
                      workspace_adjust = -enclosing_this_offset;
                      /*{{{  generate information for the parameters*/
                      {
                        treenode *n = NTypeOf(nptr);
                        if (TagOf(nptr) != N_PROCDEF)
                          n = FnParamsOf(n);
                        while (!EndOfList(n))
                          {
                            treenode *thisparam = ThisItem(n);
                            if (!ishiddenparam(thisparam))
                              {
                                /* add_to_index_table( thisparam,
                                   debuginfo_index); */
                                gendebugvar(thisparam);
                              }
                            n = NextItem(n);
                          }
                      }
                      /*}}}*/
                      /* Note the entry point of the proc/function */
                      /* We need another hashed value to be able to retrieve this debuginfo_index */
                      /* Because nptr has already been used, we don't have any more treenodes around */
                      /* So we add 1 (UGH) to its value to create a different `address' */
                      /* This would break if treenodes were only 1 byte long */
                      gendebuglocn(LocnOf(tptr), (treenode *)(((char *)nptr)+1));
                      if (enclosing_this_offset != ZERO32)
                        {
                          gendebugworkspace(RTL_SEGTYPE_SEQ, enclosing_this_offset, ZERO32);
                          workspace_adjust = ZERO32;
                          /* note the entry point after the ajw */
                          /* See above comment about nptr+1 */
                          gendebuglocn(LocnOf(tptr), (treenode *)(((char *)nptr)+2));
                        }
                      gendebug(DValOf(tptr));/* Do the routine body */
                      if (enclosing_this_offset != ZERO32) gendebugendseg();
                      gendebugendseg();
                      enclosing_this_offset = savedoffset;
                      workspace_adjust = savedworkspace;
                    }
                    /*}}}*/
                  #endif
                }
            }
            break;
          /*}}}*/
          /*{{{  protocol definition*/
          case S_SPROTDEF:
            add_to_index_table( DNameOf(tptr), debuginfo_index);
            gendebugvar(DNameOf(tptr));
            break;
          case S_TPROTDEF:
            {
              treenode *n = DNameOf(tptr);
              gendebugvar(n);
              n = NTypeOf(n);
              while (!EndOfList(n))
                {
                  treenode *thistag = ThisItem(n);
                  /* add_to_index_table( thistag, debuginfo_index); */
                  gendebugvar(thistag);
                  n = NextItem(n);
                }
            }
            break;
          /*}}}*/
        }
      tptr = DBodyOf(tptr);
    }
  return tptr;
}
/*}}}*/
/*{{{  PRIVATE void gendebugalt (tptr, enabling, replcount)*/
/*****************************************************************************
 *
 *  gendebugalt takes a parse tree, for an ALT 'tptr' and generates the symbolic
 *           debugging information for it.
 *
 *****************************************************************************/
PRIVATE void gendebugalt ( treenode *tptr, int enabling, int replcount )
{
  /*{{{  create linemark if required*/
  /* specification nodes sometimes don't need linemarks so the specification
     case of the switch worries about this */
  if (!isspecification(tptr))
    gendebuglocn(LocnOf(tptr), tptr);
  /*}}}*/
  switch(TagOf(tptr))
    /*{{{  cases*/
    {
      default:  /* Specification*/
        DEBUG_MSG(("gendebugalt: specification\n"));
        gendebugscope();
        tptr = gendebugspecs(tptr);
        gendebugalt(tptr, enabling, replcount);
        gendebugendscope();
        return;
      /*{{{  ALT PRIALT*/
      case S_ALT: case S_PRIALT:
        DEBUG_MSG(("gendebugalt: ALT\n"));
        tptr = CBodyOf(tptr);
        while (!EndOfList(tptr))
          {
            gendebugalt(ThisItem(tptr), enabling, replcount);
            tptr = NextItem(tptr);
          }
        return;
      /*}}}*/
      /*{{{  REPLALT PRIREPLALT*/
      case S_REPLALT: case S_PRIREPLALT:
        DEBUG_MSG(("gendebugalt: repl ALT\n"));
        {
          gendebug(ReplCStartExpOf(tptr));
          gendebug(ReplCLengthExpOf(tptr));
          /*{{{  do the replicated process*/
          {
            treenode *replicator = ReplCNameOf(tptr);
            INT32 reploffset = NVOffsetOf(replicator);
            int base = NVOffsetOf(ReplCTempOf(tptr));
            SetNVOffset(replicator, enabling ? reploffset : (base + replcount));
            gendebugscope();
            gendebugvar(replicator);
            gendebugalt(ReplCBodyOf(tptr), enabling, replcount+1);
            gendebugendscope();
            SetNVOffset(replicator, reploffset);
          }
          /*}}}*/
        }
        return;
      /*}}}*/
      /*{{{  ALTERNATIVE*/
      case S_ALTERNATIVE:
        {
          treenode *input = AltInputOf(tptr);
          int endscope = FALSE;
          DEBUG_MSG(("gendebugalt: ALTERNATIVE\n"));
          if (isspecification(input))
            {
              gendebugscope();
              endscope = TRUE;
              input = gendebugspecs(input);
            }
          if (enabling)
            {
              gendebug(AltGuardOf(tptr));
            #if 0 /* bug 779 2/11/90 */
              switch(inputtypeof(input))
                {
                  default: /* Any channel or port input */
                    gendebug(LHSOf(input)); /* channel */
                    break;
                  case INP_SKIP: case INP_DELAYED_INPUT:
                    break;
                }
            #else
              gendebug(AltChanExpOf(tptr)); /* bug 779 2/11/90 */
            #endif
            }
          else
            {
              gendebug(input);
              gendebug(AltBodyOf(tptr));
            }
          if (endscope)
            gendebugendscope();
        }
        return;
      /*}}}*/
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void gendebug (tptr)*/
/*****************************************************************************
 *
 *  gendebug takes a parse tree, 'tptr' and generates the symbolic
 *           debugging information for it.
 *           The file table and the code offsets are output separately.
 *
 *****************************************************************************/
PRIVATE void gendebug ( treenode *tptr )
{
  while (tptr != NULL)
  {
    /*{{{  create linemark if required*/
    /* specification nodes sometimes don't need linemarks so the specification
       case of the switch worries about this */
    if (need_locate(TagOf(tptr)) && !isspecification(tptr))
      gendebuglocn(LocnOf(tptr), tptr);
    /*}}}*/
    switch(TagOf(tptr))
      /*{{{  cases*/
      {
        default:
          return;
        /*{{{  STOP SKIP GUYCODE GUYSTEP*/
        case S_STOP:
          /*{{{  comment*/
          /* STOP creates an explicit linemark because it requires locate but is not
             included in need_locate(...), This is because the code to produce
             codemarks, tstop() in gen1, produces its own explicit codemarks
             independent of need_locate to prevent duplicate codemarks being produced
             for explicit STOP statements */
          /*}}}*/
          gendebuglocn(LocnOf(tptr), tptr);
          return;
        case S_SKIP:
        case S_CONSTEXP:
        case S_GUYCODE:
        case S_GUYSTEP:
        /* Case S_LABELDEF just returns */
          return;
        /*}}}*/
        /*{{{  SEQ IF GUY ASM ALT PRIALT*/
        case S_SEQ: case S_IF: case S_GUY: case S_ASM:
          tptr = CBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  REPLSEQ REPLIF*/
        case S_REPLSEQ:
        case S_REPLIF:
          /* add_to_index_table( ReplCNameOf(tptr), debuginfo_index); */
          gendebugscope();
          gendebugvar(ReplCNameOf(tptr));
          gendebug(ReplCStartExpOf(tptr));
          gendebug(ReplCLengthExpOf(tptr));
          gendebug(ReplCBodyOf(tptr));
          gendebugendscope();
          return;
        /*}}}*/
        /*{{{  ALTs of any kind*/
        case S_ALT: case S_PRIALT: case S_REPLALT: case S_PRIREPLALT:
          gendebugalt(tptr, TRUE, 0);  /* enabling guards and channels */
          gendebugalt(tptr, FALSE, 0); /* inputs and bodies */
          return;
        /*}}}*/
        /*{{{  PAR PRIPAR*/
        case S_PAR:
        case S_PRIPAR:
          {
            INT32 parsize = ((TagOf(tptr) == S_PRIPAR) && (code_style_flags & CODE_STYLE_ALT_PRI_PAR)) ?
                                DS_IO : ZERO32;
            INT32 segment_type = (TagOf(tptr) == S_PRIPAR) ? RTL_SEGTYPE_PRIPAR : RTL_SEGTYPE_PAR;
            tptr = CBodyOf(tptr);
            while (!EndOfList(tptr))
              /*{{{  do a branch of the PAR*/
              {
                treenode *sp = ThisItem(tptr);
                INT32 thisbranchwsp  = SpMaxwspOf(sp);
                INT32 thisbranchsize = SpDatasizeOf(sp);
                INT32 savedoffset    = enclosing_this_offset;
                INT32 savedworkspace = workspace_adjust;
                enclosing_this_offset = parsize + thisbranchwsp + enclosing_this_offset;
                workspace_adjust = parsize + thisbranchwsp + workspace_adjust;
                gendebugworkspace(segment_type, enclosing_this_offset, ZERO32);
                /*{{{  make sure only first branch of PRI PAR is marked as PRI*/
                segment_type = RTL_SEGTYPE_PAR;
                /*}}}*/
                gendebug(SpBodyOf(sp)); /* do the PAR branch */
                gendebugendseg();
                enclosing_this_offset = savedoffset;
                workspace_adjust = savedworkspace;
                parsize += thisbranchsize;
                tptr = NextItem(tptr);
              }
              /*}}}*/
          }
          return;
        /*}}}*/
        /*{{{  REPLPAR*/
        case S_REPLPAR:
          {
            treenode *sp = ReplCBodyOf(tptr);
            treenode *replicator = ReplCNameOf(tptr);
            INT32 thisbranchwsp = SpMaxwspOf(sp);
            INT32 savedoffset = enclosing_this_offset;
            INT32 prev_offset = enclosing_this_offset;
            INT32 savedworkspace = workspace_adjust;
            int replparslots = (SpVSUsageOf(sp) == ZERO32) ? MIN_REPLPAR_SPECIALS :
                                                             MIN_REPLPAR_SPECIALS + 1;
            gendebugscope();
            gendebugvar(replicator);
            gendebug(ReplCStartExpOf(tptr));
            gendebug(ReplCLengthExpOf(tptr));
            /* gendebugendscope(); */
            enclosing_this_offset = thisbranchwsp - replparslots + REPLPAR_STATICLINK;
            gendebugworkspace(RTL_SEGTYPE_REPL, enclosing_this_offset, prev_offset);
            workspace_adjust = ZERO32;
            /*{{{  send replicator debug info.*/
            {
              INT32 offset = NVOffsetOf(replicator);
              SetNVOffset(replicator, thisbranchwsp - replparslots + REPLPAR_REPLICATOR);
              /* gendebugscope(); */
              gendebugvar(replicator);
              SetNVOffset(replicator, offset);
            }
            /*}}}*/
            gendebug(SpBodyOf(sp));
            /* gendebugendscope(); */
            gendebugendseg();
            gendebugendscope();
            enclosing_this_offset = savedoffset;
            workspace_adjust = savedworkspace;
          }
          return;
        /*}}}*/
        /*{{{  WHILE CHOICE SELECTION*/
        case S_WHILE:
        case S_CHOICE:
        case S_SELECTION:
          gendebug(CondGuardOf(tptr));
          tptr = CondBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  VARIANT*/
        case S_VARIANT:
          gendebug(VRTaggedListOf(tptr));
          tptr = VRBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  FINSTANCE PINSTANCE*/
        case S_FINSTANCE:
        case S_PINSTANCE:
          tptr = IParamListOf(tptr);
          break;
        /*}}}*/
        /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT CASE_INPUT CASE*/
        case S_ASS:
        case S_OUTPUT:
        case S_INPUT:
        case S_TAGGED_INPUT:
        case S_DELAYED_INPUT:
        case S_CASE_INPUT:
        case S_CASE:
          gendebug(LHSOf(tptr)); /* Added to fix bug 778 29/10/90 */
          tptr = RHSOf(tptr);
          break;
        /*}}}*/
        /*{{{  LIST*/
        case S_LIST:
          gendebug(ThisItem(tptr));
          tptr = NextItem(tptr);
          break;
        /*}}}*/
        /*{{{  PLACED*/
        #if 0 /* never used */
        case S_PLACED:
          tptr = ConfigBodyOf(tptr);
          break;
        #endif
        /*}}}*/
        /*{{{  monadic operator node*/
        case S_MOSTPOS:
        case S_MOSTNEG:
        case S_NEG:
        case S_BITNOT:
        case S_UMINUS:
        case S_NOT:
        case S_SIZE:
        case S_VAL:
        /*case S_VAR:*/
        case S_CONSTRUCTOR:
        case S_EXACT: case S_ROUND: case S_TRUNC:
          tptr = OpOf(tptr);
          break;
        /*}}}*/
        /*{{{  dyadic operator node*/
        case S_AND: case S_OR:
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_LSHIFT: case S_RSHIFT:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_EQ: case S_NE:
        case S_LS: case S_LE:
        case S_GR: case S_GE:
        case S_AFTER:
        case S_COLON2:
        case S_CSUB0: case S_CCNT1:
        case S_EVAL: /* added 9/10/90 bug 1007 */
          gendebug(LeftOpOf(tptr));
          tptr = RightOpOf(tptr);
          break;
        /*}}}*/
        /*{{{  ARRAYSUB ARRAYITEM*/
        case S_ARRAYSUB:
        case S_ARRAYITEM:
          gendebug(ASExpOf(tptr));
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        /*{{{  valof node*/
        case S_VALOF:
          gendebug(VLBodyOf(tptr));
          tptr = VLResultListOf(tptr);
          gendebuglocn(LocnOf(tptr), tptr);  /* linemark for RESULT line */
          break;
        /*}}}*/
        /*{{{  specification node*/
        case S_DECL:
        case S_VALABBR: case S_ABBR:
        case S_VALRETYPE:case S_RETYPE:
        case S_PROCDEF:
        case S_SFUNCDEF: case S_LFUNCDEF:
        case S_SPROTDEF: case S_TPROTDEF:
          gendebugscope();
          tptr = gendebugspecs(tptr);
          gendebug(tptr);
          gendebugendscope();
          return;
        /*}}}*/
        /*{{{  PLACE WSPLACE VSPLACE*/
        case S_PLACE:
        case S_WSPLACE: case S_VSPLACE:
          tptr = DBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  segment node*/
        case S_SEGMENTITEM:
        case S_SEGMENT:
          /* debuginfo_index++;*/
          gendebug(SNameOf(tptr));
          gendebug(SStartExpOf(tptr));
          gendebug(SLengthExpOf(tptr));
          gendebug(SCheckExpOf(tptr)); /* added 9/10/90 bug 1007 */
          if (TagOf(tptr) == S_SEGMENT)
            return;
          tptr = SSubscriptExpOf(tptr);
          break;
        /*}}}*/
        /*{{{  temp node*/
        case T_TEMP:
        case T_REGTEMP:
        case T_PREEVALTEMP:
          tptr = NDeclOf(tptr);
          break;
        /*}}}*/
        /*{{{  space usage node*/
        case S_SPACEUSAGE:
          tptr = SpBodyOf(tptr);
          break;
        /*}}}*/
        /* namenodes just return */
      }
      /*}}}*/
  }
  return;
}
/*}}}*/
/*}}}*/
/*{{{  PUBLIC routines*/
/*{{{  PUBLIC void gencodemark (address, offset)*/
/*****************************************************************************
 *
 * gendebugline generates a codemark record to the debug file
 *
 * codemark     index ptr offset
 *
 *****************************************************************************/
PUBLIC void gencodemark ( treenode *address , BIT32 offset )
{
  int line;

  if (source_output)
    return;
  line = get_from_index_table(address);
  if (debug_diagnostics)
    fprintf(outfile, "%4d Codemark : %d %ld\n", debuginfo_index, line, (long)offset);
  else if (debug_to_file)
    /*{{{  write a record to the object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_CODEMARK;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, line);
      b = addbuf_3L_num(b, offset);
      kp = add_to_debug_buffer(buffer, b - buffer, BUF_KEEP);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PUBLIC void genlibpatch (line, offset)*/
/*****************************************************************************
 *
 * genlibpatch  generates a libpatch record to the debug file
 *
 * libpatch     index address
 *
 * N.B. The 'line' parameter of this procedure is not required by the
 *      debugger and so is not included in the libpatch record. It is left
 *      in the parameter list to accomodate future changes
 *      Andy Whitlow - 8.9.88
 *
 *****************************************************************************/
PUBLIC void genlibpatch ( int line , BIT32 offset )
{
  line = line; /* stop unused variable warning */
  if (source_output)
    return;
  if (debug_diagnostics)
    fprintf(outfile, "%4d Libpatch : %ld\n", debuginfo_index, (long)offset);
  else if (debug_to_file)
    /*{{{  write a record to the object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_LIBPATCH;
      b = addbuf_3L_num(b, debuginfo_index);
      /* b = addbuf_3L_num(b, line);  don't need line for current debugger */
      b = addbuf_3L_num(b, offset);
      kp = add_to_debug_buffer(buffer, b - buffer, BUF_KEEP);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/
/*{{{  PUBLIC void genaddressfix (address, offset)*/
/*****************************************************************************
 *
 * gennewaddressfix generates a addressfix record to the debug file
 *
 * newlibpatch     index ptr offset
 *
 *****************************************************************************/
PUBLIC void genaddressfix ( treenode *address , BIT32 offset )
{
  int line;
  if (source_output || minimal_debugoutput || !symbolic_debugoutput) /* bug 1160 - 18/2/91 */
    return;
  line = get_from_index_table(address);
  if (debug_diagnostics)
    fprintf(outfile, "%4d Addressfix : %d %ld\n", debuginfo_index, line, (long)offset);
  else if (debug_to_file)
    /*{{{  write a record to the object file*/
    {
      char buffer[MAX_DEBUG_STRING_SIZE];
      char *b = buffer;
    
      *b++ = RTL_ADDRESS_FIX;
      b = addbuf_3L_num(b, debuginfo_index);
      b = addbuf_3L_num(b, line);
      b = addbuf_3L_num(b, offset);
      tp = add_to_debug_buffer(buffer, b - buffer, BUF_THROW);
    }
    /*}}}*/
  debuginfo_index++;
}
/*}}}*/

/*{{{  index hash table public routine*/
/*{{{  PUBLIC int get_from_index_table(address)*/
/*{{{  comment*/
/******************************************************************************
*
*    get_from_index_table(address)
*
*    address = address of treenode.
*    finds the record corresponding to address in the index hash table and
*    returns its index field.
*    Andy Whitlow 19.10.88
*
******************************************************************************/
/*}}}*/
PUBLIC int get_from_index_table ( treenode *address )
{
  index_hash_t *hptr;
  BIT32 hash_value = hash_function(address);

#ifdef DEBUG
  if ( (((BIT32)address) & 3) == 0 )
    DEBUG_MSG(("get_from_index_table: treenode: %lX, tag: %s, hash: %ld\n",
               (BIT32)address, itagstring(TagOf(address)), hash_value));
  else
    DEBUG_MSG(("get_from_index_table: treenode: %lX, tag: %s, hash: %ld (offset)\n",
               (BIT32)address, itagstring(TagOf((treenode *)(((BIT32)address) & ~3))), hash_value));
#endif
  for (hptr = index_hash_table[hash_value]; hptr != NULL; hptr = hptr->next)
    if (hptr->address == address)
      return(hptr->index);

  geninternal_i(GEN_INDEX_HASH_TABLE_ERR, TagOf(address));
  return (0); /* not reached */
}
/*}}}*/
/*}}}*/
/*{{{  source code output*/
/*{{{  linemark list public routine*/
/*{{{  PUBLIC int get_from_linemark_list(index)*/
/*{{{  comment*/
/******************************************************************************
*
*    get_from_linemark_list(index)
*
*    index = index of codemark record.
*    Finds the linemark record corresponding to the given index and returns
*    the line number.
*    Andy Whitlow 15.11.88
*
******************************************************************************/
/*}}}*/
PUBLIC int get_from_linemark_list ( int index )
{
  linemark_cell_t *lptr;

  for (lptr = linemark_list; lptr != NULL; lptr = lptr->next)
    if (lptr->index == index)
      return(lptr->linenumber);
  geninternal(GEN_LINEMARK_LIST_ERR);
  return (0); /* not reached */
}
/*}}}*/
/*}}}*/
/*{{{  file list public routine*/
/*{{{  PUBLIC int get_file_number(index)*/
/*{{{  comment*/
/******************************************************************************
*
*    get_file_number(this_index)
*
*    this_index = index of codemark record.
*    Finds the filenumber of the file which contains the linemark with index
*    of 'this_index'.
*    Andy Whitlow 15.12.88
*
******************************************************************************/
/*}}}*/
PUBLIC int get_file_number ( int this_index )
{
  file_cell_t *fptr = file_list;
  int curr_fileid, prev_fileid;

  /* printf("\n@@@ searching for filenumber for index %d", this_index); */
  curr_fileid = fptr->fileid;
  /* this_index = fptr->index; */
  fptr = fptr->next;
  while (this_index > fptr->index)
  {
    /*{{{  check next file list cell*/
    if (fptr->tag == FILE_START)
    {
      /*{{{  found start file marker*/
      prev_fileid = curr_fileid;
      curr_fileid = fptr->fileid;
      /*}}}*/
    }
    else
      /*{{{  found end file marker*/
      curr_fileid = prev_fileid;
      /*}}}*/
    fptr = fptr->next;
    /*}}}*/
  }
  return(curr_fileid);
}
/*}}}*/
/*}}}*/
/*}}}*/

/*{{{  PUBLIC int need_locate(tag)*/
/*{{{  comment*/
/******************************************************************************
*
*  need_locate(tag)
*  tag = tag field of tree node being examined.
*  returns true if run time locate information is required for that node.
*  Andy Whitlow 20.10.88
*
******************************************************************************/
/*}}}*/
PUBLIC int need_locate (int tag )
{
  switch (tag)
  /*{{{  cases*/
  {
    /*{{{  don't need locate info*/
    default:
      return(FALSE);
    /*}}}*/
    /*{{{  do need locate info*/
    /* case S_STOP : STOP does need locate info but this is provided for explicitly
                     in tstop(), (gen1), if we were to include STOP here we would
                     obtain two codemarks for an explicit STOP statement */
    case S_SKIP :
    case S_SEQ :
    case S_REPLSEQ :
    case S_IF :
    case S_REPLIF :
    case S_ALT :
    case S_REPLALT :
    case S_PRIALT :
    case S_PRIREPLALT :
    case S_WHILE :
    case S_PINSTANCE :
    case S_FINSTANCE :
    case S_ASS :
    case S_OUTPUT :
    case S_INPUT :
    case S_TAGGED_INPUT :
    case S_DELAYED_INPUT :
    case S_CASE :
    case S_CASE_INPUT :
    case S_VALABBR :
    case S_VALRETYPE :
    case S_TPROTDEF :
    case S_SPROTDEF :
    case S_DECL :
    case S_SFUNCDEF :
    case S_ABBR :
    case S_RETYPE :
    case S_SEGMENT :
    case S_SEGMENTITEM :
    case S_VALOF :
    case S_CHOICE :
    case S_ALTERNATIVE :
    case S_SELECTION :
    case S_VARIANT :
    case S_GUY :
    case S_ASM :
    case S_GUYCODE :
    case S_GUYSTEP :
    /*case S_LABELDEF :  This never actually exists! */
    case S_PAR :
    case S_REPLPAR :
    case S_PRIPAR :
    /* case S_CONSTEXP : */
      return(TRUE);
    /*}}}*/
  }
  /*}}}*/
}
/*}}}*/

/*{{{  PUBLIC void flush_debug_buffers()*/
/*{{{  comment*/
/******************************************************************************
*
*  flushes debug buffers to output after the code generation phase.
*  called by harness.tsr after the call to tmain.
*
******************************************************************************/
/*}}}*/
PUBLIC void flush_debug_buffers ( void )
{
  if (kp != keep)  write_debug_string(keep,  kp - keep );
  if (tp != throw) write_debug_string(throw, tp - throw);
}
/*}}}*/

/*{{{  PUBLIC void debugmain (tptr)*/
/*****************************************************************************
 *
 * debugmain writes out all the symbolic debugging information for tree tptr.
 *
 *****************************************************************************/
PUBLIC void debugmain ( treenode *tptr )
{
  int i;
  debuginfo_index = 0;
  #ifdef CONFIG
  /*{{{  (re)initialise index hash table(s)*/
  /* If configuring, this is done more than once, so we have to keep
     re-initialising the hash table
  */
  for (i=0; i < INDEX_HASH_TABLE_SIZE; i++)
    {
      index_hash_t *hptr = index_hash_table[i];
      while (hptr != NULL)
        {
          index_hash_t *temp = hptr->next;
          freevec(hptr, sizeof(*hptr));
          hptr = temp;
        }
      index_hash_table[i] = NULL;
    }
  while (linemark_list != NULL)
    {
      linemark_cell_t *lptr = linemark_list;
      linemark_list = linemark_list->next;
      freevec(lptr, sizeof(*lptr));
    }
  while (file_list != NULL)
    {
      file_cell_t *fptr = file_list;
      file_list = file_list->next;
      freevec(fptr, sizeof(*fptr));
    }
  /*}}}*/
  #endif

  debug_buffer_size = tcoff_obj_format ? TCOFF_MAX_DEBUG_STRING_SIZE :
                                           LFF_MAX_DEBUG_STRING_SIZE ;
  if (keep == NULL)  keep  = memalloc(debug_buffer_size);
  if (throw == NULL) throw = memalloc(debug_buffer_size);
  kp = keep;     /* initialise keep pointer */
  tp = throw;    /* initialise throw pointer */

  debug_to_file = (!assembly_output && !disassemble);
  enclosing_this_offset = ZERO32;
  workspace_adjust = ZERO32;
  filenum = 0;
  nextfile = 1;

  DEBUG_MSG(("debugmain: start of debug info\n"));
  gendebugversion();
  gendebugstartfold(0);
  gendebug(tptr);
  i = numberoffiles();
  while (nextfile < i)  /* Finish any blank files */
    if (filemodeoffile(nextfile) == LEX_SOURCE)
      {
        DEBUG_MSG(("debugmain: tidying up blank file %d\n", nextfile));
        gendebugfoldmarks(filenum, nextfile);
      }
    else
      {
        DEBUG_MSG(("debugmain: ignoring non-source file %d\n", nextfile));
        nextfile++;
      }
  DEBUG_MSG(("debugmain: tidying up the endfolds\n"));
  gendebugfoldmarks(filenum, 0);        /* Back to the start */
  gendebugendfold();
}
/*}}}*/
#ifdef CONFIG
/*{{{  PUBLIC void debugfree*/
PUBLIC void debugfree(void)
{
  if (keep != NULL)  { memfree(keep);  keep  = NULL; }
  if (throw != NULL) { memfree(throw); throw = NULL; }
}
/*}}}*/
#endif
/*}}}*/
