/*#define DEBUG*/
/******************************************************************************
*
*  code1 - instruction generation, code adjustment, code output
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lex1def.h"
# include "genhdr.h"
# include "instruct.h"
# include "instdef.h"
# include "generror.h"
# include "lexconst.h"
# include "chkdef.h"
# include "desc1def.h"
# include "gen1def.h"
# include "gen2def.h"
# include "codehdr.h"
# include "code1def.h"
# include "bind2def.h"
# include "bind3def.h"
# include "debugdef.h"
# include "srcoutde.h"
#ifdef MSC
# include "memory.h"
#endif
#ifdef CONFIG
# include "confdef.h"
#endif
/*}}}*/

/*{{{  definitions*/
/* The actual code buffer size is CODE_SIZE * sizeof(INT32) */
#ifdef MSC
  #define CODE_SIZE 7500 /* approx 30K bytes */
  #define HUGE 
#else
  #ifdef CONFIG
    #define CODE_SIZE 10000 /* approx 40K bytes */
  #else
    #define CODE_SIZE 60000 /* approx 240K bytes */
  #endif
#endif
PUBLIC  int req_code_size = 0;         /* code size requested (in K) */
PRIVATE int max_code_size = CODE_SIZE; /* buffer size (in INT32s)    */


#define MAX_BYTES_IN_INSTRUCTION 8  /* 32 bit target wordsize maximum */
#define MAX_CONSTANT_SIZE        8  /* 8 bytes for a REAL64 etc */

/*{{{  fields in a setlab record*/
#define SL_SIZE     3                                 /* Length of the record */
#define SL_ADDR     1                /* Last calculated address of the setlab */
#define SL_NEXT     2                 /* Pointer to next item on adjust chain */
/*}}}*/
/*{{{  fields in a uselab record*/
#define U_SIZE     6                                 /* Length of the record */
#define U_ADDR     1           /* Last calculated address of the label usage */
#define U_NEXT     2                 /* Pointer to next item on adjust chain */
#define U_DEF      3                          /* Pointer to label definition */
#define U_ADJ      4  /* Current estimated length of instruction using label */
#define U_OP       5                              /* Instruction using label */
/*}}}*/
/*{{{  fields in a uselabdiff record*/
#define UD_SIZE    7                                 /* Length of the record */
#define UD_ADDR    1       /* Last calculated address of the labeldiff usage */
#define UD_NEXT    2                 /* Pointer to next item on adjust chain */
#define UD_DEF1    3                 /* Pointer to definition of first label */
#define UD_ADJ     4  /* Current estimated length of instruction using label */
#define UD_OP      5                          /* Instruction using labeldiff */
#define UD_DEF2    6                /* Pointer to definition of second label */
/*}}}*/
/*{{{  fields in a label record*/
#define LB_SIZE    3                                 /* Length of the record */
#define LB_NAME    0                                           /* Label name */
#define LB_PRAD    1                  /* Address of the corresponding setlab */
#define LB_DEF     2                  /* Pointer to the corresponding setlab */
/*}}}*/
/*{{{  fields in a table record*/
#define T_SIZE     3                                 /* Length of the record */
#define T_SCALE    1            /* Pointer to field in related scale record  */
#define T_NEXT     2                 /* Pointer to next item on adjust chain */
/*}}}*/
/*{{{  fields in a scale record*/
#define SC_SIZE    4                                 /* Length of the record */
#define SC_VALUE   1              /* Last calculated size of the scale value */
#define SC_NEXT    2                 /* Pointer to next item on adjust chain */
#define SC_OP      3                         /* Instruction using case scale */
/*}}}*/
/*{{{  fields in a debug record*/
#define DB_SIZE    5                                 /* Length of the record */
#define DB_ADDR    1            /* Last calculated address of the debug item */
#define DB_NEXT    2                 /* Pointer to next item on adjust chain */
#define DB_LINE    3                     /* Source line number of debug item */
#define DB_DATA    4                       /* Label of associated code chunk */
/* bug 1158 means that multiple tables can have the same addresses;
   these are marked with address fixes, where the DB_DATA field is the label
   number of the section of code for that table
   CON 14/2/91
*/
/*}}}*/


/*{{{  segment definitions*/
typedef struct seginfo_struct
  {
    struct seginfo_struct *seg_next;
    treenode *seg_name;
    INT32 seg_entry;
  } seginfo_t;
/*}}}*/

#define IBUF_SIZE 256  /* amount copied at one time to code file */
/* This must be small enough to fit into the output file buffer */

/*}}}*/
/*{{{  local variables*/
PRIVATE int dead;
PRIVATE treenode *libentrychain;
/*{{{  data for code buffer handling*/
PRIVATE INT32 *code = NULL;          /* Buffer to hold code during compaction */

PRIVATE int cstart;          /* Pointer to first free location in code buffer */
PRIVATE int cptr;                    /* Pointer to top of code in code buffer */
PRIVATE int lptr;          /* Pointer to bottom of label info. in code buffer */
PRIVATE int topl;       /* Pointer to top of chain of label defns. and usages */
PRIVATE int start; /* Pointer to start of current code segment in code buffer */
PRIVATE int coden;         /* Number of bytes of code in current code segment */
             /* Word used to pack bytes of code before placing in code buffer*/
PRIVATE BIT32 codew;
PRIVATE int partword;                         /* Used during packing of codew */
PRIVATE INT32 address;            /* Current estimate of code address within SC */
                     /* Pointer to label which marks entry point for section */
PRIVATE int section_label;
            /* Pointer to most recently declared scale record in code buffer */
PRIVATE int case_scale;
PRIVATE INT32 getword;              /* Word of four bytes read from code buffer */
                    /* Bytes packed into getword still to be written to ibuf */
PRIVATE int bytesleft;
PRIVATE int getptr;     /* Next word to be taken from code buffer by put_code */

PRIVATE BIT32 section_size; /* Current estimate of size of current code section */
PRIVATE BIT32 total_code_size;                             /* Total code size */
PRIVATE BIT32 bytes_output;
PRIVATE BYTE libstub[] = { I_PFIX, I_PFIX, I_PFIX, I_PFIX,
                           I_PFIX, I_PFIX, I_PFIX, I_PFIX };
  /* libstub is also used for entry point zero padding */

/*{{{  STOP on error handling*/
PRIVATE int int_error_status;
PRIVATE int fpu_error_status;

#define ERROR_STATUS_START 0   /* At the start of a block    */
#define ERROR_STATUS_CLEAN 1   /* Error is known to be clear */
#define ERROR_STATUS_DIRTY 2   /* Error may have been set    */

#define LOCALISE_ERROR_HANDLING TRUE
/* TRUE if we want to wait until generating an FPU instruction before clearing
   the FPU error flag
   BUT we need a way to clear FPU error without clobbering integer stack.
   There isn't one, cos both FPTESTERR and FPUCLRERR hit 1 stack slot.
   AHAH! We now use the fact that FPU loads always use a CPU slot,
   so we can use that after the load has happened!  CO'N 12/2/90
   We can't use this system for normal errors, cos the 'TESTERR' instruction
   clobbers the stack
*/

#if LOCALISE_ERROR_HANDLING
#define ERROR_STATUS_FP_INIT ERROR_STATUS_START
#else
#define ERROR_STATUS_FP_INIT ERROR_STATUS_CLEAN
#endif
/*}}}*/

/*}}}*/
/*{{{  data for segment handling*/
PRIVATE seginfo_t *seginfo;
/*}}}*/
/*}}}*/

/*{{{  private routines*/
/*{{{  picture of the code buffer*/
/*****************************************************************************
 *
 *   <---- Code                                              Labels    ---->
 * ---------------------------------------------------------------------------
 * |  |   |    |                     |       |            | | | | | | | | |  |
 * |  |   |    | current code        |       | current    |   entry points   |
 * |  |   |    |     --->            |       | labels     |   for previously |
 * |  |   |    |                     |       |  <---      |   compacted      |
 * |  |   |    |                     |       |            |   routines       |
 * |  |   |    |                     |       |            | | | | | | | | |  |
 * |  |   |    |                     |       |            | | | | | | | | |  |
 * ---------------------------------------------------------------------------
 *  previously  ^                     ^       ^          ^
 *  compacted   cstart                cptr    lptr       section
 *  routines                                             label
 *
 *****************************************************************************/

/*{{{  format of previously compacted routines*/
/*****************************************************************************
 *
 * ---------------------------------------------------------------------------
 * | |                      | # |            | |
 * | |                      | # |            | |
 * | |                      | # |            | |
 * | |                      | # |            | |
 * | |                      | # |            | |
 * ---------------------------------------------------------------------------
 *  ^                        | ^              |
 *  |                        | |              |
 *  -------------------------  ----------------
 *  ^                          ^
 *  |                          |
 *  ---bytes in this routine ---
 *                           ^                ^
 *                           |                |
 *                      pointer to start of routine
 *****************************************************************************/
/*}}}*/

/*}}}*/
/*{{{  PRIVATE int isbranchop(op)*/
/*****************************************************************************
 *
 *  isbranchop if the opcode 'op' represents a branch instruction.
 *
 *****************************************************************************/
PRIVATE int isbranchop (const int op )
{
  return((op == I_J) || (op == I_CJ));
}
/*}}}*/
/*{{{  low-level code buffer handling*/
/*{{{  PRIVATE BYTE get_code()*/
/**************************************************************************
 *
 * get_code returns the next byte from the code buffer.
 *          The current position in the code buffer is represented by
 *          (getptr, getword, bytesleft).
 *
 **************************************************************************/
PRIVATE BYTE get_code ( void )
{
  BYTE r;
  if (bytesleft == 0)
    {
      getword = code[getptr++];
      bytesleft = 4;
    }
  r = (BYTE)(getword & 0xff);
  getword >>= 8;
  bytesleft--;
  return(r);
}
/*}}}*/
/*{{{  PRIVATE void add_code(c)*/
/***************************************************************************
 *
 * add_code adds a byte of code to the code buffer. Every time four bytes
 *          are assembled, they are written to the code buffer at cptr.
 *
 **************************************************************************/
PRIVATE void add_code (const BYTE c )
{
  codew |= ((BIT32)c << partword);
  coden++;
  address++;
  if (partword == 24)
    /*{{{  write codew to code buffer*/
    {
      code[cptr++] = codew;
      if (cptr > lptr)
        generr_i(GEN_CODE_BUFFER_FULL, cptr*sizeof(INT32));
      partword = 0;
      codew = ZERO32;
    }
    /*}}}*/
  else
    {
      if (start == C_NIL)
        /*{{{  start new code section*/
        {
          start = cptr;
          cptr++;      /* Save word for code length */
        }
        /*}}}*/
      partword += 8;
    }
}
/*}}}*/
/*}}}*/
/*{{{  input*/
/*{{{  code input*/
/*{{{  support*/
/*{{{  PRIVATE int assemble_instruction(v)*/
/***************************************************************************
 *
 * assemble_instruction adds the prefix bytes required for operand to the
 *        buffer, and returns the length
 *
 **************************************************************************/
PRIVATE int assemble_instruction (BYTE buf[], const int instruction, INT32 operand )
{
  int ptr;

  if (operand > 0)  /* most common */
    {
      ptr = 0;
      while (operand > 0)
        {
          buf[ptr++] = I_PFIX | (operand & 0xf);
          operand >>= 4;
        }
    }
  else if (operand == 0)
    {
      buf[0] = 0;
      ptr = 1;
    }
  else if (operand >= -16)
    {
      buf[1] = I_NFIX | 0;
      buf[0] = operand & 0xf;
      ptr = 2;
    }
  else
    {
      ptr = 0;
      operand = ~operand;
      while (operand > 15)
        {
          buf[ptr++] = I_PFIX | ((~operand) & 0xf);
          operand >>= 4;
        }
      buf[ptr++] = (I_NFIX | operand);
    }

  buf[0] = instruction | (buf[0] & 0xf);

  return ptr;
}
/*}}}*/
/*{{{  PRIVATE void put_pfix (instruction, operand, size)*/
/*{{{  comment*/
/***************************************************************************
 *
 *  put_pfix adds the primary (instruction, operand) into the code buffer,
 *           prefixed if neccessary so that it occupies 'size' bytes.
 *
 **************************************************************************/
/*}}}*/
PRIVATE void put_pfix (const int instruction, const INT32 operand, const int size )
{
  int i;
  if (operand >= ZERO32)
    for (i = 1; i < size; i++)
      add_code(I_PFIX | (BYTE)((operand >> (( size - i) * 4)) & 0xf));
  else
    {
      int shifts = size - 2;
      add_code(I_NFIX | (BYTE)((~(operand >> ((size - 1) * 4))) & 0xf));
      for (i = 0; i < shifts; i++)
        add_code(I_PFIX | (BYTE)((operand >> ((shifts - i) * 4)) & 0xf));
    }
  add_code(instruction | (BYTE)(operand & 0xf));
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void code_primary(instruction, operand)*/
/***************************************************************************
 *
 * code_primary adds primary instruction, operand to the code buffer.
 *
 **************************************************************************/
PRIVATE void code_primary (const int instruction, const INT32 operand )
{
  BYTE buf[MAX_BYTES_IN_INSTRUCTION];
  int ptr;
  if ((operand == ZERO32) && ((instruction == I_LDNLP) ||
                              (instruction == I_AJW) ||
                              (instruction == I_ADC)))
    /* Null instruction */
    return;

  ptr = assemble_instruction(buf, instruction, operand);

  while (--ptr >= 0)
    add_code(buf[ptr]);
}
/*}}}*/
/*{{{  PRIVATE void code_secondary(instruction)*/
/***************************************************************************
 *
 * code_secondary adds secondary instruction to the code buffer.
 *
 **************************************************************************/
PRIVATE void code_secondary (const int instruction )
{
  /*code_primary(I_OPR, instruction);*/
  code_primary(I_OPR, (instruction & I_NEGATIVE) ? -(instruction & (~I_NEGATIVE)) :
                                                    instruction);
}
/*}}}*/
/*}}}*/
/*{{{  labels and tables input*/
/*{{{  support*/
/*{{{  PRIVATE int mark_code(type, size)*/
/***************************************************************************
 *
 * mark_code inserts a record of type type and size size into the code
 *           buffer, and returns the index of the start of the record.
 *
 **************************************************************************/
PRIVATE int mark_code (const int type, const int size )
{
  int ptr;
  if (partword != 0)
    /*{{{  write partial word of code to code buffer*/
    {
      code[cptr++] = (INT32)codew;
      if (cptr > lptr)
        generr_i(GEN_CODE_BUFFER_FULL, cptr*sizeof(INT32));
      partword = 0;
      codew = ZERO32;
    }
    /*}}}*/
  if (start != C_NIL)
    /*{{{  close up code segment*/
    {
      if (coden == 0)
        cptr = start;   /* Empty code segment, erase it */
      else
        code[start] = coden; /* Save length of code segment */
      start = C_NIL;
    }
    /*}}}*/
  coden = 0;
  ptr = cptr;
  if ((cptr + size) < lptr)
    /*{{{  create node, add it to adjust node chain*/
    {
      code[cptr] = type;
      /* Add node to adjust node chain */
      if (topl != C_NIL) code[topl + U_NEXT] = cptr;
      topl = cptr;
      /* Reserve space for the node */
      cptr += size;
    }
    /*}}}*/
  else
    generr_i(GEN_CODE_BUFFER_FULL, cptr*sizeof(INT32));
  return(ptr);
}
/*}}}*/
/*{{{  PRIVATE int addlabel(label)*/
/**************************************************************************
 *
 *  addlabel creates a new label record with name 'label', and returns
 *           the index of the new label.
 *
 **************************************************************************/
PRIVATE int addlabel (const int label )
{
  lptr -= LB_SIZE;
  if (lptr > cptr)
    {
      code[lptr + LB_NAME] = label;
      code[lptr + LB_DEF] = C_NIL;
      code[lptr + LB_PRAD] = 0;
    }
  else
    generr_i(GEN_LABEL_BUFFER_FULL, lptr);
  return(lptr);
}
/*}}}*/
/*{{{  PRIVATE int findlabel(label)*/
/**************************************************************************
 *
 * findlabel searches for label 'label' and if found returns the index
 *           of the label record. If not found, a new record is created
 *           for 'label' and its index is returned.
 *
 **************************************************************************/
PRIVATE int findlabel (const int label )
{
  int l = lptr;
  while ((l < max_code_size) && (code[l + LB_NAME] != label))
    l += LB_SIZE;
  if (l < max_code_size)    /* Label found */
    return(l);
  else
    {                  /* Create new label node */
      return(addlabel(label));
    }
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void label_def(label, type)*/
/***************************************************************************
 *
 * label_def inserts a setlab record for label 'label' into the code buffer.
 *
 ***************************************************************************/
PRIVATE void label_def (const int label, const int type )
{
  const int lab = findlabel(label);
  if (code[lab + LB_DEF] != C_NIL)
    geninternal_i(GEN_MULTIPLY_DEFINED_LABEL, label);
  else
    {
      int ptr = mark_code(type, SL_SIZE);
      code[ptr + SL_ADDR] = address;
      code[lab + LB_DEF] = ptr;
    }
}
/*}}}*/
/*{{{  PRIVATE void section_label_def(label, type)*/
/***************************************************************************
 *
 * section_label_def inserts a setlab record for label 'label'
 *                   into the code buffer, and sets it to
 *                   be the start of the current code section.
 *
 **************************************************************************/
PRIVATE void section_label_def (const int label, const int type )
{
  label_def(label, type);
  section_label = lptr;
}
/*}}}*/
/*{{{  PRIVATE void label_ref(instruction, label)*/
/***************************************************************************
 *
 * label_ref inserts a uselab record for label 'label' into the code buffer.
 *
 **************************************************************************/
PRIVATE void label_ref (const int instruction, const int label )
{
  const int l = findlabel(label);
  const int forward_reference = (code[l + LB_DEF] == C_NIL);
  const int ptr = mark_code(C_USELAB, U_SIZE);
  int adj;

  if (forward_reference)
    adj = 0; /* assume null jump */
  else
    adj = 2; /* backwards jumps require nfix and op */

  code[ptr + U_DEF] = (forward_reference) ? l : code[l + LB_DEF];
  code[ptr + U_ADDR] = address;
  code[ptr + U_OP] = instruction;
  code[ptr + U_ADJ] = adj;

  address += adj;
}
/*}}}*/
/*{{{  PRIVATE void label_diff(instruction, label1, label2)*/
/***************************************************************************
 *
 * label_diff inserts a uselabdiff record for label 'label'
 * into the code buffer.
 *
 **************************************************************************/
PRIVATE void label_diff (const int instruction, const int label1, const int label2 )
{
  const int ptr = mark_code(C_USELABDIFF, UD_SIZE);
  int lab;

  code[ptr + UD_ADDR] = address;
  code[ptr + UD_ADJ] = 0;
  /*{{{  set UD_DEF1*/
  lab = findlabel(label1);
  if (code[lab + LB_DEF] == C_NIL)
    code[ptr + UD_DEF1] = lab;
  else
    code[ptr + UD_DEF1] = code[lab + LB_DEF];
  /*}}}*/
  /*{{{  set UD_DEF2*/
  lab = findlabel(label2);
  if (code[lab + LB_DEF] == C_NIL)
    code[ptr + UD_DEF2] = lab;
  else
    code[ptr + UD_DEF2] = code[lab + LB_DEF];
  /*}}}*/
  code[ptr + UD_OP] = instruction;
}
/*}}}*/
/*{{{  PRIVATE void table_start()*/
/***************************************************************************
 *
 *  table_start inserts a table record marking the start of a table, into
 *              the code buffer.
 *
 **************************************************************************/
PRIVATE void table_start ( void )
{
  int ptr = mark_code(C_TABLE, T_SIZE);
  code[ptr + T_SCALE] = case_scale;
}
/*}}}*/
/*{{{  PRIVATE void table_end()*/
/***************************************************************************
 *
 *  table_end inserts a table record marking the end of a table, into
 *            the code buffer.
 *
 **************************************************************************/
PRIVATE void table_end ( void )
{
  mark_code(C_TABLE, T_SIZE);
}
/*}}}*/
/*{{{  PRIVATE void scale_ref(instruction)*/
/***************************************************************************
 *
 *  scale_ref inserts a scale record for the scale of the forthcoming table
 *            into the code buffer.
 *
 **************************************************************************/
PRIVATE void scale_ref (const int instruction )
{
  case_scale = mark_code(C_SCALE, SC_SIZE);
  code[case_scale + SC_VALUE] = ZERO32;
  code[case_scale + SC_OP] = instruction;
  address++;
}
/*}}}*/
/*}}}*/
/*{{{  debug information input*/
/*{{{  PRIVATE void add_debug_info (type, line)*/
/*****************************************************************************
 *
 *  add_debug_info inserts a debug record into the code buffer,
 *                 'type' indicates the type of debug record, line gives
 *                 the corresponding source file line number.
 *
 *****************************************************************************/
PRIVATE void add_debug_info (const int type , treenode *const line, const int data )
{
  const int ptr = mark_code(type, DB_SIZE);
  code[ptr + DB_ADDR] = address;
  code[ptr + DB_LINE] = (INT32)line;
  code[ptr + DB_DATA] = data;
}
/*}}}*/
/*}}}*/
/*}}}*/
/*{{{  code adjustment*/
/*{{{  variables for adjust*/
PRIVATE int extra,                /* Extra bytes of code added this iteration */
           chptr;                 /* Pointer to current node on adjust chain */
/*}}}*/
/*{{{  routines for adjust*/
/*{{{  PRIVATE int getaddr(ldef)*/
/***************************************************************************
 *
 * getaddr returns the address (offset from beginning of current code
 *         section) of the label ldef.
 *
 **************************************************************************/
PRIVATE INT32 getaddr (const int ldef )
{
  if (ldef >= section_label)
    /*{{{  previous routine*/
    return(section_size + extra + code[ldef + LB_PRAD]);
    /*}}}*/
  else if (ldef >= lptr)
    /*{{{  unresolved forward reference*/
    {
      /* resolve forward reference */
      INT32 l = code[ldef + LB_DEF];                  /* point l to the setlab */
      if (l == C_NIL)
        geninternal_i(GEN_ADJUST_ERROR, code[ldef + LB_NAME]);  /* no setlab */
      if (code[chptr] == C_USELAB)
        code[chptr + U_DEF] = l;           /* Point the uselab to the setlab */
      else
        /*{{{  point the uselabdiff to the setlab*/
        {
          if (code[chptr + UD_DEF1] == ldef)
            code[chptr + UD_DEF1] = l;
          else
            code[chptr + UD_DEF2] = l;
        }
        /*}}}*/
    
      return(code[l + SL_ADDR] + extra);
    }
    /*}}}*/
  else if (ldef > chptr)
    /*{{{  resolved forward reference*/
    return(code[ldef + SL_ADDR] + extra);
    /*}}}*/
  else
    /*{{{  backward reference*/
    return(code[ldef + SL_ADDR]);
    /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void adjust()*/
/***************************************************************************
 *
 * adjust adjusts the label offsets in the current code section to their
 *        optimal size.
 *
 **************************************************************************/
PRIVATE void adjust ( void )
{
  int passno = 0;
  /* Perform code compaction */
  extra = -1;
  section_size = address;
  code[topl + U_NEXT] = C_NIL; /* Mark end of list */
  while (extra != 0)
    /*{{{  iteration of code adjustment*/
    {
      passno++;
      if (diagnostics)
        fprintf(outfile, "\nCode adjustment, pass %d...", passno);
      chptr = cstart;
      extra = 0;
      while (chptr != C_NIL)
        {
          switch(code[chptr])
            {
              /*{{{  C_SETLAB C_STARTFILE C_ENDFILE C_LOCATE C_LIBPATCH C_ADDRESSFIX*/
              case C_SETLAB:
              case C_STARTFILE:
              case C_ENDFILE:
              case C_LOCATE:
              case C_LIBPATCH:
              case C_ADDRESSFIX:
                code[chptr + SL_ADDR] += extra;
                break;
              /*}}}*/
              /*{{{  C_USELAB*/
              case C_USELAB:
               {
                 INT32 addr, diff;
                 int newadj;
              
                 addr = getaddr(code[chptr + U_DEF]);
                 code[chptr + U_ADDR] += extra;
                 diff = addr - (code[chptr + U_ADDR] + code[chptr + U_ADJ]);
                 if (isbranchop(code[chptr + U_OP]) && (diff == ZERO32))
                   newadj = 0; /* we optimise jump 0 out */
                 else
                   newadj = ilength(diff);
                 if (newadj > code[chptr + U_ADJ])
                   {
                     extra += newadj - code[chptr + U_ADJ];
                     code[chptr + U_ADJ] = newadj;
                   }
               }
               break;
              /*}}}*/
              /*{{{  C_USELABDIFF*/
              case C_USELABDIFF:
                {
                  INT32 addr1, addr2, diff;
                  int newadj;
              
                  addr1 = getaddr(code[chptr + UD_DEF1]);
                  addr2 = getaddr(code[chptr + UD_DEF2]);
                  diff = addr1 - addr2;
                  code[chptr + UD_ADDR] += extra;
              
                  if (isbranchop(code[chptr + UD_OP]) && (diff == ZERO32))
                    newadj = 0; /* we optimise jump 0 out */
                  else
                    newadj = ilength(diff);
              
                  if (newadj > code[chptr + UD_ADJ])
                    {
                      extra += newadj - code[chptr + UD_ADJ];
                      code[chptr + UD_ADJ] = newadj;
                    }
                }
                break;
              /*}}}*/
              /*{{{  C_TABLE*/
              case C_TABLE:
                {
                  int case_start = chptr,
                      case_extra = -1,
                      max_size = 0,
                      first = TRUE;
              
                  while (case_extra != 0)
                    {
                      chptr = code[case_start + U_NEXT];
                      case_extra = 0;
                      while (code[chptr] != C_TABLE)
                        /*{{{  visit each uselab record in adjust chain*/
                        {
                          INT32 addr, diff;
                          int newadj;
                        
                          addr = getaddr(code[chptr + U_DEF]);
                          if (first)
                            code[chptr + U_ADDR] += extra + case_extra;
                          else
                            code[chptr + U_ADDR] += case_extra;
                          diff = addr - (code[chptr + U_ADDR] + code[chptr + U_ADJ]);
                          newadj = ilength(diff);
                          newadj = max(newadj, max_size);
                          if (newadj > code[chptr + U_ADJ])
                            {
                              case_extra += newadj - code[chptr + U_ADJ];
                              code[chptr + U_ADJ] = newadj;
                            }
                          max_size = max(max_size, code[chptr + U_ADJ]);
                          chptr = code[chptr + U_NEXT];
                        }
                        /*}}}*/
                      extra += case_extra;
                      first = FALSE;
                    }
              
                  code[code[case_start + T_SCALE] + SC_VALUE] = max_size;
                }
                break;
              /*}}}*/
              /*{{{  C_SCALE*/
              case C_SCALE:
                /* Scale is set when we go round the associated table */
                break;
              /*}}}*/
              default:
                generr_i(GEN_ADJUST_TAG_ERROR, code[chptr]);
            }
          chptr = code[chptr + U_NEXT];
        }
      /* Move address up the number of bytes we have inserted */
      address += extra;
      section_size += extra;
    }
    /*}}}*/
  if (diagnostics)
    fprintf(outfile, "\nAdjustment complete.\n");
}
/*}}}*/
/*}}}*/
/*{{{  code compression*/
/*{{{  PRIVATE void compress_code_buffer()*/
/*{{{  comment*/
/***************************************************************************
 *
 * compress_code_buffer compresses the adjusted code section to the front
 *                      of the code buffer (cstart onwards), and moves
 *                      cstart to point to the first free word.
 *
 **************************************************************************/
/*}}}*/
PRIVATE void compress_code_buffer ( void )
{
  int topc = mark_code(C_SETLAB, SL_SIZE);
  int chptr = cstart;                    /* Index into the uncompressed code */
  cptr = cstart;                       /* Where we write the compressed code */
  start = C_NIL;
  /*{{{  update labels above section_label*/
  /* Update labels above section_label, ie. entry points of other code
     sections, so that their LB_PRAD field contains the offset from the
     beginning of the current code section */
  {
    int l = section_label + LB_SIZE;
    while (l < max_code_size)
      {
        code[l + LB_PRAD] += section_size;
        l += LB_SIZE;
      }
  }
  /*}}}*/
  /*{{{  set up codew*/
  codew = 0; coden = 0; partword = 0;
  /*}}}*/
  /*{{{  patch uselab and uselabdiff operands*/
  while (chptr < topc)
    /*{{{  patch operands*/
    {
      switch((int)code[chptr])
        {
          /*{{{  C_SETLAB*/
          case C_SETLAB:
            chptr += SL_SIZE;
            break;
          /*}}}*/
          /*{{{  C_USELAB*/
          case C_USELAB:
            {
              INT32 addr = code[code[chptr + U_DEF] + SL_ADDR],
                    diff = addr - (code[chptr + U_ADDR] + code[chptr + U_ADJ]);
              code[chptr + U_DEF] = diff;
              chptr += U_SIZE;
            }
            break;
          /*}}}*/
          /*{{{  C_USELABDIFF*/
          case C_USELABDIFF:
            {
              INT32 addr1 = code[code[chptr + UD_DEF1] + SL_ADDR],
                    addr2 = code[code[chptr + UD_DEF2] + SL_ADDR];
              code[chptr + UD_DEF1] = addr1 - addr2;
              chptr += UD_SIZE;
            }
            break;
          /*}}}*/
          /*{{{  C_TABLE*/
          case C_TABLE:
            chptr += T_SIZE;
            break;
          /*}}}*/
          /*{{{  C_SCALE*/
          case C_SCALE:
            chptr += SC_SIZE;
            break;
          /*}}}*/
          /*{{{  C_STARTFILE C_ENDFILE C_LOCATE C_LIBPATCH C_ADDRESSFIX*/
          case C_STARTFILE:
          case C_ENDFILE:
          case C_LOCATE:
          case C_LIBPATCH:
          case C_ADDRESSFIX:
            chptr += DB_SIZE;
            break;
          /*}}}*/
          default:   /* code chunk */
            {
              int len = code[chptr++];
              int words = len / 4;
              if ((len % 4) != 0) /* allow for partword */
                words++;
              chptr += words;
            }
            break;
        }
    }
    /*}}}*/
  /*}}}*/
  /*{{{  compress code buffer*/
  chptr = cstart;
  while (chptr < topc)
    switch((int)code[chptr])
      {
        /*{{{  C_SETLAB*/
        case C_SETLAB:
          chptr += SL_SIZE;
          break;
        /*}}}*/
        /*{{{  C_USELAB*/
        case C_USELAB:
          {
            int op = code[chptr + U_OP];
            int adj = code[chptr + U_ADJ];
            INT32 diff = code[chptr + U_DEF];
            /*{{{  a bit of peephole optimisation to remove j 0 and cj 0*/
            if (!isbranchop(op) || (diff != ZERO32) || (adj != ZERO32))
              put_pfix(op, diff, adj);
            /*}}}*/
            chptr += U_SIZE;
          }
          break;
        /*}}}*/
        /*{{{  C_USELABDIFF*/
        case C_USELABDIFF:
          {
            INT32 diff = code[chptr + UD_DEF1];
            code_primary((int)code[chptr + UD_OP], diff);
            chptr += UD_SIZE;
          }
          break;
        /*}}}*/
        /*{{{  C_SCALE*/
        case C_SCALE:
          code_primary(code[chptr + SC_OP], code[chptr + SC_VALUE]);
          chptr += SC_SIZE;
          break;
        /*}}}*/
        /*{{{  C_TABLE*/
        case C_TABLE:
          chptr += T_SIZE;
          break;
        /*}}}*/
        /*{{{  C_LOCATE C_LIBPATCH C_ADDRESSFIX*/
        /* case C_STARTFILE:
           case C_ENDFILE: */
        case C_LOCATE:
          if (debugoutput)
            gencodemark((treenode *)code[chptr + DB_LINE],
             (bytes_output + section_size) - code[chptr + DB_ADDR]);
            /* write_locate_item(rtltag(code[chptr]), code[chptr + DB_LINE],
                              code[chptr + DB_ADDR]); */
          chptr += DB_SIZE;
          break;
        case C_LIBPATCH:
          if (debugoutput)
            genlibpatch(code[chptr + DB_LINE],
             (bytes_output + section_size) - code[chptr + DB_ADDR]);
          chptr += DB_SIZE;
          break;
        case C_ADDRESSFIX:
          if (debugoutput)
            {
              const int label = code[chptr + DB_DATA];
              BIT32 offset;
              DEBUG_MSG(("addressfix: label: %d, ", label));
              if (label == (-1)) /* normal addressfix */
                offset = (bytes_output + section_size) - code[chptr + DB_ADDR];
              else /* bug 1158 - extra stuff */
                {
                #if 0
                  DEBUG_MSG(("findlabel: %d, ",                  findlabel(label)));
                  DEBUG_MSG(("code(DEF): %d, ",             code[findlabel(label)+LB_DEF]));
                  DEBUG_MSG(("code(PRAD): %d\n",            code[findlabel(label)+LB_PRAD]));
                  DEBUG_MSG(("code(SL_ADDR): ld, ",         code[findlabel(label)+SL_ADDR]));
                  DEBUG_MSG(("addr(SL_ADDR): %d, ", getaddr(code[findlabel(label)+SL_ADDR])));
                  DEBUG_MSG(("addr(PRAD): %d\n",    getaddr(code[findlabel(label)+LB_PRAD])));
                  DEBUG_MSG(("section_size: %d, bytes_output: %d\n", section_size, bytes_output));
                  DEBUG_MSG(("Correct: %d, ", (bytes_output + section_size) - code[findlabel(label)+LB_PRAD]));
                #endif
                  /* This is a horrible piece of black magic:
                     What we want is the address (offset from the end of code)
                     of a particular label.
                     After much experimentation, this expression gives the
                     correct result. Blame Martin Day for designing this
                     code cruncher in this way in the first place.
                     CON - 14/2/91
                  */
                  offset = (bytes_output + section_size) -
                           code[findlabel(label)+LB_PRAD];
                }
              DEBUG_MSG(("offset: %ld\n", offset));
              genaddressfix((treenode *)code[chptr + DB_LINE], offset);
            }
          chptr += DB_SIZE;
          break;
        /*}}}*/
        default:   /* code chunk */
          /*{{{  disassemble code chunk*/
          {
            int len = code[chptr++];
            int i;
            int bytesleft = 0;
            BIT32 putword;
            for (i = 0; i < len; i++)
              {
                if (bytesleft == 0)
                  {
                    putword = code[chptr++];
                    bytesleft = 4;
                  }
                add_code((BYTE)(putword & 0xff));
                bytesleft--;
                putword >>= 8;
              }
          }
          break;
          /*}}}*/
      }
  /*}}}*/
  /*{{{  tidy up codew*/
  if (partword != 0)
    {
      code[cptr++] = codew;
      partword = 0;
      codew = 0;
    }
  /*}}}*/
  if (start != C_NIL)
    {
      code[cptr++] = start;       /* Append pointer to start of code section */
      code[start] = section_size;              /* Insert code section length */
    }
  cstart = cptr;
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void diag_write_spaces*/
PRIVATE void diag_write_spaces(void)
{
  fputc('\t', outfile);
}
/*}}}*/
/*{{{  PRIVATE void diag_write_nl*/
PRIVATE void diag_write_nl(void)
{
  fputc('\n', outfile);
  diag_write_spaces();
}
/*}}}*/
/*{{{  PRIVATE void diag_write_comment*/
PRIVATE void diag_write_comment(void)
{
  diag_write_spaces();
  fputs("-- ", outfile);
}
/*}}}*/
/*{{{  PRIVATE void diag_write_nl_comment*/
PRIVATE void diag_write_nl_comment(void)
{
  fputc('\n', outfile);
  diag_write_comment();
}
/*}}}*/
/*{{{  PRIVATE void diag_write_nl_string*/
PRIVATE void diag_write_nl_string(char *str)
{
  diag_write_nl();
  fputs(str, outfile);
}
/*}}}*/
/*{{{  PUBLIC void gencommentn(c, ...)*/
/***************************************************************************
 *
 *  gencommentn writes the printf format string 'c' (with parameters p1, p2,
 *             pn) as a comment to the debugging file.
 *
 **************************************************************************/
/* Write a comment */
PUBLIC void gencomment0 ( const char *const c )
{
  if (assembly_output)
    {
      diag_write_comment();
      fputs(c, outfile);
    }
}

/* Write a comment */
PUBLIC void gencomment1 (const char *const c, const BIT32 p1 )
{
  if (assembly_output)
    {
      diag_write_comment();
      fprintf (outfile, c, p1);
    }
}
#if 0  /* gencomment2 and gencomment3 are never used */
/* Write a comment */
PUBLIC void gencomment2 (const char *const c, const BIT32 p1, const BIT32 p2 )
{
  if (assembly_output)
    {
      diag_write_comment();
      fprintf (outfile, c, p1, p2);
    }
}

/* Write a comment */
PUBLIC void gencomment3 (const char *const c, const BIT32 p1, const BIT32 p2, const BIT32 p3 )
{
  if (assembly_output)
    {
      diag_write_comment();
      fprintf (outfile, c, p1, p2, p3);
    }
}
#endif
/*}}}*/
/*{{{  code disassembly*/
/*{{{  PRIVATE int look_at_instruction(length, operand)*/
/***************************************************************************
 *
 *  look_at_instruction disassembles the next instruction in the code
 *                      buffer (represented by getptr, bytesleft, getword)
 *                      into an instruction value and an operand value.
 *                      The instruction bytes are placed in buf.
 *                      The instruction len is returned.
 *
 **************************************************************************/
PRIVATE int look_at_instruction ( int *const instruction, INT32 *const operand,
        BYTE buf[], const int maxlen)
{
  int len = 0;
  *operand = 0;
  while (len < maxlen)
    /*{{{  read instruction byte*/
    {
      BYTE b = get_code();
      buf[len++] = b;
      *operand |= (INT32)(b & 0x0f);
      *instruction = b & 0xf0;
      if (*instruction == I_PFIX)
        *operand <<= 4;
      else if (*instruction == I_NFIX)
        *operand = (~(*operand)) << 4;
      else
        return len;
    };
    /*}}}*/
  return len;
}
/*}}}*/
/*{{{  PRIVATE int print_instruction(address)*/
/***************************************************************************
 *
 *  print_instruction disassembles the next instrution
 *                    and writes it to outfile.
 *                    Returns the length of the instruction disassembled.
 *
 **************************************************************************/
PRIVATE int print_instruction (const INT32 address, const int bytesleft )
{
  int i_value, i_len;
  int i;
  INT32 operand;
  BYTE buf[MAX_BYTES_IN_INSTRUCTION];

  fprintf(outfile, "%8ld: ", address);
  i_len = look_at_instruction (&i_value, &operand, buf,
                               min(MAX_BYTES_IN_INSTRUCTION, bytesleft));

  for (i = 0; i < i_len; i++)
    fprintf (outfile, "%.2X ", (int)buf[i]);
  for (; i < MAX_BYTES_IN_INSTRUCTION; i++)
    fprintf (outfile, "   ");

  if (i_value == I_OPR)
    fprintf (outfile, "%-8s\n", secondaryname((int)operand));
  else
    fprintf (outfile, "%-8s%ld\n", primaryname(i_value), operand);

  return (i_len);
}
/*}}}*/
/*}}}*/
/*}}}*/
#ifdef CONFIG
/*{{{  PUBLIC void patch_jump(INT32 source_addr, INT32 target_addr)*/
PUBLIC void patch_jump(const INT32 source_addr, const INT32 target_addr)
{
  INT32 operand;
  int len, ptr;
  int codeptr = 0;
  BYTE buf[MAX_BYTES_IN_INSTRUCTION];

  DEBUG_MSG(("Patching a jump from %ld to %ld\n", source_addr, target_addr));

  operand = target_addr - source_addr;
  len = 1;
  while ((ilength(operand - len) > len) && (len < MAX_BYTES_IN_INSTRUCTION))
    len++;

  operand -= len;

  if (ilength(operand) < len)  /* allow for the smaller operand being */
    {                          /* a smaller length */
      /* Note - This can never happen for backward jumps */
      buf[0] = (I_PFIX | 0);
      ptr = 1;
    }
  else
    ptr = 0;
  /* len now holds the length of the patch to be inserted */
  if (assembly_output)
    fprintf(outfile, "\n-- Inserting %d byte patch at offset %ld for jump %ld to %ld",
            len, source_addr, operand, target_addr);

  len = assemble_instruction(&(buf[ptr]), I_J, operand);
  len += ptr;

  {  /* find the start of the correct code section */
    int section_start = cstart;
    INT32 code_addr = 0;
    while (section_start != 0)
      /*{{{  find a code section*/
      {
        INT32 code_length;
        section_start = code[section_start - 1];
        code_length = code[section_start];
        if (code_addr == source_addr)
          {
            codeptr = section_start + 1;
            section_start = 0;  /* force a bomb out of loop */
            if (code_length < len)
              generr_i(GEN_CONF_PATCH_TOO_SMALL, len);
          }
        else
          code_addr += code_length;
      }
      /*}}}*/
    if (codeptr == 0)
      generr_i(GEN_CONF_CANT_FIND_PATCH, source_addr);
  }
  { /* patch the jump onto the code section */
    INT32 mask = 0xff;
    int shift = 0;
    while (--len >= 0)
      {
        code[codeptr] = (code[codeptr] & ~mask) | (buf[len] << shift);
        shift += 8; mask <<= 8;
        if (shift == 32)
          {
            shift = 0;
            mask = 0xff;
            codeptr++;
          }
      }
  }
}
/*}}}*/
#endif
/*{{{  PRIVATE void patch_lib_offsets ()*/
/***************************************************************************
 *
 *  patch_lib_offsets writes the offset of each library call stub into its
 *                    corresponding name node.
 *
 **************************************************************************/
PRIVATE void patch_lib_offsets (void)
{
  treenode *libcall = libentrychain;
  while (libcall != NULL)
    {
      int labelpos = findlabel(NPLabelOf(libcall));
      SetNLEntryOffset(libcall, code[labelpos + LB_PRAD]);
      libcall = NLEntryNextOf(libcall);
    }
#ifdef CONFIG
  pseudo_link(libentrychain);
#endif
}
/*}}}*/

/*{{{  public routines*/
/*{{{  PRIVATE void output_code_size()*/
/***************************************************************************
 *
 *  output_code_size writes the total code size to the object file
 *
 **************************************************************************/
PRIVATE void output_code_size ( void )
{
  /*INT32 nested_sc_size = sc_size();*/
#ifndef CONFIG
  if (information)
    {
      /* fprintf(outfile, "Module code size is %d, nested SC code size is %d\n",
               total_code_size, nested_sc_size); */
      fprintf(outfile, "Module code size is %ld bytes\n", total_code_size);
      #if 0
      if (nested_sc_size != ZERO32)
        fprintf(outfile, ", nested SC code size is %ld bytes", nested_sc_size);
      fprintf(outfile, "\n");
      #endif
    }
#endif
  write_total_code(total_code_size/*, nested_sc_size*/);
}
/*}}}*/
/*{{{  PRIVATE void calculate_code_size*/
PRIVATE void calculate_code_size(void)
{
  int section_start = cstart;
	
  total_code_size = 0;
  while (section_start != 0)
    {
      section_start = code[section_start - 1];
      total_code_size += code[section_start];
    }
}
/*}}}*/
/*{{{  PRIVATE void output_code()*/
/**************************************************************************
 *
 * output_code writes the entire code buffer to the object file
 *
 **************************************************************************/
PRIVATE void output_code ( void )
{
  int section_start = cstart;
  INT32 disassembly_address = 0;

  if (!disassemble) write_code_block_start(total_code_size);

  while (section_start != 0)
    /*{{{  output a code section*/
    {
      INT32 code_length;
      section_start = code[section_start - 1];
      code_length = code[section_start];
      /*{{{  set up for get_code*/
      getptr = section_start + 1; bytesleft = 0;
      /*}}}*/
      if (!disassemble)
        /*{{{  write code section to object file*/
        {
          BIT32 l = code_length;
          DEBUG_MSG(("output_code: writing a section of %ld bytes\n", l));
          while (l > 0)
            {
              BYTE ibuf[IBUF_SIZE];
              size_t len = min(IBUF_SIZE, l);
              int i;
              for (i = 0; i < len; i++)
                ibuf[i] = get_code();  /* this allows for endian-ness */
              write_code_block(len, ibuf);
              l -= len;
            }
        }
        /*}}}*/
      else
        /*{{{  disassemble code section to outfile*/
        {
          int i = 0;
          while (i < code_length)
            i += print_instruction(disassembly_address + i, code_length - i);
          disassembly_address += code_length;
        }
        /*}}}*/
    }
    /*}}}*/
}
/*}}}*/

/*{{{  PRIVATE int name_is_visible*/
PRIVATE int name_is_visible(const wordnode *const name, const seginfo_t *rest)
/* Returns TRUE if the name is not descoped by a later externally visible name */
{
  while ((rest != NULL) && (name != NNameOf(rest->seg_name)))
    rest = rest->seg_next;
  return (rest == NULL);  
}
/*}}}*/
/*{{{  PRIVATE void output_entry_points(pass)*/
/*{{{  comment*/
/***************************************************************************
 *
 *  output_entry_points writes all the declared entry points into the
 *                      object file. Must be called after output_code, as
 *                      the section labels must have been adjusted for
 *                      word alignment.
 *                      Pass one writes descriptor information,
 *                      pass two writes entry point information.
 *
 **************************************************************************/
/*}}}*/
PRIVATE void output_entry_points (const int pass )
{
  seginfo_t *seg;

  /* first reverse the list back into the correct order */
  if (pass == 1)
    {
      seginfo_t *p = seginfo, *reverser = NULL, *temp;
      while (p != NULL)
        {
          temp        = p->seg_next;
          p->seg_next = reverser;
          reverser    = p;
          p           = temp;
        }
      seginfo = reverser;
    }

  seg = seginfo;
  while (seg != NULL)
    {
      treenode *n  = seg->seg_name;
      INT32 offset = code[seg->seg_entry + LB_PRAD];
      if (name_is_visible(NNameOf(n), seg->seg_next))
        {
#ifdef CONFIG
          if (pass == 1)
            save_entrydata(n, offset, total_code_size);
#else
          if (information && (pass == 1))
            fprintf(outfile, "%s %s requires %ld workspace and %ld vectorspace words\n",
                    (TagOf(n) == N_PROCDEF) ? "PROC" : "FUNCTION",
                    WNameOf(NNameOf(n)), NPDatasizeOf(n), NPVSUsageOf(n));
#endif
          if (disassemble)
            {
              diag_write_comment();
              fprintf(outfile, "Entry point %s, offset %ld\n",
                      WNameOf(NNameOf(n)), offset);
            }
          else
            write_entry_desc(n, offset, pass);
        }
      seg = seg->seg_next;
    }
}
/*}}}*/

/*{{{  PUBLIC void compress_code()*/
/***************************************************************************
 *
 * compress_code compresses the current code section down to the front
 *               of the code buffer, and prepares the code buffer for the
 *               input of a new code section.
 *
 **************************************************************************/
PUBLIC void compress_code ( void )
{
  adjust();
  compress_code_buffer();
  /*{{{  cut back labels*/
  /* The label at section_label is the entry point which we must keep.
     Save the entry point offset in this label.
     Delete all subsequent labels. */
  code[section_label + LB_DEF] = C_NIL;
  code[section_label + LB_PRAD] = 0;
  lptr = section_label;
  /*}}}*/
  /*{{{  cut back code*/
  bytes_output += section_size;
  cptr = cstart;
  start = C_NIL;
  topl = C_NIL;
  codew = 0;
  coden = 0;
  partword = 0;
  address = 0; /* New code section starts at offset 0 */
  /*}}}*/
}
/*}}}*/
/*{{{  instruction generation*/
/*{{{  PRIVATE void clean_fpu_flag*/
PRIVATE void clean_fpu_flag(const int instruction)
{
  if (inside_asm) return;

  if (fpinline && (fpu_error_status == ERROR_STATUS_START) &&
      (errormode & ERRORMODE_TIMESLICECHECK))
    {
      gensecondary(instruction);
      fpu_error_status = ERROR_STATUS_CLEAN;
    }
}
/*}}}*/
/*{{{  PUBLIC void genstartblock()*/
/*****************************************************************************
*
*  genstartblock is used in STOP mode to remember to clear the error flag
*                after a process could have been timesliced by a jump
*                to the start of the block
*
******************************************************************************/
PUBLIC void genstartblock ( void )
{
  /* We cannot localise handling of the normal error flag, cos
     the TESTERR instruction clobbers the stack;
     For the FPU we can use FPUCLRERR which leaves all stacks alone */

  if (inside_asm) return;

  int_error_status = ERROR_STATUS_CLEAN;
  if (errormode & ERRORMODE_TIMESLICECHECK)
    gensecondary(I_TESTERR);

  fpu_error_status = ERROR_STATUS_START;
#if !LOCALISE_ERROR_HANDLING
  clean_fpu_flag(I_FPTESTERR);  /* FPTESTERR is faster than FPUCLRERR */
#endif
}
/*}}}*/
/*{{{  PRIVATE void dirty_error_flag(int fpu_not_int)*/
/*****************************************************************************
*
*  dirty_error_flag is used incase we've never got around to clearing
*                   the error flag in STOP mode.
*
******************************************************************************/
PRIVATE void dirty_error_flag (const int fpu_not_int )
{
  /* We cannot localise handling of the normal error flag, cos
     the TESTERR instruction clobbers the stack;
     For the FPU we can use FPUCLRERR which leaves all stacks alone */

  if (inside_asm) return;

  if (fpu_not_int)
    {
#if 0  /* LOCALISE_ERROR_HANDLING */
      /* This doesn't work, cos FPUCLRERR DOES hit the CPU stack! */
      clean_fpu_flag(I_FPUCLRERR);
#endif
      fpu_error_status = ERROR_STATUS_DIRTY;
    }
  else
    int_error_status = ERROR_STATUS_DIRTY;
}
/*}}}*/
/*{{{  PUBLIC void mark_flag_clean*/
PUBLIC void mark_flag_clean(int fpu_not_int)
{
  if (fpu_not_int)
    fpu_error_status = ERROR_STATUS_CLEAN;
  else
    int_error_status = ERROR_STATUS_CLEAN;
}
/*}}}*/
/*{{{  PUBLIC void markdeadcode*/
PUBLIC void markdeadcode()
{
  dead = !inside_asm;
}
/*}}}*/
/*{{{  PUBLIC void genprimary (instruction, operand)*/
/*{{{  comment*/
/***************************************************************************
 *
 *  genprimary adds code byte representing primary (instruction, operand)
 *             to the code buffer
 *
 **************************************************************************/
/*}}}*/
PUBLIC void genprimary (const int instruction, const INT32 operand )
{
  if (!dead)
    {
      if (instruction == I_ADC) dirty_error_flag(FALSE);
      code_primary(instruction, operand);
      if (assembly_output)
        {
          if ((operand == ZERO32) && ((instruction == I_LDNLP) ||
                                      (instruction == I_AJW) ||
                                      (instruction == I_ADC)))
            /* Null instruction */
            return;
          else
            {
              diag_write_nl();
              fprintf(outfile, "%-8s", primaryname(instruction));
              if ((alloc_strategy & ALLOC_NOOPERANDS) == 0)
                {
                  fprintf(outfile, "%ld", operand);
                  /* display large(ish) numbers in hex too */
                  if (operand > 4000 || operand < -4000)
                    gencomment1("#%lX", operand);
                }
            }
        }
    }
}
/*}}}*/
/*{{{  PUBLIC void genbranch (instruction, label)*/
/***************************************************************************
 *
 *  genbranch adds a record (representing a primary 'instruction' with
 *            operand the offset between the next instruction and the
 *            label 'label') to the code buffer.
 *
 **************************************************************************/
PUBLIC void genbranch (const int instruction, const int label )
{
  if (!dead)
    {
      if (instruction == I_ADC) dirty_error_flag(FALSE);
      label_ref(instruction, label);
      /*{{{  debugging*/
      if (assembly_output)
        {
          diag_write_nl();
          fprintf(outfile, "%-8sL%d", primaryname(instruction), label);
        }
      /*}}}*/
      if (instruction == I_J) markdeadcode();
    }
}
/*}}}*/
/*{{{  PUBLIC void genjump*/
PUBLIC void genjump(const int label)
/* This is used to insert a jump. If code style flag is set,
   creates a non-timeslicable jump using ldc 0; cj
*/
{
  if (code_style_flags & CODE_STYLE_CJ_NOT_J)
    {
      genprimary(I_LDC, 0);
      genbranch(I_CJ, label);
    }
  else
    genbranch(I_J, label);
}
/*}}}*/
/*{{{  PUBLIC void genlabeldiff (instruction, label1, label2)*/
/***************************************************************************
 *
 *  genlabeldiff adds a record (representing a primary 'instruction' with
 *               operand the offset between 'label1' and 'label2')
 *               to the code buffer.
 *
 **************************************************************************/
PUBLIC void genlabeldiff (const int instruction, const int label1, const int label2 )
{
  if (!dead)
    {
      if (instruction == I_ADC) dirty_error_flag(FALSE);
      label_diff(instruction, label1, label2);
      if (assembly_output)
        {
          diag_write_nl();
          fprintf(outfile, "%-8sL%d - L%d",
                           primaryname(instruction), label1, label2);
        }
    }
}
/*}}}*/
/*{{{  PUBLIC void gensecondary (instruction)*/
/***************************************************************************
 *
 *  gensecondary adds the secondary 'instruction' to the code buffer.
 *
 **************************************************************************/
PUBLIC void gensecondary (int instruction )
{
  if (!dead)
    {
      /*{{{  check whether instruction may set error*/
      switch (instruction)
        {
          default:
            break;
          case I_CSUB0: case I_CCNT1: case I_SETERR:
          case I_ADD:   case I_SUB:   case I_MUL:   case I_DIV:  case I_REM:
          case I_CWORD: case I_CSNGL:
          case I_LADD:  case I_LSUB:  case I_LDIV:   case I_FMUL:
          case I_CFLERR:
          case I_FPCHKERR:
            dirty_error_flag(FALSE); break;
          case I_FPADD: case I_FPSUB: case I_FPMUL: case I_FPDIV: case I_FPREMFIRST:
          case I_FPLDNLADDSN:         case I_FPLDNLADDDB:
          case I_FPLDNLMULSN:         case I_FPLDNLMULDB:
          case I_FPGT:  case I_FPEQ:
          case I_FPINT: case I_FPRTOI32:
            dirty_error_flag(TRUE);  break;
        }
      /*}}}*/
      code_secondary(instruction);
      if (assembly_output)
        {
          diag_write_nl();
          if (instruction & I_STEP_BIT)
            {
               fputs("step ", outfile);
               instruction &= ~(I_STEP_BIT);
            }
          fprintf(outfile, "%-8s", secondaryname(instruction));
        }
      /*{{{  check if we have been timesliced, also for loading FPU values*/
      switch (instruction)
        {
          default:
            break;
          case I_OUTWORD: case I_OUTBYTE: case I_OUT:
          case I_IN: case I_TIN:
          case I_ALTWT: case I_TALTWT:
            genstartblock();
            break;
          /* We rely on the fact that after an FPU load, there is always
             a spare register, so an FPUCLRERR is OK.
             Note that regsfor gives 1 for CONSTEXP, so the loads
             of zeroes are OK too. */
          case I_FPLDNLSN: case I_FPLDNLDB: case I_FPLDNLSNI: case I_FPLDNLDBI:
          case I_FPLDNLADDSN: case I_FPLDNLMULSN:
          case I_FPLDNLADDDB: case I_FPLDNLMULDB:
          case I_FPLDZEROSN: case I_FPLDZERODB:
            clean_fpu_flag(I_FPUCLRERR);
            break;
        #if 0
          /* NO - we can't blatantly clobber stopp, because we use it in
             PRIPAR and in RESCHEDULE, so we do this in the STOP code */
          case I_STOPP:  /* added for bug 1057 27/11/90 */
            markdeadcode();
            break;
        #endif
          case I_SETERR: /* added for bug 1057 27/11/90 */
            if (errormode & ERRORMODE_HALT) markdeadcode();
            break;
        }
      /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC void checkerror ()*/
PUBLIC void checkerror ( void )
{
  if (inside_asm) return;

  if (NEED_ERRORS && !H1L_process)
    {
      if (fpu_error_status == ERROR_STATUS_DIRTY)
        {
          /* This will automagically dirty the other flag */
          gensecondary(I_FPCHKERR);
          fpu_error_status = ERROR_STATUS_CLEAN;
        }
      if ((errormode & ERRORMODE_NEED_STOPERR) &&
          (int_error_status == ERROR_STATUS_DIRTY))
        {
          gensecondary(I_STOPERR);
          int_error_status = ERROR_STATUS_CLEAN;
        }
    }
}

/*}}}*/
/*{{{  PUBLIC void setlab (label)*/
/***************************************************************************
 *
 *  setlab adds a record representing the setting of label 'label' to the
 *         code buffer.
 *
 **************************************************************************/
PUBLIC void setlab (const int label )
{
  dead = FALSE;
  label_def(label, C_SETLAB);
  if (assembly_output)
    fprintf(outfile, "\nL%d:", label);
}
/*}}}*/
/*{{{  PUBLIC void setsectionlab (label)*/
/***************************************************************************
 *
 *  setsectionlab adds a record representing the setting of label 'label'
 *                to the code buffer, and marks this label as indicating
 *                the start of the current code section (routine, table...)
 *
 **************************************************************************/
PUBLIC void setsectionlab (const int label )
{
  dead = FALSE;
  section_label_def(label, C_SETLAB);
  if (assembly_output)
    {
      fprintf(outfile, "\nL%d:", label);
      gencomment0("section label");
    }
}
/*}}}*/
/*{{{  PUBLIC void commentexp (tptr)*/
/***************************************************************************
 *
 *  commentexp writes the expression 'tptr' as a comment to the debugging
 *             file.
 *
 **************************************************************************/
PUBLIC void commentexp ( treenode *tptr )
{
  if (!source_output)
  {
    diag_write_nl_comment();
    printexp (tptr);
  }
}
/*}}}*/
/*{{{  PUBLIC void add_entry_point(nptr)*/
/***************************************************************************
 *
 *  add_entry_point adds the entry point of the current code section
 *                  (given by the current section_label)
 *                  to the list of entry points.
 *
 **************************************************************************/
PUBLIC void add_entry_point ( treenode *nptr )
{
  seginfo_t *seg =
 (seginfo_t *)newvec(sizeof(seginfo_t));
  seg->seg_next  = seginfo;
  seg->seg_name  = nptr;
  seg->seg_entry = section_label; /* Pointer (ie. subscript in code buffer)
                                     to current section_label, this will be
                                     changed to offset from start of module
                                     when code is output. */
  seginfo = seg;
}
/*}}}*/
/*{{{  PUBLIC void add_code_block(l, p)*/
/***************************************************************************
 *
 *  add_code_block adds l bytes of code, pointed to by p, to the code
 *                 buffer.
 *
 **************************************************************************/
PUBLIC void add_code_block ( const int l , const BYTE *const p )
{
  int i;
  for (i = 0; i < l; i++)
    add_code(p[i]);
  if (assembly_output)
    /*{{{  print the block to outfile*/
    {
      for (i = 0; i < l; i++)
        {
          if ((i & 0xF) == 0) diag_write_nl_string(".BYTE");
          /*fprintf (outfile, "#%-3x", p[i]);*/
          fprintf (outfile, " #%.2X", p[i]);
        }
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int get_padlen()*/
PUBLIC int get_padlen(const int len, const int alignment)
{
  /*return (alignment - (len % alignment)) % alignment;*/
  if (alignment > 1)
    return (alignment - (len & (alignment-1))) & (alignment-1);
  else
    return 0;
}
/*}}}*/
/*{{{  PRIVATE void subwordalign*/
PRIVATE void subwordalign(const int padlen, const BYTE *const padbytes)
{
  if (padlen > 0)
    {
      add_code_block(padlen, padbytes);
      gencomment1("add %d bytes for alignment", padlen);
    }
}
/*}}}*/
/*{{{  PUBLIC void genpadding*/
PUBLIC void genpadding(const int len)
{
  static BYTE padding[4] = { 0, 0, 0, 0 };
  assert(len <= 4);
  subwordalign(len, padding);
}
/*}}}*/
/*{{{  PUBLIC int endofsection_padlen*/
PUBLIC int endofsection_padlen(const int alignment)
/* This returns the number of bytes which have to be added to the previous
   section to make it end on that alignment
*/
{
  return get_padlen(bytes_output, alignment);
}
/*}}}*/
/*{{{  PUBLIC void alignendofsection()*/
/***************************************************************************
 *
 *  alignendofsection ensures that the last byte of the
 *               current code section is word aligned.
 *               It must be called after the code has been put in the
 *               code buffer, but before it is compressed.
 *
 *               In effect, it ensures that the end of the previous section
 *               is on a word boundary.
 *
 **************************************************************************/
PUBLIC void alignendofsection (const int alignment )
{
  genpadding(endofsection_padlen(alignment));
}
/*}}}*/
/*{{{  PUBLIC void alignwholemodule()*/
PUBLIC void alignwholemodule(void)
/* This is used at the end of the object file to ensure that the whole
   object file is word aligned
*/
{
  const int padlen = endofsection_padlen(bytesperword);
  if (padlen > 0)
    {
      const int l = newlab();
      setsectionlab(l);
      /*alignendofsection(bytesperword);*/ /* bug 1066 11/12/90 */
      subwordalign(padlen, libstub);
      compress_code();
    }
}
/*}}}*/
/*{{{  PUBLIC int genconstant(cptr)*/
/***************************************************************************
 *
 *  genconstant adds bytes representing the constant cptr to the code
 *              buffer (least significant byte first). Returns the
 *              number of WORDS written to the code.
 *
 **************************************************************************/
PUBLIC int genconstant ( treenode *cptr )
{
  BYTE v[MAX_CONSTANT_SIZE];
  BIT32 l;
  int i;
  INT32 w = wordsin(gettype(cptr));
  /*{{{  put the code bytes representing cptr into v*/
  l = LoValOf(cptr);
  for (i = 0; i < 4; i++)
    {
      v[i] = l & 0xff;
      l >>= 8;
    }
  l = HiValOf(cptr);
  for (i = 4; i < 8; i++)
    {
      v[i] = l & 0xff;
      l >>= 8;
    }
  /*}}}*/
  add_code_block(w * bytesperword, v);
  return(w);
}
/*}}}*/
/*{{{  PUBLIC void genstartjumptable ()*/
/***************************************************************************
 *
 *  genstartjumptable adds a record to the code buffer marking the start
 *                    of a jump table.
 *
 **************************************************************************/
PUBLIC void genstartjumptable ( void )
{
  if (!dead)
    {
      table_start();
      if (assembly_output) diag_write_nl_string("-- Jump table");
    }
}
/*}}}*/
/*{{{  PUBLIC void genjumptableentry (instruction, label)*/
/***************************************************************************
 *
 *  genjumptableentry adds a record representing a jump table entry
 *                    to the code buffer.
 *
 **************************************************************************/
PUBLIC void genjumptableentry (const int instruction, const int label )
{
  if (!dead)
    {
      genbranch(instruction, label);
      dead = FALSE;
    }
}
/*}}}*/
/*{{{  PUBLIC void genendjumptable ()*/
/***************************************************************************
 *
 *  genendjumptable adds a record to the code buffer marking the end
 *                  of a jump table.
 *
 **************************************************************************/
PUBLIC void genendjumptable ( void )
{
  if (!dead)
    {
      table_end();
      if (assembly_output) diag_write_nl_string("-- End of jump table");
    }
}
/*}}}*/
/*{{{  PUBLIC void gencasescale (instruction)*/
/***************************************************************************
 *
 *  gencasescale adds a record representing a primary 'instruction'
 *               to the code buffer, whose operand is the size
 *               in bytes of a forthcoming jump table.
 *
 **************************************************************************/
PUBLIC void gencasescale (const int instruction )
{
  if (!dead)
    {
      scale_ref(instruction);
      if (assembly_output) diag_write_nl_string("ldc     case.scale");
    }
}
/*}}}*/
/*{{{  PUBLIC void genfpuentry (int instruction)*/
/*****************************************************************************
 *
 *  genfpuentry generates an fpuentry instruction
 *
 *****************************************************************************/
PUBLIC void genfpuentry (const int instruction )
{
  if (!dead)
    {
      /*{{{  check whether instruction may set error*/
      switch (instruction)
        {
          default:
            break;
          case I_FPUSQRTFIRST:
          case I_FPUMULBY2:   case I_FPUDIVBY2:
          case I_FPUEXPINC32: case I_FPUEXPDEC32:
          case I_FPUABS:
          case I_FPUSETERR:
          case I_FPUR32TOR64: case I_FPUR64TOR32:
          case I_FPUCHKI32: case I_FPUCHKI64:
            dirty_error_flag(TRUE);
            break;
        }
      /*}}}*/
      /*code_primary(I_LDC, instruction & 0xff);*/
      code_primary(I_LDC, instruction & ~I_FPU_ENTRY_BIT);
      code_secondary(I_FPENTRY);
      if (assembly_output)
        {
          diag_write_nl();
          fprintf(outfile, "%-8s", secondaryname(instruction));
        }
    }
}
/*}}}*/
/*{{{  PUBLIC void declarelabel(label, offset)*/
/***************************************************************************
 *
 *  declarelabel declares the label 'label' at offset 'offset' from the
 *               start of code.
 *
 **************************************************************************/
PUBLIC void declarelabel (const int label, const BIT32 offset )
{
  int lptr = addlabel(label);
  code[lptr + LB_PRAD] = offset;
}
/*}}}*/
/*}}}*/
/*{{{  run time locate information*/
/*{{{  PUBLIC void genlocate (address) - takes treenode address as its param*/
/*****************************************************************************
 *
 *  genlocate inserts a runtime locate item record into the code buffer
 *
 *****************************************************************************/
PUBLIC void genlocate ( treenode *address )
{
  add_debug_info(C_LOCATE, address, 0);
  /*{{{  debugging*/
  if (diagnostics)
    {
      diag_write_nl_string("-- Locate");
      DEBUG_MSG((" address: %lX", (BIT32)address));
    }
  /*}}}*/
  /*{{{  source output*/
  if (source_output)
    output_source_code(get_from_index_table(address), C_LOCATE);
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void genaddrfix (address) - takes treenode address as its param*/
/*****************************************************************************
 *
 *  genaddrfix inserts an addressfix item record into the code buffer
 *
 *****************************************************************************/
PUBLIC void genaddrfix ( treenode *address, const int label )
{
  add_debug_info(C_ADDRESSFIX, address, label);
  /*{{{  debugging*/
  if (diagnostics)
    diag_write_nl_string("-- Addressfix");
  /*}}}*/
  /*{{{  source output*/
  if (source_output)
    output_source_code(get_from_index_table(address), C_ADDRESSFIX);
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void genlibrpatch ()*/
/*****************************************************************************
 *
 *  genlibrpatch inserts a libpatch item record into the code buffer
 *
 *****************************************************************************/
PUBLIC void genlibrpatch ( void )
{
  add_debug_info(C_LIBPATCH, 0, 0);
  /*{{{  debugging*/
  if (diagnostics)
    diag_write_nl_string("-- Libpatch");
  /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  library handling*/
/*{{{  PUBLIC void add_to_libentries(nptr)*/
/*****************************************************************************
 *
 *  add_to_libentries adds the library entrypoint 'nptr' to the list of used
 *                    library entry points if it is not aready there.
 *
 *****************************************************************************/
PUBLIC void add_to_libentries ( treenode *nptr )
{
  treenode *l = libentrychain;
  while (l != NULL)
    {
      if (l == nptr)
        return;
      l = NLEntryNextOf(l);
    }
  SetNLEntryNext(nptr, libentrychain);
  libentrychain = nptr;                   /* Add to the entry point list */
}
/*}}}*/
/*{{{  PUBLIC void genlibstubs()*/
/*****************************************************************************
 *
 *  genlibstubs adds a stub for each library call to the code buffer
 *
 *****************************************************************************/
PUBLIC void genlibstubs ( void )
{
  treenode *libentry;
  for (libentry = libentrychain; libentry != NULL;
       libentry = NLEntryNextOf(libentry))
    {
      int l = newlab();
      SetNPLabel(libentry, l);          /* give the entry point a label */
      setsectionlab(l);
      gencomment1("Stub for library call %s", (BIT32)WNameOf(NNameOf(libentry)));
      /*{{{  debugging*/
      if (debugoutput)
        genlibrpatch();
      /*}}}*/
      add_code_block(libpatchsize > sizeof(libstub) ? sizeof(libstub) : libpatchsize,
                     libstub);
      compress_code();  /* this must be done once for each library stub */
    }
}
/*}}}*/
/*}}}*/

/*{{{  PUBLIC void write_object_file()*/
/*****************************************************************************
 *
 * write_object_file writes out the object file and closes it
 *
 *****************************************************************************/
PUBLIC void write_object_file ( void )
{
  calculate_code_size();

  if (assembly_output)
    fprintf(outfile, "%sModule code size is %ld bytes",
                     diagnostics ? "" : "\n-- ", total_code_size);

  patch_lib_offsets();
  if (!disassemble && !assembly_output)
    {
      output_code();
      if (debugoutput) flush_debug_buffers();

      write_library_calls(libentrychain, I_J);
      output_entry_points(1);
      output_code_size();
      output_entry_points(2);

#ifdef CONFIG
      config_link_map();  /* write the linker map into the file */
#endif

      /*  This is a nop if creating a 3l format output file.
          Otherwise it finishes off the file nicely! */
      write_end_module ();
    }
  else if (disassemble)
    {
      output_code(); /* disassemble code buffer contents to outfile */
      output_entry_points(1);  /* pass 1 */
    }
}
/*}}}*/
/*{{{  PUBLIC void close_object_file()*/
/*{{{  comment*/
/*****************************************************************************
*
*  close the object file - called by the harness
*
******************************************************************************/
/*}}}*/
PUBLIC void close_object_file (FILE *fptr, const char *const filename)
{
  if (fclose(fptr) == EOF)
    generr_s(GEN_CANNOT_CLOSE_OBJECT_FILE, filename);
}
/*}}}*/
/*{{{  PUBLIC FILE *open_object_file()*/
/*****************************************************************************
 *
 * open_object_file opens the object file for writing
 *
 *****************************************************************************/
PUBLIC FILE *open_object_file (const char *const filename)
{
  FILE *fptr;
#ifdef DEC
  fptr = fopen(filename, "wb"); /* The C compiler uses this! */
  /*fptr = fopen(filename, "w", "rfm = udf");*/
#else
  fptr = fopen(filename, "wb");
#endif
  if (fptr == NULL)
    generr_s(GEN_CANNOT_OPEN_OBJECT_FILE, filename);
  return fptr;
}
/*}}}*/

/*{{{  PUBLIC void initcode()*/
/***************************************************************************
 *
 * initcode initialises the coder.
 *
 **************************************************************************/
PUBLIC void initcode ( void )
{
  if (req_code_size != 0)
    {
      /* we multiply by 1126 (1024 + 10%) so that in effect we add 10%
         to the buffer size.
         (requested in K, so multiply by 1024 * 1.1)
         This allows for the other crap which is used in the code buffer */
      /* bug 1021 - 16/10/90 */
        max_code_size = req_code_size * (1126 / sizeof(INT32));
    }
  if (code == NULL) code = (INT32 *)memalloc(max_code_size * sizeof(INT32));
  cstart = 0;
  cptr = 0;
  lptr = max_code_size;
  start = C_NIL;
  topl  = C_NIL;
  codew = 0;
  coden = 0;
  partword = 0;
  address  = 0;
  case_scale = 0;
  seginfo = NULL;
  bytes_output = ZERO32;
  /* debugnamesptr = 0; */
  libentrychain = NULL;
  dead = FALSE;
  int_error_status = ERROR_STATUS_CLEAN;
  fpu_error_status = ERROR_STATUS_FP_INIT;
}
/*}}}*/
#ifdef CONFIG
/*{{{  PUBLIC void freecode*/
PUBLIC void freecode(void)
{
  if (code != NULL)
    {
      memfree(code);
      code = NULL;
    }
}
/*}}}*/
#endif
/*}}}*/

