/*#define DEBUG*/
/*****************************************************************************
 *
 *  bind1 - workspace allocation
 *
 *****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "extlib.h"

# include "generror.h"
# include "genhdr.h"
# include "instruct.h"
# include "instdef.h"
# include "lex1def.h"
# include "desc1def.h"
# include "syndef.h"
# include "syn1def.h"
# include "chkdef.h"
# include "predefhd.h"
# include "trandef.h"
# include "bind1def.h"
# include "bind2def.h"
# include "bind3def.h"
# include "gen1def.h"
# include "gen2def.h"
# include "gen8def.h"
# include "gen9def.h"
# include "gen10def.h"
# include "gen11def.h"
# include "gen12def.h"
/*}}}*/

/*{{{  constants*/
#define VAR_USAGE_PER_LOOP ((INT32)8)
/* Average number of times loops are executed; see Roger about this value! */

/* This is used to optimise VALOFs which expand into constants. */
#define OPTIMISE_VALOFS TRUE
/*}}}*/

/*{{{  public variables*/
/* Global variables used by mapvariant */
PUBLIC int globchannelmode;
PUBLIC treenode **globchannelptr;
PUBLIC treenode *constantnptr = NULL;
PUBLIC treenode *consttablechain;

/* This flag is set while inside ASM, and inside the rhs of a RETYPE.
   If a channel is encountered while it is set, it is known that we can't
   perform fancy optimisations on that channel.
*/
PUBLIC int flag_scalar_chans = FALSE;

PUBLIC treenode *templist;
/*}}}*/

/*{{{  PRIVATE variables*/
typedef struct libentry_s
  {
    struct libentry_s *l_next;
    treenode          *l_name;
  } libentry_t;

PRIVATE libentry_t *stdlibentries = NULL;
PRIVATE int stdlibsloaded = FALSE;

#ifdef VIRTUALCHANS
  PRIVATE libentry_t *vlibentries = NULL;
  PRIVATE int vlibsloaded = FALSE;
#endif

PRIVATE treenode *namelist;

/* PRIVATE treenode *scopelist;*/ /* Not used*/
/*}}}*/

/*{{{  routines*/
/*{{{  std library loading*/
/*{{{  PRIVATE treenode *maplibtree(libtree)*/
PRIVATE treenode *maplibtree ( treenode *libtree )
{
  if (libtree != NULL)
    {
      int olddisassemble = disassemble, oldassembly_output = assembly_output;
      int olddiagnostics = diagnostics;
      disassemble = FALSE; assembly_output = FALSE; diagnostics = FALSE;
      transmain(libtree);
      mapnestedblocks(libtree);
      disassemble = olddisassemble; assembly_output = oldassembly_output;
      diagnostics = olddiagnostics;
    }
  return libtree;
}
/*}}}*/
/*{{{  PRIVATE treenode *getstdlib*/
PRIVATE treenode *getstdlib(wordnode *libcallstr, SOURCEPOSN locn,
        int *enabled,
        int *loaded, char *filename, libentry_t **libentries)
/*****************************************************************************
 *
 *  getstdlib returns a pointer to the name node representing extended type
 *           function call 'libcallstr'.
 * Used to load both the standard library, and the virtual i/o library
 *
 *****************************************************************************/
{
  libentry_t *entry;
  if (!(*enabled))
    {
      genlocn = locn;
      generr_s(GEN_STDLIBS_NOT_ENABLED, WNameOf(libcallstr));
      /* This is a fatal error, so never returns */
    }
  if (!(*loaded))
    /*{{{  load in standard libraries, set up libentries*/
    {
      treenode *libtree = maplibtree(loadstdlib(filename));
      treenode *s;
      for (s = libtree; s != NULL && TagOf(s) != S_END; s = DBodyOf(s))
        {
          libentry_t *this = (libentry_t *)newvec(sizeof(libentry_t));
          this->l_next = *libentries;
          this->l_name = DNameOf(s);
          *libentries = this;
        }
      *loaded = TRUE;
    }
    /*}}}*/

  entry = *libentries;
  while ((entry != NULL) && (NNameOf(entry->l_name) != libcallstr))
    entry = entry->l_next;

  if (entry == NULL)
    {
      genlocn = locn;
      generr_s(GEN_MISSING_STDLIB_ENTRY, WNameOf(libcallstr));
      /* this is a fatal error */
    }

  return (entry->l_name);
}
/*}}}*/
#ifdef VIRTUALCHANS
/*{{{  PUBLIC treenode *vlibentry(libcallstr)*/
/*****************************************************************************
 *
 *  vlibentry returns a pointer to the name node representing extended type
 *           function call 'libcallstr'.
 *
 *****************************************************************************/
PUBLIC treenode *vlibentry ( wordnode *libcallstr )
{
  return getstdlib(libcallstr, genlocn,
                   &vlibsenabled, &vlibsloaded, vlibsfilename, &vlibentries);
}
/*}}}*/
#endif
/*{{{  PUBLIC treenode *libentry (libcallstr)*/
/*****************************************************************************
 *
 *  libentry returns a pointer to the name node representing extended type
 *           function call 'libcallstr'.
 *
 *****************************************************************************/
PUBLIC treenode *libentry (wordnode *libcallstr, SOURCEPOSN locn)
{
  return getstdlib(libcallstr, locn,
                   &stdlibsenabled, &stdlibsloaded, extlibfilename, &stdlibentries);
}
/*}}}*/
/*}}}*/
/*{{{  PRIVATE treenode *transformalt(spec, alt)*/
/*****************************************************************************
 *
 *  transformalt takes a list of specifications, spec, preceding an
 *               ALTERNATIVE, alt, and attempts to move some or all of the
 *               specifications inside the alt input.  The updated
 *               tree is returned.
 *               N.B. The specification list is terminated by alt.
 *
 *****************************************************************************/
PRIVATE treenode *transformalt ( treenode *spec , treenode *alt )
{
  treenode *result = spec; /* assume we return what we were given unless we */
                           /* find out otherwise */
  if (spec != alt)  /* There are specifications */
    /*{{{  see if we can move this specification to the input*/
    {
      treenode *n = DNameOf(spec);
      treenode *input = AltInputOf(alt);  /* take a copy before we move the specs */
      SetDBody(spec, transformalt(DBodyOf(spec), alt));
      switch(TagOf(n))
        {
          case N_ABBR:   case N_VALABBR:
          case N_RETYPE: case N_VALRETYPE:
          case N_DECL:
            /*{{{  see if it can be moved*/
            {
              treenode *next = DBodyOf(spec);
              int moveable  = !usedin(n, AltGuardOf(alt));
              int inputtype = inputtypeof(input);
              while (moveable && (next != alt))
                {  /* check if this spec is used in a following spec */
                  switch (TagOf(DNameOf(next)))
                    {
                      default:
                        break;
                      case N_ABBR: case N_VALABBR: case N_DECL:
                      case N_RETYPE: case N_VALRETYPE:
                        moveable = !usedin(n, DValOf(next));
                        break;
                    }
                  next = DBodyOf(next);
                }
              switch(inputtype)
                {
                  case INP_SKIP: case INP_DELAYED_INPUT:
                    break;
                  default:  /* any channel or port input */
                  #ifdef DEBUG
                    fprintf(outfile, "transformalt: spec: %s alternative is:", WNameOf(NNameOf(n)));
                    printtree(0, LHSOf(input));
                    fputc('\n', outfile);
                  #endif
                    moveable = moveable && !usedin(n, LHSOf(input)); /* channel */
                    break;
                }
              if (moveable)
                /*{{{  transform*/
                {
                  DEBUG_MSG(("transformalt: moving %s\n", WNameOf(NNameOf(n))));
                  result = DBodyOf(spec);
                  SetDBody(spec, AltInputOf(alt));
                  SetAltInput(alt, spec);
                }
                /*}}}*/
            }
            /*}}}*/
            break;
          default:
            break;
        }
    }
    /*}}}*/
  return(result);
}
/*}}}*/
/*{{{  PRIVATE int addstaticandvec(nptr)*/
PRIVATE int addstaticandvec ( treenode *nptr, INT32 vs )
{
  treenode *fparams = NParamListOf(nptr);
  int nparams = 0;

  if (vs != ZERO32)
    /*{{{  add the vector space pointer*/
    {
      /*treenode *vsnode = newleafnode(S_PARAM_VSP, NOPOSN);*/
      /* bug 1141 4/2/91 - make sure VSP is always a hiddenparamnode */
      treenode *vsnode = newhiddenparamnode(S_PARAM_VSP, NOPOSN, NULL, 0);
      if (NLexLevelOf(nptr) == initlexlevel || separatelycompiled(nptr))
        /* At the outer level, put the vector space pointer at the end
           of the parameters, to preserve compatibility with existing
           toolset SCs and libraries. */
        fparams = appendnode(vsnode, fparams);
      else
        fparams = addtofront(vsnode, fparams);
    }
    /*}}}*/

  if (NPSLUsageOf(nptr) != STATICLINK_NOT_USED) /* set up ok for libraries */
    fparams = addtofront(newleafnode(S_PARAM_STATICLINK, 0), fparams);

  /* Update param list on definition */
  SetNParamList(nptr, fparams);

  /* Count Number of parameters */
  while (!EndOfList(fparams))
    {
      /* Don't count TIMER parameterse */
      if (basetype(gettype(ThisItem(fparams))) != S_TIMER)
        nparams++;
      fparams = NextItem(fparams);
    }
  return(nparams);
}
/*}}}*/
/*{{{  PRIVATE void mapname(nptr)*/
/*****************************************************************************
 *
 *  mapname gives a workspace allocation number to a name node, and reserves
 *          an appropriate number of workspace slots for that name.
 *          Sets the name mode, if not already set.
 *
 *****************************************************************************/
PRIVATE void mapname ( treenode *nptr )
{
  switch (TagOf(nptr))
    {
      /*{{{  N_DECL*/
      case N_DECL:
        if (isplaced(nptr))
          /*{{{  allocate absolutely*/
          {
            BIT32 placeval = NVOffsetOf(nptr);
            int error;
            BIT32 plhi, pllo;
            BIT32 mostneg = (targetintsize == S_INT32) ? MOSTNEG_INT32 : MOSTNEG_INT16;
          
            I32ToI64(&plhi, &pllo, placeval);
            DEBUG_MSG(("mapname: PLACED at hi:%lu, lo%lu\n", plhi, pllo));

            if (iobycall && (plhi == 0) && (pllo < (BIT32)(NO_OF_LINKS * 2)))
              genwarn_s(GEN_PLACE_ON_LINKS, WNameOf(NNameOf(nptr)));
          
            Int64Mul(&error, &plhi, &pllo, plhi, pllo, ZERO32, bytesperword);
            Int64Add(&error, &plhi, &pllo, plhi, pllo, 0xffffffffl, mostneg);
          
            if (checkbounds(S_INT64, targetintsize, plhi, pllo) != 0)
              genwarn_s(GEN_PLACE_OVERFLOW, WNameOf(NNameOf(nptr)));

            SetNVOffset(nptr, pllo);
          }
          /*}}}*/
        else 
          /*{{{  in workspace or vectorspace*/
          {
            if (iswsplaced(nptr)) /* allocate at fixed point in workspace */
              alloc_fixedws(NVOffsetOf(nptr), allocsize(nptr));
            else
              create_var(nptr);
            create_var_subsidiary(nptr);  /* create the vectorspace and initialiser */
          }
          /*}}}*/
        break;
      /*}}}*/
      /*{{{  N_REPL*/
      case N_REPL:
        /* Replicator occupies two words, one for base, one for count */
        create_var(nptr);
        break;
      /*}}}*/
      default:
        badtag(genlocn, TagOf(nptr), "mapname");
    }
}
/*}}}*/
/*{{{  PRIVATE void mapsingledecl(tptr)*/
PRIVATE void mapsingledecl ( treenode *tptr )
{
#ifdef DEBUG
  {
    treenode *name = DNameOf(tptr);
    if (TagOf(name) == S_LIST)
      name = ThisItem(name);
    DEBUG_MSG(("mapsingledecl: mapping %s\n", WNameOf(NNameOf(name))));
  }
#endif
  switch (TagOf(tptr))
    {
      /*{{{  S_DECL*/
      case S_DECL:
        /*{{{  map each of the variables in the declaration list*/
        /* Map each of the variables in the declaration list */
        {
          treenode *t = DNameOf(tptr);
          if (TagOf(t) == S_LIST)
            while (!EndOfList(t))
              {
                mapname(ThisItem(t));
                t = NextItem(t);
              }
          else
            mapname (t);
        }
        /*}}}*/
        break;
      /*}}}*/
      /*{{{  S_VALABBR S_ABBR S_VALRETYPE S_RETYPE*/
      case S_VALABBR: case S_ABBR: case S_VALRETYPE: case S_RETYPE:
        {
          treenode *nptr = DNameOf(tptr);
          const int retype = (TagOf(nptr) == N_VALRETYPE) || (TagOf(nptr) == N_RETYPE);
          treenode *rhs = DValOf(tptr);
          const int am = abbrevmode(tptr);
          int type = typeof(nptr);
          const int saved_flag = flag_scalar_chans;
          flag_scalar_chans = retype; /* check for chans on rhs of a RETYPE */
          if (retype)
            /*{{{  bring hidden dimension temporaries into scope*/
            {
              treenode *type = NTypeOf(nptr);
              while (TagOf(type) == S_ARRAY)
                {
                  if (TagOf(ARDimLengthOf(type)) == T_TEMP)
                    {
                      treenode *temp = ARDimLengthOf(type);
                      DEBUG_MSG(("Allocating temp for hidden dim of %s, current lexlevel is %d\n", WNameOf(NNameOf(nptr)), lexlevel));
                      DEBUG_MSG(("vars lexlevel: %d, temp lexlevel:%d\n", NLexLevelOf(nptr), NLexLevelOf(temp)));

                    #if 1
                      alloctemp(temp);
                      addtemplist(temp);
                    #else
                      create_var(temp);
                    #endif
                    }
                  type = ARTypeOf(type);
                }

              /* now map the hidden length (bug 836 19/12/90) */
              {
                treenode *rhstype = gettype(rhs);
                treenode *t;
                int opendim = FALSE;
                for (t = rhstype; TagOf(t) == S_ARRAY; t = ARTypeOf(t))
                  if (ARDimOf(t) == (-1))
                    opendim = TRUE;
                if (opendim)
                  {
                    treenode *dims = unknowndimsof(rhstype);
                    DEBUG_MSG(("mapsingledecl: mapping unknown dims of rhs of RETYPE\n"));
                    mapexp(&dims);
                  }
              }
            }
            /*}}}*/
          switch(am)
          {
          case AM_CONST:
            /*{{{  no space allocated, no offset*/
            SetNVOffset(nptr, NO_SLOT);
            break;
            /*}}}*/
          case AM_ISRHS:
            /*{{{  no space allocated, set offset to offset of rhs*/
            {
              /* Abbreviation inherits attributes of rhs */
              /* rhs is either a namenode, or an ARRAYITEM node with constant offset */
              /* Note that we don't do this if rhs is a segment */
              treenode *rhsname = nameof(rhs);

              SetNVOffset(nptr, (TagOf(rhs) == S_ARRAYITEM) ? ASOffsetOf(rhs) : 0);

              /* Bug 1135 31/1/91 if rhs is also an abbreviation, we must add in
                 the rhs's NVOffset.
              */
              /* Ditto if rhsname is PLACE'd AT WORKSPACE n, so nptr must be at
                 n + possible array offset */
              if (NVOffsetOf(rhsname) != NO_SLOT)
                {
                  DEBUG_MSG(("mapsingledecl: adding rhs offset %ld to is ISRHS abbreviation %s\n",
                             NVOffsetOf(rhsname), WNameOf(NNameOf(nptr))));
                  SetNVOffset(nptr, NVOffsetOf(nptr) + NVOffsetOf(rhsname));
                }

              if (!iswsplaced(rhsname))
                {
                  /* Chain this name onto rhsname: when rhsname is allocated, nptr will
                     be at the same place + possible array offset */
                  addnamechain(rhsname, nptr); /* Put the abbrev onto name chain of rhs */
                }
              if (retype && (TagOf(NTypeOf(rhsname)) == S_CHAN))
                {
                  DEBUG_MSG(("mapsingledecl: marking scalar channel %s\n", WNameOf(NNameOf(rhsname))));
                  SetNChanMark(rhsname, TRUE);
                }
            }
            break;
            /*}}}*/
          case AM_PTR:
            /*{{{  allocate space for a pointer*/
            create_var(nptr);
            mapexpopd(P_PTR, DValAddr(tptr)); /* Map the loading of a pointer */
            break;
            /*}}}*/
          default: /*case AM_VAL:*/
            /*{{{  allocate in workspace or vectorspace*/
            {
              treenode *lhstype = NTypeOf(nptr);
              int w = allocsize(nptr);
              if (w == (-1))
                geninternal_s(GEN_UNSIZED_ABBREV, WNameOf(NNameOf(nptr)));
              create_var(nptr);
              if (isinvectorspace(nptr)) /* Can never happen cos constructors are currently put in WS */
                SetNVVSOffset(nptr, newvs(w));
              if (retype)
                /* Pretend the lhs has the same type as the rhs so the mapping works
                   correctly. */
                {
                  treenode *rhstype = gettype(rhs);
                  type = TagOf(rhstype);
                  SetNType(nptr, rhstype);
                }
              /*{{{  map the initialisation*/
              mapsimpleassign(type, P_EXP, DNameAddr(tptr), P_EXP, DValAddr(tptr));
              /*}}}*/
              SetNType(nptr, lhstype);
            }
            break;
            /*}}}*/
          }
          flag_scalar_chans = saved_flag;
          DEBUG_MSG(("mapsingledecl: %s : abbrevmode is %d, mode is %d\n",
                   WNameOf(NNameOf(nptr)), am, NModeOf(nptr)));
        }
        break;
      /*}}}*/
      /*{{{  S_PROCDEF S_SFUNCDEF S_LFUNCDEF*/
      case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
        /* Workspace allocated already */
        break;
      /*}}}*/
      /*{{{  S_TPROTDEF S_SPROTDEF S_PLACE S_WSPLACE S_VSPLACE*/
      case S_TPROTDEF: case S_SPROTDEF: case S_PLACE: case S_WSPLACE: case S_VSPLACE:
        break;
      /*}}}*/
      default:
        badtag(genlocn, TagOf(tptr), "mapsingledecl");
    }
}
/*}}}*/
/*{{{  PUBLIC void mapdeclandbody(tptr, mbody)*/
PUBLIC void mapdeclandbody ( treenode *tptr , void (*mbody )PARMS ((treenode *)),
                             int preprocess_first, int preprocess_rest)
{
  BIT32 savevsp = vsp;
  int *savelive_var = save_scope();
  int not_first_time = FALSE;
  DEBUG_MSG(("mapdeclandbody: saving scope as: %lX\n", (long)savelive_var));
  while (isspecification(tptr))
    {
      genlocn = LocnOf(tptr);
      if (preprocess_rest && (not_first_time || preprocess_first))
        mappreprocess(tptr); /* bug 776 5/10/90 */
      not_first_time = TRUE;
      mapsingledecl(tptr);                /* Allocate for this declaration */
      tptr = DBodyOf(tptr);
    }
  genlocn = LocnOf(tptr);
  (*mbody)(tptr);
  DEBUG_MSG(("mapdeclandbody: restoring scope to: %lX\n", (long)savelive_var));
  restore_scope(savelive_var);
  vsp = savevsp;
}
/*}}}*/
/*{{{  construct mapping*/
/*{{{  PUBLIC void addtemplist(tptr)*/
PUBLIC void addtemplist ( treenode *tptr )
{
  DEBUG_MSG(("addtemplist: adding temporary %lX\n", (long)tptr));
  SetNVNextTemp(tptr, templist);
  templist = tptr;
}
/*}}}*/
/*{{{  PUBLIC void freetemplist(tptr)*/
PUBLIC void freetemplist ( treenode *tptr )
{
  while (tptr != NULL)
    {
      DEBUG_MSG(("freetemplist: freeing temporary %lX\n", (long)tptr));
      if (TagOf(tptr) == T_PREEVALTEMP)
        SetTag(tptr, T_TEMP);
      freetemp(tptr);
      tptr = NVNextTempOf(tptr);
    }
}
/*}}}*/
/*{{{  PUBLIC void mapconstruction(tptr, mapproc)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  mapconstruction maps out the workspace for a list of processes
 *                  which are NOT performed in parallel, ie. a list of
 *                  processes in sequence, a list of choices, alternatives
 *                  or selections.  'mapproc' is called to map each item
 *                  of the list.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void mapconstruction ( treenode *tptr , void (*mapproc )(treenode *))
{
  while (!EndOfList(tptr))
    /*{{{  map each item on the list*/
    {
      treenode *thisbranch = ThisItem(tptr);
      (*mapproc)(thisbranch);

      /*{{{  COMMENT */
      /**********************  Start comment out ****************************
      @*{{{  *@
      {
        treenode *spptr;
        (*mapproc)(thisbranch);
      @* Note that the fields maxvsp are not used in this
         space usage node: they are only used in PROC headers and repl PAR
         space usage nodes. *@
        spptr = newspacenode(S_SPACEUSAGE, 0, thisbranch, ZERO32,
                              ZERO32, ZERO32, ZERO32,
                              NULL, 0);
        NewItem(spptr, tptr);
      }
      @*}}}*@
       **********************   End comment out  ****************************/
      /*}}}*/

      tptr = NextItem(tptr);
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void maprepl(tptr, mbody)*/
/*****************************************************************************
 *
 *  maprepl maps a replicator node, tptr: the replicator body is mapped
 *          using the function parameter mbody.
 *
 *****************************************************************************/
PUBLIC void maprepl ( treenode *tptr , void (*mbody )(treenode *))
  {
    INT32 oldweight = loop_weight;
    mapname(ReplCNameOf(tptr));
    mapexp(ReplCStartExpAddr(tptr));
    mapexp(ReplCLengthExpAddr(tptr));
    if (isconst(ReplCLengthExpOf(tptr)))
      uploop_weight(LoValOf(ReplCLengthExpOf(tptr)));
    else
      uploop_weight(VAR_USAGE_PER_LOOP);
    /* We have to access the loop variable to test for end of loop */
    SetNVUseCount(ReplCNameOf(tptr), loop_weight);
    (*mbody)(ReplCBodyOf(tptr));
    loop_weight = oldweight;
    kill_var(ReplCNameOf(tptr));
  }
/*}}}*/
/*{{{  PRIVATE void mapreplpar(tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  mapreplpar allocates workspace for a replicated PAR contruct, tptr.
 *             The space usage is placed in a spacenode inserted in front
 *             of the replicated process.
 *
 *****************************************************************************/
/*}}}*/
PRIVATE void mapreplpar ( treenode *tptr )
{
  INT32 saveddatasize, savedvsp, savedmaxvsp;
  INT32 replcount = LoValOf(ReplCLengthExpOf(tptr));
  INT32 parspace, parvspace;
  INT32 maxwsp;
  INT32 savedweight;
  INT32 savedusecount;
  treenode *savednamelist;
  int savedvar_base;
  treenode *pprocess = ReplCBodyOf(tptr);

  /*{{{  debug message*/
  if (diagnostics)
    fprintf (outfile,
             "\nAllocating workspace for replicated parallel process\n");
  /*}}}*/
  /*{{{  save old environment*/
  saveddatasize = datasize;
  savedvsp = vsp; savedmaxvsp = maxvsp;
  savedvar_base = var_base;
  savedweight = loop_weight;
  savedusecount = NVUseCountOf(constantnptr);
  savednamelist = namelist;
  /*}}}*/
  /*{{{  initialise new environment*/
  initvsp(0);                       /* Initialise vector space usage to zero */
  datasize = DS_MIN;                /* Mimumum below workspace usage         */
  lexlevel++;
  var_base = num_var;
  uploop_weight(replcount);
  SetNVUseCount(constantnptr, ZERO32);
  namelist = NULL;
  /*}}}*/
  /*{{{  map replicated process*/
  {
    int rpspecials = MIN_REPLPAR_SPECIALS;
    treenode *spptr;
    mapprocess(pprocess);
    /* datasize contains our below workspace requirement for PAR branch */
    /* maxvsp gives the vector space usage for the PAR branch */
    /*{{{  check whether we need a constant pointer*/
    {
      treenode *cptr = consttablechain;
      while ((cptr != NULL) && (TagOf(cptr) != S_CONSTEXP))
        cptr = CTNextOf(cptr);

      if (cptr != NULL)
        create_immortalvar(constantnptr);
      else
        SetNVOffset(constantnptr, CONSTANTPOINTER_NOT_USED);
    }
    /*}}}*/
    maxwsp = allocvars(var_base, 0, TRUE);
    if (maxvsp != ZERO32)
      rpspecials++;                       /* Allocate a slot for the vsp */
    maxwsp = maxwsp + rpspecials;         /* Add special slots to maxwsp */
    spptr = newspacenode(S_SPACEUSAGE, 0, pprocess, maxwsp, maxwsp + datasize,
                          maxvsp, savedvsp, consttablechain,
                          NVOffsetOf(constantnptr));
    SetReplCBody(tptr, spptr);
  }
  /*}}}*/
  /*{{{  debug message*/
  if (diagnostics)
    fprintf(outfile,
            "Workspace for replpar: maxwsp = %ld, datasize = %ld, maxvsp = %ld\n\n",
            maxwsp, datasize, maxvsp);
  /*}}}*/
  /*{{{  restore & update environment*/
  free_fixedwslist();
  parspace = ((maxwsp + datasize) * replcount) + DS_MIN;
  parvspace = maxvsp * replcount;
  datasize = max(saveddatasize, parspace);
  vsp = savedvsp;
  maxvsp = max(savedmaxvsp, vsp + parvspace);
  lexlevel--;
  num_var = var_base;
  var_base = savedvar_base;
  SetNVUseCount(constantnptr, savedusecount);
  loop_weight = savedweight;
  namelist = savednamelist;
  /*}}}*/
  /*{{{  map replicator*/
  mapname(ReplCNameOf(tptr));
  upusecount(ReplCNameOf(tptr), replcount);
  mapexp(ReplCStartExpAddr(tptr));
  mapexp(ReplCLengthExpAddr(tptr));
  /*}}}*/
  /*{{{  map slots used in setting up the replicated PAR*/
  /* Map slots for the PAR join and count
     temporary used to calculate replicated wptr's,
     and possibly temporary used to calculate replicated vsp's */
  {
    int replparslots = MIN_REPLPAR_SLOTS;
    if (parvspace != ZERO32)
      /* Allocate a slot for the vsp temp */
      replparslots++;
    reservelowworkspace(replparslots);
  }
  /*}}}*/
  /*{{{  descope the replicator*/
  kill_var(ReplCNameOf(tptr));
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int setup_asm_operands()*/
PUBLIC int setup_asm_operands(treenode *tptr, treenode **operands[MAXREGS], int opmodes[MAXREGS])
{
  int i, ops = 0;
  while (!EndOfList(tptr))
    {
      if (TagOf(ThisItem(tptr)) == S_ADDRESSOF)
        {
          opmodes [ops] = P_PTR;
          operands[ops] = OpAddr(ThisItem(tptr));
        }
      else
        {
          operands[ops] = ThisItemAddr(tptr);
          opmodes [ops] = P_EXP;
        }
      tptr = NextItem(tptr);
      ops++;
    }

  /* These are generated in reverse order, so swap the list */
  for (i = 0; i < ops / 2; i++)
    {
      /* swap item i with item j (j is i'th from top) */
      int j = (ops - 1) - i;
      int opmode_temp         = opmodes[i];
      treenode **operand_temp = operands[i];
      opmodes [i] = opmodes[j];
      operands[i] = operands[j];
      opmodes [j] = opmode_temp;
      operands[j] = operand_temp;
    }

  return ops;
}
/*}}}*/
/*{{{  PRIVATE void mapguy_or_asm*/
PRIVATE void mapguy_or_asm (treenode *tptr, int guy_not_asm)
{
  treenode *operand = RightOpOf(tptr);
  int instruction = DOpTypeOf(tptr);
  if ((instruction & I_PRIMARY) && (TagOf(ThisItem(operand)) != N_LABELDEF))
    {
      instruction &= INST_MASK;
      if (guy_not_asm && (instruction == I_LDLP || instruction == I_LDNLP) &&
            !isconst(ThisItem(operand)))
        mapaddr(ThisItemAddr(operand));
      else
        mapexp(ThisItemAddr(operand));
    }
  else if ((instruction & I_PSEUDO_OP) && (operand != NULL))
    {
      int loadseq;
      treenode **operands[MAXREGS];
      int        opmodes[MAXREGS];
      int ops = setup_asm_operands(operand, operands, opmodes);
      switch (instruction & INST_MASK)
        {
          default:
            break;
          case I_LD:
            if (opmodes[0] == P_EXP)
              mapexp(operands[0]);
            else
              mapaddr(operands[0]);
            break;
          case I_LDAB:
            loadseq = mapload2regs(opmodes[0], operands[0],
                                   opmodes[1], operands[1]);
            break;
          case I_LDABC:
            loadseq = mapload3regs(opmodes[0], operands[0],
                                   opmodes[1], operands[1],
                                   opmodes[2], operands[2]);
            break;
          case I_ST: case I_STAB: case I_STABC:
            mapstoreregs(operands, ops);
            break;
        }
    }
  else if ((instruction & (I_PRIMARY | I_PSEUDO_OP | I_FPU_ENTRY_BIT)) == 0)
    {  /* it's a secondary */
       switch (instruction & INST_MASK)
         {
           case I_IN: case I_OUT: case I_OUTBYTE: case I_OUTWORD:
           case I_ALT: case I_ALTWT: case I_ENBC: case I_ENBS:
           case I_DISC: case I_DISS:
             datasize = max(datasize, DS_IO);
             break;
           case I_TIN: case I_TALT: case I_TALTWT: case I_ENBT: case I_DIST:
             datasize = max(datasize, DS_WAIT);
             break;
           default:
             break;
         }
    }
}
/*}}}*/
/*{{{  PRIVATE void mappreprocesslist(tptr)*/
/*****************************************************************************
 *
 *  mappreprocesslist takes a process list tptr, and calls mappreprocess
 *                     for each item on the list, leaving a list of
 *                     temporaries allocated, but not freed, by mappreprocess
 *
 *****************************************************************************/
PRIVATE void mappreprocesslist ( treenode *tptr )
{
  if (tptr != NULL)
    while (!EndOfList(tptr))
      {
        mappreprocess(ThisItem(tptr));
        tptr = NextItem(tptr);
      }
  return;
}
/*}}}*/
/*{{{  PRIVATE void mappresubscripts (tptr)*/
/*****************************************************************************
 *
 *  mappresubscripts performs mappreprocess on segments and subscripts.
 *                   For segments, temporaries are inserted where needed on
 *                   the start and length expressions.
 *                   No tree transformation is done, this is performed by
 *                   trans
 *
 *****************************************************************************/
PRIVATE void mappresubscripts ( treenode *tptr )
{
  while (TRUE)
    switch(TagOf(tptr))
      {
        /*{{{  SEGMENT SEGMENTITEM*/
        case S_SEGMENT:
        case S_SEGMENTITEM:
          {
            treenode *lengthexp = SLengthExpOf(tptr);
            treenode *startexp = SStartExpOf(tptr);
        
            mappreprocess(TagOf(startexp) == T_TEMP ? NDeclOf(startexp) : startexp);
            mappreprocess(TagOf(lengthexp) == T_TEMP ? NDeclOf(lengthexp) : lengthexp);
            mappresubscripts(SNameOf(tptr));
        
            /*{{{  map start and length expressions, put them in temporaries if necessary*/
            /* Mark the start and length temporaries (if there are any) as
               preevaluated so that we correctly map them when used later.
               These tags are changed back to T_TEMP when the temporaries are freed. */
            
            if (TagOf(startexp) == T_TEMP)
              /*{{{  load start expression to a temporary*/
              /* we would evaluate it twice - once in the range check and once in the
                 base, otherwise */
              {
                DEBUG_MSG(("mappresubscripts: mapping/allocating segment start\n"));
                mapexp(NDeclAddr(startexp));
                alloctemp(startexp);
                SetTag(startexp, T_PREEVALTEMP);
                addtemplist(startexp);
              }
              /*}}}*/
            if (TagOf(lengthexp)== T_TEMP)
              /*{{{  insert temporary on slength*/
              {
                DEBUG_MSG(("mappresubscripts: mapping segment length\n"));
                mapexp(NDeclAddr(lengthexp));
                DEBUG_MSG(("mappresubscripts: allocating segment length\n"));
                alloctemp(lengthexp);
                SetTag(lengthexp, T_PREEVALTEMP);
                addtemplist(lengthexp);
              }
              /*}}}*/
            /*}}}*/
        
            mappreprocess(SSubscriptExpOf(tptr));
            if (SCheckExpOf(tptr) != NULL)
              mapexp(SCheckExpAddr(tptr));
            return;
          }
        /*}}}*/
        /*{{{  ARRAYSUB*/
        case S_ARRAYSUB:
        case S_ARRAYITEM:
          mappreprocess(ASExpOf(tptr));
          tptr = ASBaseOf(tptr);
          break;
        /*}}}*/
        default:
          return;
      }
}
/*}}}*/
#if 0 /* this is never used */
/*{{{  PRIVATE treenode *mappreprocesses*/
PRIVATE treenode *mappreprocesses(treenode *tptr)
{
  /* changed because mappreprocess only does a single specification at a time:
     bug 776 05/10/90 */
  while (isspecification(tptr))
    {
      mappreprocess(tptr);
      tptr = DBodyOf(tptr);
    }
  return tptr;
}
/*}}}*/
#endif
/*{{{  PUBLIC void mappreprocess(tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  mappreprocess maps out workspace requirements for code which has to be
 *                pulled out in front of a process.
 *                At the moment this is segment base and length checking.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void mappreprocess ( treenode *tptr )
{
  while (tptr != NULL)
  {
    DEBUG_MSG(("mappreprocess: %s\n", itagstring(TagOf(tptr))));
    switch(TagOf(tptr))
      {
        default:
          return;
        /*{{{  specification         break / return*/
        case S_ABBR: case S_VALABBR:
        case S_RETYPE: case S_VALRETYPE:
          /* changed to only do a single specification:
             bug 776 05/10/90 */
        #if 0
          mappreprocess(DValOf(tptr));
          if (!isspecification(DBodyOf(tptr)))
            return;
          tptr = DBodyOf(tptr);
        #else
          tptr = DValOf(tptr);
        #endif
          break;

          /* changed to only do a single specification:
             bug 776 05/10/90 */
          /* They now fall through to the default, which is to return; */
        #if 0
        case S_DECL:
        case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
        case S_TPROTDEF: case S_SPROTDEF:

          if (!isspecification(DBodyOf(tptr)))
            return;
          tptr = DBodyOf(tptr);
          break;
        #endif
        /*}}}*/
        /*{{{  process               break / return*/
        /*{{{  IF                          return*/
        case S_IF:
          /*{{{  transform choice list*/
          /* This whole lot should be moved into trans! */

          #if 0 /* This leaves in FALSE guards, but turns them into SKIPs */
          {
            treenode *choicelist = CBodyOf(tptr);
            while (!EndOfList(choicelist))
              {
                treenode *choicenode = ThisItem(choicelist);
                choicenode = skipspecifications(choicenode);
                if (TagOf(choicenode) == S_CHOICE)
                  {
                    if (istrueguard(CondGuardOf(choicenode)))
                      NewNextItem(NULL, choicelist);
                    else if (isfalseguard(CondGuardOf(choicenode)))
                      SetCondBody(choicenode, newleafnode(S_SKIP, LocnOf(choicenode)));
                  }
                choicelist = NextItem(choicelist);
              }
          }
          #else
          {  /* This removes FALSE guards; CO'N 31/5/90 bug 307 */
            treenode **choicelist = CBodyAddr(tptr);
            while (!EndOfList(*choicelist))
              {
                treenode *choicenode = skipspecifications(ThisItem(*choicelist));
                treenode **next      = NextItemAddr(*choicelist);
                if (TagOf(choicenode) == S_CHOICE)
                  {
                    if (istrueguard(CondGuardOf(choicenode)))
                      NewNextItem(NULL, *choicelist);
                    else if (isfalseguard(CondGuardOf(choicenode)))
                      {
                        if (choicenode != ThisItem(*choicelist)) /* added 28/8/90 */
                          { /* there are leading specifications which may set error */
                            SetCondBody(choicenode, newleafnode(S_SKIP, LocnOf(choicenode)));
                          }
                        else /* remove that guard completely */
                          {
                            next = choicelist;
                            *choicelist = NextItem(*choicelist);
                          }
                      }
                  }
                choicelist = next;
              }
          }
          #endif
          /*}}}*/
          /* This has been removed to avoid calling mappreprocess twice
             on the guards */
          /* bug 776 8/10/90 */
          /*mappreprocesslist(CBodyOf(tptr));*/
          return;
        /*}}}*/
        /*{{{  ALT PRIALT                    return*/
        case S_ALT: case S_PRIALT:
          /*{{{  transform alternative list*/
          {
            /* This should really be done in trans (CON 8/10/90) */
            treenode *altlist = CBodyOf(tptr);
            while (!EndOfList(altlist))
              {
                treenode *aptr = ThisItem(altlist);
                treenode *altnode = skipspecifications(aptr);
                if (TagOf(altnode) == S_ALTERNATIVE)
                  /*{{{  try and move specifications in front of the input*/
                  /* We move specifications in front of the input providing:
                       1. They are used in the input item list
                       2. They are not used in the guard or channel
                   */
                  {
                    treenode *t = transformalt(aptr, altnode);
                    NewItem(t, altlist);
                  }
                  /*}}}*/
                altlist = NextItem(altlist);
              }
          }
          /*}}}*/
        /* This has been removed (bug 776 5/10/90) so that mappreprocess
           is called for these when actually mapping */
        #if 0  
          /* mappreprocess would not normally go inside an alternative with
             leading specifications, so we have to force it in with the
             following code. */
          {
            treenode *altlist = CBodyOf(tptr);
            while (!EndOfList(altlist))
              {
                treenode *aptr = ThisItem(altlist);
                aptr = mappreprocesses(aptr);
                mappreprocess(aptr);
                altlist = NextItem(altlist);
              }
          }
        #endif
          return;
        /*}}}*/
        /*{{{  REPLSEQ REPLPAR PRIREPLPAR    break*/
        case S_REPLSEQ:
        case S_REPLPAR:
        case S_PRIREPLPAR:
        case S_REPLIF:                     /* bug 776 8/10/90 */
        case S_REPLALT: case S_PRIREPLALT: /* bug 776 5/10/90 */
          mappreprocess(ReplCStartExpOf(tptr));
          tptr = ReplCLengthExpOf(tptr);
          break;
        /*}}}*/
        #if 0 /* This made the same as S_REPLSEQ - bug 776 8/10/90 */
        /*{{{  REPLIF              break*/
        case S_REPLIF:
          mappreprocess(ReplCStartExpOf(tptr));
          mappreprocess(ReplCLengthExpOf(tptr));
          tptr = ReplCBodyOf(tptr);
          break;
        /*}}}*/
        #endif
        #if 0 /* now ALTs are the same as any other replicator (bug 776 5/10/90)*/
        /*{{{  REPLALT PRIREPLALT  break*/
        case S_REPLALT:
        case S_PRIREPLALT:
          mappreprocess(ReplCStartExpOf(tptr));
          mappreprocess(ReplCLengthExpOf(tptr));
        /* This has been removed (bug 776 5/10/90) so that mappreprocess
           is called for these when actually mapping */
        #if 0
          tptr = ReplCBodyOf(tptr);
          /* mappreprocess would not normally go inside an alternative with
             leading specifications, so we have to force it in with the
             following code. */
          tptr = mappreprocesses(tptr);
          break;
        #else
          return;
        #endif
        /*}}}*/
        #endif
        /*{{{  WHILE CHOICE        break*/
        case S_WHILE:
        case S_CHOICE:
          tptr = CondGuardOf(tptr);
          break;
        /*}}}*/
        /*{{{  ALTERNATIVE         break*/
        case S_ALTERNATIVE:
          mappreprocess(AltGuardOf(tptr));
        #if 0
          /* This has been changed so that it only maps the input and guard
             (bug 776 5/10/90) */
          tptr = mappreprocesses(AltInputOf(tptr));
        #else
          tptr = skipspecifications(AltInputOf(tptr));
        #endif
          break;
        /*}}}*/
        /*{{{  PINSTANCE FINSTANCE         return*/
        case S_PINSTANCE: case S_FINSTANCE:
          /*  Must make sure that the parameter expressions are transformed before
              the parameter list is augmented, as hidden params may point to
              proper params. N.B. also that a segment param must have had a tempnode
              inserted on its length already (this is done by 'trans')
              if we are to use the temp as a hidden param.
           */
          mappreprocesslist(IParamListOf(tptr));
          return;
        /*}}}*/
        /*{{{  ASS                 break / return*/
        case S_ASS:
          {
            treenode *lhs = LHSOf(tptr),
                     *rhs = RHSOf(tptr);
            if (TagOf(lhs) == S_LIST)
              {
                mappreprocesslist(lhs);
                mappreprocesslist(rhs);
                return;
              }
            else
              {
                mappreprocess(lhs);
                tptr = rhs;
              }
            break;
          }
        /*}}}*/
        /*{{{  OUTPUT INPUT TAGGED_INPUT   return*/
        case S_OUTPUT: case S_INPUT: case S_TAGGED_INPUT:
          mappreprocess(LHSOf(tptr));
          mappreprocesslist(RHSOf(tptr));
          return;
        /*}}}*/
        /*{{{  DELAYED_INPUT       break*/
        case S_DELAYED_INPUT:
          mappreprocess(LHSOf(tptr));
          tptr = RHSOf(tptr);
          break;
        /*}}}*/
        /*{{{  CASE                break*/
        case S_CASE:
          tptr = LHSOf(tptr);
          break;
        /*}}}*/
        /*{{{  CASE_INPUT                  return*/
        case S_CASE_INPUT:
          mappreprocess(LHSOf(tptr));
          mappreprocesslist(RHSOf(tptr));
          return;
        /*}}}*/
        /*{{{  VARIANT                     return*/
        case S_VARIANT:
          mappreprocesslist(VRTaggedListOf(tptr));
          return;
        /*}}}*/
        /*}}}*/
        /*{{{  expression            break / return*/
        /*{{{  monadics            break*/
        case S_NEG: case S_BITNOT: case S_NOT: case S_SIZE: case S_UMINUS:
        case S_EXACT: case S_TRUNC: case S_ROUND: case S_ADDRESSOF:
          tptr = OpOf(tptr);
          break;
        /*}}}*/
        /*{{{  dyadics             break*/
        case S_AND: case S_OR:
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
        case S_COLON2:
        case S_CSUB0:
        case S_LSHIFT: case S_RSHIFT:
          mappreprocess(LeftOpOf(tptr));
          tptr = RightOpOf(tptr);
          break;
        case S_EVAL:
          mappreprocess(LeftOpOf(tptr));
          mapexp(LeftOpAddr(tptr));
          tptr = RightOpOf(tptr);
          break;
        /*}}}*/
        /*{{{  CONSTRUCTOR                 return*/
        case S_CONSTRUCTOR:
          mappreprocesslist(OpOf(tptr));
          return;
        /*}}}*/
        /*{{{  VALOF                       return*/
        case S_VALOF:
          mappreprocesslist(VLResultListOf(tptr));
          return;
        /*}}}*/
        /*}}}*/
        /*{{{  element                       return*/
        case S_ARRAYITEM:
        case S_SEGMENTITEM:
          mappresubscripts(tptr);
          return;
        /*}}}*/
        /*{{{  configuration         break*/
        #if 0 /* these never get to the back end! CON 8/10/90 */
        case S_PROCESSOR:
          tptr = ProcessorExpOf(tptr);
          break;
        #endif
        #if 0
          /* changed to only do a single process at a time */
          /* bug 776 5/10/90 */
          /* They now fall through to the default, which is to return; */
        case S_PLACE:
        case S_WSPLACE:
        case S_VSPLACE:
          tptr = DBodyOf(tptr);
          break;
        #endif
        /*}}}*/
      }
  }
}
/*}}}*/
/*{{{  PRIVATE void mapprocessbody(tptr)*/
/* Allocate workspace offsets for the process tptr */
PRIVATE void mapprocessbody ( treenode *tptr )
{
  static int guy_not_asm = FALSE; /* Yes, I mean local to this function */
  int priparflag = FALSE;
  DEBUG_MSG(("mapprocessbody: %s\n", itagstring(TagOf(tptr))));
  while (TRUE)
    switch (TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  abbreviation, retype, declaration, procedure, function; return*/
        case S_VALABBR: case S_ABBR:
        case S_VALRETYPE: case S_RETYPE:
        case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
        case S_DECL:
        case S_TPROTDEF: case S_SPROTDEF:
        case S_PLACE: case S_WSPLACE: case S_VSPLACE:
          mapdeclandbody(tptr, mapprocess, FALSE, TRUE);
          return;
        /*}}}*/
        /*{{{  S_STOP S_SKIP S_END   return*/
        case S_STOP:
        case S_SKIP:
        case S_END: /* Marks the end of an SC */
          return;
        /*}}}*/
        /*{{{  S_SEQ/IF                 return*/
        case S_SEQ: case S_IF:
          mapconstruction(CBodyOf(tptr), mapprocess);
          return;
        /*}}}*/
        /*{{{  S_REPLSEQ/S_REPLIF             return*/
        case S_REPLSEQ: case S_REPLIF:
          maprepl(tptr, mapprocess);
          return;
        /*}}}*/
        /*{{{  S_PAR S_PRIPAR        return*/
        case S_PAR:
        case S_PRIPAR:
          /* Workspace requirement is : local 0 and local 1 for housekeeping,
               plus the space required by each branch of the PAR */
          if (TagOf(tptr) == S_PRIPAR) priparflag = TRUE;
          {
            treenode *parlist = CBodyOf(tptr);
            int maxwsp;
            int npars = listitems(parlist);
            if (npars > 0)
              {
                /* Reserve slot for process count, and slot for join address */
                /* If PRI PAR via an ALT, still use two slots, one for the channel,
                   and one for the ALT clobbering workspace zero */
                reservelowworkspace(2);
                /*{{{  map each process*/
                {
                  INT32 saveddatasize = datasize;
                  INT32 savedvsp = vsp, savedmaxvsp = maxvsp;
                
                  /* If we are doing a PRI PAR with an ALT, the process doing the setting up
                     will be descheduled (by the ALT) before it terminates,
                     so it needs another DS_IO workspace slots  - CO'N */
                  INT32 parspace = (priparflag && (code_style_flags & CODE_STYLE_ALT_PRI_PAR)) ?
                                   DS_IO : ZERO32;
                  int savednum_var = num_var, savedvar_base = var_base;
                  /*{{{  debug message*/
                  if (diagnostics)
                    fprintf (outfile, "\nAllocating workspace for parallel process %d, %d\n",
                             var_base, num_var);
                  /*}}}*/
                  maxvsp = vsp;
                  /*{{{  map each branch of the PAR*/
                  while(!EndOfList(parlist))
                    {
                      treenode *pprocess = ThisItem(parlist);
                      treenode *spptr;
                      treenode *nameliststart = namelist;
                      /*{{{  initialise environment*/
                      num_var = savednum_var;
                      var_base = savednum_var;
                      datasize = DS_MIN;                          /* Minimum below workspace usage */
                      /*}}}*/
                      mapprocess(pprocess);
                      maxwsp = allocvars(var_base, 0, TRUE);
                      /*{{{  debug message*/
                      if (diagnostics)
                        fprintf (outfile,
                                 "for branch of par maxwsp = %d, datasize = %ld, maxvsp = %ld\n",
                                             maxwsp, datasize, maxvsp);
                      
                      /*}}}*/
                      /* maxwsp now contains our maximum workspace requirement for PAR branch */
                      /* The field vsusage is not needed in this space node */
                      spptr = newspacenode(S_SPACEUSAGE, 0, pprocess, maxwsp,
                                         maxwsp + datasize, ZERO32, ZERO32, NULL, 0);
                      NewItem(spptr, parlist);
                      /*{{{  move all vars declared within this branch below ws already allocated*/
                      {
                        int i;
                        treenode *var;
                        int adjust = maxwsp+parspace;
                        for (i = var_base; i < num_var; i++)
                          { /* Add these vars to the par namelist */
                            var = get_var(i);
                            if (TagOf(var) != T_RESERVEDWS)
                              while (var != NULL)
                                {
                                  treenode *nextvar = NVNextOf(var);
                                  SetNVNext(var, namelist);
                                  namelist = var;
                                  var = nextvar;
                                }
                          }
                        /* Move all variables that have been added in this branch */
                        var = namelist;
                        while (var != nameliststart)
                          {
                            if (NVOffsetOf(var) != NO_SLOT)
                              {
                                SetNVOffset(var, NVOffsetOf(var) - adjust);
                                if (diagnostics)
                                  /*{{{  print some diagnostics*/
                                  {
                                    if (TagOf(var) == T_TEMP)
                                      fprintf(outfile, "$temp%ld", NVVarNumOf(var));
                                    else
                                      fprintf(outfile, WNameOf(NNameOf(var)));
                                    fprintf (outfile, " moved to workspace offset %ld\n", NVOffsetOf(var));
                                  }
                                  /*}}}*/
                              }
                            var = NVNextOf(var);
                          }
                      }
                      /*}}}*/
                      parspace += maxwsp + datasize;
                      vsp = maxvsp;
                      parlist = NextItem(parlist);
                      free_fixedwslist();
                    }
                  /*}}}*/
                  /*{{{  restore & update environment*/
                  var_base = savedvar_base;
                  num_var = savednum_var;
                  vsp = savedvsp;
                  maxvsp = max(maxvsp, savedmaxvsp);
                  
                  /* the parspace goes below workspace */
                  datasize = max(saveddatasize, parspace);
                  /*}}}*/
                }
                /*}}}*/
              }
          }
          return;
        /*}}}*/
        /*{{{  S_REPLPAR             return*/
        case S_REPLPAR:
          /* Workspace requirement is : local 0 and local 1 for housekeeping,
               local 2 (and possibly local 3) as temporaries
               plus the space required for all replications of the PAR,
               this last goes below our current workspace. */
          /* Vector space requirement is: a workspace slot for a vsp for each
             replicated process (if vector space is used), and
               vector.space for repl.process * repl.count . */
          mapreplpar(tptr);
          return;
        /*}}}*/
        /*{{{  S_WHILE               return*/
        case S_WHILE:
          {
            int oldweight = loop_weight;
            uploop_weight(VAR_USAGE_PER_LOOP);
            mapbool(CondGuardAddr(tptr));
            mapprocess(CondBodyOf(tptr));
            loop_weight = oldweight;
          }
          return;
        /*}}}*/
        /*{{{  S_CHOICE              return*/
        case S_CHOICE:
          mapbool(CondGuardAddr(tptr));
          mapprocess(CondBodyOf(tptr));
          return;
        /*}}}*/
        /*{{{  S_SELECTION           return*/
        case S_SELECTION:
          {
            /* Map selection list to put constants in a constant table, if necessary */
            treenode *cguard = CondGuardOf(tptr);
            if (TagOf(cguard) != S_ELSE)
              mapexplist(cguard);
            mapprocess(CondBodyOf(tptr));
          }
          return;
        /*}}}*/
        /*{{{  S_VARIANT             return*/
        case S_VARIANT:
          {
            treenode *inputlist = NextItem(VRTaggedListOf(tptr));
                                                                /* Skip over the tag */
            while (!EndOfList(inputlist))
              {
                mapinputitem(globchannelmode, globchannelptr,
                             P_PTR, ThisItemAddr(inputlist));
                inputlist = NextItem(inputlist);
              }
            mapprocess(VRBodyOf(tptr));
          }
          return;
        /*}}}*/
        /*{{{  S_ALT S_REPLALT S_PRIALT S_PRIREPLALT  return*/
        case S_ALT: case S_PRIALT:
        case S_REPLALT: case S_PRIREPLALT:
          mapalt(tptr);
          return;
        /*}}}*/
        /*{{{  S_OUTPUT              return*/
        case S_OUTPUT:
          mapoutput(tptr);
          return;
        /*}}}*/
        /*{{{  S_INPUT S_DELAYED_INPUT S_CASE_INPUT S_TAGGED_INPUT return*/
        case S_INPUT:
        case S_DELAYED_INPUT:
        case S_CASE_INPUT:
        case S_TAGGED_INPUT:
          mapinput(tptr);
          return;
        /*}}}*/
        /*{{{  S_ASS                 return*/
        case S_ASS:
          mapassign(LHSAddr(tptr), RHSAddr(tptr));
          return;
        /*}}}*/
        /*{{{  S_PINSTANCE           return*/
        case S_PINSTANCE:
          if ((TagOf(INameOf(tptr)) == N_PREDEFPROC) && mappredef(tptr, NULL) )
            return;  /* Exit if done inline */
          SetIParamList(tptr, augmentparams(IParamListOf(tptr),
                                            NTypeOf(INameOf(tptr)), NULL));
          mapinstance(tptr);
          return;
        /*}}}*/
        /*{{{  S_CASE                return*/
        case S_CASE:
          mapcase(tptr);
          return;
        /*}}}*/
        /*{{{  guys*/
        case S_GUY: case S_ASM:
          {
            const int saved_flag = flag_scalar_chans;
            flag_scalar_chans = TRUE;
            guy_not_asm = (TagOf(tptr) == S_GUY);
            walklist(mapprocessbody, CBodyOf(tptr));
            flag_scalar_chans = saved_flag;
          }
          return;
        case S_LABELDEF:
          SetNVOffset(DNameOf(tptr), newlab());
          return;
        case S_GUYCODE:
          /* guy_not_asm was set by the `enclosing' S_GUY or S_ASM */
          mapguy_or_asm(tptr, guy_not_asm);
          return;
        case S_GUYSTEP:
          return;
        /*}}}*/
        default:
          badtag(genlocn, TagOf(tptr), "mapprocessbody");
      }
      /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void mapprocess(tptr)*/
/*****************************************************************************
 *
 *  mapprocess maps out workspace requirements for the process tptr,
 *             including pre- and post- code.
 *
 *****************************************************************************/
PUBLIC void mapprocess ( treenode *tptr )
{
  treenode *savedtemplist = templist;
  templist = NULL;
  genlocn = LocnOf(tptr);
  mappreprocess(tptr);
  mapprocessbody(tptr);
  freetemplist(templist);
  templist = savedtemplist;
}
/*}}}*/
/*{{{  PUBLIC void mapblock(tptr)         Andy's new one*/
/*****************************************************************************
 *
 *  mapblock maps out the workspace for the PROC/FUNCTION definition tptr
 *
 *****************************************************************************/
PUBLIC void mapblock ( treenode *tptr )
{
  treenode *n = DNameOf(tptr);
  int nt = TagOf(n);
  treenode *body = DValOf(tptr);
#ifdef COMMENT
  if (usedags)
    /*{{{  create basic blocks & reorder*/
    {
      extern treenode *make_blocklist();
      treenode *blist = make_blocklist(body);
      reorderlist(blist);
      SetDVal(tptr, blist);
      body = blist;
    }
    
    /*}}}*/
#endif
  /*{{{  debug message*/
  if (diagnostics)
    fprintf (outfile, "\nAllocating workspace for routine %s at lexlevel %d\n",
                  WNameOf(NNameOf(n)), lexlevel-1);
  current_proc = n;
  /*}}}*/
  /*{{{  initialise environment*/
  initvsp(0); /* Initialise vector space usage to zero */
  datasize = DS_MIN;
  staticlinkused = STATICLINK_NOT_USED;
  consttablechain = NULL;
  namelist = NULL;
  var_base = 0;
  num_var = 0;
  loop_weight = 1;
  init_mapping();
  SetNVUseCount(constantnptr, 0);
  /*}}}*/
  /*{{{  map out the routine body*/
  if (nt == N_PROCDEF)
    mapprocess(body);   /* Map out the body of tptr */
  else
    {
      treenode *savedtemplist = templist;
      /*{{{  map leading specifications*/
      #if 0 /* bug 867 28/1/91 */
      if (isspecification(body))
        /*{{{  prewalk leading specifications*/
        mappreprocess(body);
        /*}}}*/
      #endif
      while (isspecification(body))
        {
          mappreprocess(body); /* bug 867 28/1/91 */
          mapsingledecl(body);
          body = DBodyOf(body);
        }
      /*}}}*/
      /*{{{  map the valof*/
      {
        treenode *resultlist = VLResultListOf(body);
        struct { treenode **opd; int opdmode; } regresults[MAXREGS];
        int nregresults = 0;
        mapprocess(VLBodyOf(body));
        mappreprocess(body);  /* Mappreprocess on the result list */
        if
          /*{{{  special case single-valued functions on an fp processor*/
          (fpinline && listitems(resultlist) == 1 &&
           isreal(typeof(ThisItem(resultlist))))
            mapfpexp(ThisItemAddr(resultlist));
          /*}}}*/
        else
          {
            /* destlist moved here: bug 1012 24/10/90 */
            treenode *destlist = firstresultof(FnParamsOf(NTypeOf(n)));
            while (!EndOfList(resultlist))
              /*{{{  assign this result through a pointer, or save it for a register*/
              {
                treenode *thisresult = ThisItem(resultlist);
                /*treenode *destlist = firstresultof(FnParamsOf(NTypeOf(n)));*/ /* bug 1012 24/10/90 */
                int type = typeof(thisresult);
                if ((nregresults < MAXREGS) && fitsinregister(type))
                  {
                    regresults[nregresults].opd = ThisItemAddr(resultlist);
                    regresults[nregresults].opdmode = P_EXP;
                    nregresults++;
                  }
                else
                  {
                    mapsimpleassign(type, P_EXP, ThisItemAddr(destlist),
                                          P_EXP, ThisItemAddr(resultlist));
                    DEBUG_MSG(("mapblock: getting next destlist\n"));
                    destlist = nextresultof(destlist);
                  }
                resultlist = NextItem(resultlist);
              }
              /*}}}*/
            /*{{{  load register results*/
            /*{{{  preevaluate non-addressable real-valued results on an fp processor*/
            if (fpinline)
              {
                int i;
                for (i = 0; i < nregresults; i++)
                  {
                    treenode **expaddr = regresults[i].opd;
                    if (needtemptoload(regresults[i].opdmode, *expaddr))
                      {
                        mapfpexp(expaddr);
                        *expaddr = gettemp(*expaddr, NM_WORKSPACE);
                        upusecount(*expaddr, 1);
                        regresults[i].opdmode = P_TEMP;
                      }
                  }
              }
            /*}}}*/
            {
              int loadseq;
              /* We go to all the trouble of working the load sequence out,
                 and then throw it away */
              switch(nregresults)
                {
                  case 0: break;
                  case 1: mapexpopd(regresults[0].opdmode, regresults[0].opd);
                          loadseq = 1;
                          break;
                  case 2: loadseq = mapload2regs(regresults[1].opdmode, regresults[1].opd,
                                                 regresults[0].opdmode, regresults[0].opd);
                          break;
                  case 3: loadseq = mapload3regs(regresults[2].opdmode, regresults[2].opd,
                                                 regresults[1].opdmode, regresults[1].opd,
                                                 regresults[0].opdmode, regresults[0].opd);
                          break;
                }
            }
            /*{{{  free temporaries holding preevaluated reals*/
            if (fpinline)
              {
                int i;
                for (i = 0; i < nregresults; i++)
                  if (regresults[i].opdmode == P_TEMP)
                    freetemp(*(regresults[i].opd));
              }
            /*}}}*/
            /*}}}*/
          }
      }
      /*}}}*/
      freetemplist(templist);
      templist = savedtemplist;
    }
  /*{{{  check whether we need a constant pointer*/
  {
    treenode *cptr = consttablechain;
    while ((cptr != NULL) && (TagOf(cptr) != S_CONSTEXP))
      cptr = CTNextOf(cptr);
    if (cptr != NULL)
      create_immortalvar(constantnptr);
    else
      SetNVOffset(constantnptr, CONSTANTPOINTER_NOT_USED);
  }
  /*}}}*/
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void mapnestedblocks(tptr)*/
/*****************************************************************************
 *
 *  mapnestedblocks walks the tree 'tptr' mapping nested routines
 *
 *
 *****************************************************************************/
PUBLIC void mapnestedblocks ( treenode *tptr )
{
  while (tptr != NULL)
    {
    genlocn = LocnOf(tptr);
    switch(TagOf(tptr))
      /*{{{  cases*/
      {
        default:
          return;
        /*{{{  specification*/
        /*{{{  routine specification Andy's*/
        case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          /* Allocate workspace for procedure or function definition */
          {
            treenode *n = DNameOf(tptr);
            /*{{{  initialise params*/
            {
              treenode *fparams = NParamListOf(n);
              while (!EndOfList(fparams))
                {
                  treenode *thisfparam = ThisItem(fparams);
                  if (TagOf(thisfparam) == N_PARAM || TagOf(thisfparam) == N_VALPARAM)
                    {
                      SetNVNext(thisfparam, NULL);
                      SetNVUseCount(thisfparam, 0);
                      /* we now set up the param's mode in augmentformals (in trans phase) */
                      SetNVOffset(thisfparam, NO_SLOT); /* bug 1135 31/1/91 */
                    }
                  fparams = NextItem(fparams);
                }
            }
            /*}}}*/
            if (separatelycompiled(n))
              /*{{{  map as separately compiled*/
              {
                INT32 ws, vs;
                /* vector space pointer onto front (or end) of params */
                if (!compiledforcorrectproc(n))
                  vs = ZERO32;
                else
                  getprocwsandvs(n, &ws, &vs);
                if (diagnostics) /* added to track down a bug */
                  fprintf(outfile, "Workspace allocation for library %s; ws:%ld, vs:%ld\n",
                          WNameOf(NNameOf(n)), ws, vs);
                SetNPSLUsage(n, STATICLINK_NOT_USED);
                SetNPParams(n, addstaticandvec(n, vs));
                initallocvars(); /* added for bug 1061 30/11/90 */
                allocparams(n);
              }
              /*}}}*/
            else if (!inline(n))
              /*{{{  map normal routine*/
              {
                INT32 maxwsp, numparams;
                lexlevel++;  /* Formal parameters are at a higher lexical level */
                mapnestedblocks(DValOf(tptr));
                mapblock(tptr);
                if (staticlinkused >= lexlevel)
                  staticlinkused = STATICLINK_NOT_USED;
                SetNPSLUsage(n, staticlinkused);
                SetNPVSUsage(n, maxvsp);
                numparams = addstaticandvec(n, maxvsp);
                SetNPParams(n, numparams);
                /* All outer level procs mustn't use param slots as variables: */
                if ((lexlevel - 1) == initlexlevel)
                   /* && strcmp(WNameOf(NNameOf(n)), MAIN_ENTRY_POINT) == 0) */
                  numparams = REG_PARAMS;
                initallocvars(); /* added for bug 1061 30/11/90 */
                maxwsp = allocvars(0, REG_PARAMS - numparams, FALSE);
                SetNPConstTables(n, consttablechain);
                SetNPCPOffset(n, NVOffsetOf(constantnptr));
                SetNPMaxwsp(n, maxwsp);
                SetNPDatasize(n, datasize + maxwsp);
                free_fixedwslist();
                allocparams(n);
                if (diagnostics)
                  fprintf(outfile,
                          "Workspace for routine: maxwsp = %ld, datasize = %ld, maxvsp = %ld\n",
                          maxwsp, datasize, maxvsp);
                lexlevel--;
              }
              /*}}}*/
          }
          tptr = DBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  other specification*/
        case S_DECL:
          /*{{{  initialise all use counts to zero*/
          {
            treenode *t = DNameOf(tptr);
            if (TagOf(t) == S_LIST)
              for (; !EndOfList(t); t = NextItem(t))
                {
                  SetNVUseCount(ThisItem(t), 0);
                  /*(void)isinvectorspace(ThisItem(t));*/ /* does SetNMode */ /* bug 1156 - now done in trans */
                  if (TagOf(NTypeOf(ThisItem(t))) == S_CHAN) /* scalar channel */
                    SetNChanMark(ThisItem(t), FALSE);
                }
            else
              {
                SetNVUseCount(t, 0);
                /*(void)isinvectorspace(t);*/ /* does SetNMode */ /* bug 1156 - now done in trans */
                if (TagOf(NTypeOf(t)) == S_CHAN) /* scalar channel */
                  SetNChanMark(t, FALSE);
              }
            tptr = DBodyOf(tptr);
          }
          break;
          /*}}}*/
        
        case S_VALABBR:
        case S_VALRETYPE:
        case S_ABBR:
        case S_RETYPE:
          {
            treenode *rhs = DValOf(tptr);
            treenode *nptr = DNameOf(tptr);
            if (TagOf(NTypeOf(nptr)) == S_CHAN) /* scalar channel */
              SetNChanMark(nptr, FALSE);
          #if 0 /* This is now done in trans - bug 1156 15/2/91 */
            /*{{{  Mode must be set before mapping nested functions*/
            switch(abbrevmode(tptr))
              {
                case AM_CONST:
                  /* Though it isn't really, just give it a valid mode */
                  SetNMode(nptr, NM_WORKSPACE);
                  break;
                case AM_ISRHS:
                  SetNMode(nptr, NModeOf(nameof(rhs)));
                  break;
                case AM_PTR:
                  {
                  #if 0 /* removed 19/9/90 */
                    treenode *rhsname = nameof(rhs);
                    if (TagOf(rhsname) != S_CONSTCONSTRUCTOR && TagOf(rhsname) != S_STRING &&
                        TagOf(rhsname) != S_CONSTRUCTOR      && isinvectorspace(rhsname))
                      SetNMode(nptr, NM_VECSPACE);
                    else
                  #endif
                      SetNMode(nptr, NM_POINTER);
                  }
                  break;
                default: /* case AM_VAL: */
                  (void)isinvectorspace(nptr); /* sets up NM_WORKSPACE or NM_VECSPACE */
                  break;
              }
            /*}}}*/
          #endif
            mapnestedblocks(rhs);
          #if 0 /* This has already been done in trans */
            /*{{{  fill in the unknown array dimensions*/
            {
              int tag = TagOf(nptr);
              if ((tag == N_VALABBR) || (tag == N_ABBR))
                /*{{{  copy dimensions from the type of the right-hand side*/
                {
                  treenode *type = NTypeOf(nptr);
                  int dimension = 0;
                  while (TagOf(type) == S_ARRAY)
                    {
                      if (ARDimOf(type) == (-1))
                        SetARDimLength(type, dimexpof(rhs, dimension));
                      dimension++;
                      type = ARTypeOf(type);
                    }
                }
                /*}}}*/
              else  /* RETYPE */
                /*{{{  there is at most one hidden dimension in a retype*/
                {
                  treenode *type = NTypeOf(nptr);
                  while ((type != NULL) && (TagOf(type) == S_ARRAY))
                    {
                      if (ARDimOf(type) == (-1))
                        /* Make up a temporary for the open dimension */
                        {
                          treenode *temp = newtempnode(T_TEMP, dummyexp_p, NM_WORKSPACE);
                          SetARDimLength(type, temp);
                        }
                      type = ARTypeOf(type);
                    }
                }
                /*}}}*/
            }
            /*}}}*/
          #endif
            tptr = DBodyOf(tptr);
          }
          break;
        case S_WSPLACE:
        case S_VSPLACE:
        case S_TPROTDEF:
        case S_SPROTDEF:
          tptr = DBodyOf(tptr);
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  SEQ IF ALT PAR PRIALT PRIPAR*/
        case S_PAR:
        case S_SEQ:
        case S_IF:
        case S_ALT:
        case S_PRIALT:
        case S_PRIPAR:
          tptr = CBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  REPLSEQ REPLIF REPLALT REPLPAR PRIREPLALT PRIREPLPAR*/
        case S_REPLPAR:
        case S_REPLSEQ:
        case S_REPLIF:
        case S_REPLALT:
        case S_PRIREPLALT:
        case S_PRIREPLPAR:
          mapnestedblocks(ReplCStartExpOf(tptr));
          mapnestedblocks(ReplCLengthExpOf(tptr));
          if (parrepl(TagOf(tptr)))
            {
              lexlevel++;
              mapnestedblocks(ReplCBodyOf(tptr));
              lexlevel--;
              return;
            }
          tptr = ReplCBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  WHILE CHOICE*/
        case S_WHILE:
        case S_CHOICE:
          mapnestedblocks(CondGuardOf(tptr));
          tptr = CondBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  SELECTION*/
        case S_SELECTION:
          tptr = CondBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  VARIANT*/
        case S_VARIANT:
          mapnestedblocks(VRTaggedListOf(tptr));
          tptr = VRBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  ALTERNATIVE*/
        case S_ALTERNATIVE:
          mapnestedblocks(AltGuardOf(tptr));
          mapnestedblocks(AltInputOf(tptr));
          tptr = AltBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  LIST*/
        case S_LIST:
          mapnestedblocks(ThisItem(tptr));
          tptr = NextItem(tptr);
          break;
        /*}}}*/
        /*{{{  configuration*/
        /*{{{  PLACE*/
        case S_PLACE:
          mapnestedblocks(DValOf(tptr));
          tptr = DBodyOf(tptr);
          break;
        /*}}}*/
        /*{{{  PROCESSOR*/
        case S_PROCESSOR:
          tptr = ProcessorBodyOf(tptr);
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  instance*/
        /*{{{  PINSTANCE FINSTANCE*/
        case S_PINSTANCE:
        case S_FINSTANCE:
          tptr = IParamListOf(tptr);
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  action*/
        /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT CASE CASEINPUT*/
        case S_ASS:
        case S_OUTPUT:
        case S_INPUT:
        case S_TAGGED_INPUT:
        case S_DELAYED_INPUT:
        case S_CASE:
        case S_CASE_INPUT:
          mapnestedblocks(LHSOf(tptr));
          tptr = RHSOf(tptr);
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  monadic*/
        case S_NEG:
        case S_BITNOT:
        case S_UMINUS:
        case S_NOT:
        /* SIZE ELSIZE and SEGSTART all return */
        case S_EXACT:
        case S_ROUND:
        case S_TRUNC:
        case S_CONSTRUCTOR:
          tptr = OpOf(tptr);
          break;
        /*}}}*/
        /*{{{  dyadic*/
        case S_AND: case S_OR:
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_LSHIFT: case S_RSHIFT:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE:
        case S_AFTER:
        case S_COLON2:
          mapnestedblocks(LeftOpOf(tptr));
          tptr = RightOpOf(tptr);
          break;
        /*}}}*/
        /*{{{  valof*/
        case S_VALOF:
          mapnestedblocks(VLBodyOf(tptr));
          tptr = VLResultListOf(tptr);
          break;
        /*}}}*/
        /*{{{  subscript*/
        case S_ARRAYSUB:
        case S_ARRAYITEM:
          mapnestedblocks(ASBaseOf(tptr));
          tptr = ASExpOf(tptr);
          break;
        /*}}}*/
        /*{{{  segment*/
        case S_SEGMENTITEM:
        case S_SEGMENT:
          mapnestedblocks(SNameOf(tptr));
          mapnestedblocks(SStartExpOf(tptr));
          tptr = SLengthExpOf(tptr);
          break;
        /*}}}*/
        /*{{{  temporary*/
        case T_TEMP:
        case T_PREEVALTEMP:
          tptr = NDeclOf(tptr);
          break;
        /*}}}*/
      }
      /*}}}*/
    }
}
/*}}}*/
/*}}}*/
/*{{{  PUBLIC void mapmain(tptr)*/
/* Allocate workspace offsets for main body */
PUBLIC void mapmain ( treenode *tptr )
{
  consttablechain = NULL;
  /* when configuring, we don't need a new copy for every processor */
  if (constantnptr == NULL)
    constantnptr = newnamenode(T_TEMP, NOPOSN, tempname_p, newleafnode(S_INT, NOPOSN),
                               NULL, 0, 0, NM_DEFAULT);
  staticlinkused = STATICLINK_NOT_USED;
#ifdef COMMENT
  if (usedags)
    initdfa();
#endif
  initvsp(0);
  datasize = DS_MIN;
  lexlevel = initlexlevel;
  templist = NULL;
  mapnestedblocks(tptr);

  /*{{{  COMMENT not needed at present, as there is no global process*/
  /**********************  Start comment out ****************************
  @*{{{  *@
  globmaxwsp = maxwsp;
  globdatasize = maxwsp + datasize;
  @*}}}*@
   **********************   End comment out  ****************************/
  /*}}}*/
}
/*}}}*/

/*{{{  PRIVATE void clearlibentries*/
#ifdef CONFIG
PRIVATE void clearlibentries(libentry_t *libentries)
{
  /* stdlibentries holds a list of entries in the reverse order to
     the order in which they were read in.
     A tree still exists which holds a copy of all the stuff which was
     read in.
     This piece of code points decltree at the original tree so that it
     can be cleaned up, also cleans up the stdlibentries list at the
     same time.
     CON 8/1/91
  */
  while (libentries != NULL)
    {
      libentry_t *this   = libentries;
      treenode *decltree = NDeclOf(libentries->l_name);
      libentries         =         libentries->l_next;
      SetDBody(decltree, NULL);
      freetree(&decltree);
      freevec(this, sizeof(libentry_t));
    }
}
#else
/* Since this is only called once in the real compiler, we needn't
   clean it up the following times.
*/
#define clearlibentries(x) /* empty */
#endif
/*}}}*/
/*{{{  PUBLIC void initbind(void)*/
PUBLIC void initbind ( void )
{
  /* we re-load these for each invocation, if configuring, because their
     formal parameter lists might be augmented differently.
  */
  if (stdlibentries != NULL)
    {
      clearlibentries(stdlibentries);
      stdlibentries = NULL;
    }
  stdlibsloaded = FALSE;
#ifdef VIRTUALCHANS
  if (vlibentries != NULL)
    {
      clearlibentries(vlibentries);
      vlibentries = NULL;
    }
  vlibsloaded = FALSE;
#endif
}
/*}}}*/
