#include "extlib.h"

/*{{{  static procedures */
/*{{{  Int64UnsDiv */
/* Unsigned divide using Knuth's divide algorithm, with error checking */

#ifdef ANSI
PRIVATE VOID Int64UnsDiv (BOOL *Error, BIT32 *QuotHi, BIT32 *QuotLo, BIT32 *RemHi, BIT32 *RemLo, BIT32 DvdHi, BIT32 DvdLo, BIT32 DvsrHi, BIT32 DvsrLo)
#else
PRIVATE VOID Int64UnsDiv (Error, QuotHi, QuotLo, RemHi, RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo)
    BOOL *Error;
    BIT32 *QuotHi, *QuotLo, *RemHi, *RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo;
#endif
{
    *Error = FALSE;

    if ((DvsrLo == ZERO32) && (DvsrHi == ZERO32))
        *Error = TRUE;
    else
    {
        /*{{{   */
        BIT32 Count, X, U[3], W[3], V[2];
        
        if (DvsrHi == ZERO32)
        {
            /*{{{  double by single */
            *RemHi = ZERO32;
            
            LONGDIV(QuotHi, &U[1], ZERO32, DvdHi, DvsrLo);
            LONGDIV(QuotLo, RemLo, U[1], DvdLo, DvsrLo);
            /*}}}*/
        }
        else
        {
            /*{{{  double by double */
            /*{{{  normalise the operands */
            NORMALISE(&Count, &V[1], &V[0], DvsrHi, DvsrLo);
            if (Count == ZERO32)
            {
                /*{{{  already normalised */
                U[2] = ZERO32;
                U[1] = DvdHi;
                U[0] = DvdLo;
                /*}}}*/
            }
            else
            {
                /*{{{  shift the dividend */
                SHIFTLEFT(&U[2], &X, ZERO32, DvdHi, Count);
                SHIFTLEFT(&U[1], &U[0], DvdHi, DvdLo, Count);
                /*}}}*/
            }
            /*}}}*/
            
            /*{{{  evaluate the quotient */
            *QuotHi = ZERO32;
            
            if (U[2] == V[1])
                *QuotLo = MaxUnsignedInt32;
            else
                LONGDIV(QuotLo, &X, U[2], U[1], V[1]);
            
            if (*QuotLo != ZERO32)
            {
                /*{{{  evaluate and check for overflow */
                LONGPROD(&W[1], &W[0], *QuotLo, V[0], ZERO32);
                LONGPROD(&W[2], &W[1], *QuotLo, V[1], W[1]);
                
                LONGDIFF(&X, &U[0], U[0], W[0], ZERO32);
                LONGDIFF(&X, &U[1], U[1], W[1], X);
                LONGDIFF(&X, &U[2], U[2], W[2], X);
                
                while ((U[2] & SignBit32) != ZERO32)
                {
                    /*{{{  add back as required */
                    *QuotLo = *QuotLo - ONE32;
                    
                    LONGSUM(&X, &U[0], U[0], V[0], ZERO32);
                    LONGSUM(&X, &U[1], U[1], V[1], X);
                    LONGSUM(&X, &U[2], U[2], ZERO32, X);
                    /*}}}*/
                }
                /*}}}*/
            }
            /*}}}*/
            
            /*{{{  evaluate the remainder */
            if (Count == ZERO32)
            {
                /*{{{   */
                *RemLo = U[0];
                *RemHi = U[1];
                /*}}}*/
            }
            else
                SHIFTRIGHT(RemHi, RemLo, U[1], U[0], Count);
            /*}}}*/
            /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/

/*{{{  Int64DivRem */
/* Signed divide from Knuth (algorithm D, section 4.3.1 - vol 2, pp 257) */

#ifdef ANSI
PRIVATE VOID Int64DivRem (BOOL *Error, BIT32 *QuotHi, BIT32 *QuotLo, BIT32 *RemHi, BIT32 *RemLo, BIT32 DvdHi, BIT32 DvdLo, BIT32 DvsrHi, BIT32 DvsrLo)
#else
PRIVATE VOID Int64DivRem (Error, QuotHi, QuotLo, RemHi, RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo)
    BOOL *Error;
    BIT32 *QuotHi, *QuotLo, *RemHi, *RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo;
#endif
{
    BOOL NegRem, NegQuot;
    BIT32 Borrow, U[2], V[2];

    if ((DvdHi & SignBit32) != ZERO32)
    {
        /*{{{   */
        NegRem = TRUE;
        
        LONGDIFF(&U[1], &U[0], ZERO32, DvdLo, ZERO32);
        LONGDIFF(&Borrow, &U[1], ZERO32, DvdHi, U[1]);
        /*}}}*/
    }
    else
    {
        /*{{{   */
        NegRem = FALSE;
        
        U[0] = DvdLo;
        U[1] = DvdHi;
        /*}}}*/
    }

    if ((DvsrHi & SignBit32) != ZERO32)
    {
        /*{{{   */
        NegQuot = ! NegRem;
        
        LONGDIFF(&V[1], &V[0], ZERO32, DvsrLo, ZERO32);
        LONGDIFF(&Borrow, &V[1], ZERO32, DvsrHi, V[1]);
        /*}}}*/
    }
    else
    {
        /*{{{   */
        NegQuot = NegRem;
        
        V[0] = DvsrLo;
        V[1] = DvsrHi;
        /*}}}*/
    }

    Int64UnsDiv(Error, QuotHi, QuotLo, RemHi, RemLo, U[1], U[0], V[1], V[0]);

    if (! *Error)
    {
        /*{{{   */
        if (((*QuotHi & SignBit32) != ZERO32) && (! NegQuot))
            *Error = TRUE;
        else
        {
            /*{{{   */
            if (NegQuot)
            {
                /*{{{  check if result has negative quotient */
                LONGDIFF(&Borrow, QuotLo, ZERO32, *QuotLo, ZERO32);
                LONGDIFF(&Borrow, QuotHi, ZERO32, *QuotHi, Borrow);
                /*}}}*/
            }
            
            if (NegRem)
            {
                /*{{{  check if result has negative remainder */
                LONGDIFF(&Borrow, RemLo, ZERO32, *RemLo, ZERO32);
                LONGDIFF(&Borrow, RemHi, ZERO32, *RemHi, Borrow);
                /*}}}*/
            }
            /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/
/*}}}*/

/*{{{  global procedures */
/*{{{  Int64Add */
#ifdef ANSI
PUBLIC VOID Int64Add (BOOL *Error, BIT32 *SumHi, BIT32 *SumLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Add (Error, SumHi, SumLo, AHi, ALo, BHi, BLo)
    BOOL *Error;
    BIT32 *SumHi, *SumLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 Carry;

    LONGSUM(&Carry, SumLo, ALo, BLo, ZERO32);
    LONGADD(SumHi, AHi, BHi, Carry);

    *Error = ((((AHi ^ BHi) & SignBit32) == ZERO32) &&
              (((AHi ^ *SumHi) & SignBit32) != ZERO32));
}
/*}}}*/

/*{{{  Int64Plus */
#ifdef ANSI
PUBLIC VOID Int64Plus (BIT32 *SumHi, BIT32 *SumLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Plus (SumHi, SumLo, AHi, ALo, BHi, BLo)
    BIT32 *SumHi, *SumLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 Carry;

    LONGSUM(&Carry, SumLo, ALo, BLo, ZERO32);
    LONGSUM(&Carry, SumHi, AHi, BHi, Carry);
}
/*}}}*/

/*{{{  Int64Sub */
#ifdef ANSI
PUBLIC VOID Int64Sub (BOOL *Error, BIT32 *DiffHi, BIT32 *DiffLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Sub (Error, DiffHi, DiffLo, AHi, ALo, BHi, BLo)
    BOOL *Error;
    BIT32 *DiffHi, *DiffLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 Borrow;

    LONGDIFF(&Borrow, DiffLo, ALo, BLo, ZERO32);
    LONGDIFF(&Borrow, DiffHi, AHi, BHi, Borrow);

    *Error = ((((AHi ^ BHi) & SignBit32) != ZERO32) &&
              (((AHi ^ *DiffHi) & SignBit32) != ZERO32));
}
/*}}}*/

/*{{{  Int64Minus */
#ifdef ANSI
PUBLIC VOID Int64Minus (BIT32 *DiffHi, BIT32 *DiffLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Minus (DiffHi, DiffLo, AHi, ALo, BHi, BLo)
    BIT32 *DiffHi, *DiffLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 Borrow;

    LONGDIFF(&Borrow, DiffLo, ALo, BLo, ZERO32);
    LONGDIFF(&Borrow, DiffHi, AHi, BHi, Borrow);
}
/*}}}*/

/*{{{  Int64Mul */
#ifdef ANSI
PUBLIC VOID Int64Mul (BOOL *Error, BIT32 *ProdHi, BIT32 *ProdLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Mul (Error, ProdHi, ProdLo, AHi, ALo, BHi, BLo)
    BOOL *Error;
    BIT32 *ProdHi, *ProdLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 W[4], Carry1, Carry2, Carry3, Dummy, Asign, Bsign, Borrow;

    Asign = AHi & SignBit32;
    Bsign = BHi & SignBit32;

    LONGPROD(&Carry1, &W[0], ALo, BLo, ZERO32);
    LONGPROD(&Carry1, &W[1], AHi, BLo, Carry1);
    LONGPROD(&Carry2, &W[1], ALo, BHi, W[1]);
    LONGPROD(&Carry3, &Carry2, AHi, BHi, Carry2);

    LONGSUM(&Carry1, &W[2], Carry2, Carry1, ZERO32);
    LONGSUM(&Dummy, &W[3], Carry3, Carry1, ZERO32);

    if (Asign != ZERO32)
    {
        /*{{{  if A < 0 subtract B from carry part */
        LONGDIFF(&Borrow, &W[2], W[2], BLo, ZERO32);
        LONGDIFF(&Borrow, &W[3], W[3], BHi, Borrow);
        /*}}}*/
    }

    if (Bsign != ZERO32)
    {
        /*{{{  if B < 0 subtract A from carry part */
        LONGDIFF(&Borrow, &W[2], W[2], ALo, ZERO32);
        LONGDIFF(&Borrow, &W[3], W[3], AHi, Borrow);
        /*}}}*/
    }

    /*{{{  check for overflow */
    *Error = FALSE;
    
    if (((INT32) W[1]) < ZERO32)
    {
        /*{{{   */
        if ((W[2] & W[3]) != -ONE32)
            *Error = TRUE;
        /*}}}*/
    }
    else
    {
        /*{{{   */
        if ((W[2] | W[3]) != ZERO32)
            *Error = TRUE;
        /*}}}*/
    }
    /*}}}*/

    *ProdLo = W[0];
    *ProdHi = W[1];
}
/*}}}*/

/*{{{  Int64Times */
#ifdef ANSI
PUBLIC VOID Int64Times (BIT32 *ProdHi, BIT32 *ProdLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Times (ProdHi, ProdLo, AHi, ALo, BHi, BLo)
    BIT32 *ProdHi, *ProdLo, AHi, ALo, BHi, BLo;
#endif
{
    BIT32 Carry;

    LONGPROD(&Carry, ProdLo, ALo, BLo, ZERO32);
    LONGPROD(&Carry, ProdHi, AHi, BLo, Carry);
    LONGPROD(&Carry, ProdHi, ALo, BHi, *ProdHi);
}
/*}}}*/

/*{{{  Int64Div */
#ifdef ANSI
PUBLIC VOID Int64Div (BOOL *Error, BIT32 *QuotHi, BIT32 *QuotLo, BIT32 DvdHi, BIT32 DvdLo, BIT32 DvsrHi, BIT32 DvsrLo)
#else
PUBLIC VOID Int64Div (Error, QuotHi, QuotLo, DvdHi, DvdLo, DvsrHi, DvsrLo)
    BOOL *Error;
    BIT32 *QuotHi, *QuotLo, DvdHi, DvdLo, DvsrHi, DvsrLo;
#endif
{
    BIT32 DummyHi, DummyLo;

    Int64DivRem(Error, QuotHi, QuotLo, &DummyHi, &DummyLo,
                DvdHi, DvdLo, DvsrHi, DvsrLo);
}
/*}}}*/

/*{{{  Int64Rem */
#ifdef ANSI
PUBLIC VOID Int64Rem (BOOL *Error, BIT32 *RemHi, BIT32 *RemLo, BIT32 DvdHi, BIT32 DvdLo, BIT32 DvsrHi, BIT32 DvsrLo)
#else
PUBLIC VOID Int64Rem (Error, RemHi, RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo)
    BOOL *Error;
    BIT32 *RemHi, *RemLo, DvdHi, DvdLo, DvsrHi, DvsrLo;
#endif
{
    BIT32 DummyHi, DummyLo;

    Int64DivRem (Error, &DummyHi, &DummyLo, RemHi, RemLo, 
                 DvdHi, DvdLo, DvsrHi, DvsrLo);
}
/*}}}*/

/*{{{  Int64Eq */
#ifdef ANSI
PUBLIC VOID Int64Eq (BOOL *Equal, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Eq (Equal, AHi, ALo, BHi, BLo)
    BOOL *Equal;
    BIT32 AHi, ALo, BHi, BLo;
#endif
{
    *Equal = ((ALo == BLo) && (AHi == BHi));
}
/*}}}*/

/*{{{  Int64Gt */
#ifdef ANSI
PUBLIC VOID Int64Gt (BOOL *Greater, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Gt (Greater, AHi, ALo, BHi, BLo)
    BOOL *Greater;
    BIT32 AHi, ALo, BHi, BLo;
#endif
{
    *Greater = (AHi == BHi) ? (ALo > BLo) : ((INT32) AHi > (INT32) BHi);
}
/*}}}*/

/*{{{  Int64BitAnd */
#ifdef ANSI
PUBLIC VOID Int64BitAnd (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64BitAnd (ResHi, ResLo, AHi, ALo, BHi, BLo)
    BIT32 *ResHi, *ResLo, AHi, ALo, BHi, BLo;
#endif
{
    *ResLo = ALo & BLo;
    *ResHi = AHi & BHi;
}
/*}}}*/

/*{{{  Int64BitOr */
#ifdef ANSI
PUBLIC VOID Int64BitOr (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64BitOr (ResHi, ResLo, AHi, ALo, BHi, BLo)
    BIT32 *ResHi, *ResLo, AHi, ALo, BHi, BLo;
#endif
{
    *ResLo = ALo | BLo;
    *ResHi = AHi | BHi;
}
/*}}}*/

/*{{{  Int64Xor */
#ifdef ANSI
PUBLIC VOID Int64Xor (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo, BIT32 BHi, BIT32 BLo)
#else
PUBLIC VOID Int64Xor (ResHi, ResLo, AHi, ALo, BHi, BLo)
    BIT32 *ResHi, *ResLo, AHi, ALo, BHi, BLo;
#endif
{
    *ResLo = ALo ^ BLo;
    *ResHi = AHi ^ BHi;
}
/*}}}*/

/*{{{  Int64BitNot */
#ifdef ANSI
PUBLIC VOID Int64BitNot (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo)
#else
PUBLIC VOID Int64BitNot (ResHi, ResLo, AHi, ALo)
    BIT32 *ResHi, *ResLo, AHi, ALo;
#endif
{
    *ResLo = ~ALo;
    *ResHi = ~AHi;
}
/*}}}*/

/*{{{  Int64LShift */
#ifdef ANSI
PUBLIC VOID Int64LShift (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo, BIT32 B)
#else
PUBLIC VOID Int64LShift (ResHi, ResLo, AHi, ALo, B)
    BIT32 *ResHi, *ResLo, AHi, ALo, B;
#endif
{
    SHIFTLEFT(ResHi, ResLo, AHi, ALo, B);
}
/*}}}*/

/*{{{  Int64RShift */
#ifdef ANSI
PUBLIC VOID Int64RShift (BIT32 *ResHi, BIT32 *ResLo, BIT32 AHi, BIT32 ALo, BIT32 B)
#else
PUBLIC VOID Int64RShift (ResHi, ResLo, AHi, ALo, B)
    BIT32 *ResHi, *ResLo, AHi, ALo, B;
#endif
{
    SHIFTRIGHT(ResHi, ResLo, AHi, ALo, B);
}
/*}}}*/
/*}}}*/
