/* arg.c -- Implement the user command line argument parsing */
/* Copyright Inmos 1989, 1990 */
/* Version 1.0 -- 19900129 */
/* History */
/* ahp: First version written */

#include "arg.h"
#include "host_os.h"    /* define HOST by macro definition on command line */
#include <ctype.h>
#include <stddef.h>

#if (HOST != SUN_OS)
#include <stdlib.h>
#endif

#include <string.h>

#if (HOST == SERVER)
#include <host.h>
#endif

#define NUL '\0'

#ifdef SUN4
#ifdef tolower
#undef tolower
#endif
static int tolower (int c)
{
  if ((c >= 'A') && (c <= 'Z'))
    return (c + ((int)'a' - (int)'A'));
  return (c);
}
#endif

#ifdef SUN3
#ifdef tolower
#undef tolower
#endif
static int tolower (int c)
{
  if ((c >= 'A') && (c <= 'Z'))
    return (c + ((int)'a' - (int)'A'));
  return (c);
}
#endif

typedef enum {false = 0, true = 1} bool;

typedef enum {ident = 0, prefix = 1, noteq = 2} str_compare;

static arg_control do_argument (int const, char const * [],
          arg_descriptor const [], int *);
static arg_control do_token (char const *, arg_descriptor const []);
static arg_control do_help (arg_descriptor const []);
static arg_control do_end (arg_descriptor const []);
static arg_control do_error (char const * const, arg_descriptor const []);

static str_compare compare (char const * const, char const * const);
static char * new_string (char const * const);

arg_parse_result arg_parse
  (
    int const argc,           /* number of tokens on command line     */
    char const * argv[],      /* token strings                        */
    arg_descriptor const argd[] /* description of expected arguments  */
  )
{
  char const * token;
  char prefix_character;
  int tk_index;
  
#if ((HOST == MS_DOS) || (HOST == VMS))
  prefix_character = '/';
#elif ((HOST == SUN_OS) || (HOST == HELIOS))
  prefix_character = '-';
#elif ((HOST == SERVER))
  {
    int host, os, board;
    
    host_info (&host, &os, &board);
    switch (os)
    {
      case _IMS_OS_DOS:
      case _IMS_OS_VMS:
      {
        prefix_character = '/';
        break;
      }
      
      case _IMS_OS_HELIOS:
      case _IMS_OS_SUNOS:
      {
        prefix_character = '-';
        break;
      }
    }
  }
#else
#include <assert.h>
  assert(0);
#endif
  
  if (argc == 0)                    /* if no tokens are in the command  */
  {                                 /* then help is requested.          */
    do_help (argd);
    return (arg_parse_help);        /* tell user that it was help       */
  }
  else                              /* there are some tokens to look at */
  {
    tk_index = 0;
    while (tk_index < argc)         /* examine each token in turn       */
    {
      token = argv[tk_index];
      if (token[0] == prefix_character) /* look for argument in user list*/
        if (do_argument (argc, argv, argd, &tk_index) ==
                         arg_terminate)
          return (arg_parse_error);
        else
          ;
      else                          /* no prefix switch character       */
        if (do_token (token, argd) == arg_terminate)
          return (arg_parse_error);
      tk_index += 1;                /* now look at next token on line   */
    }
    switch (do_end (argd))          /* normal exit                      */
    {
      case arg_continue:
        return (arg_parse_ok);
      case arg_terminate:
        return (arg_parse_error);
    }
  }
}

/* This function checks out an argument that start with the prefix      */
/* character. It searches the descriptor table to find the matching     */
/* string and calls the processing routine to perform the job.          */
 
static arg_control do_argument (int const argc, char const * argv[],
          arg_descriptor const argd[], int *tk_index)
{
  arg_descriptor const * desc;      /* desc. currently examined         */
  char const * token;
  
  desc = argd;
  token = argv[*tk_index];          /* token in command line            */
  while (true)
  {
    switch (desc->arg_type)
    {
      case arg_single:              /* stand alone -- see if same       */
      {
        switch (compare (desc->arg_string, token+sizeof(char)))
        {
          case ident:
            return ((*(desc->arg_fn))(NULL)); /* process and return     */
          
          case prefix:
            return (do_error (token, argd)); /* get user to do error    */
          
          case noteq:
            break;                  /* no action to take                */
        }
        break;
      }

      case arg_operand:             /* argument needs operand           */
      {
        switch (compare (desc->arg_string, token+sizeof(char)))
        {
          case ident:               /* operand in next token (if one)   */
          {
            *tk_index += 1;
            if (*tk_index >= argc)
              return (do_error (token, argd)); /* last on command line  */
            else
            {
              token = argv[*tk_index]; /* user processing and return    */
              return ((*(desc->arg_fn))(new_string(token)));
            }
          }
          
          case prefix:              /* followed in same token           */
            return ((*(desc->arg_fn)) /* user processes and return      */
                (new_string(token + sizeof(char) + strlen(desc->arg_string))));
          
          case noteq:
            break;                  /* nothing doing this time          */
        }
        break;
      }

      case arg_end:                 /* no match found in user's list    */
        return (do_error (token, argd));

      case arg_token:
      case arg_help:
      case arg_error:
        break;                      /* not relevant this time round     */
    }
    desc += 1;
  }
  return (arg_continue);
}

/* This function processes a token which does not start with a prefix */
/* character. It searches the descriptor table for the relevant       */
/* function and invokes it.                                           */

static arg_control do_token (char const * token, arg_descriptor const argd[])
{
  arg_descriptor const * desc;
  
  desc = argd;                      /* descriptor currently examined    */
  while (true)
  {
    switch (desc->arg_type)
    {
      case arg_single:
      case arg_operand:
      case arg_help:
      case arg_error:
        break;                      /* nothing relevant this time       */
      
      case arg_token:
        return ((*(desc->arg_fn))(new_string(token))); /* process by user*/

      case arg_end:
        return (do_error (token, argd)); /* could not find argument     */

    }
    desc += 1;
  }
}

/* This function invokes the user's help function and returns what    */
/* the user says should be returned.                                  */

static arg_control do_help (arg_descriptor const argd[])
{
  arg_descriptor const * desc;
  
  desc = argd;                      /* current descriptor               */
  while (true)
  {
    switch (desc->arg_type)
    {
      case arg_single:
      case arg_operand:
      case arg_token:
      case arg_error:
        break;                      /* not relevant                     */
      
      case arg_help:
        return ((*(desc->arg_fn))(NULL)); /* process and return         */

      case arg_end:
        return (arg_continue);      /* none found - ignore it           */

    }
    desc += 1;
  }
}

/* This function invokes the user's end function, if it exists.       */
/* It searches the descriptor table to find the last entry and        */
/* invokes it if it not null.                                         */

static arg_control do_end (arg_descriptor const argd[])
{
  arg_descriptor const * desc;
  
  desc = argd;                      /* current descriptor               */
  while (true)
  {
    switch (desc->arg_type)
    {
      case arg_single:
      case arg_operand:
      case arg_token:
      case arg_error:
      case arg_help:
        break;                      /* not relevant                     */
      
      case arg_end:
        if (desc->arg_fn != NULL)   /* process final call (if one)      */
          return ((*(desc->arg_fn))(NULL));
        else
          return (arg_continue);

    }
    desc += 1;
  }
}

/* This function invokes the user's error function, if it exists.     */
/* It searches the descriptor table to find the error entry; if it    */
/* does not exist or is NULL, then arg_terminate is returned instead. */

static arg_control do_error (char const * const token,
                            arg_descriptor const argd[])
{
  arg_descriptor const * desc;
  
  desc = argd;                      /* current descriptor               */
  while (true)
  {
    switch (desc->arg_type)
    {
      case arg_single:
      case arg_operand:
      case arg_token:
      case arg_help:
        break;                      /* not relevant                     */
      
      case arg_error:
        if (desc->arg_fn != NULL)   /* user processing                  */
          return ((*(desc->arg_fn))(new_string(token)));
        else
          return (arg_terminate);   /* none given - take default action */

      case arg_end:
        return (arg_terminate);     /* none specified -- default action */
      
    }
    desc += 1;
  }
}

/* This function compares trwo strings. If they are identical, then   */
/* the result is 'ident'. If the first is a prefix substring of the   */
/* second, then the result is 'prefix'. Otherwise, the result is      */
/* 'noteq'.                                                           */

static str_compare compare (char const * const sub, char const * const str)
{
  char const * sub_ch, * str_ch;
  
  sub_ch = sub;                     /* expected substring               */
  str_ch = str;                     /* check start of this string       */
  while (true)
  {
    if (*sub_ch == NUL)             /* end of substring                 */
      if (*str_ch == NUL)           /* also end of the string           */
        return (ident);             /* they are identical               */
      else
        return (prefix);            /* substring is prefix of string    */
    else
      ;
    
    if (tolower((int)*sub_ch) != tolower((int)*str_ch))
      return (noteq);               /* difference found                 */
    
    sub_ch += 1;                    /* keep searching                   */
    str_ch += 1;
  }
}

/* This function creates a new string to be passed to the user for      */
/* processing. This ensures that the original command line tokens are   */
/* not changed by the parser -- they may of course be clobbered by      */
/* the user directly.                                                   */

static char * new_string (char const * const old_str)
{
  char * new_str;
  
  new_str = (char *) malloc (sizeof(char) * (strlen(old_str)+1));
  strcpy (new_str, old_str);
  return (new_str);
}
